#include "protos.h"

/*
 * This software is Copyright (C) 1988 by Steven Dorner and the
 * University of Illinois Board of Trustees.  No warranties of any
 * kind are expressed or implied.  No support will be provided.
 * This software may not be redistributed for commercial purposes.
 * You may direct questions to nameserv@uiuc.edu
 */

#include <signal.h>

static int ChangeEntries __P((long *, ARG *));

extern int end;
extern int InputType;

/*
 * Do a change command
 */
void 
DoChange(arg)
	ARG *arg;
{
	long	*entries;
	int	changedCount;
	int	foundCount;

	if (ReadOnly)
	{
		DoReply(LR_READONLY, "changes not allowed to read-only database.");
		return;
	}
	if (!AmHero && !AmOpr && !User)
	{
		DoReply(LR_NOTLOG, "You must be logged in to use this command.");
		return;
	}
	if (!ValidQuery(arg, C_CHANGE))
	{
		DoReply(LR_SYNTAX, "Did not understand change command.");
		return;
	} else if (!GonnaWrite())
	{
		/* GonnaWrite will issue an error message */ ;
		return;
	}
	if (!(entries = DoLookup(arg)))
	{
		DoReply(LR_NOMATCH, "No matches to specification.");
		goto done;
	}
	while (arg->aType != RETURN)
		arg = arg->aNext;	/* skip query */
	arg = arg->aNext;

	foundCount = length(entries);
	if (OP_VALUE(LIMIT_OP) && foundCount > atoi(OP_VALUE(LIMIT_OP)))
		DoReply(LR_LIMIT, "Too many entries (%d) selected; limit is %s.",
			foundCount, OP_VALUE(LIMIT_OP));
	else if (foundCount == (changedCount = ChangeEntries(entries, arg)))
		DoReply(LR_OK, "%d entr%s changed.", changedCount,
			changedCount > 1 ? "ies" : "y");
	else if (changedCount)
		DoReply(LR_ERROR, "Only %d entr%s changed (%d found)", changedCount,
			changedCount > 1 ? "ies" : "y", foundCount);
	else
		DoReply(LR_ERROR, "%d entr%s found, none changed.",
			foundCount, foundCount > 1 ? "ies" : "y");

      done:
	Unlock();
	free((char *) entries);
}

/*
 * Change selected fields in entries
 */
static int 
ChangeEntries(entries, InArgs)
	long *entries;
	ARG *InArgs;
{
	ARG	*arg;
	QDIR	dir;
	QDIR	oldDir;
	int	successes;
	int	entryDirty;
	int	indexDirty;
#ifndef KERBEROS
	int	pwDirty;
#endif /* !KERBEROS */
	int	reason;
	int	oldMask;

	/* loop through all requested entries */
	for (successes = 0; *entries; entries++)
	{
		/* retrieve the entry */
		if (!next_ent(*entries))
		{
			DoReply(-LR_TEMP, "%d:couldn't fetch.", *entries);
			IssueMessage(LOG_WARNING, "%d:couldn't fetch.", *entries);
			continue;
		}
		getdata(&dir);
		getdata(&oldDir);
		if (!UserCanChange(dir))
		{
			IssueMessage(LOG_INFO, "%s not authorized to change entry for %s.",
				     UserAlias, FINDVALUE(dir, F_ALIAS));
			DoReply(-LR_AENTRY, "%s:You may not change this entry.\n",
				FINDVALUE(dir, F_ALIAS));
		} else
		{
#ifdef KERBEROS
			oldMask = entryDirty = indexDirty = 0;
#else /* !KERBEROS */
			oldMask = entryDirty = indexDirty = pwDirty = 0;
#endif /* KERBEROS */
			for (arg = InArgs; arg; arg = arg->aNext)
			{
				if (CanChange(dir, arg->aFD))
				{
					if (OP_VALUE(ADDONLY_OP) && *FINDVALUE(dir, arg->aFD->fdId))
					{
						DoReply(-LR_ADDONLY, "Field has a value.");
					} else
					{
						if (arg->aFD->fdId == F_ALIAS)
						{
							if (reason = BadAlias(arg->aSecond))
							{
								switch (reason)
								{
								    case 1:
									DoReply(-LR_VALUE, "Alias is too long or too short.");
									break;
								    case 2:
									DoReply(-LR_VALUE, "Only alphanumerics and ``-'' are allowed in aliases.");
									break;
								    default:
									DoReply(-LR_VALUE, "Alias is bad.");
									break;
								}
								continue;
							} else if (AliasIsUsed(arg->aSecond, *entries))
							{
								DoReply(-LR_ALIAS, "Alias %s conflicts with other users.",
									arg->aSecond);
								continue;
							}
						}
#ifndef KERBEROS
						else if (arg->aFD->fdId == F_PASSWORD)
						{
							register char *k;

							for (k = arg->aSecond; *k; k++)
								if (*k < ' ' || *k > '~')
								{
									DoReply(-LR_VALUE, "Passwords must use only printable characters; sorry.");
									DoReply(-LR_VALUE, "If your password met this rule, reissue your login command, and try again.");
									break;
								}
							if (*k)
								continue;
# ifdef PRE_ENCRYPT
							{
								char	pwCrypt[14];

								strncpy(pwCrypt, crypt(arg->aSecond, arg->aSecond), 13);
								pwCrypt[13] = 0;
								free(arg->aSecond);
								arg->aSecond = make_str(pwCrypt);
							}
# endif /* PRE_ENCRYPT */
						}
#endif /* !KERBEROS */
						else if (arg->aFD->fdId == F_EMAIL)
						{
							char	scratch[256];

							strcpy(scratch, "@");
							strcat(scratch, MAILDOMAIN);
							if (issub(arg->aSecond, scratch))
							{
								DoReply(-LR_VALUE, "Your email field must not contain addresses ending in @%s;", MAILDOMAIN);
								DoReply(-LR_VALUE, "Use a specific login and machine instead.");
								break;
							}
						}
#ifdef DOSOUND
						else if (arg->aFD->fdId == F_NAME)
						{	/* change sound, too */
							if (!ChangeDir(&dir, FindFDI(F_SOUND), phonemify(arg->aSecond)))
								IssueMessage(LOG_WARNING, "Couldn't change sound.");
						}
#endif
						if (!ChangeDir(&dir, arg->aFD, arg->aSecond))
						{
							IssueMessage(LOG_WARNING, "Couldn't change dir entry.");
							DoReply(-LR_TEMP, "Having difficulties.  Change has failed.");
						}
#ifndef KERBEROS
						pwDirty = pwDirty || arg->aFD->fdId == F_PASSWORD;
#endif /* !KERBEROS */
						entryDirty = 1;
						indexDirty = indexDirty || arg->aFD->fdIndexed;
					}
				} else
					DoReply(-LR_ACHANGE, "%s:you may not change this field.",
						arg->aFD->fdName);
			}
			if (entryDirty)
			{
				if (!putdata(dir))
					DoReply(-LR_TEMP, "%s:Couldn't store.",
						FINDVALUE(dir, F_ALIAS));
				else
				{
					successes++;
#ifdef SIGXCPU
					oldMask = sigblock(sigmask(SIGXCPU));
#endif
					set_date(1);
					store_ent();
					if (indexDirty)
					{
						MakeLookup(oldDir, *entries, unmake_lookup);
						MakeLookup(dir, *entries, make_lookup);
					}
#ifdef SIGXCPU
					sigsetmask(oldMask);
#endif
					if (OP_VALUE(VERBOSE_OP))
						DoReply(LR_PROGRESS, "%s:changed.", FINDVALUE(dir, F_ALIAS));
					if (*entries == UserEnt)
					{	/* replace User dir with new dir */
						FreeDir(&User);
						FreeDir(&oldDir);
						User = dir;
						dir = NULL;
#ifndef KERBEROS
						if (pwDirty)
							crypt_start(PasswordOf(User));
#endif /* !KERBEROS */
						UserAlias = FINDVALUE(User, F_ALIAS);
						continue;	/* avoid freeing dir, which is now User */
					}
				}
			}
		}
		FreeDir(&dir);
		FreeDir(&oldDir);
	}
	return (successes);
}

/*
 * change a dir entry
 */
int 
ChangeDir(dir, fd, value)
	QDIR *dir;
	FDESC *fd;
	char *value;
{
	char	**ptr;
	int	indx;
	int	count;
	char	scratch[MAX_LEN];

	if ((indx = FindField(*dir, fd->fdId)) >= 0)
	{
		ptr = (*dir) + indx;
		free(*ptr);
	} else if (!*value || strcmp(value, "none"))
	{
		count = length((long *) *dir) + 2;
		*dir = (QDIR) realloc((char *) *dir, (unsigned) (count * sizeof (char *)));

		ptr = (*dir) + count - 1;
		*ptr = NULL; /* dir terminator */
		ptr--;	/* back up to right spot */
	} else
	{
		/* we are deleting, && no value presently exists.  do nothing */
		return (1);
	}

	/* ptr now points to the proper char pointer */
	if (strcmp(value, "none"))
	{
		sprintf(scratch, "%d:%s", fd->fdId, value);
		/* enforce max length */
		scratch[fd->fdMax + index(scratch, ':') - scratch + 1] = 0;
		*ptr = make_str(scratch);
	} else
	{
		/* remove pointer at ptr */
		do
		{
			*ptr = *(ptr + 1);
		}
		while (*++ptr);
	}
	return (1);
}

/*
 * Check to see if alias is in use as an alias or name by any record
 * other than the one belonging to the current user.  (New JLN version).
 */
int 
AliasIsUsed(alias, requestor)
	char *alias;
	long requestor;
{
	QDIR	user;
	ARG	*argList;
	ARG	*arg;
	long	*entry;
	int	result;
	char	*p;

	if (user = GetAliasDir(alias))
	{
		FreeDir(&user);
		result = CurrentIndex() == requestor ? 0 : 1;
	} else
	{
		arg = argList = FreshArg();
		arg->aType = COMMAND;
		arg->aFirst = make_str("query");
		arg->aNext = FreshArg();
		arg = arg->aNext;
		arg->aType = VALUE | EQUAL | VALUE2;
		arg->aFirst = make_str("name");
		arg->aSecond = make_str(alias);
		while (1)
		{
			(void) ValidQuery(argList, C_QUERY);
			if ((entry = DoLookup(argList)) != NULL)
			{
				if (length(entry) == 1)
				{
					next_ent(*entry);
					result = CurrentIndex() == requestor ? 0 : 1;
				} else
				{
					result = 1;
				}
				free((char *) entry);
				break;
			}
			if (p = index(arg->aSecond, '-'))
			{
				*p = 0;
				if (p[1])
				{
					arg->aNext = FreshArg();
					arg = arg->aNext;
					arg->aType = VALUE | EQUAL | VALUE2;
					arg->aFirst = make_str("name");
					arg->aSecond = make_str(p + 1);
				}
				continue;
			}
			result = 0;
			break;
		}
		FreeArgs(argList);
	}
	if (requestor && CurrentIndex() != requestor)
	{
		if (!result)
			IssueMessage(LOG_INFO, "Caught one; %x was going to be stored in %x.",
				     requestor, CurrentIndex());
		if (!next_ent(requestor))
		{
			DoReply(-LR_ERROR, "Fatal database error.\n");
			cleanup(-1);
		}
	}
	return result;
}

/*
 * Check to see if an alias is reasonable
 */
int 
BadAlias(alias)
	char *alias;
{
	char	*cp;
	int	len;

	len = strlen(alias);
	if (len < MIN_ALIAS && !(len == 0 && AmHero) || len > FindFDI(F_ALIAS)->fdMax)
		return (1);

	for (cp = alias; *cp; cp++)
		if (!isalnum(*cp) && *cp != '-')
			return (2);

	return (0);
}
