/******************************************************************************
 Phq
	
	This is a simple (ie. no error handling) Gopher server that makes Ph look 
	like any other Gopher search type. The advantage of this is that the client
	need not support Ph queries and the CWIS Administrator can mung the Ph
	data as needed. In addition, the code supports the folder type; this can 
	be used to return a canned query, for example, a department's staff.

	Phq should be run from inetd.
	
	Your Gopher directory should have .Links file entries to support this 
	server. The following shows two Brown uses:
	
	SEARCH
	
		Name=Search the Brown University Online Directory
		Type=7
		Path=7
		Host=gopher.brown.edu
		Port=1070
		
	FOLDER
	
		Name=News Bureau Staff
		Type=1
		Path=1department="News Bureau"
		Host=gopher.brown.edu
		Port=1070

	Since no two Ph implimentions are alike, you will need to modify the
	code to make it usable at your site. The changes are:

	1.  Change the "gthishost" and "gthisport" to the domain name and port
		that this server is running on. There should be code written to find out
		the name of the host, but the port would still have to be hand coded.

	2.	Change the "gphhost" and "gphport" to the domain name and port of your 
		ph server.

	3.	Change "gselectorfield" to the name of a ph field that uniquely 
		identifies a ph record. This field is used as the selector in the
		response to the query. At Brown University, a user's "alias" is unique.
	
	4.	Printlist() assumes you want the response to include the user's name
		and department in the title. At Brown, rather than "name" we are using
		"alias". If you want to list "name" at your site than change all 
		occurences of "alias" to "name". Double check that printlist() is doing 
		what you want.

	NOTE: As a last minute change I have added command line support for 
	overriding gthishost, gthisport, gphhost, gphport, and gselectorfield.

	Good luck.

	-- 
		
	Copyright (C) 1992 by Brown University. All rights reserved.

	Permission is granted to any individual or institution to use, copy,
	or redistribute the binary version of this software and its
	documentation provided this notice and the copyright notices are
	retained.  Permission is granted to any individual or non-profit
	institution to use, copy, modify, or redistribute the source files
	of this software provided this notice and the copyright notices are
	retained.  This software may not be distributed for profit, either
	in original form or in derivative works, nor can the source be
	distributed to other than an individual or a non-profit institution.
	Any  individual or group interested in seeing and/or using these
	source files but who are prevented from doing so by the above
	constraints should contact Don Wolfe, Vice-President for Computer 
	Systems at Brown University, (401) 863-7247, for possible
	software licensing of the source developed at Brown.

	Brown University and Andrew James Gilmartin make no representations
	about the suitability of this software for any purpose.
 
 	BROWN UNIVERSITY AND ANDREW JAMES GILMARTIN GIVE NO WARRANTY, EITHER
	EXPRESS OR IMPLIED, FOR THE PROGRAM AND/OR DOCUMENTATION PROVIDED,
	INCLUDING, WITHOUT LIMITATION, WARRANTY OF MERCHANTABILITY AND
	WARRANTY OF FITNESS FOR A PARTICULAR PURPOSE.

    AUTHOR: Andrew_Gilmartin@Brown.Edu
	REVISION: 2
	
******************************************************************************/

static char rcsid[] = "$Id: phq.c,v 1.8 1992/10/27 22:35:46 gopher Exp gopher $";

#include <stdlib.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <netdb.h>
#include <netinet/in.h>
#include <stdio.h>
#include <string.h>

	/* I don't know why this is not in stddef.h... */
	
typedef int BOOLEAN;
#define TRUE  (0==0)
#define FALSE (0!=0)


	/* Host and port this server is running on */

char* gthishost = "gopher.brown.edu"; 
int gthisport = 1070;


	/* Host and port of your Ph server */
	
char* gphhost = "ns.brown.edu";
int gphport = 105;


	/* Field that uniquely identifies a Ph record */
	
char* gselectorfield = "alias";


	/* HACK: Used by printsummary() */

BOOLEAN gRequestFolder = TRUE;


	/* Set c-string to zero length */

#define strempty( _string ) ( (_string)[0] = '\0' )



/******************************************************************************
 trim

	Remove the white space from the end of the given string.
******************************************************************************/

char* trim( char* line )
{
	char* cursor = &line[ strlen( line ) -1 ];

		/* Remove trailing white space */

	while ( cursor != line && isspace( *cursor ) )
		cursor--;

	*(cursor +1) = '\0';

	return line;

} /* trim */



/******************************************************************************
 fopensocket

	Return a FILE handle to a socket opened on the given host and port. When
	using the returned handled remember to fflush() every now and then; Is
	there a way to do this automatically?
******************************************************************************/

FILE* fopensocket( char* hostname, int port )
{
	struct hostent* host;
	struct sockaddr_in server;
	FILE* file;
	int sock;
	
		/* Resolve host's name */
		
	host = gethostbyname( hostname );
	if ( host == NULL )
		return NULL;
	
		/* Configure server information */
		
	bzero( &server, sizeof( server ) );
	
	memcpy( &server.sin_addr, host->h_addr, host->h_length );
	server.sin_port = htons( port );
	server.sin_family = host->h_addrtype;
	
		/* Create the socket */
		
	sock = socket( AF_INET, SOCK_STREAM, IPPROTO_TCP );
	if ( sock == EOF )
		return NULL;
	
		/* Connect to the server */
		
	if ( connect( sock, &server, sizeof( server ) ) == EOF )
		return NULL;
	
		/* Open the file number as an ANSI file descriptor */
		
	file = fdopen( sock, "r+" );
	if ( file == NULL )
		return NULL;
		
		/* Done */
		
	return file;
	
} /* fopensocket */



/******************************************************************************
 parsephresponse

	Parse a response line from Ph. Most lines have the form
	
		return-code:record-sequence:field-name:field-value
******************************************************************************/

#define SKIP_CHAR( _cursor, _char )\
	if ( *_cursor == _char )\
		_cursor++

#define SKIP_SPACE( _cursor )\
	while( *_cursor == ' ' )\
		_cursor++

#define SKIP_FIELD( _cursor )\
	 while ( *_cursor != ':' && *_cursor != '\0' )\
	 	_cursor++;\
	 SKIP_CHAR( _cursor, ':' )

void parsephresponse
	( char* line
	, int* code 
	, int* index
	, char* field
	, char* value )
{
		/* Get code */
		
	*code = atoi( line );
	
	SKIP_FIELD( line );
	
		/* Get index */
		
	*index = atoi( line );

	SKIP_FIELD( line );
	
		/* Get field */
		
	SKIP_SPACE( line );
	
	while( *line != ':' && *line != '\0' )
		*field++ = *line++;
	*field = '\0';
	
	SKIP_FIELD( line );
	
		/* Get value */

	SKIP_SPACE( line );
	
	while( *line != '\n' && *line != '\0' )
		*value++ = *line++;
	*value = '\0';
		
		/* Done */

} /* parsephresponse */



/******************************************************************************
 printsummary

	Print a gopher item for the given record information. HACK: The global
	variavle gRequestFolder is used to determine with the user's title or
	department should appear in the parenthesis.
******************************************************************************/

void printsummary( char* name, char* title, char* department, char* selector )
{
	char* parentheticaltext = gRequestFolder ? title : department;
	
	if ( *parentheticaltext == '\0' )
		printf( "0%s\t0%s\t%s\t%d\r\n", name, selector, gthishost, gthisport );
	else
		printf( "0%s (%s)\t0%s\t%s\t%d\r\n"
			, name, parentheticaltext, selector, gthishost, gthisport );

} /* printsummary */



/******************************************************************************
 printlist

	Print a list of gopher items representing the response to the given query.
******************************************************************************/

void printlist( FILE* phserver, char* query )
{
	char name[128];
	char title[128];
	char department[128];
	char selector[128];
	char line[128];
	int result;
	int index, previousindex;
	char field[128];
	char value[128];
	
		/* Ask server for a list of matching records */
		
		/*
			NOTE: If you change "alias" and "department" here, you will 
			also need to change them further down in this function. If you 
			do choose another field to use for name and/or department, I 
			would suggest that you DON'T change the names of the variables.
		*/
	
	fprintf
		( phserver
#ifdef BROWN_U
		, "query %s return alias title department %s\r\n"
#else
		, "query %s return name title department %s\r\n"
#endif
		, query
		, gselectorfield );
	fflush( phserver );

		/* Did the server find any matches? */
		
	fgets( line, sizeof( line ), phserver );	
	if ( atoi( line ) == 102 )
	{
			/* Send the client the list of matching records */
			
		strempty( name );
		strempty( department );
		strempty( selector );
		previousindex = 1;

		for(;;)
		{
				/* Get line of record */
				
			fgets( line, sizeof( line ), phserver );

				/* Is it the terminating line? */
				
			if ( atoi( line ) > 0 )
			{
					/* Print previous record */
					
				printsummary( name, title, department, selector );
				
					/* Stop looping */
					
				break;
			}

				/* Parse ph response */

			parsephresponse( line, &result, &index, field, value );

				/* Is this line the start of a new record? */
				
			if ( index != previousindex )
			{
					/* Print previous record */
					
				printsummary( name, title, department, selector );

				strempty( name );
				strempty( department );
				strempty( selector );
				previousindex = index;
			}

				/* Does the line contain useful information? */
				
			if ( result == -200 )
			{
					/* Save the field's value */

#ifdef BROWN_U
				if ( strcmp( field, "alias" ) == 0 )
#else
				if ( strcmp( field, "name" ) == 0 )
#endif
					strcpy( name, value );						
				else if ( strcmp( field, "department" ) == 0 )
					strcpy( department, value );
				else if ( strcmp( field, "title" ) == 0 )
					strcpy( title, value );

				if ( strcmp( field, gselectorfield ) == 0 )
					strcpy( selector, value );
			}
		}
	}
	else
	{
		printf( "3%s\r\n", trim( strchr( line, ':' ) + 1 ) );
	}

		/* Finished with Ph */
		
	fprintf( phserver, "quit\r\n" );
	fflush( phserver );

		/* Finished sending list to client */
		
	printf( ".\r\n" );
	
} /* printlist */



/******************************************************************************
 printrecord

	Print the actual Ph entry for the given user (via selector).
******************************************************************************/

void printrecord( FILE* phserver, char* selector )
{
	char line[128];

		/* Ask server for particular record */
		
	fprintf
		( phserver
		, "query %s=\"%s\" return all\r\n"
		, gselectorfield
		, selector );
	fflush( phserver );
	
		/* Did the server find the record? */
		
	fgets( line, sizeof( line ), phserver );
	if ( atoi( line ) == 102 )
	{
			/* Send the client the lines of the record */
			
		for(;;)
		{
			fgets( line, sizeof( line ), phserver );
			
			if ( atoi( line ) == 200 )
				break;

				/* Messy!! Skips over the first two fields */

			printf
				( "%s\r\n"
				, trim( (char*) strchr( strchr( line, ':' ) + 1, ':' ) + 1 ) );
		}
	}
	else
	{
		printf( "\
Can't find any information about \"%s\". \
Please contact the CWIS Administrator about this problem.\r\n"
			, selector );
	}

		/* Finished with Ph */
		
	fprintf( phserver, "quit\r\n" );		
	fflush( phserver );

		/* Finished sending record to client */
		
	puts( ".\r\n" );
		
} /* printrecord */



/******************************************************************************
 parsecommandline

	Parse command line arguments.
******************************************************************************/

void parsecommandline( int argc, char** argv )
{
	extern char *optarg;
	char c;

	while ( ( c = getopt( argc, argv, "?H:h:P:p:s:" ) ) != EOF )
		switch ( c )
		{
			case 'H':
				gthishost = optarg;
				break;
				
			case 'h':
				gthisport = atoi( optarg );
				break;
				
			case 'P':
				gphhost = optarg;
				break;
				
			case 'p':
				gphport = atoi( optarg );
				break;
				
			case 's':
				gselectorfield = optarg;
				break;
			
			default:
				printf( "Unknown command line option %c\n\n", c );
				
			case '?':
				printf( "\
Usage:\n\
\n\
	phq [-options]\n\
\n\
Options:\n\
\n\
    -H <host>               domain name of host running phq\n\
                            (%s)\n\
    -h <port>               port on which inetd is listening (%d)\n\
\n\
    -P <Ph host>            domain name of host servicing Ph\n\
                            (%s)\n\
    -p <Ph port>            port on which Ph is listening (%d)\n\
\n\
    -s <Ph selector field>  Ph field to use as Gopher selector (%s)\n\
\n\
$Revision: 1.8 $\n"
					, gthishost
					, gthisport
					, gphhost
					, gphport
					, gselectorfield );
				exit( 1 );
		}

} /* parsecommandline */



/******************************************************************************
 main

******************************************************************************/

int main( int argc, char** argv )
{
	FILE* phserver;
	char selector[256];

	parsecommandline( argc, argv );

	gets( selector );
	
	switch( *selector )
	{
		case '7':
			if ( phserver = fopensocket( gphhost, gphport ) )
			{
				gRequestFolder = FALSE;
				
					/* 7\t<query>\n */
				printlist( phserver, trim( &selector[2] ) );
				fclose( phserver );
			}
			break;
	
		case '1':
			if ( phserver = fopensocket( gphhost, gphport ) )
			{
				gRequestFolder = TRUE;

					/* 1<query>\n */
				printlist( phserver, trim( &selector[1] ) );
				fclose( phserver );
			}
			break;
		
		case '0':
			if ( phserver = fopensocket( gphhost, gphport ) )
			{
					/* 0<selector>\n */
				printrecord( phserver, trim( &selector[1] ) );
				fclose( phserver );
			}
			break;
			
		default:
				/* Return some error information */
			break;
	}

} /* main */

