/* Copyright (c) 1992 Vincent Cate
 * All Rights Reserved.
 *
 * Permission to use and modify this software and its documentation
 * is hereby granted, provided that both the copyright notice and this
 * permission notice appear in all copies of the software, derivative works
 * or modified versions, and any portions thereof, and that both notices
 * appear in supporting documentation.  This software or any derivate works
 * may not be sold or distributed without prior written approval from
 * Vincent Cate.
 *
 * THE SOFTWARE IS PROVIDED "AS IS" AND VINCENT CATE DISCLAIMS ALL
 * WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING ALL IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS.  IN NO EVENT SHALL
 * VINCENT CATE BE LIABLE FOR ANY SPECIAL, DIRECT, INDIRECT, OR
 * CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM LOSS
 * OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE
 * OR OTHER TORTIOUS ACTION, ARISING OUT OF OR IN CONNECTION WITH THE USE
 * OR PERFORMANCE OF THIS SOFTWARE.
 *
 * Users of this software agree to return to Vincent Cate any improvements
 * or extensions that they make and grant Vincent Cate the rights to
 * redistribute these changes.
 *
 */



/* UNFSD - copyright Mark A Shand, May 1988.
 *         copyright Donald J. Becker, Harris Corp.  Jan 1989
 * This software maybe be used for any purpose provided
 * the above copyright notice is retained.  It is supplied
 * as is, with no warranty expressed or implied.
 *
 *  Originally written by Mark Shand.
 *  Modified for Alex purposes by Vince Cate.
 */

/*
 *  FILE HANDLE PACKAGE FOR USER-LEVEL NFS SERVER
 *
 *  Interfaces:
 *      fh_init
 *          Initializes the queues
 *
 *      fh_pr
 *          converts file handle into a printable text string
 *
 *      fh_create
 *          establishes initial file handle; called from mount daemon
 *
 *      fh_path
 *          returns unix path corresponding to fh
 *
 *      fh_fd
 *          returns open file descriptor for given file handle;
 *          provides caching of open files
 *
 *      fh_compose
 *          construct new file handle from existing file handle and
 *          directory entry
 *
 *  Internal:
        fh_delete

        fh_lookup
            given an inode looks up fh

 *  
 */

#include "alexincs.h"
#include "alex.h"
#include "nfs_prot.h"
#include "fh.h"


#define FHC_XONLY_PATH  01
#define FHC_BUSY    02  /* NOT USED */

/* #define CACHE_SIZE_BASE     301                     /* small test number              */
#define CACHE_SIZE_BASE     2003                       /* nice if prime                  */
#define LOWAT_CACHE_SIZE    3*CACHE_SIZE_BASE          /* flush down to this many        */
#define HIWAT_CACHE_SIZE    4*CACHE_SIZE_BASE          /* when gets this high flush some */
#define HASH_TAB_SIZE       5*CACHE_SIZE_BASE          /*                                */

/*
 * Paths constructed in this system always consist of real directories
 * (excepting the last element) i.e. they do not contain symbolic links.
 * This is guaranteed by the way NFS constructs the paths.
 * As a consequence we may assume that
 *  /x/y/z/.. == /x/y
 * and  /x/y/z/. == /x/y/z
 * provided that z != . && z != ..
 * These relations are exploited in fh_compose.
 *
 * Further assumptions:
 *  All cached pathnames consist of a leading /
 *  followed by zero or more / separated names
 *  s.t.
 *      name != .
 *      name != ..
 *      index(name, '/') == 0
 */

typedef struct fhcache
{
    struct  fhcache  *next;
    struct  fhcache  *prev;
    struct  fhcache  *hash_next;
    svc_fh  h;
    int     fd;
    int     omode;
    char    *path;
  /*  int     PartsInHostName;   was thinking of adding this but don't really need it ... */
    time_t  last_used;
}
    fhcache;

static fhcache fh_head, fh_tail, *last_flushable;
static fhcache *fh_hashed[HASH_TAB_SIZE];
static int fh_list_size=0;
static time_t   curtime;
extern int errno;

void fh_move_to_front(fhc)
fhcache *fhc;
{
    fhc->prev->next = fhc->next;                           /* Remove from current posn */
    fhc->next->prev = fhc->prev;

    fhc->prev = &fh_head;                                  /* Insert at head */
    fhc->next = fh_head.next;
    fhc->prev->next = fhc;
    fhc->next->prev = fhc;
}

void fh_inserthead(fhc)
fhcache *fhc;
{
    fhcache **hash_ptrptr;
    int hashindex;
    char tmpstr[MAXPATH];

    fhc->prev = &fh_head;                                   /* Insert at head */
    fhc->next = fh_head.next;
    fhc->prev->next = fhc;
    fhc->next->prev = fhc;
    fh_list_size++;

    sprintf(tmpstr, "fh_inserthead  %lu %lu %lu %lu %lu %lu %lu %lu ",
                 fhc->h.InodeArray[0], fhc->h.InodeArray[1], fhc->h.InodeArray[2],
                 fhc->h.InodeArray[3], fhc->h.InodeArray[4], fhc->h.InodeArray[5],
                 fhc->h.InodeArray[6], fhc->h.InodeArray[7]);
    Log(tmpstr);

    hashindex = fhc->h.InodeArray[0] % HASH_TAB_SIZE;   
    hash_ptrptr = &(fh_hashed[hashindex]);  
    fhc->hash_next = *hash_ptrptr;                          /* Insert into hash tab */
    *hash_ptrptr = fhc;
}


fhcache *AddRoot()
{
    static fhcache fhc;

    ToLog(DBMAJOR, "AddRoot adding %s Inode= %d\n", CACHEDIRVAR, ROOTINODE);

    fhc.path = strsave(CACHEDIRVAR);
    fhc.fd = -1;
    fhc.last_used = curtime;
    fhc.h.InodeArray[0]=ROOTINODE; 

    fh_inserthead(&fhc);

    return(&fhc);
}

/*
 *      fh_init
 *          Initializes the queues 
 */
fh_init()
{
    InitReadOnlyVariables(); 
    Log("fh_init");

    fh_head.next = fh_tail.next = &fh_tail;
    fh_head.prev = fh_tail.prev = &fh_head;
    last_flushable = &fh_tail;

    (void) AddRoot();

    Log("fh_init is done");
}

/*  Input: inode
 *  Returns:  fhc or NULL
 */
fhcache *fh_lookup(inode)
u_long inode;
{
    fhcache *fhc;

    fhc = fh_hashed[inode % HASH_TAB_SIZE];              /* go to right linked list */

    while ((fhc != NULL) && (fhc->h.InodeArray[0] != inode)) {  /* follow till find inode */
        fhc = fhc->hash_next;
    }

    if (fhc != NULL) {
        LogN("fh_lookup returning ", (int) fhc->h.InodeArray[0]);
    } else {
        LogN("fh_lookup could not find ", (int) inode);
        if (inode == ROOTINODE) {
            LogN("fh_lookup adding ROOTINODE ", (int) inode);
            fhc=AddRoot();
        }
    }

    return(fhc);
}

void fh_delete(fhc)
fhcache *fhc;
{
    fhcache **hash_slot;

    Log("fh_delete");

    fhc->prev->next = fhc->next;                    /* Remove from current posn */
    fhc->next->prev = fhc->prev;
    fh_list_size--;

    hash_slot = &(fh_hashed[fhc->h.InodeArray[0] % HASH_TAB_SIZE]);  /* Remove from hash tab */
    while (*hash_slot != NULL && *hash_slot != fhc) {
        hash_slot = &((*hash_slot)->hash_next);
    }

    if (*hash_slot == NULL) {
         ToLog(DBERROR, "internal inconsistency -- fhc not in hash table %d\n", (int) fhc);
         exit(-9);
    } else {
        *hash_slot = fhc->hash_next;
    }

    if (fhc->path != NULL) {                                 /* Free storage */
        ToLog(10, "fh_delete is flushing: %s  Inode= %d \n", fhc->path, fhc->h.InodeArray[0]);
        free(fhc->path);
    }
    free((char *) fhc);

    Log("fh_delete done ");
}



/* flush_cache() is invoked on demand from fh_find. 
 */
int flush_cache(DontFlushInode)
u_long DontFlushInode;
{
    fhcache *fhc;
    int CacheSize, Inode;

    Log("flush_cache");

    time(&curtime);
    fh_fdclose();                                                     /* close open files */

    fhc = fh_head.next;             /* works in empty case because: fh_tail.next = &fh_tail */
    CacheSize=0;
    while (fhc->next != &fh_tail) {
        if (CacheSize < LOWAT_CACHE_SIZE) {
            CacheSize++;                                      /* keep first LOWAT_CACHE_SIZE entries */
            fhc = fhc->next;
        } else {
            Inode = fhc->h.InodeArray[0];                             /* save his inode number       */
            fhc = fhc->next;                                          /* point to next               */
            if ((Inode != DontFlushInode) && (Inode != ROOTINODE)) {  /* if previous was not special */
                fh_delete(fhc->prev);                                 /* then remove him             */
            } else {
                CacheSize++;                                          /* do need to count him        */
            }
        }
    }

    if (fh_list_size != CacheSize) {
        ToLog(DBERROR, "flush_cache ERROR BUG fh_list_size= %d CacheSize= %d \n", 
                              fh_list_size, CacheSize);
    }

    fh_list_size = CacheSize;
}



/* given a svc_fh create the path for that inode
 *
 * returns: 
 *    AOK and *returnpath if works
 *   AWORKING - if work in progress
 *   ANFSSTALE - if must be stale
 *   AFAIL     - other failure
 *
 * The svc_fh has 32 bytes.  We use those to hold 8 inode numbers, the one for
 * the current path, the parent, the parents parent, etc.
 * If any of the ansestors are in the cache we are fine.
 * alex.nanny does a "/bin/ls -l /alex" to get the top 2 levels in the cache.
 * This means that even after a reboot we can deal with file handles as long as:
 * 
 *       /alex/edu/cmu/cs/nectar/furmint/tmp/tmp2/tmp3
 *
 * This is really good enough for most places.
 *
 */
int fh_buildpath(h, returnpath)
svc_fh  *h;
char **returnpath;
{
    int i, Status;
    u_long  RootInode;
    char    *path;
    struct  stat    sbuf;
    char    pathbuf[MAXPATHLEN+MAXNAMLEN+1];
    struct ParsedDir Current;
    struct ActiveAlexInfo AAI;
    int KnownRelative, dist, HaveThisPart, OkSoFar;
    struct fhcache *fhc;

    Log("fh_buildpath");

    for (i=0; i<INODEARRAYLEN; i++) {
        if (h->InodeArray[i] != 0) {
            LogN("fh_buildpath InodeArray ", (int) h->InodeArray[i]);
        }
    }


    Status=AlexInfoCompatStat(CACHEDIRVAR, &sbuf, (struct ParsedDir *) NULL, LastUidStr, NORECURSION);
    if (Status != AOK) { 
        LogT("fh_buildpath ERROR could not AlexInfoCompatStat CACHEDIRVAR ", Status);
        flushLog();
        exit(-1);
    }

    RootInode= (u_long) sbuf.st_ino;

    LogN("fh_buildpath  / inode number is ", (int) RootInode);

    KnownRelative=0;
    for (dist=0 ; !KnownRelative && (dist<INODEARRAYLEN); dist++) { /* go up the tree         */
        fhc=fh_lookup(h->InodeArray[dist]);                         /* looking for a Realtive */
        if (fhc != NULL) {                                          /* who is in the cache    */
            KnownRelative=1;
        }
    }
    dist--;

    if (!KnownRelative) {
        LogN("fh_buildpath could find no relatives for ", (int) h->InodeArray[0]);
        return(ANFSSTALE);
    }

    if (dist==0) {                                      /* his own inode? */
        *returnpath=fhc->path;
        Log("fh_buildpath found in cache - 2");         /* then we are done */
        return(AOK);
    }

    Status=ANFSSTALE;
    OkSoFar=1;
    (void) strcpy(pathbuf, fhc->path);                  /* copy ansestors path            */
    for (i = dist-1; OkSoFar && (i >= 0) ; i--) {       /* work back down directory tree  */
        Status=AlexInfoCompatStat(pathbuf, &sbuf, (struct ParsedDir *) NULL, LastUidStr, RECURSIONOK); 
        if (Status==AOK && ((Status=AlexOpenDir(pathbuf, &AAI, LastUidStr)) == AOK)) {

            LogN("fh_buildpath current pathbuf has an inode of ", (int) sbuf.st_ino);
            LogN("fh_buildpath is looking for an inode of ", (int) h->InodeArray[i]);

            HaveThisPart=0;
            while ((!HaveThisPart) && (AlexInfoInNext(&AAI, &Current) == AOK)) {
                if (h->InodeArray[i] == Current.Inode) {
                    Log2("fh_buildpath got another part of path ", Current.Name);
                    (void) strcat(pathbuf, "/");                 
                    (void) strcat(pathbuf, Current.Name);                 
                    HaveThisPart=1;
                }
            }
    
            (void) AlexInfoInClose(&AAI);

            if (!HaveThisPart) {
                OkSoFar=0;
            }
       } else {
            OkSoFar=0;
       }
    }

    if ( OkSoFar &&                                                      /* if everything is good */
        (AlexInfoCompatStat(pathbuf, &sbuf, (struct ParsedDir *) NULL, LastUidStr, RECURSIONOK) == 0) && 
        (sbuf.st_ino == h->InodeArray[0])) {
            path = strsave(pathbuf);
            *returnpath=path;                                         /* return a pointer to path */
            Log2("fh_buildpath GOT IT ", *returnpath);
            return(AOK);

    }    
    LogN("fh_buildpath giving up - could not find ", (int)  h->InodeArray[0]);
    *returnpath=NULL;
    if (Status == AWORKING) {
        return(Status);
    } else {
        return(ANFSSTALE);
    }
}


/* if create==1 we will make a new one if we can not find it 
 *
 */
int fh_find(h, create, returnfhc)
svc_fh  *h;
int create;
fhcache **returnfhc;
{
    fhcache *fhc;
    int Status;

    LogN("fh_find start   create=", create);

    if (fh_list_size > HIWAT_CACHE_SIZE) {
        ToLog(DBMAJOR, "fh_find calling flush_cache but protecting %d \n", h->InodeArray[0]);
        flush_cache(h->InodeArray[0]);                        /* but not h->inode */
    }

    Log("fh_find mid");

    time(&curtime);
    if ((fhc = fh_lookup(h->InodeArray[0])) != NULL) {
        Log("fh_find got right inode ");

        /* if (bcmp((char *) h->InodeArray, (char *) fhc->h.InodeArray, INODEARRAYBYTES) != 0) { */
        /* no longer check rest of what was hash_path and is now InodeArray since only hint      */

        if (fhc != fh_head.next) {
            fh_move_to_front(fhc);                                /* true LRU */
        }

        fhc->last_used = curtime;
        ToLog(10, "fh_find done AOK  Inode= %d  path=%s \n", h->InodeArray[0], fhc->path);
        *returnfhc = fhc;
        return(AOK);
    }

    if (create) {
        if ((fhc = (fhcache *) malloc((unsigned) sizeof(*fhc))) == NULL) {
            MallocDeath("fh_find 1");
        }
        fhc->path = NULL;
        fhc->last_used = curtime;
        fhc->h = *h;
        fh_inserthead(fhc);
    } else {
        char    *path;                                 

        Status= fh_buildpath(h, &path);                /* attempt to contruct from InodeArray */
        if (Status != AOK) {
            LogT("fh_find done 2 tried fh_buildpath", Status);
            return(Status);
        }

        if ((fhc = (fhcache *) malloc((unsigned) sizeof(*fhc))) == NULL) {
            MallocDeath("fh_find 2");
        }
        fhc->path = path;
        fhc->fd = -1;
        fhc->last_used = curtime;
        fhc->h = *h;
        fh_inserthead(fhc);
    }


    ToLog(10, "fh_find done  AOK3    Inode= %d  path=%s \n", h->InodeArray[0], fhc->path);

    *returnfhc=fhc;
    return(AOK);
}







/*      fh_pr
 *          debugging primitive; converts file handle into a printable
 *          text string
 */
char *fh_pr(fh)
nfs_fh  *fh;
{
    static char *p;
    nfsstat status;
    int Status;

    Log("fh_pr entering ");

    Status = fh_path(fh, &status, &p);

    if (((Status != AOK) || (status != NFS_OK)) || (p == NULL) || (*p == 0)) {
        LogT("fh_pr Status= ", Status);
        LogN("fh_pr status= ", status);
        p= "###STALE###";                         /* might be AWORKING */
    }

    Log2("fh_pr done", p);
    return(p);
}


/*      fh_create
 *          establishes initial file handle; called only from mount daemon
 */
int fh_create(fh, path)
nfs_fh  *fh;
char    *path;                                /* note that (dirpath) == (char*)   */
{
    svc_fh  *key = (svc_fh *) fh;
    fhcache *h;
    u_long inode;
    nfsstat status;
    int Status;

    Log2("fh_creat called with ", path);

    bzero((char *) fh, sizeof fh);
    status = NFS_OK;

    Status = path_inode(CACHEDIRVAR, &status, &inode);
    if (Status != AOK) {
        ToLog(DBERROR, "fh_create  ERROR BUG this should never happen %s\n", ATypeToString(Status));
        return((int) status);
    }

    key->InodeArray[0] = inode;
    Status  = fh_find(key, 1, &h);

    if (Status != AOK) {                        
        ToLog(DBERROR, "fh_creat ERROR BUG this should always work");
        status = NFSERR_STALE;
    } else {
        if (h->path == NULL) {
            h->fd = -1;
            h->path = strsave(CACHEDIRVAR);
        }
    }

    LogN("fh_creat returning ", (int) status);
    return((int) status);
}

/*      fh_path
 *          returns unix path corresponding to fh
 */
extern int fh_path(fh, status, returnpath)
nfs_fh  *fh;
nfsstat *status;
char **returnpath;
{
    fhcache *h;
    int Status;

    Log("fh_path entering ");

    Status=fh_find((svc_fh *) fh, 0, &h);

    if (Status == AOK) {
        *status = NFS_OK;
        *returnpath = h->path;
    } else {
        *returnpath = "BogusPath";
        if (Status != AWORKING) {
            *status = NFSERR_STALE;
             Status = ANFSSTALE;
        }
    }

    LogT("fh_path done ", Status);
    return(Status);
}


  
/*      fh_fd
 *          returns open file descriptor for given file handle;
 *          provides caching of open files
 * 
 *  Currently we cache 1 open file descriptor.
 * in the future we could allocate an array of size
 * getdtablesize() which would contain inode's to provide
 * an mapping from descriptor to inode's.
 * Then we could maintain many concurrently open files.
 */

extern int fh_fd(fh, status, omode, returnfd)
nfs_fh  *fh;
nfsstat *status;
int omode;
int *returnfd;
{
    fhcache     *h;
    static int  fd = -1;
    static unsigned int  inode = 0;
    int Status;

    Log("fh_fd");

    if (fh == NULL) {                    /* special case -- request to flush fd cache */
        if (fd >= 0) {
            (void) close(fd);
        }
        fd = -1;
        *returnfd=fd;
        return(AOK);
    }

    Status = fh_find((svc_fh *) fh, 0, &h);
    if (Status != AOK) {
        *status = NFSERR_STALE;
        if ((Status != ANFSSTALE) &&  (Status != AWORKING)) {
            LogT("fh_fd expected stale or aworking", Status);
        }
        return(Status);
    } 
       

    if (fd >= 0) {                      
        if (inode == h->h.InodeArray[0] && h->omode == omode) {
            *returnfd=fd;
            *status = NFS_OK;
            return(AOK);
        }
        (void) close(fd);
    }

    errno = 0;
    fd = open(h->path, omode);
    if (fd >= 0) {
        Status=AOK;
        *status = NFS_OK;
    } else {
        Status=AFAIL;
        *status = (nfsstat) errno;    
    }
    h->fd = fd;
    h->omode = omode;
    inode = h->h.InodeArray[0];
    *returnfd=fd;

    LogT("fh_fd done ", Status);
    return(Status);
}

int fh_fdclose()
{
    int fd;

    return( fh_fd((nfs_fh *) NULL, (nfsstat *) NULL, 0, &fd) );
}


/*      fh_compose
 *          construct new file handle from existing file handle and
 *          directory entry
 */
extern int fh_compose(dopa, new_fh, returnstatus)
diropargs  *dopa;
nfs_fh  *new_fh;
nfsstat *returnstatus;
{
    svc_fh  *key;
    fhcache *dirfhc, *h;
    char    *sindx;
    int is_dd;
    char    pathbuf[MAXPATHLEN+MAXNAMLEN+1];
    int Status;
    u_long  Inode;

    Log("fh_compose entering");

    Status = fh_find((svc_fh *) &(dopa->dir), 0, &dirfhc );
    if (Status != AOK) {
        LogT("fh_compose returning STALE ", Status);
        *returnstatus = NFSERR_STALE;
        return(Status);
    }

    /* Construct path */

    if (strcmp(dopa->name, ".") == 0) {
        *new_fh = dopa->dir;
        Log("fh_compose returning - was a .");
        *returnstatus = NFS_OK;
        return(AOK);
    }

    if (strcmp(dopa->name, "..") == 0) {
        is_dd = 1;
        sindx = rindex(dirfhc->path, '/');
        if (sindx == dirfhc->path) {
            (void) strcpy(pathbuf, "/");
        } else {
            int len = sindx - dirfhc->path;
            (void) strncpy(pathbuf, dirfhc->path, len);
            pathbuf[len] = '\0';
        }
    } else {
        int len = strlen(dirfhc->path);

        is_dd = 0;
        if (dirfhc->path[len-1] == '/') {
            len--;
        }
        (void) strncpy(pathbuf, dirfhc->path, len);
        pathbuf[len] = '/';
        (void) strcpy(pathbuf + (len+1), dopa->name);
    }
                                                        /* pathbuf has the path     */

    *new_fh = dopa->dir;                                /* copy old InodeArray info */

    key = (svc_fh *) new_fh;
    Status = path_inode(pathbuf, returnstatus, &Inode);
    if (Status != AOK) {
        LogT("fh_compose returing after calling path_inode", Status);
        *returnstatus=NFSERR_STALE;
        return(Status);
    }

    if (is_dd) {
        int i;
        for (i=0; i<INODEARRAYLEN-1; i++) {
            key->InodeArray[i]=key->InodeArray[i+1];
        }
        key->InodeArray[INODEARRAYLEN-1]=0;               /* wrong but only a hint anyway */
    } else {
        int i;
        for (i=INODEARRAYLEN-1; i>0; i--) {
            key->InodeArray[i]=key->InodeArray[i-1];
        }
    }
    key->InodeArray[0]=Inode;                             /* only [0] really matters */


    Status = fh_find(key, 1, &h);                         /* use InodeArray key to get good fhc */
    if (Status != AOK) {                                   
        *returnstatus=NFSERR_STALE;
    } else {
        if (h->path == 0) {
            h->fd = -1;
            h->path = strsave(pathbuf);
        }
        *returnstatus=NFS_OK;
    }

    LogT("fh_compose done ", Status);
    return(Status);
}



/* 
 */
extern int path_inode(path, status, returninode)
char    *path;
nfsstat *status;
u_long *returninode;
{
    struct stat sbuf;
    u_long Result;
    int Status;

    Log2("path_inode starting on ", path);

    Status=AlexInfoCompatStat(path, &sbuf, (struct ParsedDir *) NULL, LastUidStr, RECURSIONOK);
    if (Status ==  AOK) { 
        *status = (nfsstat) NFS_OK;
        Result=sbuf.st_ino;
    } else {
        *status = (nfsstat) NFSERR_NOENT;                  /* XXXXX what error should it be? */
        Result=0;                                          /* zero as failure                */
    }

    LogN("path_inode returning inode ", (int) Result);
    *returninode=Result;
    LogT("path_inode ", Status);
    return(Status);
}


