/*
 * Copyright 1992 SynOptics Communications, Inc.  All Rights Reserved.
 * SynOptics grants a non-exclusive license to use, copy, modify, and
 * distribute this software for any purpose and without fee, provided
 * that this copyright notice and license appear on all copies and
 * supporting documentation.
 * SynOptics makes no representations about the suitability of this
 * software for any particular purpose.  The software is supplied
 * "AS IS", and SynOptics makes no warranty, either express or implied,
 * as to the use, operation, condition, or performance of the software.
 * SynOptics retains all title and ownership in the software.
 *
 * file: SMIC.C - SNMP Concise MIB Compiler
 *
 * Use:  SMIC <options> <file>
 *
 * $Revision:   1.4  $ $Date:   27 Jul 1992 19:49:34  $
 * $Log:   R:/MIBTOOLS/V1.0/SMIC/SRC/SMIC.C_V  $
 * 
 *    Rev 1.4   27 Jul 1992 19:49:34   gfoster
 * Changed the -7 option to make stricter checking
 * of index items instead of looser checking.
 * 
 * Added brackets around the word "options" in
 * the usage line.
 * 
 *    Rev 1.3   08 Jul 1992 17:31:48   gfoster
 * Removed unnecessary revision comment lines added by
 * PVCS to make revision history easier to read.
 * 
 *    Rev 1.2   08 Jul 1992 16:53:42   gfoster
 * Added new option "0" to assign message (error)
 * file to output file.
 * 
 *    Rev 1.1   19 Jun 1992 16:07:40   gfoster
 * Copyright text was reformated.
 * 
 * Name of the variable "usOptChar" changed to "chOptChar".
 * 
 * Added support for environment variable for the include
 * file directory.
 * 
 * Changed program help for OID list to indicate it
 * also prints traps.
 * 
 *    Rev 1.0   27 May 1992 15:57:36   gfoster
 * Initial revision.
 *
*/

#include <stdio.h>
#include <stdlib.h>

#include <string.h>
#include <ctype.h>

#include "tds.h"
#include "smscdefs.h"
#include "smstdefs.h"
#include "smsydefs.h"
#include "smic.h"


/* BOOL fNoStr;*/               /* suppress strings in scanner */
extern BOOL fNoStrSav;          /* saved value when processing directives
                                    (or any thing else where suppressing
                                    is a problem) */

#define MXOPT 4                 /* depth of options stack */
USHORT afOpt[MXOPT];            /* stack for options */
USHORT iOpt;                    /* current depth of option stack */

typedef struct _optEntry {
    BOOL *pbOpt;                /* ptr to option */
    USHORT fVal;                /* value for option */
    UCHAR chVal;                /* char value for option */
} OPTENTRY;

OPTENTRY optTab[] = {
    {&fNoStrSav,        0x0001, 'X'}, /* the "saved" value of fNoStr must
                                         be used since the directives
                                         include strings */
    {&fCheckRoIndx,     0x0002, 'R'},
    {&fCheckSEQ,        0x0004, 'W'},
    {&fCheckSR,         0x0008, 'C'},
    {&fCheckISR,        0x0010, 'B'},
    {&fCheckIndxSeq,    0x0020, '7'},
    {&fNoCheckTab,      0x0040, '6'},
    {&fNoSeqTcErr,      0x0080, 'E'},
    {&fAllowIndx,       0x0100, '3'},
    {&fNoUnUsedWarn,    0x0200, 'G'},
    {&fAllowAccess,     0x0400, '4'},
    {&fAllowOpt,        0x0800, '5'},
    {&fAllowDV,         0x1000, 'J'}
};


/** pushOpt - push options on stack
*
* returns:
*   TRUE - options pushed
*   FALSE - an error
*/
    BOOL
#ifdef __STDC__
pushOpt(VOID)
#else
pushOpt()
#endif /* __STDC__ */
{
    USHORT fOpt;
    USHORT i;


    if (iOpt >= MXOPT) {
        yyerror("Option stack already at max depth");
        return(FALSE);
    }

    for (i = 0, fOpt = 0; i < sizeof(optTab)/sizeof(OPTENTRY); i++) {
        if (*(optTab[i].pbOpt))
            fOpt |= optTab[i].fVal;
    }

    afOpt[iOpt++] = fOpt;

    return(TRUE);

} /* pushOpt */


/** popOpt - pop options from stack
*
* returns:
*   TRUE - options popped
*   FALSE - an error
*/
    BOOL
#ifdef __STDC__
popOpt(VOID)
#else
popOpt()
#endif /* __STDC__ */
{
    USHORT fOpt;
    USHORT i;


    if (iOpt == 0) {
        yyerror("Option stack empty");
        return(FALSE);
    }

    for (i = 0, fOpt = afOpt[--iOpt]; i < sizeof(optTab)/sizeof(OPTENTRY); i++) {
        if (fOpt & optTab[i].fVal)
            *(optTab[i].pbOpt) = TRUE;
        else
            *(optTab[i].pbOpt) = FALSE;
    }

    return(TRUE);

} /* popOpt */


/** printOpt - print options
*
*/
    VOID
#ifdef __STDC__
printOpt(VOID)
#else
printOpt()
#endif /* __STDC__ */
{
    USHORT i;
    BOOL fNone;

    fprintf(fhMsg, "options:");
    for (i = 0, fNone = TRUE; i < sizeof(optTab)/sizeof(OPTENTRY); i++) {
        if (*(optTab[i].pbOpt)) {
            fNone = FALSE;
            fprintf(fhMsg, " %c", optTab[i].chVal);
        }
    }
    if (fNone)
        fprintf(fhMsg, " **none set**");
    fprintf(fhMsg, "\n");

} /* printOpt */


/** addOpt - set on specified options
*
* call with:
*   pVal - string of options to turn on
*
* returns:
*   TRUE - options set
*   FALSE - an error
*/
    BOOL
#ifdef __STDC__
addOpt(STRTAB *pVal)
#else
addOpt(pVal)
    STRTAB *pVal;
#endif /* __STDC__ */
{
    USHORT usOpt;
    PSZ pszOpt;
    BOOL bRt;
    USHORT i;


    for (bRt = TRUE, pszOpt = pVal->pszVal; *pszOpt != 0; pszOpt++) {
        if (isspace(*pszOpt))
            continue;

        usOpt = toupper(*pszOpt);
        for (i = 0; i < sizeof(optTab)/sizeof(OPTENTRY); i++) {
            if (optTab[i].chVal == (UCHAR)usOpt)
                break;
        }
        if (i == sizeof(optTab)/sizeof(OPTENTRY)) {
            yyerror("unknown option \'%c\' in \"%s\"",
                    *pszOpt, pVal->pszVal);
            bRt = FALSE;
            continue;
        }
        *(optTab[i].pbOpt) = TRUE;  /* turn on option */
    }

    return(bRt);

} /* addOpt */


/** removeOpt - turn off specified options
*
* call with:
*   pVal - string of options to turn off
*
* returns:
*   TRUE - options turned off
*   FALSE - an error
*/
    BOOL
#ifdef __STDC__
removeOpt(STRTAB *pVal)
#else
removeOpt(pVal)
    STRTAB *pVal;
#endif /* __STDC__ */
{
    USHORT usOpt;
    PSZ pszOpt;
    BOOL bRt;
    USHORT i;


    for (bRt = TRUE, pszOpt = pVal->pszVal; *pszOpt != 0; pszOpt++) {
        if (isspace(*pszOpt))
            continue;

        usOpt = toupper(*pszOpt);
        for (i = 0; i < sizeof(optTab)/sizeof(OPTENTRY); i++) {
            if (optTab[i].chVal == (UCHAR)usOpt)
                break;
        }
        if (i == sizeof(optTab)/sizeof(OPTENTRY)) {
            yyerror("unknown option \'%c\' in \"%s\"",
                    *pszOpt, pVal->pszVal);
            bRt = FALSE;
            continue;
        }
        *(optTab[i].pbOpt) = FALSE; /* turn off option */
    }

    return(bRt);

} /* removeOpt */


/** dirHelp - print help for directives
*
*/
    VOID
#ifdef __STDC__
dirHelp(VOID)
#else
dirHelp()
#endif
{
    fprintf(fhMsg, "Directives\n");
    fprintf(fhMsg, "  #help -- print this list\n");
    fprintf(fhMsg, "  #include \"<fileName>\" -- process include file\n");
    fprintf(fhMsg, "  #aliasModule <module> <alias> -- alias a module\n");
    fprintf(fhMsg, "  #aliasSymbol <module> <object> <alias> -- alias an object\n");
    fprintf(fhMsg, "  #pushOpt -- push options\n");
    fprintf(fhMsg, "  #popOpt -- pop options\n");
    fprintf(fhMsg, "  #addOpt \"<options>\" -- add one or more options\n");
    fprintf(fhMsg, "  #removeOpt \"<options>\" -- remove one or more options\n");
    fprintf(fhMsg, "  #printOpt -- print options\n");

} /* dirHelp */


/** GetArgs - parse program arguments
*
* call with:
*   argc - number of arguments
*   argv - program arguments
*
* returns:
*   TRUE - arguments parsed OK
*   FALSE - error with argument syntax
*/
    BOOL
#ifdef __STDC__
GetArgs(INT argc, CHAR *argv[])
#else
GetArgs(argc, argv)
    INT argc;
    CHAR *argv[];
#endif /* __STDC__ */
{
    SHORT i;
    USHORT usOpt;
    PSZ pszArg;
    BOOL fSameFile = FALSE;


    /* check out the options */
    for (i = 1; i < argc; i++) {
        pszArg = argv[i];

        if ((pszArg[0] != '-') && (pszArg[0] != '/')) {
            break;
        }

        if (strlen(pszArg) != 2) {
            fprintf(fhMsg, "Illegal option: \"%s\"\n",
                    pszArg);
            return(FALSE);
        }

        usOpt = toupper(pszArg[1]);
        switch(usOpt) {
        case '0':       /* set the message file to be the same
                            as the output file */
            fSameFile = TRUE;
            break;

        case '1':       /* print version and copyright */
            prVer();
            prCpyrt();
            exit(0);
            break;
    
        case '2':       /* print version */
            prVer();
            break;

        case '3':       /* allow INDEX clause on leaf objects */
            fAllowIndx = TRUE;
            break;

        case '4':       /* allow non-std access on leaf objects */
            fAllowAccess = TRUE;
            break;

        case '5':       /* allow "optional" status */
            fAllowOpt = TRUE;
            break;

        case '6':       /* no check that table, row, seq have related names */
            fNoCheckTab = TRUE;
            break;

        case '7':       /* check that all index items for a row are in seq */
            fCheckIndxSeq = TRUE;
            break;

        case '8':       /* output in mosy format */
            fDumpMosy = TRUE;
            break;

        case '9':       /* output in Extended mosy format */
            fDumpExtMosy = TRUE;
            break;

        case 'A':       /* dump alias names */
            fDebugAL = TRUE;
            break;

        case 'B':       /* strong check size/range of index items */
            fCheckISR = TRUE;
            break;

        case 'C':       /* check size/range */
            fCheckSR = TRUE;
            break;

        case 'D':       /* dump all */
            fDebugStrTab = TRUE;
            fDebugMOD = TRUE;
            fDebugOID = TRUE;
            fDebugOIDtree = TRUE;
            fDebugName = TRUE;
            fDebugIMP = TRUE;
            fDebugAL = TRUE;
            fDebugSMI = TRUE;
            fDebugTRAP = TRUE;
            fDebugTC = TRUE;
            fDebugSEQ = TRUE;
            fDebugStats = TRUE;
            break;

        case 'E':       /* no exact syntax match for TCs in seq */
            fNoSeqTcErr = TRUE;
            break;

        case 'F':       /* name of output file */
            /* check for filename */
            if (((i+1) < argc) && (*(argv[i+1]) != '/') &&
                    (*(argv[i+1]) != '-')) {
                /* get filename */
                i++;
                fhOut = fopen(argv[i], "w");
                if (fhOut == NULL) {
                    fprintf(fhMsg, "Error opening file \"%s\"\n", argv[i]);
                    return(FALSE);
                }
            } else {
                fprintf(fhMsg, "Name of output file expected\n");
                return(FALSE);
            }
            break;

        case 'G':       /* no warnings for unused Imports and TCs */
            fNoUnUsedWarn = TRUE;
            break;

        case 'I':       /* print include file names */
            fPrintIname = TRUE;
            break;

        case 'J':       /* allow DEFVAL for counters and gauges */
            fAllowDV = TRUE;
            break;

        case 'K':       /* dump SMI names */
            fDebugSMI = TRUE;
            break;

        case 'L':       /* dump OID item names */
            fDebugName = TRUE;
            break;

        case 'M':       /* dump module names */
            fDebugMOD = TRUE;
            break;

        case 'N':       /* dump OID tree */
            fDebugOIDtree = TRUE;
            break;

        case 'O':       /* dump OID table */
            fDebugOID = TRUE;
            break;

        case 'P':       /* dump Trap names */
            fDebugTRAP = TRUE;
            break;

        case 'Q':       /* dump Sequences */
            fDebugSEQ = TRUE;
            break;
 
        case 'R':       /* check index objs for read-only */
            fCheckRoIndx = TRUE;
            break;

        case 'S':       /* dump string table */
            fDebugStrTab = TRUE;
            break;

        case 'T':       /* dump IMPORT names */
            fDebugIMP = TRUE;
            break;

        case 'U':       /* dump resource usage statistics */
            fDebugStats = TRUE;
            break;

        case 'V':       /* dump Textual conventions */
            fDebugTC = TRUE;
            break;

        case 'W':       /* strong check of sequence items syntax */
            fCheckSEQ = TRUE;
            break;

        case 'X':       /* suppress strings in scanner */
            fNoStr = TRUE;
            break;

        case 'Z':       /* output in intermediate format */
            fDumpInterm = TRUE;
            break;

        default:
            fprintf(fhMsg, "Unknown option: \"%s\"\n",
                    pszArg);
            return(FALSE);
        }
    }

    /* check for valid number of parameters */
    if (i >= argc) {
        fprintf(fhMsg, "Need file name\n");
        return(FALSE);
    }

    /* open filename and check for errors */
    pszInFile = argv[i];
    pszBaseFile = pszInFile;
    fhIn = fopen(pszInFile, "rb");
    if (fhIn == NULL) {
        fprintf(fhMsg, "Error opening file \"%s\"\n", pszInFile);
        return(FALSE);
    }

    /* check if message file should be mapped to output file */
    if (fSameFile)
        fhMsg = fhOut;

    /* everything OK */
    return(TRUE);

} /* GetArgs */


/** MAIN for program
*
*
*/
    VOID
#ifdef __STDC__
main(INT argc, CHAR *argv[])
#else
main(argc, argv)
    INT argc;
    CHAR *argv[];
#endif /* __STDC__ */
{
    /* get ptr to environment variable for include files */
    pszSmicIncl = getenv(SMICINCL);

    /* get program arguments */
    if (!GetArgs(argc, argv)) {
        prCpyrt();
        prVer();
        fprintf(fhMsg,
            "Usage: SMIC [<options>] <inputFile>\n");
        fprintf(fhMsg,
            "  where <options> are the following:\n");
        fprintf(fhMsg,
            "    %c1 print copyright     %c2 print version     %cN dump OID tree and traps\n",
            chOptChar, chOptChar, chOptChar);
        fprintf(fhMsg,
            "    %cS dump string table   %cP dump Trap names   %cO dump OID items\n",
            chOptChar, chOptChar, chOptChar);
        fprintf(fhMsg,
            "    %cT dump IMPORT names   %cK dump SMI names    %cM dump module names\n",
            chOptChar, chOptChar, chOptChar);
        fprintf(fhMsg,
            "    %cA dump Alias names    %cQ dump Sequences    %cV dump Textual conventions\n",
            chOptChar, chOptChar, chOptChar);
        fprintf(fhMsg,
            "    %cL dump OID names      %cD dump all\n",
            chOptChar, chOptChar);
        fprintf(fhMsg,
            "    %cC check size/range              %cR check INDEX objs for read-only\n",
            chOptChar, chOptChar);
        fprintf(fhMsg,
            "    %c3 allow INDEX clause on objects %c4 allow non-std access for objects\n",
            chOptChar, chOptChar);
        fprintf(fhMsg,
            "    %c5 allow 'optional' status       %c6 no check on table, row, seq names\n",
            chOptChar, chOptChar);
        fprintf(fhMsg,
            "    %c7 chck for all indx itms in seq %cE allow seq item syntax to match TC\n",
            chOptChar, chOptChar);
        fprintf(fhMsg,
            "    %cJ allow DEFVAL on Counter/Gauge %cW strong check of seq item syntax\n",
            chOptChar, chOptChar);
        fprintf(fhMsg,
            "    %cB strong check size/range of index items\n",
            chOptChar);
        fprintf(fhMsg,
            "    %cG turn off warnings for unused IMPORTs and textual conventions\n",
            chOptChar);
        fprintf(fhMsg,
            "    %cX suppress strings in scanner   %cU print resource usage statistics\n",
            chOptChar, chOptChar);
        fprintf(fhMsg,
            "    %c8 Output in Mosy Format         %c9 Output in Extended Mosy Format\n",
            chOptChar, chOptChar);
        fprintf(fhMsg,
            "    %cZ Output to file in intermediate format\n",
            chOptChar);
        fprintf(fhMsg,
            "    %cF <file> Name of output file    %cI print names of include files\n",
            chOptChar, chOptChar);
        fprintf(fhMsg,
            "    %c0 Set message(and error) file to output file\n",
            chOptChar);
        exit(1);
    }


    /* do initializations */
    if (!yylexinit()) {         /* init scanner */
        fprintf(fhMsg, "Scanner Initialization error\n");
        exit(1);
    }
    initOIDroot();              /* init root of OID tree */


    /* parse the MIB file */
    if (yyparse() != 0) {
        fprintf(fhMsg, "Unsuccessful parsing\n");
        exit(1);
    }

    if ((cErr == 0) && (cWarn == 0)) 
        fprintf(fhMsg, "%s parsed successfully\n", pszInFile);
    else
        fprintf(fhMsg, "\n*** %s parsed with %u error%s and %u warning%s\n",
                pszInFile, cErr, (cErr != 1) ? "s" : "",
                cWarn, (cWarn != 1) ? "s" : "");


    /* dump string table */
    if (fDebugStrTab)
        StrTabDump();

    /* dump modules */
    if (fDebugMOD)
        dumpMOD();

    /* dump OID objects */
    if (fDebugOID)
        dumpOID();

    /* dump OID tree */
    if (fDebugOIDtree)
        dumpOIDtree();

    /* dump OID Names */
    if (fDebugName)
        dumpOIDname();

    /* dump Imports */
    if (fDebugIMP)
        dumpIMP();

    /* dump Aliases */
    if (fDebugAL)
        dumpAL();

    /* dump SMI names */
    if (fDebugSMI)
        dumpSMI();

    /* dump Trap names */
    if (fDebugTRAP)
        dumpTRAP();

    /* dump Textual conventions */
    if (fDebugTC)
        dumpTC();

    /* dump Sequences */
    if (fDebugSEQ)
        dumpSEQ();

    /* dump resource usage stats */
    if (fDebugStats)
        dumpStats();

    /* dump in mosy format */
    if (fDumpMosy)
        dumpMosy();

    /* dump in Extended mosy format */
    if (fDumpExtMosy)
        dumpExtMosy();

    /* dump in intermediate format */
    if (fDumpInterm)
        dumpInterm();

    /* close input file and exit */
    fclose(fhIn);

    exit(0);

} /* main */


/* end of file: SMIC.C */
