/*
 * $Header: /noc/network/netlog/src/RCS/process_tickets.c,v 2.0 1992/05/10 16:52:12 aggarwal Exp $
 */

/* Copyright 1992 JvNCnet, Princeton University */

/*+
** FUNCTION:
**	This is for processing a ticket and printing it out. Given a
** range of ticket numbers, it prints out all entries made for that
** ticket (by consulting the INDEXFILE).
**
**	Vikas Aggarwal, vikas@jvnc.net
**/

/*
 *
 *	$Log: process_tickets.c,v $
 * Revision 2.0  1992/05/10  16:52:12  aggarwal
 * Cleaned up and restructured for 'netlog v2.0'
 *
 * Revision 1.7  1991/10/18  19:47:34  aggarwal
 * Piped the output through 'more' in case displaying to the terminal.
 *
 * Revision 1.5  90/06/21  11:28:33  aggarwal
 * Added option to print out the output file directly on
 * the 'lpr' printer by specifying 'lpr' as the output file.
 * The file is printed and then removed.
 * 
 * Revision 1.3  90/03/05  11:49:13  aggarwal
 * Added a line so that if processing to a file, it prints out '.' to
 * indicate that program is still alive and working.
 * 
 * Revision 1.1  90/03/01  14:32:21  aggarwal
 * Initial revision
 * 
 *
 */

/*  */

#ifndef lint
 static char rcsid[] = "$RCSfile: process_tickets.c,v $ $Revision: 2.0 $ $Date: 1992/05/10 16:52:12 $" ;
#endif

#include "netlog.h"

process_tickets()
{
    int sttkt, endtkt, outfd ;
    int printout = 0, ttyout= 0 ;		/* If to print output 	*/
    char *reply, temp[MAXLINE], cmd[BUFSIZ] ;
    FILE *outfp ;				/* output file pointer	*/


    printf("\n\n\t\t%s\n\n", "** Ticket Processing **");
    printf(" You can specify that the output be printed directly on the\n");
    printf(" printer (lpr), by specifying 'lpr' as the output filename\n\n");

 redo:
    if ( (endtkt = get_lastkt ()) == -1)	/* get last ticket num	*/
      return (-1) ;

    reply = get_reply("Enter starting ticket number", "0", C_DIGIT);
    sscanf(reply, "%d", &sttkt);
    sprintf(temp,"%d\0", endtkt) ;		/* for default response */
    reply = get_reply("Enter ending ticket number", temp, C_DIGIT);
    sscanf(reply, "%d", &endtkt);

    /*
    ** Do some input checks
    */
    if ( sttkt > endtkt )			/* screwed up values	*/
    {
	printf ("Error: starting ticket %d is greater than ending %d\n",
		 sttkt, endtkt) ;
	goto redo ;
    }

 redo1:
    reply = get_reply("Enter output file name", "/dev/tty",
		      C_DIGIT | C_ALPHA | C_PUNCT) ;
    /*
     * Make sure user gives full path
     */
    if (reply[0] != '/' && strncmp(reply, "lpr", strlen("lpr")) != 0)
    {
	printf("Error: Please give full pathname while specifying file\n");
	goto redo1;
    }

    printf("Starting tkt: %d\nEnding tkt: %d\nOutput file '%s' [confirm]: ",
	   sttkt, endtkt, reply);
    gets(temp);
    if (*temp == NULL || *temp == 'y' || *temp == 'Y')
      ;
    else
      goto redo ;

    /* To print directly on printer or on display, create tmp file first */
    if (strncmp(reply, "lpr", strlen("lpr")) == 0)
    {						/* create temp file	*/
	printout = 1 ;
	sprintf(reply, "/var/tmp/netlog-lpr.%d\0", getpid());
    }
    else if (strcmp(reply, "/dev/tty") == 0)
    {
	ttyout = 1;
	sprintf(reply, "/var/tmp/netlog-tty.%d\0", getpid());
    }

    if (access(reply, F_OK) != 0)
      if (create_blank_file(reply, 0664) == -1)	/* world gets no-write */
	return (-1);

    if (access(reply, W_OK) != 0)		/* check if file writable */
    {
	fprintf(stderr, "(process_tkt): no write access\n") ;
	perror(reply);
	return (-1);
    }
    sprintf(cmd, "%s %s %s\0", COPY, Netloghdr, reply) ;
    system(cmd);				/* copy header file	*/

    if ((outfp = fopen(reply, "a")) == NULL)	/* open for append mode */
    {
	fprintf (stderr, "(process_tkt) fopen  ");
	perror(reply);
	return(-1);
    }

    while (sttkt <= endtkt)			/* for all the tickets	*/
      if (process_write_ticket(sttkt++, outfp) == -1)
	return (-1);
      else if ( (sttkt % 10) == 1 )		/* Indicate that working */
	printf("%d, ", (sttkt - 1) ), fflush(stdout);

    if (outfp != stdout)			/* else in trouble	*/
      fclose(outfp);

    printf("\n") ;
    if (printout)				/* to be printed	*/
    {
	/* Print and remove file */
	sprintf(cmd, " %s %s; %s %s\0", PRINTCMD, reply, RM, reply);
	system (cmd) ;
	printf( "\nFile printed using: '%s'\n", PRINTCMD);
    }
    else if (ttyout)
    {
	/* Display and remove file, pipe thru more so person cannot 'vi' */
	sprintf(cmd, " %s %s | %s; %s %s\0", CAT, reply, MORE, RM, reply);
	system (cmd);
    }
    return(0);
}						/* end:	process_tickets	*/

/*+ 			get_lastkt
** FUNCTION:
** 	Returns the last ticket from the file LASTKT. Returns zero if
** no file.
**/
get_lastkt()
{
    int tkt;
    FILE *lastktfp ;

    if ((lastktfp = fopen(Prevtkt, "r")) == NULL)
    {					
	fprintf(stderr, "(get_lastkt) fopen ");
	perror(Prevtkt) ;
	return (-1);
    }
    
    fscanf(lastktfp, "%d", &tkt) ;		/* get the last ticket	*/
    fclose(lastktfp);
    return (tkt) ;
}					/* end:  get_lastkt()		*/

/*+		process_write_ticket_()
** FUNCTION:
**	To process a ticket number passed to it and put out all the
** entries for the ticket on the file provided (notice that this is
** an output file). Have to supply an OPENed FILE pointer.
**
** The routine also takes care of the case when the number of entries
** exceeds the value MAXTKTENTRIES - it then searches for the ticket
** number in increasing order of file dates.
**
** It uses the files INDEXFILE and the directory LOG.
**/

process_write_ticket(ticket, outfp)
     int ticket;			/* ticket to process		*/
     FILE *outfp;			/* opened file pointer		*/
{
    struct index_entry index;
    int indexfd, indexsz = sizeof(index),
	search = 0;			/* flag indicates > MAXTKTENTRY */
    register i ;
    char buffer[MAXLINE];		/* copy GREPREC output over	*/
#ifdef NFS
    struct flock ilock;			/* Test for any write lock	*/
#endif

    if (outfp == NULL)			/* Not possible to write file	*/
    {
	fprintf(stderr, "(process_write_ticket) output file ptr is NULL\n");
	return (-1);
    }
    if ((indexfd = open(Indexfile, O_RDONLY)) < 0)
    {
	fprintf(stderr, "(process_write_ticket) open ");
	perror(Indexfile);
	return (-1);
    }
    bzero((char *)&index, indexsz);
    if (lseek(indexfd, (off_t)INDEXOFFSET(ticket), L_SET) < 0)
    {					/* protect against bad indexfd	*/
	perror("(process_write_ticket) lseek");
	close(indexfd);
	return (-1);
    }
#ifdef NFS
    ilock.l_type = F_RDLCK;			/* read lock only	*/
    ilock.l_whence = 1 ;			/* from current offset	*/
    ilock.l_start = 0 ;				/* start from current	*/
    ilock.l_len = indexsz ;			/* len of lock		*/
    ilock.l_pid = 0 ;				/* Not used at all	*/
    fcntl(indexfd, F_SETLKW, (struct flock *)&ilock) ;	/* wait if lked */
#else
    flock(indexfd, LOCK_SH);			/* Shared lock		*/
#endif
    if (read(indexfd, &index, indexsz) < 0)
    {
	perror(Indexfile);
	close(indexfd), fclose(outfp) ;
	return (-1);
    }
    close (indexfd);				/* release the lock	*/
    if (index.tnum != ticket)			/* avoid junk		*/
      bzero ((char *)&index, indexsz) ;
    
    if (index.numdate > MAXTKTDATES)
      search = 1 ;				/* haveto searchall dir */

    for (i = 0; i < Numentries(index.numdate); ++i)	/* all entries	*/
    {
	register nread ;			/* num of char read	*/
	static char logfile[BUFSIZ],		/* store logfile path	*/
	     tmpstr[10],			/* temporary string	*/
	     grepcmd[BUFSIZ],			/* grep_record() cmd	*/
	     da[3], mo[3], yr[3] ;		/* extract mon yr etc	*/
	FILE *grepout;				/* For read pipe	*/
	
	if (index.edate[i] == 0)		/* bad date, skip */
	  continue ;

	sprintf(tmpstr,"%06.6d\0",index.edate[i]);/* extract month, etc	*/
	sscanf(tmpstr, "%2s%2s%2s", mo, da, yr) ;

	fprintf(outfp, "::: %s/%s/%s :::\n", mo,da,yr);	/* date stamp	*/
	fflush(outfp);
	
	sprintf(logfile, "%s/%s%s/netlog.%s%s%s\0", 
		LOGDIR, mo, yr, mo, da, yr);	/* create logfile name	*/

	/*
	 * Now search for entry using 'grep_record'. Look for '\t#345 :'
	 * type of expressions where 345 is the ticket number, etc.
	 */
	sprintf(grepcmd, "%s -i -s '%s' -e '%s' '\t#%d :' %s\0", 
		GREPREC, RECSEP, RECSEP, ticket, logfile) ;
	grepout = popen(grepcmd, "r");		/* Copy pipeline data	*/
	while (fgets(buffer, MAXLINE, grepout) != NULL)
	  fputs(buffer, outfp);
	pclose(grepout);
	fflush (outfp) ;
    }						/* end:	 for ...	*/

    if (search)
      search_tkt_entry(ticket, index.numdate, index.edate[i-1], outfp) ;

    return (0) ;
}		/* process_write_ticket()	*/

/*+ 		search_tkt_entry
** FUNCTION:
** 	Called when the number of date entries exceeds MAXTKTDATES.
** Can either search thru all the logs for the ticket entries, or else
** give a message and let the user do the rest.
**
**/
search_tkt_entry(tkt, numdate, lastdate, outfp)
     int tkt, lastdate, numdate;
     FILE *outfp ;				/* opened file pointer	*/
{

    printf("Number of entry dates for tkt '#%d' exceeds %d. Please run '%s'\n",
	   tkt, MAXTKTDATES, GREPREC) ;
    printf("for remaining %d entries after date %06d\n\n", 
	   (numdate - MAXTKTDATES), lastdate);

}
