/*
 * $Header: /noc/network/netlog/src/RCS/init.c,v 2.2 1992/05/14 17:46:44 aggarwal Exp $
 */

/* Copyright 1992 JvNCnet, Princeton University */

/*+ 
 ** o Gets the terminal type in case it is not set (essential for the editor).
 ** o checks for the accessibility of all essential files
 ** o Also sets the program uid to the effective uid if user belongs to
 **   an WRITE_GROUP. The real uid is saved in 'ruid', and the effective uid 
 **   is saved in euid. Note that this is a good check of whether the user
 **   is OK or not  (OK if getuid() == euid, not ok if getuid() == ruid).
 ** o sets 'Indexfile, Initialsfile' etc to fully qualified pathnames.
 ** o dumb check on the size of 'Indexfile' and if it is evenly divided
 **   by the size of the 'index_entry'
 **
 **	A return value of -1 signifies that some fatal error has
 ** occurred while initializing and the calling program should suitably
 ** exit.
 **
 **/

/*
 * $Log: init.c,v $
 * Revision 2.2  1992/05/14  17:46:44  aggarwal
 * Error in the set_terminal() function. Needed '**bad_terminals' and
 * also the testing was wrong.
 *
 * Revision 2.1  1992/05/11  23:42:09  aggarwal
 * Changes since tackon_dir() was modified to return a malloc-ed ptr.
 *
 * Revision 2.0  1992/05/10  16:28:26  aggarwal
 * Cleaned up and restructured for releasing 'netlog v2.0'
 *
 * Revision 1.1  1990/03/05  14:44:07  network
 * Initial revision
 *
 */

/*  */

#ifndef lint
  static char rcsid[] = "$RCSfile: init.c,v $ $Revision: 2.2 $ $Date: 1992/05/14 17:46:44 $" ;
#endif


#include "netlog.h"
#include <sys/param.h>				/* for getgroups()	*/
#include <grp.h>
#include <pwd.h>

init()
{
    int i, okuser ;
    struct stat buf ;			/* to get the file info */

    if (setprocessid() == -1)	   	/* id = eid if WRITE_GROUP	*/
      return (-1);
    if (setuserenviron() == -1)		/* check terminal and path */
      return (-1);

    /*
     * Do a cursory check on the important files
     */
    for (i = 0 ; *read_files[i] != NULL ; ++i )
      if (strchr (read_files[i], '/') != NULL)	/* pathname specified */
	if (access (read_files[i], R_OK) == -1)
	{
	    fprintf(stderr, "init (read access) ") ;
	    perror (read_files[i]) ;
	    return (-1);			/* fatal if not present	*/
	}

    for (i = 0 ; *exec_files[i] != NULL ; ++i )
      if (strchr (exec_files[i], '/') != NULL)
	if (access (exec_files[i], X_OK) == -1)
	{
	    fprintf(stderr, "init (exec access) ") ;
	    perror (exec_files[i]) ;
	    return (-1);			/* fatal if not present	*/
	}

    path_qualify_files() ;		/* Append dir to filenames */

    /* Now check the Indexfile size */

    if (stat(Indexfile, &buf) != 0)
    {
	fprintf(stderr, "(init) stat error for %s: %s\n",
		Indexfile, sys_errlist[errno]) ;
	return (-1);
    }

    if (((int)buf.st_size % sizeof(struct index_entry)) != 0)
      fprintf(stderr,
	      "(init) Warning: indexfile %s doesn't match entry size\n",
	      Indexfile) ;

    if (debug)
      fprintf(stderr, "(init) %s size= %d bytes, index_entry= %d bytes\n",
	      Indexfile, (int)buf.st_size, sizeof(struct index_entry)) ;

}	/* end init */


/*+ 			setprocessid
 ** FUNCTION:
 ** 	Set the uid/gid of the process to the effective uid/gid (depending
 ** on whether the program is set with the 4xxx or 2xxx mode and depending
 ** on whether the group is a member of validgroups or not.
 ** Saves real uid/gid in ruid rgid and effective user id/gid in euid, egid
 **/

setprocessid()
{
    extern int debug ;
    extern int in_write_group;			/* in netlog.h  */
    register i, j=0;				/* fast index		*/
    int ngroups, groups[NGROUPS] ;		/* defined in param.h	*/
    uid_t uid ;	
    gid_t gid ;
    struct group *usrgr;			/* to get user's info */

    in_write_group = 0 ;
    ruid = getuid() ;				/* save the real uid	*/
    euid = geteuid() ;				/* save the eff uid	*/
    rgid = getgid() ;
    egid = getegid() ;

    ngroups = getgroups(NGROUPS, groups) ;	/* get entire group list */
    while (!in_write_group && *write_groups[j] != NULL)
    {
	usrgr = getgrnam((char *)write_groups[j++]) ;
	if (usrgr == NULL)
	{
	    if (debug)
	      fprintf (stderr,"Can't get id for group %s\n", 
		       write_groups[j-1]);
	}
	else
	  for (i = 0 ; i < ngroups ; ++i)
	    if (usrgr->gr_gid == groups[i])
	    {
		in_write_group = 1;
		break ;				/* from the 'for' loop	*/
	    }

    }		/* end 'while'	*/

    if (in_write_group)				/* user	group ok */
      uid = euid , gid = egid ;			/* do a suid */
    else
      uid = ruid , gid = rgid ;			/* User's real ID */

    /*
     * Now change the UID and GID of the program for consistency,
     */
    if (setreuid(uid, uid) == -1 || setregid(gid,gid) == -1)
    {
	perror ("setreuid/setregid");
	return (-1) ;
    }

    if (debug)
    {
	for ( i = 0 ; *write_groups[i] != NULL ; ++i)
	  fprintf(stderr, "Valid groups: %s\n", write_groups[i]) ;
	for ( i = 0 ; i < ngroups ; ++i )
	{
	    usrgr = getgrgid(groups[i]);
	    fprintf (stderr, "User's groups are: %s\n", usrgr->gr_name);
	}
        fprintf(stderr,
		"((%s) Process' ruid/euid/rgid/egid= %d/%d/%d/%d\n",
                prognm, getuid(), geteuid(), getgid(), getegid() );

    }		/* end:  if (debug)	*/


}		/* end: setprocessid	*/


/*+		setuserenviron 
 ** FUNCTION
 **
 **	This function attempts to set the environment variables
 ** relevant to the program (like TERM). It modifies the structure
 ** 'environ' which is a pointer to the strings containing the
 ** variables. If an essential environment variable is not set, then
 ** it sets the variable by prompting or by a default value.
 **
 ** Sets/gets VISUAL, TERM, PATH
 **
 ** 'putenv' requires that the storage for the variable be static.
 **
 ** The environment variable VISUAL is used for the editor or else
 ** 'vi' is used.
 **
 **/

setuserenviron ()
{
    char *g, oldvar[MAXLINE], bp[1024] ;	/* needed by tgetent    */
    static char termvar[MAXLINE];		/* has to be static     */
    static char pathvar[MAXLINE];

    if ((g = (char *)getenv("TERM")) != NULL)	/* retrieve TERM type	*/
      strcpy (oldvar, g);
    else
      oldvar[0] = '\0' ;

    if (check_terminal(oldvar) == -1)		/* Extract terminal type */
      return(-1) ;

    sprintf (termvar, "TERM=%s\0", (char *)oldvar);

    /* the 'PATH' keyword is a part of BINPATH */
    if ((g = (char *)getenv("PATH")) != NULL)	/* retrieve old PATH */
      strcpy (oldvar, g);
    else
      oldvar[0] = '\0' ;

    if (debug)
      fprintf(stderr, "Old path is '%s'\n", oldvar) ;

    sprintf(pathvar, "PATH=%s:%s%s\0", BINDIR, BINPATH, oldvar) ;

    if (putenv(termvar) != 0 || putenv(pathvar) != 0)
    {
	perror("(setuserenviron) putenv") ;
	return (-1) ;
    }

    if ( (g = (char *)getenv("VISUAL")) == NULL )  /* use default editor */
      strcpy( (editor = (char *)malloc(strlen(EDITOR) + 1)), EDITOR ) ;

    else					/* use user environ */
	strcpy( (editor = (char *)malloc(strlen(g) + 1)), g ) ;

    if (debug)
      fprintf (stderr, "New TERM is '%s'\n    PATH is '%s'\n    Editor= %s\n",
	       getenv("TERM"), getenv("PATH"), editor) ;

}		/* end setuserenv() */


/*+ 		check_terminal
** FUNCTION:
** 	Check the supplied terminal string against the arrays
** 'good_term' and 'bad_terminals'. Sets the value of the terminal type.
** Return 1 if ok, -1 if error.
**/
check_terminal(ptermtype)
     char *ptermtype ;
{
    register char **p ;
    char bp[1024];		/* to extract 'tgetent' entry  */
    int tries = 2 , badterm = 1 ;

    for ( p = good_terminals ; **p != NULL ; ++p )
      if (strcmp (*p, ptermtype) == 0)
	return(1) ;

    while (tries--)
    {				
	fprintf (stderr, "Searching for terminal type '%s'...", ptermtype);

	if ( tgetent(bp, ptermtype)  == 1 )
	{
	    badterm = 0 ;			/* Okay terminal type */
	    fprintf(stderr, "\n") ;
	    /*
	     * Now check against unacceptable terminals
	     */
	    for ( p = bad_terminals ; **p != NULL ; ++p )
	      if (strcmp (*p, ptermtype) == 0)
	      {
		  badterm = 1 ;
		  break ;
	      }
	}
	else				/* Couldn't get term entry */
	{
	    badterm = 1 ;
	    fprintf(stderr, "not found\n") ;
	}

	if (!badterm)
	  return (1) ;

	if (tries)
	  strncpy(ptermtype, get_reply("Terminal type", "vt100",
				      C_ALPHA|C_DIGIT|C_PUNCT), MAXLINE -1);
	else
	  break ;

    }		/* end while() */

    strcpy(ptermtype, "dumb") ;		/* set terminal type to dumb	*/
    fprintf(stderr, "Setting terminal type to '%s'\n", ptermtype) ;
    sleep (2) ;
    if ( tgetent(bp, ptermtype) <= 0 )		/* can't even find dumb */
    {
	fprintf(stderr, "(%s) ERROR: tgetent cannot find entry for '%s'\n",
		prognm, ptermtype);
	return (-1) ;				/* error */
    }
    else
      return(1);
}		/* end check_terminal */


/*+ 		path_qualify_files
** FUNCTION:
** 	Append the respective path to the filenames. These char ptrs
** are then used in the other modules (they use fully qualified paths)
**/
path_qualify_files()
{
    /*
     * Make sure that the necessary files are fully qualified with path names
     */
    Initialsfile = tackon_dir(ETCDIR, INITIALSFILE) ;
    Indexfile = tackon_dir(ETCDIR, INDEXFILE) ;
    Netloghdr = tackon_dir(ETCDIR, NETLOGHDR) ;
    Opentickets = tackon_dir(ETCDIR, OPENTICKETS) ;
    Opentickethdrs = tackon_dir(ETCDIR, OPENTICKETHDRS) ;
    Prevtkt = tackon_dir(ETCDIR, PREVTKT) ;
    Nroffhdr = tackon_dir(ETCDIR, NROFFHDR) ;

    Otlock = tackon_dir(LOCKDIR, OTLOCK) ;
    Othlock = tackon_dir(LOCKDIR, OTHLOCK) ;
    Ptlock = tackon_dir(LOCKDIR, PTLOCK) ;
    Loglock = tackon_dir(LOCKDIR, LOGLOCK) ;


}	/* end: path_qualify_files() */
