/*
** $Header: /noc/network/netlog/src/RCS/indexing.c,v 2.2 1992/06/01 19:36:50 aggarwal Exp $
*/

/* Copyright 1992 JvNCnet, Princeton University */

/*+ 
 ** This module contains functions that are related to the index file -
 ** Creating index from list of files, adding new entry in index file
 ** and deleting an entry from the index file, parsing the index file
 ** and other lil goodies.
 ** 
 ** STRUCTURE OF INDEX FILE
 **
 **	ticket#  num_of_dates date[MAXDATES]
 **
 ** Note that the index file can contain upto a maximum to MAXDATES
 ** number of netlog dates. A record of the total dates that entries
 ** were made regarding a ticket is stored in num_of_dates - thus
 ** is would be easy to check if there are any entries not listed in
 ** the index file and must be searched for manually AFTER the last
 ** file entry.
 **
 ** Using advantage of the fact that a file in UNIX can be 'fseeked' to
 ** beyond its end (effectively leaving a black hole). To get info
 ** about ticket X, simply seek the record at X * sizeof(index_entry).
 ** This method is simple, and hopefully fast.
 **
 ** When a record is being updated, the particular section is locked
 ** and another process will be blocked while this is being updated.
 **
 ** Can supply it the name of a file (which will be created if necessary)
 ** or else an OPEN file descriptor for the index file (in the latter case
 ** set the index filename to NULL).
 **
 ** Does NOT do a sorted insert of the entry.
 **
 ** A word of caution - if changing the index struct, rebuild index !
 **
 ** RETURN VALUES
 **
 **	If any error writing to the index file, a value of -1 is 
 ** returned. You should then rebuild the index file manually using
 ** the function "build_index_file()".
 **
 ** Have fun !!
 **/

/*
 *
 *	$Log: indexing.c,v $
 * Revision 2.2  1992/06/01  19:36:50  aggarwal
 * Small corrections in the 'fprintf' and 'perror' statements.
 *
 * Revision 2.1  1992/05/11  23:39:28  aggarwal
 * Moved tackon_dir part to main(). Also initialized sttkt and endtkt to
 * zero.
 *
 * Revision 2.0  1992/05/10  16:28:26  aggarwal
 * Cleaned up and restructured for releasing 'netlog v2.0'
 *
 *
 * Revision 1.1  90/03/01  14:31:20  aggarwal
 * Initial revision
 * 
 */

/*  */
#ifndef lint
  static char rcsid[] = "$RCSfile: indexing.c,v $ $Revision: 2.2 $ $Date: 1992/06/01 19:36:50 $" ;
#endif


#include "netlog.h"
#include <fcntl.h>			/* for file locking etc.	*/

/*
 * The egrep reg exp to extract the ticket number from the log files.
 * Assuming that the ticket number will be in the file as:
 *	<TAB> #124 <SPACE>
 *
 * The tkt number should be right after the '#' mark.
 */
#define TKT_EGREP_FORMAT  "'\t#[0-9]+ :'"


/*+ 
** FUNCTION:
** 	Given a ticket number, date, index filename and index file
** descriptor, it updates the index file with the new entry. If the
** filename is NULL, it tries using the supplied file descriptor instead.
**
** If number of 'dates' for the ticket is more than MAXTKTDATES, then it
** simply updates the number of entries for that ticket.
**/
add_index_entry(tktnum, entrydate, indexfile, indexfd)
     int	tktnum ;
     int	indexfd, entrydate;		/* open index file desc	*/
     char 	*indexfile;			/* name of index file	*/
{
    struct index_entry index ;
    register int indexsz = sizeof(index) ;	/* size of index entry	*/
    int fd, nread, ticketnum ;			/* file desc+read bytes	*/
#ifdef NFS
    struct flock indexlk ;			/* to lock the record	*/
#endif

    bzero((char *)&index, indexsz);
    if (indexfile == NULL || *indexfile == NULL)/* try using given fd	*/
      fd = dup(indexfd) ;			/* dup to allow close()	*/
    else
    {
	if ((fd = open (indexfile, O_RDWR | O_CREAT, 0666)) < 0)
	{
	    Fprintf("ERROR (add_index_entry) ");
	    perror(indexfile);
	    return (-1) ;
	}
    }						/* end outer else	*/
    ticketnum = tktnum % MAXTKT ;		/* else runaway offset	*/
    if (lseek(fd, (off_t)INDEXOFFSET(ticketnum), L_SET) < 0)
    {					/* protect against bad indexfd	*/
	perror("add_index_entry (lseek)");
	return (-1);
    }
#ifdef NFS
    indexlk.l_type = F_WRLCK;
    indexlk.l_whence = 1 ;			/* from current offset	*/
    indexlk.l_start = 0 ;			/* start from current	*/
    indexlk.l_len = indexsz ;			/* len of lock		*/
    indexlk.l_pid = 0 ;				/* Not used at all	*/
    fcntl (fd, F_SETLKW, &indexlk) ;		/* wait if record lcked	*/
#else
    flock(fd, LOCK_EX);				/* lock entire file	*/
#endif

    if (read (fd, &index, indexsz) < 0 )
    {
	perror("(add_index_entry) read") ;
	close(fd);
	return (-1) ;
    }
    index.tnum = ticketnum ;

    /*
    ** If number of dates is more than MAXTKTDATES, then only increase
    ** the number of dates (index.numdate). Else..
    ** Make sure that the date has not been entered already. Else make
    ** no changes and simply write the index record back to file.
    */
    if (unique_date(&index, entrydate))
    {
	if (index.numdate == MAXTKTDATES)
	{
	    index.numdate++ ;		/* just increment total, dont add */
	    fprintf(stderr, 
		    "(add_index_entry): Warning, total dates %d for tkt #%d exceeds MAXTKTDATES (%d)\n",
		    index.numdate, ticketnum, MAXTKTDATES);
	}
	else
	  index.edate[index.numdate++] =  entrydate;
    }

    if (index.numdate >= MAXTKTDATES)
    {
	Fprintf("Please open a new ticket for this problem, since number ");
	fprintf(stderr, "of entered dates (%d) for '#%d' has reached MAX\n",
		index.numdate, ticketnum) ;
    }

    lseek(fd, (off_t)INDEXOFFSET(ticketnum), L_SET);	/* rewind the file */
    if ((write (fd, &index, indexsz) < 0))
    {
	fprintf (stderr, 
		 "add_index_entry: Fatal error writing to index file\n");
	perror("write");
	return (-1);
    }
    fsync(fd);
    close(fd);				/* auto releases all locks	*/

}		/* end: add_index_entry	*/

/*+ 			unique_date
** FUNCTION:
** 	Checks to make sure that new 'entrydate' is not already
** present in the index structure (avoid duplication). Returns a '0'
** if not unique, else returns a 1
**/
unique_date(pindex, entrydate)
     int entrydate;
     struct index_entry *pindex;
{
    register int i;

    for (i = 0; i < Numentries(pindex->numdate) ; ++i)
      if (pindex->edate[i] == entrydate)
	return(0);

    return(1);
}						/* end: unique_date	*/

/*+ 		sort_index_file
** FUNCTION:
** 	This functions sorts out all the entries of the index file
** based on increasing order of the log dates (remember that the log
** dates are of type integer. Reads in an entire block and locks the
** entire block while sorting.
**/

sort_index_file(indexfile)
     char *indexfile ;				/* name of file to sort	*/
{
    register char *pindex ;			/* pointer to the index	*/
    register int indexsz = sizeof(struct index_entry) ;
    char buf[BUFSIZ] ;		/* to read in the file's data	*/
    int nread, n, fd;
#ifdef NFS
    struct flock buflk ;		/* buffer lock on the file	*/
#endif
    
    if (indexfile == NULL || *indexfile == NULL)
      indexfile = get_reply("Index file to sort", (char *)NULL,
			      C_ALPHA | C_DIGIT | C_PUNCT) ;

    if ((fd = open (indexfile, O_RDWR)) < 0)
    {
	fprintf(stderr,"sort_index_file: %s ", indexfile);
	perror("open");
	return (-1);
    }

#ifdef NFS
    /*
    ** set up the lock structure to lock record from the present location
    */
    buflk.l_type = F_WRLCK;
    buflk.l_whence = 1 ;			/* from current offset	*/
    buflk.l_start = 0 ;				/* start from current	*/
    buflk.l_len = BUFSIZ ;			/* lock len (full buff)	*/
    buflk.l_pid = 0 ;				/* Not used at all	*/
#else
    flock(fd, LOCK_EX);				/* Lock entire file	*/
#endif
    for ( ; ; )
    {
#ifdef NFS	
	fcntl (fd, F_SETLKW, &buflk) ;		/* wait if locked	*/
#endif
	nread = read(fd, buf, BUFSIZ);		/* BUFSIZ in stdio.h	*/
	if (nread < indexsz)			/* not enuf file or eof	*/
	  break ;
	pindex = buf , n = nread ;		/* initial values	*/
	while ((nread = nread - indexsz) >= 0)	/* enuf data in buffer	*/
	{
	    sort_index(pindex);
	    pindex = pindex + indexsz ;		/* move to next i entry	*/
	}
	lseek (fd, (off_t)(-n) ,L_INCR);	/* rewind file pointer	*/
	write (fd, buf, pindex - buf);	      	/* write out sorted buf	*/
	fsync (fd);
#ifdef NFS
	fcntl (fd, F_UNLCK, &buflk) ;		/* unlock locked part	*/
#endif
    }						/* end: endless for	*/
    close (fd);
    return (0);
}						/* end sort_index_file	*/

/*+ 
** FUNCTION:
** 	Sorts just one index file entry:
**		tkt-number:	date date date ...
**/
sort_index(pindex)				/* sort an index_entry	*/
     struct index_entry *pindex;
{
    int	 compar_index_entry();			/* comparison routine	*/

    qsort( (char *)&(pindex->edate[0]),
	  Numentries(pindex->numdate), 
	  sizeof(pindex->edate[0]), 
	  compar_index_entry);
}					/* end: sort_index	*/

compar_index_entry(pa, pb)		/* Compare two index elements	*/
     int *pa, *pb;			/* ptrs to elements compared	*/
{
    if (*pa == *pb)
      return (0);
    else if (*pa > *pb)
      return (1);
    else if (*pa < *pb)
      return (-1);
}					/* end:  compar_index_entry	*/
    

/*+		build_new_index_file 
** FUNCTION:
** 	This function builds the entire index file from the input
** log files provided. The input files and the range of ticket numbers
** that are to be modified is required. 
** The destination index file is truncated and locked until the update
** is made - this ensures that any new entry being made will have to wait
** till the new indexfile is updated, and the new entry will be made in 
** the new file.
**
**/
build_new_index_file(ac, av)
     int ac;
     char **av;				/* list of all input log files	*/
{
    char temp[MAXLINE], egrepcmd[MAXLINE] ;
    char *reply;
    int sttkt, endtkt ;			/* start & end ticket numbers	*/
    int entrydate, ticketnum, indexfd ;
    FILE  *egreppipe ;			/* pipe for popen() call	*/

    if (ac <= 0)			/* no input log files	*/
    {
	fprintf (stderr,"build_new_index_file: No input log files\n");
	return (-1);
    }

 redo:
    reply = get_reply("Enter starting ticket number", "0", C_DIGIT);
    sscanf(reply, "%d", &sttkt);
    sprintf(temp,"%d\0", MAXTKT) ;		/* for default response	*/
    reply = get_reply("Enter ending ticket number", temp, C_DIGIT);
    sscanf(reply, "%d", &endtkt);

    /*
    ** Do some input checks
    */
    if ( sttkt > endtkt )			/* screwed up values	*/
    {
	printf ("Error: starting ticket %d is greater than ending %d\n",
		sttkt, endtkt) ;
	goto redo ;
    }
    reply = get_reply("Enter new index file name", (char *)NULL,
		      C_ALPHA | C_DIGIT | C_PUNCT) ;

    if (access(reply, F_OK) != -1)
      printf("Warning - file already exists and will be overwritten\n");
    printf("Starting tkt: %d\nEnding tkt: %d\nNew index file %s [confirm]: ",
	   sttkt, endtkt, reply);
    gets(temp);
    if (*temp != NULL)
      goto redo ;

    if ((indexfd = open (reply, O_CREAT | O_TRUNC | O_RDWR, 0666)) < 0)
    {
	fprintf(stderr, "Error creating indexfile ") ;
	perror(reply);
	return (-1);
    }
    /*
    ** For each input file, search for a ticket number and insert
    ** in the index_file.
    */
    while (ac--)
    {
	if (access(*av++, R_OK) != 0)		/* check read access	*/
	{
	    fprintf (stderr, "(build_new_index) Ignoring file '%s'\n", av[-1]);
	    perror ("access");
	    continue ;			/* next file	*/
	}

	entrydate = getfiledate(av[-1]);		/* strip header	*/
	if ((entrydate < 10185) || (entrydate > 123199))
	{
	    fprintf(stderr, 
		    "(build_index) Hmm, this is an interesting date %06d\n",
		    entrydate);
	    fprintf(stderr, "\t -Ignoring file %s\n", av[-1]);
	    continue ;
	}
	if (debug)
	  printf(" %s", av[-1]) ;
	sprintf(egrepcmd, "%s %s %s", EGREP, TKT_EGREP_FORMAT, av[-1]);
	egreppipe = popen (egrepcmd, "r");	/* grep ticket# in file	*/
	while (fgets(temp, MAXLINE, egreppipe) != NULL)
	{
	    sscanf(temp, "%*s #%d ", &ticketnum);/* grab ticket number	*/
	    if (add_index_entry(ticketnum, entrydate, (char *)NULL, indexfd) == -1)
	    {
		fprintf(stderr,
			"build_new_index: Fatal error while creating index file\n");
		return(-1);
	    }
	}					/* end: while fgets()	*/
	if (pclose(egreppipe) == 2)
	  fprintf(stderr, 
		  "build_new_index: WARNING '%s' return value 2 for file netlog.%d\n", 
		  EGREP, entrydate);

	putchar('.'), fflush(stdout);		/* To show that alive	*/
    }						/* end:  while ac	*/
    close (indexfd) ;
    fprintf(stderr, "\nbuild_new_index: Now sorting index file...");
    fflush(stdout);
    if (sort_index_file (reply) == -1)
    {
	fprintf(stderr, "Screwed up\n");
	return (-1);
    }
    fprintf(stderr, "Done\n");
    fprintf(stderr, "\nDone creating new index file, remember to update %s\n",
	    Prevtkt) ;
    return (0);
}						/* end: build_new_index	*/


/*+		update_index_file
** FUNCTION:
** 	This function simply updates an existing index file. It is the
** same as build_index_file except that it doesn't truncate the file 
** while opening it.
**
**/
update_index_file(ac, av)
     int ac;
     char **av;				/* list of all input log files	*/
{
    char temp[MAXLINE], egrepcmd[MAXLINE] ;
    char *reply;
    int sttkt, endtkt ;			/* start & end ticket numbers	*/
    int entrydate, ticketnum, indexfd ;
    FILE  *egreppipe ;			/* pipe for popen() call	*/

    if (ac <= 0)			/* no input log files	*/
    {
	fprintf (stderr,"update_index_file: No input log files\n");
	return (-1);
    }

 redo:
    reply = get_reply("Enter starting ticket number", "0", C_DIGIT);
    sscanf(reply, "%d", &sttkt);
    sprintf(temp,"%d\0", MAXTKT) ;		/* for default response	*/
    reply = get_reply("Enter ending ticket number", temp, C_DIGIT);
    sscanf(reply, "%d", &endtkt);

    /*
    ** Do some input checks
    */
    if ( sttkt > endtkt )			/* screwed up values	*/
    {
	printf ("Error: starting ticket %d is greater than ending %d\n",
		sttkt, endtkt) ;
	goto redo ;
    }
    reply = get_reply("Enter index file name to update", Indexfile,
		      C_ALPHA | C_DIGIT | C_PUNCT) ;

    if (access(reply, W_OK) == -1)
    {
	perror(reply);
	goto redo;
    }
    printf("Starting tkt: %d\nEnding tkt: %d\nIndex file %s [confirm]: ",
	   sttkt, endtkt, reply);
    gets(temp);
    if (*temp != NULL)
      goto redo ;

    if ((indexfd = open (reply, O_RDWR)) < 0)
    {
	perror(reply);
	return (-1);
    }
    /*
    ** For each input file, search for a ticket number and insert
    ** in the index_file.
    */
    while (ac--)
    {
	if (access(*av++, R_OK) != 0)		/* check read access	*/
	{
	    fprintf (stderr, "(update_index) Ignoring file '%s'\n", av[-1]);
	    perror ("access");
	    continue ;			/* next file	*/
	}

	entrydate = getfiledate(av[-1]);		/* strip header	*/
	if ((entrydate < 10185) || (entrydate > 123199))
	{
	    fprintf(stderr, 
		    "(update_index) Hmm, this is an interesting date %06d\n",
		    entrydate);
	    fprintf(stderr, "\t -Ignoring file %s\n", av[-1]);
	    continue ;
	}
	if (debug)
	  printf(" %s", av[-1]) ;
	sprintf(egrepcmd, "%s %s %s", EGREP, TKT_EGREP_FORMAT, av[-1]);
	egreppipe = popen (egrepcmd, "r");	/* grep ticket# in file	*/
	while (fgets(temp, MAXLINE, egreppipe) != NULL)
	{
	    sscanf(temp, "%*s #%d ", &ticketnum);/* grap ticket number	*/
	    if (add_index_entry(ticketnum, entrydate, (char *)NULL, indexfd) == -1)
	    {
		fprintf(stderr,
			"update_index: Fatal error in add_index_entry\n");
		close(indexfd);
		return(-1);
	    }
	}					/* end: while fgets()	*/
	if (pclose(egreppipe) == 2)
	  fprintf(stderr, 
		  "(update_index) WARNING '%s' return value 2 for file netlog.%d\n", 
		  EGREP, entrydate);

	putchar('.'), fflush(stdout);		/* To show that alive	*/
    }						/* end:  while ac	*/
    close (indexfd) ;
    printf("\nupdate_index: Now sorting index file...");
    fflush(stdout);
    if (sort_index_file (reply) == -1)
    {
	printf("Screwed up\n");
	return (-1);
    }
    printf("Done\n");
    printf("\nDone updating index file  %s, remember to update %s\n", 
	   reply, Prevtkt) ;
    return (0);
}			/* end: update_index	*/


/*+		getfiledate 
** FUNCTION:
** Strips the header from the name and returns 'date'. Starts from the
** end of the string and stop when it encounters the first '.'
**/

getfiledate(file)
     char *file;			/* filename as '/dira/netlog.010390' */
{
    register char *p ;
    int entrydate ;

    p = file + strlen(file);		/* work backwards to get a '.'	*/
    while (*p != '.' && p != file)
      --p;
    sscanf(p, ".%d", &entrydate);
    return (entrydate);
}			/* end: getfiledate()  */


/*+ 		print_index_file
** FUNCTION:
** 	Print out the entire index file (except bogus entries)  or a
** specific entry for a ticket number.
**/

print_index_file(indexf)
     char *indexf ;
{
    int fd, indexsz ; 
    int sttkt =0, endtkt =0;			/* star & end ticket	*/
    char *outfile, *indexfile, *reply, temp[MAXLINE] ;
    FILE *fp, *lasttktfp;			/* Output file pointer	*/
    struct index_entry index ;
    register struct index_entry *pindex = &index ;

    indexsz = sizeof(index);

    if (indexf == NULL || *indexf == NULL)
      indexfile = get_reply("Enter index file name", Indexfile,
			    C_ALPHA | C_DIGIT | C_PUNCT) ;
    else
      indexfile = indexf ;			/* Supplied filename */

    if ((fd = open(indexfile, O_RDONLY)) < 0)
    {
	Fprintf("(print_index_file) open ");
	perror(indexfile);
	return(-1);
    }

 redop:

    if ((lasttktfp = fopen(Prevtkt, "r")) == NULL)
    {						/* get last-ticket	*/
	Fprintf("(print_index_file) fopen ");
	perror(Prevtkt) ;
	endtkt = MAXTKT ;
    }
    else
    {
	fscanf(lasttktfp, "%d", &endtkt) ;
	fclose(lasttktfp);
    }

    reply = get_reply("Enter starting ticket number", "0", C_DIGIT);
    sscanf(reply, "%d", &sttkt);
    sprintf(temp,"%d\0", endtkt) ;		/* for default response	*/
    reply = get_reply("Enter ending ticket number", temp, C_DIGIT);
    sscanf(reply, "%d", &endtkt);
    /*
    ** Do some input checks
    */
    if ( sttkt > endtkt )			/* screwed up values	*/
    {
	printf ("Error: starting ticket %d is greater than ending %d\n",
		sttkt, endtkt) ;
	goto redop ;
    }
    outfile = get_reply("Enter output file name", "/dev/tty",
			C_ALPHA | C_DIGIT | C_PUNCT) ;
    printf("Starting tkt: %d\nEnding tkt: %d\nOutput file= %s [confirm]: ",
	   sttkt, endtkt, outfile);
    gets(temp);
    if (*temp != NULL)
      goto redop ;
    if ((fp = fopen(outfile, "w")) == NULL)	/* open the output file	*/
    {
	Fprintf("(print_index_file) fopen") ;
	perror(outfile);
	return(-1);
    }

    while (read(fd, pindex, indexsz) == indexsz)
      if (INDEXOFFSET(pindex->tnum) == (tell(fd) - indexsz)) /* no junk	*/
	if (pindex->tnum >= sttkt && pindex->tnum <= endtkt)
	{
	    register int i;

	    fprintf(fp,"\nTicket %d, number of entries = %d\n\t(Log dates) ",
		    pindex->tnum, pindex->numdate);
	    for (i = 0; i < Numentries(pindex->numdate); ++i)
	      fprintf(fp, "%06d  ", pindex->edate[i]);
	}	
    fprintf(fp, "\n");
    fclose(fp);				/* output file stream		*/
    close(fd);			     	/* input file descriptor	*/
}		/* end:   print_index_file	*/


/*+ 		print_ticket_entry
** FUNCTION:
** 	Prints out all information in index file for a given ticket
** number.
**/
print_ticket_entry(ticket, file)
     int ticket;
     char *file ;				/* The index file	*/
{
    struct index_entry index ;
    int fd, i, tkt, indexsz = sizeof(index) ;

    bzero(&index, indexsz);
    tkt = ticket % MAXTKT ;
    if (file == NULL || *file == NULL)
      file = get_reply("Enter index file name", (char *)NULL,
		       C_ALPHA | C_DIGIT | C_PUNCT) ;
    if ((fd = open (file, O_RDONLY)) < 0)
    {
	Fprintf("print_ticket_entry - ") ;
	perror(file);
	return (-1);
    }
    if (lseek(fd, (off_t)INDEXOFFSET(tkt), L_SET) < 0)
    {
	perror("print_ticket (lseek)");
	close (fd);
	return(-1);
    }
    if (read (fd, &index, indexsz) != indexsz)
    {
	close (fd);
	return (0) ;				/* Nothing to print	*/
    }
    printf("\nTicket #%d, number of entries = %d\n\t(Log dates) ",
	   index.tnum, index.numdate);
    for (i=0 ; i < Numentries(index.numdate) ; ++i)
      printf("%06d  ", index.edate[i]);
    printf("\n");
    fflush(stdout) ;
    close (fd);
    return (0) ;
}					/* end:  print_ticket_entry	*/

/*+ 		       	delete_index_entry
** FUNCTION:
** 	Given a ticket number and a index file zeroes the entry. If
** file is a NULL pointer, then uses open descriptor - fd.
**/
delete_index_entry(ticket,file,fd)
     int ticket, fd ;
     char *file;
{
    struct index_entry index;
    int newfd, tkt, indexsz = sizeof(index);
#ifdef NFS
    struct flock indexlk ;		/* to lock the record	*/
#endif

    if (file == NULL || *file == NULL)
      newfd = dup(fd);
    else
      newfd = open(file, O_RDWR);

    if (fd < 0)
    {
	perror("delete_index_entry");
	return(-1);
    }
    bzero((char *)&index, indexsz);
    tkt = ticket % MAXTKT ;
    index.tnum = tkt ;
    if (lseek(fd, (off_t)INDEXOFFSET(tkt), L_SET) < 0)
    {					/* protect against bad indexfd	*/
	perror("delete_index_entry");
	return (-1);
    }
#ifdef NFS
    indexlk.l_type = F_WRLCK;
    indexlk.l_whence = 1 ;			/* from current offset	*/
    indexlk.l_start = 0 ;			/* start from current	*/
    indexlk.l_len = indexsz ;			/* len of lock		*/
    indexlk.l_pid = 0 ;				/* Not used at all	*/
    fcntl (newfd, F_SETLKW, &indexlk) ;		/* wait if locked	*/
#else
    flock (newfd, LOCK_EX);			/* on BSD lock all file	*/
#endif
    if (write(newfd, &index, indexsz) < 0)
    {
	fprintf (stderr, 
		 "delete_index_entry: Fatal error writing to index file\n");
	perror("delete_index_entry");
	return (-1);
    }
    fsync(newfd);
    close(newfd);			/* auto releases all locks	*/

}			/* end delete_entry	*/
