/**** Tektronix interface for network map program.  ****/

/*	Copyright (C) 1989, 1990 Massachusetts Institute of Technology
 *		Right to copy granted under the terms described
 *		in the file Copyright accompanying this distribution.
 */


/* This is the Tektronix interface for the program that displays network
 * connectivity.
 */

#include "map.h"
#include "debug.h"


#ifdef EDIT_HISTORY

Nbr Date	Author		Description
--- ----	------		-----------
 3  20-Feb-89	M. A. Patton	First release version.
 3A  4-Mar-89	M. A. Patton	Made more independant of display stuff.
				No longer references any Vertex structures.
 4  20-Mar-89	M. A. Patton	Added pxl2rgb to remove X from TEK module.
				(From report by P. V. O'Neill)
 4A 23-Mar-89	M. A. Patton	Cosmetics from report by Chris Tengi.
 4B 31-Mar-89	M. A. Patton	Parameterized viewport, adjusted slightly.
 4C  3-Apr-89	M. A. Patton	Can't seem to get viewport to do what I want,
				fix "manually".
 4D 29-Jun-89	M. A. Patton	X calculated label pos is LL not UL now.
 4E 11-Jul-89	M. A. Patton	Unfortunately TEK wants background on 0 and
				foreground on 1, so I have to cons up a mapping
				from X pixel values to TEK color indexes.
 4F 10-Feb-90	M. A. Patton	Removed some extraneous variables.
 5   9-Aug-90	M. A. Patton	Check AlreadyPlaced flags in case a redisplay
				hasn't happened (tty mode or load/write in a
				take file).
 5A 16-Apr-91	M. A. Patton	Added support for point-to-point nets
 5B 18-Apr-91	Mike Patton	Fixed sign bug in above.

#endif

		/* These should all be computed!! */
static int TEKx0 = 819;
static int TEKy0 = 417;
static int TEKx1 = 3378;
static int TEKy1 = 3130;
static int TEKdx = 60;
static int TEKdy = 100;
static int TEKscale = 6;
#define CHAR_SCALE 2/3
static int TEKchar_height = 59 * CHAR_SCALE;
static int TEKchar_width = 39 * CHAR_SCALE;
static int TEKchar_space = 12 * CHAR_SCALE;
#undef CHAR_SCALE


extern int foreground, background, IP_pixel, CHAOS_pixel;
/**** Tektronix output ****/

typedef struct _TEKfile
{   FILE *fd;
    int x_pos;
} TEKfile;

HardcopyTEK(fname)
char *fname;
{   register Network *np;
    register Machine *hp;
    register Title *tp;
    TEKfile *file, *TEKopen();

    if ( (file=TEKopen(fname)) == NULL)
    {	fprintf(stderr,"Can't open Tektronix file %s.\n",fname);
	return;
    }
    clear_color_map();
    /* Set the basic parameters. */
    TEKputESC(file,"JL6M1:PWR"); /* Reset to initial conditions */
    TEKputESC(file,"UX00");	/* 12-bit XY coords */
    TEKputESC(file,"RE0");	/* ???? Invisible border */
    TEKputESC(file,"MQ2");	/* Stroke precision (scalable) characters */
    /* Now set up size and shape parameters */
    TEKputESC(file,"QO0");	/* Landscape mode orientation */
/*  Old PS scale stuff, should calculate TEKscale here!!!!
    TEKputint(file,display_height);
    TEKputword(file,"div");
    TEKputword(file,"exch");
    TEKputint(file,display_width);
    TEKputstr(file," div min dup scale pop pop\n");
*/
/*  TEKputESC(file,"RV#gh&L8jn:L"); /* Old Viewport setup */
    /* Viewport setup */
    TEKputESC(file,"RV");
    TEKputcoord(file, TEKx0, TEKy0);
    TEKputcoord(file, TEKx1, TEKy1);
    /* Set GraphText Size */
    TEKputESC(file,"MC");TEKputint(file,TEKchar_width);
	TEKputint(file,TEKchar_height);TEKputint(file,TEKchar_space);
#ifndef BROKEN_TEK_COLOR
    /* Text is foreground */
    TEKputESC(file,"MT");TEKputint(file,TEK_color(foreground));
    /* Line's are too */
    TEKputESC(file,"ML");TEKputint(file,TEK_color(foreground));
#endif
    /* Now clear page and start printing stuff. */
    TEKputESC(file,"\014");
    TEKbreak(file);
    MAPCAR (np,net_base)
	TEK_DrawNetwork(file,np);
    MAPCAR (hp,machine_base)
	TEK_DrawMachine(file,hp);
    MAPCAR (tp,title_base)
	TEK_DrawTitle(file,tp);
    TEKbreak(file);
    dump_color_map(file);
    TEKbreak(file);
    TEKputESC(file,"KH2\n");
    TEKclose(file);
}
/****	Color map manipulation	****/



static short color_map[256];	/* Maps X pixel to TEK index */
static unsigned char r[256], g[256], b[256]; /* Actual RGB for TEK */
static short highest_index;	/* Higest index used in RGB tables */


static
clear_color_map()
{   int i;

    for (i=0;i<256;i++)
	color_map[i] = -1;
    highest_index = 0;
    if (DBG_COLOR_P)
	fprintf(stderr,"TEK color map cleared.\n");
    if (TEK_color(background) != 0)
	error("TEK background came out wrong index.\n");
    /* !!!! Hack for now we force background white!!!! */
    r[0]=255;g[0]=255;b[0]=255;			/**/
    if (TEK_color(foreground) != 1)
	error("TEK foreground came out wrong index.\n");
}


static
dump_color_map(file)
FILE *file;
{   int i;

    if (DBG_COLOR_P)
	fprintf(stderr,"dumping TEK color map.\n");
    /* Now set up color parameters */
    TEKputESC(file,"TM411");	/* Color mode 8-bit RGB, Opaque */
    TEKputESC(file,"TG1");	/* Color map for surface 1 */
    TEKputint(file,highest_index*4);
    for (i=0; i<highest_index; i++)
    {	if (DBG_COLOR_P)
	    fprintf(stderr,"index %d maps to (%d, %d, %d).\n",
				i, r[i], g[i], b[i] );
	TEKputint(file,i);
	TEKputint(file,r[i]);
	TEKputint(file,g[i]);
	TEKputint(file,b[i]);
    }
}



int
TEK_color(val)
int val;
{   int rd, gn, bl;

    if (color_map[val] != -1) return (color_map[val]);
    if (!pxl2rgb(val, &rd, &gn, &bl))
    {   fprintf(stderr, "Can't get X color for pixel value %d.\n", val);
	return (-1);
    }
    color_map[val] = highest_index;
    r[highest_index] = rd>>8;
    g[highest_index] = gn>>8;
    b[highest_index] = bl>>8;
    return(highest_index++);
}
/****	Top level for each type	****/


TEK_DrawNetwork(file,np)
TEKfile *file;
Network *np;
{   if (! (np->geo_flag&AlreadyPlaced) )
	PlaceNetwork(np);
    if (! (np->geo_flag&AlreadyPlaced) )
	return;
    if (np->geo_flag&P_P_NET)
	TEK_DrawP2Pnetwork(file,np);
    else
	TEK_DrawFlatNetwork(file,np);
}


static
TEK_DrawFlatNetwork(file,np)
TEKfile *file;
Network *np;
{   register int drawn = FALSE;
    register AddressList *ap;
    int a, b, t;

    TEKcomment(file,np->name);
    a = np->side1;
    b = np->side2;
    t = 3;
    MAPCAR (ap,np->addresses)
    {   if ( (show_chaos && ap->address.class==AF_CHAOS) ||
	     (show_IP && ap->address.class==AF_INET) )
	{	if (!drawn)
	    {   TEKTextLL(file,np->label_x,np->label_y,np->name);
		if (np->vertical)
		    TEKrect(file,a,np->y1,b,np->y2,TEK_color(np->media->pixel));
		else
		    TEKrect(file,np->x1,a,np->x2,b,TEK_color(np->media->pixel));
		drawn = TRUE;
	    }
	    if (np->vertical)
		TEKrect(file,a-t,np->y1,a,np->y2,
		    (ap->address.class==AF_INET)  ?
			    TEK_color(IP_pixel)  :
			    TEK_color(CHAOS_pixel)
		    );
	    else
		TEKrect(file,np->x1,a-t,np->x2,a,
		    (ap->address.class==AF_INET)  ?
			    TEK_color(IP_pixel)  :
			    TEK_color(CHAOS_pixel)
		    );
	    {int tmp=a-t;a=b;b=tmp;}
	    t = -t;
	}
    }
    if ( !drawn && np->addresses==NULL)
    {   TEKTextLL(file,np->label_x,np->label_y,np->name);
	if (np->vertical)
	    TEKrect(file,a,np->y1,b,np->y2,TEK_color(np->media->pixel));
	else
	    TEKrect(file,np->x1,a,np->x2,b,TEK_color(np->media->pixel));
    }
}



static int
isqrt(n)
int n;
{   if (n<0)
    {	fprintf(stderr, "SQRT approx. out of bounds, val=%d.\n",n);
	return(0);
    }
    if (n==0) return(0);
    if (n<3) return(1);
    if (n<6) return(2);
    if (n<13) return(3);
    if (n<21) return(4);
    if (n<31) return(5);
    if (n<43) return(6);
    if (n<57) return(7);
    if (n<73) return(8);
    if (n<91) return(9);
    fprintf(stderr, "SQRT approx. out of bounds, val=%d.\n",n);
    return(10);
}


static
TEK_DrawP2Pnetwork(file,np)
TEKfile *file;
Network *np;
{   /* for now we only draw the "media" part (and always draw it). */
    /* Special case for horizontal and vertical... */
    if (np->y1==np->y2)
    {	int a = np->y1 - np->width/2;
	int b = a+np->width;
	TEKrect(file,np->x1,a,np->x2,b,TEK_color(np->media->pixel));
    }
    else if (np->x1==np->x2)
    {	int a = np->x1 - np->width/2;
	int b = a+np->width;
	TEKrect(file,a,np->y1,b,np->y2,TEK_color(np->media->pixel));
    }
    else
    {	/* Compute the corners */
	int dx = np->x2-np->x1;
	int dy = np->y2-np->y1;
	int d2 = dx*dx + dy*dy;
	int w2 = np->width*np->width;
	int numy = dx*dx*w2;
	int numx = dy*dy*w2;
	int iy = numy/d2;
	int ix = numx/d2;
	int adjy = isqrt(iy);
	int adjx = isqrt(ix)*(ordered_p(dx,0,dy)*2-1);
	int xa = np->x1-adjx/2, ya = np->y1-adjy/2;
	int xd = xa+adjx, yd = ya+adjy;
	int xb = np->x2-adjx/2, yb = np->y2-adjy/2;
	int xc = xb+adjx, yc = yb+adjy;
	TEKputESC(file, "MP");	/* Select "fill" color */
	TEKputint(file, -TEK_color(np->media->pixel));
	TEKputESC(file, "LP");	/* Start panel */
	TEKputXY(file, xa, ya);
	TEKputint(file, 0);	/* No border */
	TEKputchar(file,29);	/* Enter vector mode */
	TEKputXY(file, xb, yb);
	TEKputXY(file, xc, yc);
	TEKputXY(file, xd, yd);
	TEKputESC(file, "LE");	/* End panel */
    }
}




TEK_DrawMachine(file,hp)
TEKfile *file;
register Machine *hp;
{   register int drawn = FALSE;
    Host *hst = hp->host;
    register AddressList *ap;

    if (! (hst->geo_flag&AlreadyPlaced) )
	PlaceHost(hst);
    TEKcomment(file,hp->name);
    MAPCAR (ap,hst->addresses)
    {   if ( (show_chaos && ap->address.class==AF_CHAOS) ||
	      (show_IP && ap->address.class==AF_INET) )
	{	if (!drawn)
	    {   TEKBox(file, hst->x1, hst->y1, hst->x2, hst->y2);
		TEKTextCTR(file, hst->x_pos, hst->y_pos, hp->name);
		drawn = TRUE;
	    }
	    if (ap->net_p != NULL)
	    {   if (ap->net_p->vertical)
		    if (ap->net_p->x_pos < hst->x_pos)
			TEKTap(file,hst->x1,hst->y_pos,
					ap->net_p->side2,hst->y_pos);
		    else
			TEKTap(file,hst->x2,hst->y_pos,
					ap->net_p->side1,hst->y_pos);
		else
		    if (ap->net_p->y_pos < hst->y_pos)
			TEKTap(file,hst->x_pos,hst->y1,
					hst->x_pos,ap->net_p->side2);
		    else
			TEKTap(file,hst->x_pos,hst->y2,
					hst->x_pos,ap->net_p->side1);
	    }
	}
    }
}


TEK_DrawTitle(file,tp)
TEKfile *file;
register Title *tp;
{   if (! (tp->geo_flag&AlreadyPlaced) )
	PlaceTitle(tp);
    TEKTextLL(file, tp->text_x,tp->text_y,tp->text);
/* Shouldn't reference (WidthValue|HeightValue) here (they're X)
    if ( (WidthValue|HeightValue)&tp->geo_flag)
	TEKrect(file, tp->x1, tp->y1, tp->x2, tp->y2);
 */
}
/****	High level Tektronix operations, draw.	****/


/* Filled in rectangle */
static
TEKrect(fd,x1,y1,x2,y2, color)
TEKfile *fd;
int x1, y1, x2, y2;
int color;			/* From fill-pattern chart */
{   if (fd->x_pos > 50) TEKbreak(fd);
    TEKputESC(fd, "MP");
    TEKputint(fd, -color);
    TEKputESC(fd, "UR2");
    TEKputXY(fd, x1, y1);
    TEKputXY(fd, x2, y2);
}

/* Draw Host to network tap. */
/*  This might be better off using some iconic thing at the end. */
static
TEKTap(fd,x1,y1,x2,y2)
TEKfile *fd;
int x1, y1, x2, y2;
{   TEKputchar(fd,29);		/* Enter vector mode, start new lines */
    TEKputXY(fd, x1, y1);
    TEKputXY(fd, x2, y2);
    TEKputchar(fd,29);		/* Enter vector mode, start new lines */
    if (x1==x2)
    {	TEKputXY(fd, x2-4, y2);
	TEKputXY(fd, x2, y2+ ((y1<y2)?-8:8) );
	TEKputXY(fd, x2+4, y2);
    }
    else
    {	TEKputXY(fd, x2, y2-4);
	TEKputXY(fd, x2+ ((x1<x2)?-8:8), y2 );
	TEKputXY(fd, x2, y2+4);
    }
}

/* Draw a box, given two corners. */
static
TEKBox(fd,x1,y1,x2,y2)
TEKfile *fd;
int x1,y1,x2,y2;
{   TEKputESC(fd, "MP0");	/* Fill with background */
    TEKputESC(fd, "LP");
    TEKputXY(fd, x1, y1);
    TEKputint(fd, 1);
    TEKputchar(fd,29);		/* Enter vector mode, start new lines */
    TEKputXY(fd, x2, y1);
    TEKputXY(fd, x2, y2);
    TEKputXY(fd, x1, y2);
    TEKputESC(fd, "LE");
}
/****	High level Tektronix operations, text.	****/



static
TEKTextLL(fd,x,y,str)
TEKfile *fd;
int x, y;
char *str;
{   TEKputESC(fd,"LF");TEKputXY(fd,x,y/*+TEKchar_height/TEKscale*/); /* Move */
    TEKputESC(fd,"LT");TEKputstr(fd,str); /* Graphic Text */
}


static
TEKTextCTR(fd,x,y,str)
TEKfile *fd;
int x, y;
char *str;
{   int len = strlen(str);

    TEKputESC(fd,"LF");		/* Move */
    TEKputXY(fd,
	     x-(len*(TEKchar_width+TEKchar_space)-TEKchar_space)/(TEKscale*2),
	     y+TEKchar_height/(TEKscale*2));
    TEKputESC(fd,"LT");TEKputstr(fd,str); /* Graphic Text */
}
/****	Low level Tektronix output.	****/



static TEKfile *
TEKopen(cp)
char *cp;
{   TEKfile *val;

    if ( (val=(TEKfile *)malloc(sizeof(TEKfile))) == NULL) return (NULL);
    if ( (val->fd=fopen(cp,"w")) == NULL)
    {	free(val);
	return (NULL);
    }
    val->x_pos = 0;
}

static
TEKclose(f)
TEKfile *f;
{   fclose(f->fd);
    free(f);
}





static
TEKcomment(f,cp)
TEKfile *f;
char *cp;
{   /* Can't do comments in TEK ????
    if (f->x_pos != 0)
	TEKbreak(f);
    TEKputstr(f,"% ");
    TEKputstr(f,cp);
    TEKputchar(f,'\n');
    */
}


static
putint_internal(fd,i)
TEKfile *fd;
int i;
{   char c = 64;

    c += i & 0x3F;
    i >>= 6;
    if (i > 0) putint_internal(fd,i);
    TEKputchar(fd, c);
}



static
TEKputint(fd,i)
TEKfile *fd;
int i;
{   char c = 48;

    if (i<0)
    {	c = 32;
	i = -i;
    }
    c += i & 0xF;
    i >>= 4;
    if (i > 0) putint_internal(fd,i);
    TEKputchar(fd, c);
}



static
TEKputXY(fd,x,y)
TEKfile *fd;
int x, y;
{   x *= TEKscale;
    x += TEKdx;
    if (x<0) x = 0;
    if (x>4095) x = 4095;
    y = display_height - y;
    y *= TEKscale;
    y -= TEKdy;
    if (y<0) y = 0;
    if (y>4095) y = 4095;
    TEKputcoord(fd,x,y);
}

static
TEKputcoord(fd,x,y)
TEKfile *fd;
int x, y;
{   TEKputchar(fd, 0x20 | ((0xf80&y)>>7));
    TEKputchar(fd, 0x60 | ((3&y)<<2) | (3&x));
    TEKputchar(fd, 0x60 | ((0x7c&y)>>2));
    TEKputchar(fd, 0x20 | ((0xf80&x)>>7));
    TEKputchar(fd, 0x40 | ((0x7c&x)>>2));
}



static
TEKputstr(fd,str)
TEKfile *fd;
char *str;
{   TEKputint(fd,strlen(str));
    while (*str)
	TEKputchar(fd,*str++);
}



static
TEKputESC(fd,str)
TEKfile *fd;
char *str;
{   TEKputchar(fd,033);
    while (*str)
	TEKputchar(fd,*str++);
}



static
TEKbreak(f)
TEKfile *f;
{   putc('\n',f->fd);
    f->x_pos = 0;
}

static
TEKputchar(f,c)
TEKfile *f;
char c;
{   putc(c,f->fd);
    if (c == '\n')
	f->x_pos=0;
    else
	f->x_pos++;
}
