/************************************************************************
 *		getneighbors.c	- get ISIS and ASys Neighbors		*
 *									*
 *	Added a -l option to be used to update the LINKS file		*
 ************************************************************************/

#include <stdio.h>
#include <sys/wait.h>
#include <sys/file.h>
#include <sys/types.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <sys/stat.h>
#include <time.h>

#include "CommonDefs.h"
#include "snmptable2.h"
#include "Linkfileio.h"

#define CIRCUITCHECK 1

/* Add max retries =1 since snmp_next does own retries */
char *ISISVariables[]={
		"is-isLocalAddr",
		"is-isNeighState",
		};

char *LinkTableVariables[]={
		"is-isLocalAddr",
		"is-isIndex",
		};
int LinkTableFunction=0;

char *EGPVariables[]={
		"egpNeighState",
		"egpNeighAs",
		};

char *ASysVariables[]={
		"adRouteDest"
		};

char *MIBRoutingTable[]={
		"ipRouteNextHop",
		"ifType",
		"ipAdEntIfIndex"
		};

char *progname;

struct in_addr MySubNet;

struct GenericNodeType {
        char Address[40];               /* Address */
        struct InterfaceType *Interfaces;
        struct NextHopType *Neighbors;
	struct GenericNodeType *Next;
} *GenericNodeHead=NULL;

struct InterfaceType {
        int ifnum;                      /* Interface Index */
        int ifType;                     /* Interface Type */
        char Address[40];               /* RemoteAddress */
        struct InterfaceType *Next;
};

struct NextHopType {
        char RemoteAddress[40];         /* RemoteAddress */
        struct NextHopType *Next;
};

struct ISISType {
	char RespondingAddress[40];	/* IPAddr of responding node */
	char LocalAddress[40];
	char RemoteAddress[40];		/* ISIS-RemoteAddress in the index */
	int State;
	int DSUIndex;			/* Associated Interface # (& DSU #) */
	long DSUSerialNumber;		/* and DSU identifier */
} Links[300];
int NumLinks=0;


struct EGPSessionType {
	char LocalAddress[40];
	char Address[40];
	int SessionState;
	int AnnouncedASys;
	struct EGPSessionType *Next;
} *EGPSessionHead=NULL;

struct ASysType {
	int State;
	int ASysNum;
	struct EGPSessionType *EGPSessions;
	struct ASysType *Next;
} *ASysHead=NULL;

/************* Routines for ISIS and Extended Query information ***********/

/************************************************************************
 *	FindLink() - Given a NodeAddr and Remote IPAddr			*
 *			return index for the link or -1			*
 ************************************************************************/
int FindLink(RespondingAddress, RemoteAddress)
     char *RespondingAddress;
     char *RemoteAddress;
{
int i;
	for(i=0; i<NumLinks; i++)
		if ((strcmp(RemoteAddress,Links[i].RemoteAddress)==0) &&
			(strcmp(RespondingAddress,Links[i].RespondingAddress)==0))
			return(i);
	return(-1);
}

int DSUNum(InterfaceAddress)
     char *InterfaceAddress;
{
int i;

	for(i=0; i<NumLinks; i++)
		if ( strcmp(InterfaceAddress,Links[i].LocalAddress)==0)
			return( Links[i].DSUIndex );
	return( 0 );
}

/************************************************************************
 * AllocNode() - given NodeAddr and RemoteAddr, allocate a Link Struct  *
 *			return index for Link struct			*
 ************************************************************************/
int AllocNode(RespondingAddress, RemoteAddress)
     char *RespondingAddress;
     char *RemoteAddress;
{
	strcpy( Links[NumLinks].RemoteAddress, RemoteAddress );
	strcpy( Links[NumLinks].RespondingAddress,RespondingAddress );
	Links[NumLinks].LocalAddress[0]='\0';
	Links[NumLinks].DSUSerialNumber=0;
	Links[NumLinks].DSUIndex=0;
	Links[NumLinks].State=0;
	return(NumLinks++);
}


/************************************************************************
 * AddISISData() - Add ISIS Data to appropriate Link Structure		*
 *									*
 ************************************************************************/
void AddISISData(RespondingAddress, VarName, RemoteAddress, value)
     char *RespondingAddress;
     char *VarName;
     char *RemoteAddress;
     char *value;		/* Value of Variable	*/
{
int i,j;

	if ( (i=FindLink(RespondingAddress,RemoteAddress))==-1)
		i=AllocNode(RespondingAddress,RemoteAddress);
	for(j=0; j<((sizeof(ISISVariables))/(sizeof(char *))); j++) {
		if (strcmp(ISISVariables[j],VarName)==0) {
			switch( j ) {	
			case 0: /* LocalAddr */
				strcpy(Links[i].RemoteAddress,RemoteAddress);
				strcpy(Links[i].LocalAddress,value);
				return;
				/*NOTREACHED*/
			case 1: /* NeighState */
				strcpy(Links[i].RemoteAddress,RemoteAddress);
				Links[i].State=atoi(value);
				return;
				/*NOTREACHED*/
			case 2: /* DSU Index */
				strcpy(Links[i].RemoteAddress,RemoteAddress);
				Links[i].DSUIndex=atoi(value);
				return;
				/*NOTREACHED*/
			default:
				fprintf(stderr,"Unrecognized Data Type %s\n", 
								VarName);
				return;
				/*NOTREACHED*/
			}
		}
	}
}

/************************************************************************
 * AddLinkTableData() - Add LinkTable Data to appropriate Link Structure          *
 *                                                                      *
 ************************************************************************/
void AddLinkTableData(RespondingAddress, VarName, RemoteAddress, value)
     char *RespondingAddress;
     char *VarName;
     char *RemoteAddress;
     char *value;           /* Value of Variable    */
{
int i,j;

        if ( (i=FindLink(RespondingAddress,RemoteAddress))==-1)
                i=AllocNode(RespondingAddress,RemoteAddress);
        for(j=0; j<((sizeof(LinkTableVariables))/(sizeof(char *))); j++) {
                if (strcmp(LinkTableVariables[j],VarName)==0) {
                        switch( j ) {
                        case 0: /* LocalAddr */
                                strcpy(Links[i].RemoteAddress,RemoteAddress);
                                strcpy(Links[i].LocalAddress,value);
                                return;
                                /*NOTREACHED*/
                        case 1: /* DSU Index */
                                strcpy(Links[i].RemoteAddress,RemoteAddress);
                                Links[i].DSUIndex=atoi(value);
                                return;
                                /*NOTREACHED*/
                        default:
                                fprintf(stderr,"Unrecognized Data Type %s\n",
                                                                VarName);
                                return;
                                /*NOTREACHED*/
                        }
                }
        }
}

/*************************************************************************
 * FormatLinkTableData() - Process each VarList and Binding adding it to the  *
 *                      appropriate LinkTable Data structure.                 *
 *************************************************************************/
void FormatLinkTableData(AddressListHead)
     struct AddressListType *AddressListHead;
{
struct AddressListType *A;
struct VarList *v;
struct bindings *b;

   for( A=AddressListHead; A!=NULL; A=A->Next )
        for( v=A->ResponseListHead; v!=NULL; v=v->Next)
                for( b=v->Head; b!=NULL; b=b->Next )
                         AddLinkTableData(A->Address,v->VarName,b->instance+strlen(v->Prefix),b->value);
}

#ifdef CIRCUITCHECK

/*************************************************************************
 * ProcessDSUSerialNumbers() - Process each VarList and Binding adding it*
 * to the appropriate LinkTable Data structure.                 	 *
 *************************************************************************/
void ProcessDSUSerialNumbers( AddressListHead )
struct AddressListType *AddressListHead;
{
struct AddressListType *A=AddressListHead;
struct VarList *v;
struct bindings *b;

        for( v=AddressListHead->ResponseListHead; v!=NULL; v=v->Next)
                for( b=v->Head; b!=NULL; b=b->Next )
                         UpdateDSUSerialNumbers(A->Address,b->instance+strlen(v->Prefix),b->value);
}

/************************************************************************
 * ProcessLinkType() - process returned linktype tables			*
 ************************************************************************/
ProcessLinkType( AddressListHead )
struct AddressListType *AddressListHead;
{
struct AddressListType *A;
struct VarList *v;
struct bindings *b;

   for( A=AddressListHead; A!=NULL; A=A->Next )
        for( v=A->ResponseListHead; v!=NULL; v=v->Next)
                for( b=v->Head; b!=NULL; b=b->Next )
                         UpdateLinkType( A->Address,
				atoi(b->instance+strlen(v->Prefix)),
				atoi(b->value) );
}
#endif

/*************************************************************************
 * FormatISISData() - Process each VarList and Binding adding it to the  *
 *			appropriate ISIS Data structure.		 *
 *************************************************************************/
void FormatISISData(AddressListHead)
     struct AddressListType *AddressListHead;
{
struct AddressListType *A;
struct VarList *v;
struct bindings *b;

   for( A=AddressListHead; A!=NULL; A=A->Next )
        for( v=A->ResponseListHead; v!=NULL; v=v->Next)
		for( b=v->Head; b!=NULL; b=b->Next )
			 AddISISData(A->Address,v->VarName,b->instance+strlen(v->Prefix),b->value);
}

/************************************************************************
 * AllocEGPSession() - Allocate a EGPSession structure			*
 *			return a pointer to it				*
 ************************************************************************/
struct EGPSessionType *AllocEGPSession(LocalAddress, Address)
     char *LocalAddress;
     char *Address;
{
struct EGPSessionType *E;

	if ((E=(struct EGPSessionType *)malloc(sizeof(struct EGPSessionType)))==NULL)
		exit(fprintf(stderr,"malloc failed\n"));
	strcpy( E->Address, Address );
	strcpy( E->LocalAddress, LocalAddress );
	E->SessionState=0;
	E->AnnouncedASys=0;
	E->Next=NULL;
	return(E);
}


/************************************************************************
 * AddEGPSession() - Add EGP Session to our global list.		*
 *			return a pointer to it				*
 ************************************************************************/
struct EGPSessionType *AddEGPSession(LocalAddress, Address)
     char *LocalAddress;
     char *Address;
{
struct EGPSessionType *E;

        if ( EGPSessionHead == NULL ) E=EGPSessionHead=AllocEGPSession(LocalAddress,Address);
        else {
                for( E=EGPSessionHead; E->Next!=NULL; E=E->Next );
                E->Next=AllocEGPSession( LocalAddress,Address );
                E=E->Next;
        }
	E->SessionState=0;
	E->AnnouncedASys=0;
	E->Next=NULL;
        return(E);
}

/************************************************************************
 * FindEGPSession() - Between two Nodes					*
 *			return pointer to EGPSession struct or NULL	*
 ************************************************************************/
struct EGPSessionType *FindEGPSession(LocalAddress, Address)
     char *LocalAddress;
     char *Address;
{
struct EGPSessionType *E;
	for( E=EGPSessionHead; E!=NULL; E=E->Next )
		if (( strcmp( Address, E->Address ) == 0 ) &&
			( strcmp(LocalAddress,E->LocalAddress)==0)) return( E );
	return( NULL );
}

/************************************************************************
 * AddEGPData() - AddEGPData to this EGP Session Structure		*
 ************************************************************************/
void AddEGPData(LocalAddress, VarName, Address, value)
     char *LocalAddress;
     char *VarName;
     char *Address;
     char *value;
{
int j;
struct EGPSessionType *E;

	if ( ( E = FindEGPSession(LocalAddress,Address) ) == NULL )
		E=AddEGPSession(LocalAddress,Address);
	for(j=0; j<((sizeof(EGPVariables))/(sizeof(char *))); j++) {
		if (strcmp(EGPVariables[j],VarName)==0) {
			switch( j ) {	
			case 0: /* EGP State */
				E->SessionState=atoi(value);
				return;
				/*NOTREACHED*/
			case 1: /* EGP AnnouncedASys */
				E->AnnouncedASys=atoi(value);
				return;
				/*NOTREACHED*/
			default:
				fprintf(stderr,"Unrecognized Data Type %s\n", 
								VarName);
				return;
				/*NOTREACHED*/
			}
		}
	}
}

/************************************************************************
 * AllocASys() - Allocate an ASys Structure and return a pointer to it	*
 ************************************************************************/
struct ASysType *AllocASys(ASysNum)
     int ASysNum;
{
struct ASysType *ASys;

	if ((ASys=(struct ASysType *) malloc( sizeof(struct ASysType) ))==NULL)
		exit(fprintf(stderr,"malloc failed in AllocASys\n"));
	ASys->ASysNum=ASysNum;
	ASys->State=0;
	ASys->EGPSessions=NULL;
	ASys->Next=NULL;
	return(ASys);
}

/************************************************************************
 * AddASys() - Add an ASys to our ASysHead List				*
 *		return a pointer to the new ASysstruct			*
 ************************************************************************/
struct ASysType *AddASys(ASysNum)
     int ASysNum;
{
struct ASysType *A;

	if (ASysHead == NULL)
		A=ASysHead=AllocASys( ASysNum );
	else {
		for( A=ASysHead; A->Next!=NULL; A=A->Next);
		A->Next=AllocASys(ASysNum);
		A=A->Next;
	}
	return(A);	/* return pointer to new ASys entry */
}

/************************************************************************
 * 	FindASys() - Find the ASys Structure for this ASys			*
 ************************************************************************/
struct ASysType *FindASys(ASysNum)
     int ASysNum;
{
struct ASysType *A;

	for( A=ASysHead; A!=NULL; A=A->Next)
		if ( A->ASysNum == ASysNum ) return( A );
	return( NULL );
}

/************************************************************************
 * AddEGP2ASys() - Add EGP Session structure to ASys List			*
 ************************************************************************/
void AddEGP2ASys(EGP, ASys)
     struct EGPSessionType *EGP;
     struct ASysType *ASys;
{
struct EGPSessionType *E;

	if ( ASys->EGPSessions == NULL ) ASys->EGPSessions=EGP;
	else {
		for( E=ASys->EGPSessions; E->Next!=NULL; E=E->Next);
		E->Next=EGP;
	}
	EGP->Next=NULL;
	/*                idle(1),
                          acquisition(2),
                          down(3),
                          up(4),
                          cease(5)			*/

	if ( EGP->SessionState == 4 ) ASys->State=1;
}

/************************************************************************
 * ArrangeInASysOrder() - Take all of the EGP Sessions and associate them *
 *			with their ASys structure.			*
 ************************************************************************/
struct ASysType *ArrangeInASysOrder()
{
struct EGPSessionType *E,*ENext;
struct ASysType *ASys;

	for( E=EGPSessionHead; E!=NULL; E=ENext ) {
		ENext=E->Next;
		if ( ASys = FindASys( E->AnnouncedASys ))
			AddEGP2ASys( E, ASys );
		else {
			ASys = AddASys( E->AnnouncedASys );
			AddEGP2ASys( E, ASys );
		}
	}
	return(ASysHead);
}


/************************************************************************
 * AddASysData() - Add ASys to ASys List					*
 ************************************************************************/
AddASysData(RespondingNode, Var, instance, value)
     char *RespondingNode;
     char *Var;
     char *instance;
     char *value;
{
struct EGPSessionType *E,*ENext;
struct ASysType *ASys;

int ASysNum;

	if ( ( ASysNum=atoi(value) ) == 0 ) {
		fprintf(stderr,"0 ASys Number - weird\n");
		return 1;
	}
	if (strcmp(Var, ASysVariables[0])==0) {
		if ( ASys = FindASys( ASysNum ) ) {
			if ( ASys->State == 1 ) return;
			ASys->State=1;
			/*E=AllocEGPSession(RespondingNode,ASysNum);
			E->SessionState=4;
			AddEGP2ASys( E, ASys );*/
		}
		else {
			ASys = AddASys(  ASysNum );
			ASys->State=1;
			/*E=AllocEGPSession(RespondingNode,ASysNum);
			E->SessionState=4;
			AddEGP2ASys( E, ASys );*/
		}
	}
	else fprintf(stderr,"Got back %s from %s: weird!\n",Var,RespondingNode);
	return 0;
}

void subnet(Address)
     char *Address; 
{

	MySubNet.s_addr = inet_addr( Address );
	if (IN_CLASSA(MySubNet.s_addr))
		MySubNet.s_addr &= IN_CLASSA_NET;
	else if (IN_CLASSB(MySubNet.s_addr))
		MySubNet.s_addr &= IN_CLASSB_NET;
	else if (IN_CLASSC(MySubNet.s_addr))
		MySubNet.s_addr &= IN_CLASSC_NET;
	else
		fprintf(stderr, "Unknown class address %s\n", Address);
}

int InSubNet(node)
     char *node;
{
struct in_addr addr, tmp;

        addr.s_addr = inet_addr( node );
        if (IN_CLASSA(addr.s_addr))
                addr.s_addr &= IN_CLASSA_NET;
        else if (IN_CLASSB(addr.s_addr))
                addr.s_addr &= IN_CLASSB_NET;
        else if (IN_CLASSC(addr.s_addr))
                addr.s_addr &= IN_CLASSC_NET;
        else
                fprintf(stderr, "Unknown class address %s\n", node);
	if ( addr.s_addr == MySubNet.s_addr ) return(1);
	else return(0);
}

/*
 *   FindGenericNode - search for the node with this address
 *  BEVBEV
 */
struct GenericNodeType *FindGenericNode( NodeName )
char *NodeName;
{
register struct GenericNodeType *P;

	for( P=GenericNodeHead; P; P=P->Next )
		if ( strcmp( NodeName, P->Address ) == 0 ) return( P );
	return(NULL);

}

struct GenericNodeType *AddGenericNode( NodeName )
char *NodeName;
{
struct GenericNodeType *P;

	if ((P=(struct GenericNodeType *)malloc( sizeof(struct GenericNodeType)))==NULL) {
		fprintf(stderr,"malloc failed in AddGenericNode\n");
		exit(1);
	}
	strcpy( P->Address, NodeName );
	P->Interfaces=NULL;
	P->Neighbors=NULL;
	P->Next=NULL;
	if ( GenericNodeHead==NULL ) GenericNodeHead=P;
	else {
		P->Next=GenericNodeHead;
		GenericNodeHead=P;
	}
	return( P );
}

struct InterfaceType *AddInterface(N, interfacenum)
     struct GenericNodeType *N;
     int interfacenum;
{
struct InterfaceType *I,*IPtr;

	if ((I=(struct InterfaceType *)malloc(sizeof(struct InterfaceType)))==NULL) {
		fprintf( stderr, "malloc failed in AddInterface()\n");
		exit(1);
	}
	I->ifnum=interfacenum;
	I->ifType=0;
	I->Address[0]='\0';
	I->Next=NULL;
	if ( N->Interfaces == NULL )	/* Chain in the first interface */
		N->Interfaces=I;
	else {
		for ( IPtr=N->Interfaces; IPtr->Next!=NULL; IPtr=IPtr->Next );
		IPtr->Next=I;
	}
	return( I );
}

struct InterfaceType *FindInterface(N, ifNum)
     struct GenericNodeType *N;
     int ifNum;
{
register struct InterfaceType *I;

	for( I=N->Interfaces; I != NULL; I=I->Next )
		if ( I->ifnum == ifNum ) return(I);
	return(NULL);
}

void AddIfType(N, inst, value)
     struct GenericNodeType *N;
     char *inst;
     char *value;
{
struct InterfaceType *I;

	if ( ( I=FindInterface( N, atoi(inst) ) ) == NULL )
		I=AddInterface( N, atoi(inst) );
	I->ifType=atoi(value);
	/*printf("AddIfType(): %s Interface: <%s> #%d type=%d\n",N->Address,I->Address,I->ifnum,I->ifType);*/
}

void AddIPAddr(N, inst, value)
     struct GenericNodeType *N;
     char *inst;
     char *value;
{
struct InterfaceType *I;

	if ( ( I=FindInterface( N, atoi(value) ) ) == NULL )
		I=AddInterface( N, atoi(value) );
	strcpy(I->Address,inst);
	/*printf("AddIPAddr(): %s Interface: <%s> #%d type=%d\n",N->Address,I->Address,I->ifnum,I->ifType);*/
}

void AddNextHop(RespondingNode, NextHop)
     struct GenericNodeType *RespondingNode;
     char *NextHop;
{
struct NextHopType *N,*NewHop;

	if ( strcmp( "0.0.0.0", NextHop ) == 0 ) return;

	for( N=RespondingNode->Neighbors; N!=NULL;  N=N->Next)
	    if ( strcmp(N->RemoteAddress, NextHop ) == 0 )
		return;	/* Next Hop Already in list! */

	if ((NewHop=(struct NextHopType *)malloc( sizeof(struct NextHopType))) == NULL ) {
		fprintf( stderr,"malloc() failed, exitting\n");
		exit(1);
	}
	strcpy( NewHop->RemoteAddress, NextHop );
	NewHop->Next=NULL;
	if ( RespondingNode->Neighbors == NULL ) 
		RespondingNode->Neighbors=NewHop;
	else {
		NewHop->Next=RespondingNode->Neighbors; /* Chain it in front */
		RespondingNode->Neighbors=NewHop;	/* New Head */
	}
}

void GenericReport()
{
struct GenericNodeType *N;
struct NextHopType *H;
struct InterfaceType *I;

	for( N=GenericNodeHead; N != NULL; N=N->Next )  {
		for( H=N->Neighbors; H != NULL; H=H->Next ) 
			printf("%s LINK UNKNOWNLINKTYPE %s ?? %s ??\n" ,
				N->Address,N->Address,H->RemoteAddress);
		for( I=N->Interfaces; I != NULL; I=I->Next )
			printf("%s INTERFACE %s %d %d\n" ,
				N->Address,I->Address, I->ifnum,I->ifType);
	}
}

/*
 * Search for caller's address in the Interface List for each node
 * return pointer to node and ifnum and iftype set.
 * if not found set ifType and ifNum to 0
 */
struct GenericNodeType *FindNodeWithInterface(Address, ifnum, ifType)
     char *Address;
     int *ifnum;
     int *ifType;
{
register struct GenericNodeType *NPtr;
register struct InterfaceType *I;

	*ifnum=*ifType=0;
	for( NPtr=GenericNodeHead; NPtr != NULL; NPtr=NPtr->Next )  {
		for( I=NPtr->Interfaces; I!=NULL; I=I->Next ) {
			if ( strcmp(I->Address, Address ) == 0 ) {
				*ifnum=I->ifnum;
				*ifType=I->ifType;
				return( NPtr );
			}
		}
	}
	return(NULL);
}

char *FindNodeName(NPtr)
     struct GenericNodeType *NPtr;
{
register struct InterfaceType *I;

	if ( NPtr->Interfaces == NULL ) return(NULL);
	for( I=NPtr->Interfaces; I!=NULL; I=I->Next ) {
		if (I->ifType!=22 && I->ifType!=18 && I->Address[0]!='\0' ) {
			return( I->Address );	/* return 1st nonserial */
		}
	}
	return(NPtr->Interfaces->Address);	/* return 1st interface */
}

struct InterfaceType *SearchNodeForMyInterface(N, NPtr)
     struct GenericNodeType *N;
     struct GenericNodeType *NPtr;
{
register struct NextHopType *HPtr;
register struct InterfaceType *IPtr;

	for (HPtr=N->Neighbors; HPtr!=NULL; HPtr=HPtr->Next)
		for (IPtr=NPtr->Interfaces; IPtr!=NULL; IPtr=IPtr->Next )
			if ( strcmp( IPtr->Address, HPtr->RemoteAddress ) == 0 )
				return( IPtr );
	return(NULL);
}

void LinksReport()
{
register struct GenericNodeType *NPtr;
register struct NextHopType *HPtr;
register struct GenericNodeType *N;
register struct InterfaceType *I;
struct AddressListType *AddressListHead=NULL;
char *LocalNodeName,*RemoteNodeName;
int Localifnum, ifType;
int Remoteifnum;
time_t TimeNow=time(&TimeNow);

	/*** For each NextHop that we have seen as one of our interfaces
		 add a links entry ***/
	for( NPtr=GenericNodeHead; NPtr != NULL; NPtr=NPtr->Next )  {
		if ((LocalNodeName=FindNodeName( NPtr ))==NULL) continue;
		for( HPtr=NPtr->Neighbors; HPtr != NULL; HPtr=HPtr->Next ) {
			if ((N=FindNodeWithInterface( HPtr->RemoteAddress, 
					&Remoteifnum, &ifType ))==NULL) 
				continue;  /*Can't find this next hop anywhere*/
			RemoteNodeName=FindNodeName( N );
			if ((I=SearchNodeForMyInterface( N, NPtr ))==NULL)
				continue; /* Couldn't find my interface */
			printf("%d %ld %s %s %d %d %s %s %ld 0 0\n", 
				ifType, TimeNow, LocalNodeName, I->Address, 
				I->ifnum, Remoteifnum, HPtr->RemoteAddress, 
				RemoteNodeName, TimeNow );
			/*printf("%s LINK ISISLINK %s %s %s %s UP\n", 
				LocalNodeName, LocalNodeName, I->Address, 
				HPtr->RemoteAddress, RemoteNodeName );*/
			break;  /* We found this next hop */
		}
	}
}

struct AddressListType *MakeNextHostList()
{
register struct GenericNodeType *NPtr;
register struct NextHopType *HPtr;
register struct GenericNodeType *N;
register struct InterfaceType *I;
struct AddressListType *AddressListHead=NULL;
int FoundIt;

	/*** For each NextHop that we haven't seen as one of our interfaces
		add it to the list of nodes we query  ***/
	for( NPtr=GenericNodeHead; NPtr != NULL; NPtr=NPtr->Next )  {
		for( HPtr=NPtr->Neighbors; HPtr != NULL; HPtr=HPtr->Next ) {
			/* Search for this Next Hop */
			FoundIt=0;	
			for( N=GenericNodeHead; N != NULL; N=N->Next )  {
				for( I=N->Interfaces; I!=NULL; I=I->Next ) {
					if ( strcmp( HPtr->RemoteAddress, I->Address ) == 0 ) {
						FoundIt=1;
						break;  /* We found this next hop */
					}
				}
				if ( FoundIt ) break; /* No need to look on */
			}
			if ( !FoundIt ) {
				if ( InSubNet(HPtr->RemoteAddress)) {
				if (verbose) printf("Adding %s to list to query\n",
					HPtr->RemoteAddress );
				AddressListHead=AddAddress( AddressListHead, 
					HPtr->RemoteAddress );
				} else {
					if (verbose) printf("host not in subnet: host=%s subnet=%s\n",HPtr->RemoteAddress,inet_ntoa(MySubNet));
				}
			}
		}	/* Process each Next Hop */
	}
	return(AddressListHead);
}

/************************************************************************
 * AddMIBRoutingTableData() - Add MIB Routing Table into nexthop table	*
 ************************************************************************/
void AddMIBRoutingTableData(RespondingNode, VarName, instance, value)
     char *RespondingNode;
     char *VarName;
     char *instance;
     char *value;
{
char *HewHop;
struct GenericNodeType *N;

	/* Find this Node and append the next hop if not already there */
	if (( N=FindGenericNode(RespondingNode) ) == NULL ) 
		N=AddGenericNode(RespondingNode);
	if ( strcmp(VarName,"ipRouteNextHop")==0)
		AddNextHop( N, value );
	if ( strcmp(VarName,"ifType")==0)
		AddIfType( N, instance, value );
	if ( strcmp(VarName,"ipAdEntIfIndex")==0)
		AddIPAddr( N, instance, value );
}
void DequeueNeighbor(N, Hop)
     struct GenericNodeType *N;
     struct NextHopType *Hop;
{
register struct NextHopType *H;

	if ( N->Neighbors==Hop ) N->Neighbors=Hop->Next;
	else {
	for( H=N->Neighbors; H!=NULL && H->Next!=Hop; H=H->Next );
		/*printf("checking %s against %s\n",H->RemoteAddress,Hop->RemoteAddress);*/
		if ( H == NULL ) fprintf(stderr,"Tried to dequeue a bad neighbor\n");
		else 
			H->Next=Hop->Next;	
		/* we'll leave this structure in place so the remove bad link
			loop won't choke. */
	}
}

void RemoveBadNeighbors()
{
register struct GenericNodeType *N;
register struct NextHopType *H;
register struct InterfaceType *I;

	for( N=GenericNodeHead; N != NULL; N=N->Next )  {
		for( H=N->Neighbors; H != NULL; H=H->Next ) {
			for( I=N->Interfaces; I!=NULL; I=I->Next )
				if ( ( strcmp( H->RemoteAddress, I->Address ) == 0 )  
				){
				/*if ( strcmp( H->RemoteAddress, I->Address ) == 0 ) {*/
					DequeueNeighbor( N, H );
					break;  /* This neighbor is freed */
				}
		}
	}
}


/************************************************************************
 *	FormatASysData - We did an ASys Query and found this list of ASyss	*
 ************************************************************************/
FormatASysData(AddressListHead)
     struct AddressListType *AddressListHead;
{
struct AddressListType *A;
struct VarList *v;
struct bindings *b;

   for( A=AddressListHead; A!=NULL; A=A->Next ) {
        for( v=A->ResponseListHead; v!=NULL; v=v->Next)
                for( b=v->Head; b!=NULL; b=b->Next )
                         AddASysData(A->Address,v->VarName,b->instance+strlen(v->Prefix),b->value);
   }

}


/************************************************************************
 *	FormatEGPData() - Primary Call for processing ALL EGP Data	*
 ************************************************************************/
void FormatEGPData(AddressListHead)
     struct AddressListType *AddressListHead;
{
struct AddressListType *A;
struct VarList *v;
struct bindings *b;

   for( A=AddressListHead; A!=NULL; A=A->Next ) {
        for( v=A->ResponseListHead; v!=NULL; v=v->Next)
		for( b=v->Head; b!=NULL; b=b->Next )
			 AddEGPData(A->Address,v->VarName,b->instance+strlen(v->Prefix),b->value);
   }
}

/************************************************************************
 *      FormatMIBRoutingTableData() - Primary Call for processing 	*
 *	ALL MIBRoutingTable Data      					*
 ************************************************************************/
void FormatMIBRoutingTableData(AddressListHead)
     struct AddressListType *AddressListHead;
{
struct AddressListType *A;
struct VarList *v;
struct bindings *b;

   for( A=AddressListHead; A!=NULL; A=A->Next ) {
        for( v=A->ResponseListHead; v!=NULL; v=v->Next)
                for( b=v->Head; b!=NULL; b=b->Next )
			 AddMIBRoutingTableData(A->Address,v->VarName,b->instance+strlen(v->Prefix),b->value);
   }
}



/************************************************************************
 * QAddress() - Convert 140.222 and 140.223. addresses to .62 addresses *
 ************************************************************************/
char *QAddress(address)
     char *address;
{
int octet1,octet2,octet3,octet4;
static char NodeName[100];

	if ((sscanf( address,"%d.%d.%d.%d",
		&octet1,&octet2,&octet3,&octet4))==4) {
		strcpy(NodeName,"");
		if ((octet1==140)&&((octet2==222)||(octet2==223))) 
			sprintf(NodeName,"%d.%d.%d.62",
				octet1,octet2,octet3);
		else
			if ((octet1==129)&&(octet2==140)) 
                                      sprintf(NodeName,"%d.%d.%d.1",
                                              octet1,octet2,octet3-64);
			else strcpy(NodeName,address);
	}
	else strcpy(NodeName,address);
	return(NodeName);
}

/************************************************************************
 * ISISReport() - Report information in the Links structure		*
 ************************************************************************/
void ISISReport()
{
int i;

        for(i=0; i<NumLinks; i++) {
#ifdef TEST
                printf("%s LINK ISISLINK %s %s %d %d %s %s %s\n",
                                Links[i].RespondingAddress,
                                Links[i].RespondingAddress,
                                strlen(Links[i].LocalAddress)>0 ? /* ifAvail */
                                        Links[i].LocalAddress :
                                        "NotKnown",
				DSUNum(Links[i].LocalAddress),
				DSUNum(Links[i].RemoteAddress),
                                Links[i].RemoteAddress,
                                QAddress(Links[i].RemoteAddress),
                                Links[i].State?"UP":"DOWN");

#else
                printf("%s LINK ISISLINK %s %s %s %s %s\n",
                                Links[i].RespondingAddress,
                                Links[i].RespondingAddress,
                                strlen(Links[i].LocalAddress)>0 ? /* ifAvail */
                                        Links[i].LocalAddress :
                                        "NotKnown",
                                        Links[i].RemoteAddress,
                                QAddress(Links[i].RemoteAddress),
                                Links[i].State?"UP":"DOWN");
#endif
        }
}

/************************************************************************
 * ASysReport() - Report the ASys Status - look throught he ASys Chain	*
 ************************************************************************/
void ASysReport()
{
struct ASysType *ASys;
char *address;

      for( ASys=ASysHead; ASys!=NULL; ASys=ASys->Next ) {
	     if ( ASys->ASysNum == 0 ) {
		fprintf(stderr,"ASysNum == 0 Error!\n");
		continue;
	     }
             if ( ASys->EGPSessions != NULL )
                  address=ASys->EGPSessions->LocalAddress;
	     else address="Unknown";
             if ( ASys->State == 1 ) {
		  if ( ASys->EGPSessions  != NULL )
                  printf("%s LINK ASLINK %s NotKnown NotKnown %d UP\n",
                                        address,address,ASys->ASysNum);
                  printf("%s NODE AS %d NotKnown UP\n",
                                                address,ASys->ASysNum );
             }
             else {
#ifdef SHOWASysIFDOWN
                  printf("%s LINK ASLINK %s NotKnown NotKnown %d DOWN\n",
                                address,address,ASys->ASysNum);
                  printf("%s NODE AS %d NotKnown DOWN\n",
                                                address,ASys->ASysNum );
#endif
             }
      }
}

/************************************************************************
 *	PrintReport() - Print a Formatted report of collected info	*
 *									*
 *	This consists of the Links array and the ASys Chain		*
 ************************************************************************/
void PrintReport()
{
	ISISReport();
	ASysReport();
}

#ifdef CIRCUITCHECK

/************************************************************************
 *  Update_LINK_Table() - Update Link Table with info retrieved from 	*
 *				the network.				*
 *	We'll have to note difference - no answer, missing links	*
 *	We'll want to update the times for DSU link state info		*
 ************************************************************************/
void Update_LinkDetail_Table( )
{
int i;

        for(i=0; i<NumLinks; i++) {
                if (verbose) printf("Update_LINK_Table(): %s %s %d %d %s %s\n",
                                Links[i].RespondingAddress,
                                strlen(Links[i].LocalAddress)>0 ? /* ifAvail */
                                        Links[i].LocalAddress :
                                        "NotKnown",
				DSUNum(Links[i].LocalAddress),
				DSUNum(Links[i].RemoteAddress),
                                Links[i].RemoteAddress,
                                QAddress(Links[i].RemoteAddress));
		UpdateLink( Links[i].RespondingAddress,
					Links[i].LocalAddress,
					DSUNum(Links[i].LocalAddress),
					DSUNum(Links[i].RemoteAddress),
					Links[i].RemoteAddress,
					QAddress(Links[i].RemoteAddress) );
	}
	/*  ReportDiff();    */
                                
}
#endif

void Usage()
{
	fprintf(stderr,"Usage: %s [-v] [-d] [-i] [-a] [-l] -h <node> ... -h <node> -c <community>\n",progname);
	exit(1);
}

int main( argc, argv )
int argc;
char *argv[];
{
char *community=NULL;
struct VarList *QueryListHead=NULL;
struct AddressListType *AddressListHead=NULL,*A=NULL;
int i;
int isis=0;
int as=0;
int generic=0;
extern int MaxSecsB4Retry;
extern int MaxSNMPRetries;
int circuitcheck=0;

	progname=argv[0];
        argv++; argc--;
        while( *argv!=NULL && argv[0][0] == '-') {
                switch(argv[0][1]) {
                case 'd':
                case 'v':       
				fprintf(stderr,"Verbose Mode=ON\n");
				verbose=1;
                                argv++; argc--;
                                break;
                case 'h':
                        AddressListHead=AddAddress( AddressListHead, argv[1] );
                        argv++; argc--;
                        argv++; argc--;
                        break;
                case 'c':
                        community=argv[1];
                        argv++; argc--;
                        argv++; argc--;
                        break;
		case 'i':
			isis=1;
                        argv++; argc--;
			break;
		case 'l':
			LinkTableFunction=1;
                        argv++; argc--;
			break;
		case 'g':
			generic=1;
                        argv++; argc--;
			break;
		case 'a':
			as=1;
			argv++; argc--;
			break;
                case 'r':
                        argv++; argc--;
                        if ( ( argc != 0 ) && ( atoi(argv[0]) != 0 )) {
                                MaxSNMPRetries=atoi(argv[0]);
                                argv++; argc--;
                        }
                        else fprintf(stderr,"Invalid retry specification\n");
                        break;
                case 't':
                        argv++; argc--;
                        if ( ( argc != 0 ) && ( atoi(argv[0]) != 0 )) {
                                 MaxSecsB4Retry=atoi(argv[0]);
                                argv++; argc--;
                        }
                        else fprintf(stderr,"Invalid retry specification\n");
                        break;
		case '3':
			circuitcheck=1;
			argv++; argc--;
 			break;
                default:
                        fprintf(stderr,"unknown option %s ignored\n(Valid options are: -v )\n",argv[0]);
                        argv++; argc--;
                        break;
                }
        }
	if ( community==NULL) {
		fprintf(stderr,"You MUST specify a community name \n");
		Usage();
	}
	if ((isis==0)&&(as==0)&&(generic==0)) isis=1;

#ifdef CIRCUITCHECK
	/**** Link Table Function ****/
	if ( LinkTableFunction ) {
		Read_LinkDetail_File();	/* Read the Link DB */
		for(i=0; i<((sizeof(LinkTableVariables))/(sizeof(char *))); i++)
	       	QueryListHead=AddVar(QueryListHead,LinkTableVariables[i]);
	
		AddressListHead=GetTable( AddressListHead, community, QueryListHead );
		for( A=AddressListHead; A!=NULL; A=A->Next ) {
			if (A->QueryListHead!=NULL) 
				fprintf(stderr,"%s did not respond completely to LinkTable Query\n",A->Address);
			else FormatLinkTableData( A );
		}
		Update_LinkDetail_Table( );	/* Update Link Info */

		printf("Querying for iftype\n");
                FreeAddressListsVarList( AddressListHead );
                QueryListHead=NULL;
		QueryListHead=AddVar(QueryListHead,"ifType");
		AddressListHead=GetTable( AddressListHead, community, QueryListHead );
		for( A=AddressListHead; A!=NULL; A=A->Next ) {
			if (A->QueryListHead!=NULL) 
				fprintf(stderr,"%s did not respond completely to LinkTable ifType Query\n",A->Address);
			else ProcessLinkType( A );
		}

		if ( circuitcheck ) {
			printf("Checking Circuit for DSU Serial number veracity\n");
			/*** Verify the LINKS by checking Serial Numbers at ends ***/
                	FreeAddressListsVarList( AddressListHead );
                	QueryListHead=NULL;
			QueryListHead=AddVar(QueryListHead,"t3techTopologySerialNumber");
			printf("Querying network for t3topologySerials\n");
			AddressListHead=GetTable( AddressListHead, community, QueryListHead );
			printf("Network Querying Done\n");
			for( A=AddressListHead; A!=NULL; A=A->Next ) {
				if (A->QueryListHead!=NULL) 
					fprintf(stderr,"%s did not respond completely to DSU Serial Number Query\n",A->Address);
				else ProcessDSUSerialNumbers( A );
			}
		}
		Write_LinkDetail_File( );	/* Write DB back to disk */
		exit(1);
	}
#endif

	if ( generic ) {
		MaxSecsB4Retry=10;	/* Allow a long time for queries */
		MaxSNMPRetries=10;

		for(i=0; i<((sizeof(MIBRoutingTable))/(sizeof(char *))); i++)
	       	QueryListHead=AddVar(QueryListHead,MIBRoutingTable[i]);
		subnet( AddressListHead->Address );
	
		for( ; ; ) {
			if (verbose) fprintf(stderr,"--------------- Start of collection-----------\n");
			AddressListHead=GetTable( AddressListHead, community, QueryListHead );
			for( A=AddressListHead; A!=NULL; A=A->Next ) {
				if (A->QueryListHead!=NULL) 
					fprintf(stderr,"%s did not respond completely to Neighbor Query\n",A->Address);
				else FormatMIBRoutingTableData( A );
			}
			RemoveBadNeighbors();  /* get rid of non-neighbors */
			/*GenericReport();*/
                	AddressListHead=MakeNextHostList();
			if ( AddressListHead == NULL ) break; /* All neighbors tried */
		}
		if (verbose) GenericReport();
		LinksReport();
	}

	if ( isis ) {
		for(i=0; i<((sizeof(ISISVariables))/(sizeof(char *))); i++)
	       	QueryListHead=AddVar(QueryListHead,ISISVariables[i]);
	
		AddressListHead=GetTable( AddressListHead, community, QueryListHead );
		for( A=AddressListHead; A!=NULL; A=A->Next ) {
			if (A->QueryListHead!=NULL) 
				fprintf(stderr,"%s did not respond completely to IS-IS Query\n",A->Address);
			else FormatISISData( A );
		}
	}
	if ( as ) {
		FreeAddressListsVarList( AddressListHead );   
		QueryListHead=NULL;
	
		for(i=0; i<((sizeof(EGPVariables))/(sizeof(char *))); i++)
	       	QueryListHead=AddVar(QueryListHead,EGPVariables[i]);
	
		AddressListHead=GetTable( AddressListHead, community, QueryListHead );
		if (verbose) DisplayAddressChain(AddressListHead);
		for( A=AddressListHead; A!=NULL; A=A->Next ) {
			FormatEGPData( A );
			if ((A->QueryListHead!=NULL)&&(A->ResponseListHead==NULL)) 
				fprintf(stderr,"%s did not respond \n",A->Address);
		}
		ASysHead = ArrangeInASysOrder( );
			/***** Ask for ASyss Network wide *****/
		QueryListHead=NULL;
		AddressListHead=FreeAddressList( AddressListHead );
		/*AddressListHead=AddAddress(AddressListHead,"140.222.131.62");
		AddressListHead=AddAddress(AddressListHead,"140.222.135.62");*/
		AddressListHead=AddAddress(AddressListHead,"140.222.144.62");
		for(i=0; i<((sizeof(ASysVariables))/(sizeof(char *))); i++)
	       		QueryListHead=AddVar(QueryListHead,ASysVariables[i]);
	
		AddressListHead=GetTable( AddressListHead, community, QueryListHead );
		if (verbose) DisplayAddressChain(AddressListHead);
		for( A=AddressListHead; A!=NULL; A=A->Next ) {
			FormatASysData( A );
			if ((A->QueryListHead!=NULL)&&(A->ResponseListHead==NULL)) 
				fprintf(stderr,"%s did not respond \n",A->Address);
		}
	}
	PrintReport();
	return(0);
}

