/*
 * Copyright (c) 1992 Regents of the University of Michigan.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms are permitted
 * provided that this notice is preserved and that due credit is given
 * to the University of Michigan at Ann Arbor. The name of the University
 * may not be used to endorse or promote products derived from this
 * software without specific prior written permission. This software
 * is provided ``as is'' without express or implied warranty.
 *
 *
 * This file contains modules necessary to Query the Network and
 * return a list of Neighbors Associated with a Particular Node.
 * We have the Network-Specific modules that query and convert the
 * Neighbor Table returned into our internal format.
 */

#include <stdio.h>
#include <ctype.h>
#include <string.h>
#include <sys/file.h>
#include "pathnames.h"
#include "CommonDefs.h"

extern char *Error_Log,CommunityName[],*NetworkName;

static int nq_errlog;

int SNMPRetries=5, SNMPTimeout=5;	/* Seconds before retrying queries */

/****************************************************************
 *	init_query_engine:Opens error log, and saves destriptor	*
 *			in nq_errlog            		*
 ****************************************************************/
void init_query_engine()
{
	if ((nq_errlog = open(NQ_ERRLOG, O_RDWR | O_CREAT, 0666)) < 0) {
		syserr("Opening error log: ");
		/*NOTREACHED*/
	}
/*
	XXXX Probably don't need this:

	int fd;
	fd=open( Error_Log, O_RDWR|O_CREAT, 0666 );
	close( fd );
*/
}

/************************************************************************
 ************************************************************************/
struct NetworkType *GetNSFNETNeighStates( Net )
struct NetworkType *Net;	/* Network marked all NR */
{

	register struct NodeType *Node;
	char Buf[BUFSIZ], cmd1[4*BUFSIZ], cmd2[4*BUFSIZ];
	FILE *pipe1, *pipe2;
	extern int FetchDSU, NoDiscovery;
	extern void ProcessNeighborLine();

	Net->GotOne = 0;

	/*
	 *  Build the two commands we want to issue.  Once built, we'll
	 *  use a pipe to execute them and send the results back here.
	 */
	sprintf(cmd1, "%s -i -r %d -t %d", GETNEIGHBORS2,SNMPRetries,SNMPTimeout);
	sprintf(cmd2, "%s -a -r %d -t %d", GETNEIGHBORS2,SNMPRetries,SNMPTimeout);
        for (Node = Net->NodeHead; Node != NULL; Node = Node->Next) {
                if ((Node->Queryable) && (Node->Type == NSS)) {
			strcat(cmd1, " -h ");
			strcat(cmd1, Node->IPAddress);
			strcat(cmd2, " -h ");
			strcat(cmd2, Node->IPAddress);
		}
	}
	strcat(cmd1, " -c ");
	if ( strlen(CommunityName) == 0 )
		strcat(cmd1,"lookatit");
	else
		strcat(cmd1,CommunityName);
	strcat(cmd1," &");
	strcat(cmd2, " -c ");
	if ( strlen(CommunityName) == 0 )
		strcat(cmd2,"lookatit");
	else
		strcat(cmd2,CommunityName);
	strcat(cmd2," &");

	/*
	 *  Open up the pipelines.
	 */
	if ((pipe1 = popen(cmd1, "r")) == NULL) {
		perror("popen");
		return((struct NetworkType *) NULL);
	}
	if ((pipe2 = popen(cmd2, "r")) == NULL) {
		perror("popen");
		return((struct NetworkType *) NULL);
	}

	/*
	 *  Execute the pipelines, feeding their output to
	 *  ProcessNeighborLine().  Turn of link state clash
	 *  checking before we execute the pipelines, and then
	 *  turn it off once we are done.
	 */
	SetCheckForLinkClash();
        while(fgets(Buf, sizeof(Buf), pipe1) != NULL) 
		ProcessNeighborLine(Net, Buf);
	pclose(pipe1);
        while(fgets(Buf, sizeof(Buf), pipe2) != NULL) 
		ProcessNeighborLine(Net, Buf);
	pclose(pipe2);
	ClearCheckForLinkClash();
	return(Net);
}
/************************************************************************
 ************************************************************************/
struct NetworkType *GetGENERICNetworkState( Net )
struct NetworkType *Net;	/* Network marked all NR */
{

	register struct NodeType *Node;
	char Buf[BUFSIZ], cmd1[4*BUFSIZ];
	FILE *pipe1;
	extern void ProcessNeighborLine();

	Net->GotOne = 0;

	/*
	 *  Build the two commands we want to issue.  Once built, we'll
	 *  use a pipe to execute them and send the results back here.
	 */
	sprintf(cmd1, "getlinkstate -r %d -t %d",SNMPRetries,SNMPTimeout);
	strcat(cmd1, " -c ");
	if ( strlen(CommunityName) == 0 )
		strcat(cmd1,"Merit");
	else
		strcat(cmd1,CommunityName);
	strcat(cmd1, " -n ");
	if ( strlen(NetworkName) == 0 )
		strcat(cmd1,"cicnet");
	else
		strcat(cmd1,NetworkName);
	strcat(cmd1," &");

	/*
	 *  Open up the pipelines.
	 */
	if ((pipe1 = popen(cmd1, "r")) == NULL) {
		perror("popen");
		return((struct NetworkType *) NULL);
	}

	/*
	 *  Execute the pipelines, feeding their output to
	 *  ProcessNeighborLine().  Turn of link state clash
	 *  checking before we execute the pipelines, and then
	 *  turn it off once we are done.
	 */
	SetCheckForLinkClash();
        while(fgets(Buf, sizeof(Buf), pipe1) != NULL) 
		ProcessNeighborLine(Net, Buf);
	pclose(pipe1);
	ClearCheckForLinkClash();
	return(Net);
}

/************************************************************************
 ************************************************************************/
struct NetworkType *GetMERITNetworkState( Net )
struct NetworkType *Net;	/* Network marked all NR */
{

	register struct NodeType *Node;
	char Buf[BUFSIZ], cmd1[4*BUFSIZ];
	FILE *pipe1;
	extern void ProcessNeighborLine();

	Net->GotOne = 0;

	/*
	 *  Build the two commands we want to issue.  Once built, we'll
	 *  use a pipe to execute them and send the results back here.
	 */
	sprintf(cmd1, "GetMeritNeighborTable -r %d -t %d",SNMPRetries,SNMPTimeout);
        for (Node = Net->NodeHead; Node != NULL; Node = Node->Next) {
                if (Node->Queryable) {
			strcat(cmd1, " -h ");
			/*strcat(cmd1, Node->IPAddress); future... */
			strcat(cmd1, Node->Name);
		}
	}
	strcat(cmd1, " -c ");
	if ( strlen(CommunityName) == 0 )
		strcat(cmd1,"Merit");
	else
		strcat(cmd1,CommunityName);
	strcat(cmd1," &");

	/*
	 *  Open up the pipelines.
	 */
	if ((pipe1 = popen(cmd1, "r")) == NULL) {
		perror("popen");
		return((struct NetworkType *) NULL);
	}

	/*
	 *  Execute the pipelines, feeding their output to
	 *  ProcessNeighborLine().  Turn of link state clash
	 *  checking before we execute the pipelines, and then
	 *  turn it off once we are done.
	 */
	SetCheckForLinkClash();
        while(fgets(Buf, sizeof(Buf), pipe1) != NULL) 
		ProcessNeighborLine(Net, Buf);
	pclose(pipe1);
	ClearCheckForLinkClash();
	return(Net);
}

struct NetworkType *GetNeighbors( Network )
struct NetworkType *Network;
{
	switch( Network->Type ) {
	case MERIT:	return( GetMERITNetworkState( Network ));
			/*NOTREACHED*/
	case GENERICNET:	return( GetGENERICNetworkState( Network ));
			/*NOTREACHED*/
#ifdef NOTUSEDYET
	case NSFNET:	return( GetNSFNETNeighborTable( Network ));
			/*NOTREACHED*/
	case NSFNETT3R:	return( GetNSFNETNeighborTable( Network ));
			/*NOTREACHED*/
	default:	return( GetGenericNeighborTable( Network ));
			/*NOTREACHED*/
#endif
	case NSFNETT3:	/*return( GetNSFNETNeighborTable( Network ));*/
			return( GetNSFNETNeighStates( Network ) );
			/*NOTREACHED*/
	default:	panic("GetNeighborTable: Network Type Error!!\n");
			/*NOTREACHED*/
	}
	/*NOTREACHED*/
}

#ifdef STANDALONE

static void DisplayNeighborTable( nt )
struct NeighborTableType *nt;
{
	register int i;

	for ( i = 0; nt[i].Name[0] != '\0'; i++) 
		printf("%s %2.2d %2.2d\n",nt[i].Name, nt[i].State,
							nt[i].Queryable);
}

usage()
{
	fprintf(stderr,"Usage: gnt { NSFNET | MERIT } <nodename>\n");
	exit(1);
}
 
main(argc, argv)
int argc;
char *argv[];
{
	int type;
	struct NeighborTableType *nt;
	extern void exit();

	if (argc == 3) {
		if (strcmp(argv[1], "NSFNET") == 0) 
			type = NSFNet;
		else if (strcmp(argv[1], "MERIT") == 0) 
			type = MERIT;
		else {
			usage();
			/*NOTREACHED*/
		}
		init_query_engine();
		if ((nt = GetNeighborTable(type, argv[2])) == NULL) 
			printf("ERROR\n");
		else 
			DisplayNeighborTable( nt );
	}
	exit(0);
}
#endif
