/*
 * Copyright (c) 1992 Regents of the University of Michigan.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms are permitted
 * provided that this notice is preserved and that due credit is given
 * to the University of Michigan at Ann Arbor. The name of the University
 * may not be used to endorse or promote products derived from this
 * software without specific prior written permission. This software
 * is provided ``as is'' without express or implied warranty.
 */

/************************************************************************
 *	scan.c - scan through log files and dated network status files	*
 *									*
 *   Tape Player Code - This code deals with the playback mechanism 	*
 *			in the graphical display.			*
 *									*
 ************************************************************************/

#include <stdio.h>
#include <time.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/param.h>
#include <unistd.h>
#include "pathnames.h"
#include "xmap.h"


static char *MakeLogFileName();
static char *MkNetStatusFile();

/************************************************************************
 *   SynchWNetStatusFile() - If available, read the days network *
 *				status file, and synch map with it	* 
 ************************************************************************/
SynchWNetStatusFile(CurrentDaysAgo)
int CurrentDaysAgo;
{
	char *statusfilename;
	extern void ClearAndReadStatusFile();

	return( 1 );		/* bypass this for now */
	/*NOTREACHED*/

	fprintf(stderr,
	  "Checking the network status file for %s\n",datestr(CurrentDaysAgo));
	if ((statusfilename=MkNetStatusFile(CurrentDaysAgo))==NULL) {
		fprintf(stderr,
			"network status file for %s ago is not available\n",
			datestr(CurrentDaysAgo));
	} else {
		fprintf(stderr,"StatusFile %s AVAILABLE\n",statusfilename);
		ClearAndReadStatusFile( statusfilename );
	}
}

/************************************************************************
 * OpenNextLogFile() - Going forward or backward, this routine opens the* 
 *			next available log file.			*
 ************************************************************************/
int OpenNextLogFile( CurrentDaysAgo, direction )
int *CurrentDaysAgo;
int direction;
{
	char *filename;
	static FILE *stream;

	switch (direction) {
	case FORWARD:
		if ( --*CurrentDaysAgo < 0 ) {
			fprintf(stderr, "Can't go past today!\n");
			*CurrentDaysAgo=0;
			return(NULL);
		}
		break;
	case BACKWARD:
		if ( ++*CurrentDaysAgo > 200 ) {
			printf(stderr, "Can't go more than 200 days ago!\n");
			*CurrentDaysAgo=200;
			return(NULL);
		}
		break;
	default:
		fprintf(stderr,"OpenNextLogFile(): internal error unknown direction %d\n",direction);
		exit(1);
		/*NOTREACHED*/
	}

    	filename=MakeLogFileName( *CurrentDaysAgo );
	return( ReadLogFile( filename, direction ) );
}

/************************************************************************
 * gettransitionrecord() - return a single transition record from the 	*
 *			log file.					*
 ************************************************************************/
struct transition *gettransitionrecord( direction )
int direction;
{
char *dataline, *getrecord();
struct transition *t, *ParseLogEntry();

	while( (dataline=getrecord( direction )) != NULL ) 
		if ((t=ParseLogEntry( dataline ))!=NULL) 
			break;

	if ( dataline == NULL ) 
		return( NULL );    /* No more lines in this direction ! */
        return( t );
}

/************************************************************************
 *	getrecord() - Externally callable module returns next log record* 
 *			for the	particular direction.			*
 *									*
 ************************************************************************/
char *getrecord(direction)
int direction;
{
	static char *dataptr=NULL;
	static int virgin = 1;
	static int days;

	if (virgin) {
		days = -1;	/* Go backward - start today */
		virgin = 0;
		if ((OpenNextLogFile(&days, BACKWARD)) != 0 ) {
			fprintf(stderr, "Error opening inital log files\n");
			exit(1);
		}
	}

	switch (direction) {
	case FASTFORWARD:
		if ((OpenNextLogFile(&days, FORWARD)) != 0) {
			fprintf( stderr,"Error opening next log file\n");
			return(NULL);
		}
		SynchWNetStatusFile(days);
		/* now fall through */
	case FORWARD:
		if ((dataptr=getline(FORWARD))==NULL) {
			if ((OpenNextLogFile(&days, FORWARD)) != 0) {
				fprintf(stderr, 
					"Error opening next log file\n");
				return(NULL);
			}
			SynchWNetStatusFile(days);
       			if ((dataptr = getline(FORWARD)) == NULL) {
                		fprintf(stderr,
					"Error opening next log file\n");
                		return(NULL);
			}
			break;
		}
		break;

	case FASTBACKWARD:
		SynchWNetStatusFile(days);
		if ((OpenNextLogFile(&days, BACKWARD)) != 0) {
			fprintf(stderr, "Error opening previous log file\n");
                	return(NULL);
		}
		fprintf(stderr, "Make sure you go to a new status file\n");
		/* now fall through */
	case BACKWARD:
       		if ((dataptr = getline(BACKWARD)) == NULL) {
			SynchWNetStatusFile(days);
            		if ((OpenNextLogFile(&days, BACKWARD)) != 0) {
                		fprintf(stderr,
					"Error opening previous log file\n");
                		return(NULL);
            		}
			SynchWNetStatusFile(days);
       			if ((dataptr = getline(BACKWARD)) == NULL) {
                		fprintf(stderr,
					"Error opening previous log file\n");
                		return(NULL);
			}
			break;
		}
		break; 

	default:
		fprintf(stderr,"Unknown direction: %d\n",direction);
		exit(1);
		/*NOTREACHED*/
	}
	/*printf("getrecord() returning %s\n",dataptr);*/
	return( dataptr );
}

#ifdef STANDALONE

main()
{
	char cmd[BUFSIZ];

	while (gets(cmd) != NULL) {
		if (cmd[0] == '>') 
			getrecord(FORWARD);
		else
			getrecord(BACKWARD);
	}
}

#endif

/************************************************************************
 * MakeLogFileName() - Make a log filename based on number of days ago 	*
 ************************************************************************/
static char *MakeLogFileName(DaysAgo)
int DaysAgo;
{
	extern char Error_Log[];		/* rover log file prefix */
	static char LogFile[MAXPATHLEN];
	
	sprintf(LogFile, "%s.%s", Error_Log, datestr( DaysAgo ));
	fprintf( stderr,"MakeLogFileName(): LogFile=%s\n",LogFile);
	return(LogFile);
}

/************************************************************************
 *	MkNetStatusFile() - Given a daysago parm, return the 	*
 *				name of the MkNetStatusFile of the day*
 ************************************************************************/
static char *MkNetStatusFile(CurrentDaysAgo)
int CurrentDaysAgo;
{
	static char filename[MAXPATHLEN];
	extern char progname[];
	char *getenv();
	struct stat buf;
	char *pingkydir=getenv("PINGKYDIR");

	if (pingkydir==NULL) pingkydir=DEFAULT_PINGKY_DIR;
	
	sprintf(filename, "%s/%s.%s", pingkydir, progname, datestr(CurrentDaysAgo));
	if (stat(filename, &buf) == 0) {
		printf("MkNetStatusFile(%s) FOUND!!\n", filename);
		return(filename);
	}
	else {
		printf("MkNetStatusFile(%s) *NOT* FOUND!!\n",filename);
		return(NULL);
	}
}

