
#	$RCSfile: syncdb.pl,v $
#	$Revision: 2.0 $
#	$Author: ripe-dbm $
#	$Date: 1996/08/08 10:47:30 $

# This is a client that will update objects read from a file directly
# in the database, without interference of updated.
# It will update the object in the database that is linked to the
# source field of the object, so better make sure you have a source field
# and that it has a database associated to it ...

PERL5OPTIONS

@INC = ("LIBDIR", @INC);

require "adderror.pl";
require "cldb.pl";
require "dbadd.pl";
require "dbclose.pl";
require "dbopen.pl";
require "enread.pl";
require "enwrite.pl";
require "getopts.pl";
require "misc.pl";
require "rconf.pl";
require "syslog.pl";
require "whoisqry.pl";

&Getopts('h:s:u:V');

#
# We can uncomment the next line if we have access to the
# serial file of the remote server, this is easy for 
# backup & local mirror servers
#
# opt_u works exactly the same but checks the serial file instead of the
# server for changes
#
# $SYNCFASTMODE="/ncc/dbase/log/serials/".$opt_s.$CURRENTEXTENSION;


if ((!$opt_s) || (($opt_u) && ($opt_u!~ /^\d+\:\d+$/))) {

   print <<"EOF";
    
Usage: $PROGRAMNAME [ -h hostname ] -s source [ -u period:delay ]
   
Where:

-h hostname      - hostname of server that provides updates
-s source        - which source to query for
-u period:delay  - the program will run for 'period' seconds and
                   will do a query every 'delay' seconds

EOF

   exit 1;
   
}


#
# Read config file from RIPEDBCNF, or set to default.

$conffile=$ENV{"RIPEDBCNF"};
$conffile= "DEFCONFIG" unless $conffile;
&rconf($conffile);

#
# run on low priority, whois server queries have an
# higher priority

system("$RENICECMD 10 $$ > /dev/null 2>/dev/null");

%entry=();

$locklink=$LOCKDIR.$PROGRAMNAME.".".$opt_s.".RUNNING";

$tmpdata=$TMPDIR."/".$PROGRAMNAME.".".$$;

#
# this handler is to stop updating in a as decent as possible
# when errors occur

$STOPUPDATING=0;

sub stopupdating {
    
    $STOPUPDATING=$_[0];
            
}


#
# this handler is for the the normal timeout as
# defined by $opt_u

$TIMEOUT=0;

sub timeouthandler {

    $TIMEOUT=1;
    
}

$PROBLEMS='(ERROR|Warning|Closing\s+connection)';

#
# exit syncdbase in a clean way

sub exitsyncdbase {
   local($returncode)=@_;
   
   if (!$returncode) {
      unlink($tmpdata) || warn "couldn\'t remove temporary datafile: $tmpdata\n";
   }
   
   unlink($locklink) || warn "couldn\'t remove lockfile: $locklink\n";
   
   &fatalerror($returncode) if ($returncode!~ /^\d+$/);
   
   exit(1) if (($returncode) && ($returncode!~ /^\d+$/));
   
   exit $returncode;
   
}

sub skiperrorserial {
   local(*entry,$action,$source,$msg)=@_;
   
   $entry{"pn"}="problem found" if (!&entype(*entry));
   $entry{"so"}=$source if (!$entry{"so"});
   
   &adderror(*entry,$msg);
   
   &writeseriallog($action,*entry);
     
}

#
# doaction ( )
#
# does all the work.
#

sub doaction {

    local(*entry,$action,$source,$serial) = @_;
    
    print STDERR "syncdbase - doaction($action,$serial) called\n" if $opt_V;
    
    local($returncode)=0;
    
    #
    # open database file associated with "so" for writing

    local($mirrordb)='mirrordb';
    local(%mirrordb,@mirrordb);
    
    print STDERR "syncdbase - doaction opening database\n" if $opt_V;

    local($type)=&entype(*entry);

    if (&dbopen(*mirrordb, *entry, 1)) {
       # Open classless database

       print STDERR "syncdbase - doaction opening classless db\n" if $opt_V;
       &dbclopen(*entry,1) if ($CLASSLESSDBS{$type});

       # Looks like we have some locking problem, so let's
       # lock the thing before we do anything else ...

       print STDERR "syncdbase - doaction locking databases\n" if $opt_V;
		
       # do we delete or not ?

       if ($STOPUPDATING) {
          
          $returncode="$PROGRAMNAME (cleanly) terminated by signal ($STOPUPDATING)";
       
       }
       else {
       
          if ($action=~ /^$DELETEACTION$/i) {

             print STDERR "syncdbase - doaction deleting entry ($entry{$type})\n" if $opt_V;
             $dbstat = &dbdel(*mirrordb, *entry, $type, $DELETEOPTION | $NOCHECKSOPTION);

          }
          elsif ($action=~ /^$ADDACTION$/i) {
          
             print STDERR "syncdbase - doaction adding entry ($entry{$type})\n" if $opt_V;
             $dbstat = &dbadd(*mirrordb, *entry, $type, 0);
          
          }
          
       }  

       #
       # noop, just print stat if verbose but always print errors!!!

       print STDERR "syncdbase - doaction checking stat\n" if $opt_V;

       if ($dbstat == $E_NOOP) {
          print STDERR "ERROR, incomingserial ($action: $entry{$type}): ", $LOGFILE{"INCOMINGSERIALDIR"}, $serial, " [NOOP]\n\n";
          &skiperrorserial(*entry,$action,$source,$MESSAGE[$dbstat]);
       }
       elsif ($dbstat!=$OK) {
          print STDERR "ERROR, incomingserial ($action: $entry{$type}): ", $LOGFILE{"INCOMINGSERIALDIR"}, $serial, " ", $MESSAGE[$dbstat], "\n\n";
          &skiperrorserial(*entry,$action,$source,$MESSAGE[$dbstat]);
       }
       elsif ($opt_V) {
          print STDERR "OK, ($action: $entry{$type}) serial: $serial\n";
       }

       &dbclose(*mirrordb);
       &dbclclose() if ($CLASSLESSDBS{$type});

    }
    
    return $returncode;

}

sub UpdateVersionOne {
   local($input,$line,$opt_s,*from,$to,$period)=@_;

   local(%entry,%oldentry);
   local($first,$last,$source,$type,$oldtype);
   local($i,$oldi,$action,$oldaction);
   
   local($returncode)=0;
   
   $line=~ /^\%START\s+Version:\s*\d+\s+(\S+)\s+(\d+)\-(\d+)\s*$/;
   
   $source=$1;
   $first=$2;
   $last=$3;
   
   $basename=$LOGFILE{"SERIALINCOMINGDIR"}.$source;
   
   print STDERR "basename: $basename source: $source first: $first last: $last\n" if $opt_V;
      
   $i=$first;
   $line=<$input>;
   
   &exitsyncdbase("got other start serial ($first) then expected ($from)") if ($first!=$from);
   &exitsyncdbase("got other end serial ($last) then expected ($to)") if (($to!~ /^LAST$/i) && ($last!=$to));
   &exitsyncdbase("got source ($source) then expected ($opt_s)") if ($source ne $opt_s); 
   
   
   while ((!$STOPUPDATING) &&
          ($i<=$last) &&
          ($line=<$input>) &&
          ($line!~ /^\s*$/) &&
          ($line!~ /^\%END/)) {
   
      print STDERR "update: $basename.$i\n" if $opt_V;
      
      if ($line=~ /^$ADDACTION$/i) {
         $action=$ADDACTION;
      }
      elsif ($line=~ /^$DELETEACTION$/i) {
         $action=$DELETEACTION;
      }
      else {
         return $line;
      }
      
      open(OUTP,">$basename.$i");
      print OUTP $line;
      
      $line=<$input>;
      
      return $line if ($line!~ /^\s*$/);
      
      $type=&enread($input, *entry, -1);
      
      return $type if (!defined($OBJATSQ{$type}));
      
      &enwrite(OUTP,*entry,0,1,0);
      
      close(OUTP);
      
      print STDERR "action: $action\n" if $opt_V;  
      
      &doaction(*oldentry,$oldtype,$oldaction,$source,$oldi) if ($i!=$first);
   
      %oldentry=%entry;
      $oldtype=$type;
      $oldaction=$action;
      $oldi=$i;
   
      $i++;
      
   }
   
   # print STDERR "last line: $line";

   # handy for debugging!

   local($offset)=tell(TMPFILE);

   if ($line=~ /^\s*$/) {
      $line=<TMPFILE>;
      print STDERR "read extra line: ", $line if $opt_V;
   }

   if ($STOPUPDATING) {
      
      $returncode="$PROGRAMNAME (cleanly) terminated by signal ($STOPUPDATING)";
      
      close(TMPFILE);
      
   }
   elsif ($line!~ /^\%END/) {
   
      local(@lines)=();
      local($oldline)=$line;
      
      if (($line=~ /^\d+$/) && ($offset>$line)) {
         
         seek(TMPFILE, $line, 0);
         
         while (($line=<TMPFILE>) && ($offset<=tell(TMPFILE))) {
            push(@lines, $line);
         }
         
      }
      else {
         @lines=($line);
      }
   
      if ((@lines=grep((/^\%/) && (/$PROBLEMS/i), @lines))) {
      
         if (grep(/ERROR/i, @lines)) {
         
            $returncode="Error found: ".join("", @lines);
            
         }
         else {
            
            print STDERR "Warning (from raw input data): ".join("", @lines);
            
         }
      
      }
      else {
      
         $returncode="Error in output from whois ($basename.$i)\nnear offset: $offset line: ", $oldline, "\n";
         
      }
      
      close(TMPFILE);
      
   }
   else {
      
      close(TMPFILE);
      $returncode=&doaction(*oldentry,$oldtype,$oldaction,$source,$oldi) if ((!$returncode) && ($i!=$first));
   
   }
   
   $from=$last+1;
   
   return $returncode;
}


#
# Main program

#
#
# check if this source *cannot* be updated with other methods...

if ($CANUPD{$opt_s}) {
   print STDERR "You cannot allow updates to database: $opt_s\n";
   print STDERR "*and* mirror this database at the same time.\n\n";
   print STDERR "Please remove your CANUPD variable for this database in the \'config\' file.\n";

   exit 1;
}

#
# create lock

# print STDERR "files: $locklink and $tmpdata -$$-\n";

if (!symlink($tmpdata,$locklink)) {
   
   local($linkdata)=readlink($locklink);
   
   local($pid)=$linkdata=~ /\.(\d+)$/;
   
   print STDERR "***ERROR***: found other $PROGRAMNAME (PID=$pid) running\n";
   
   print STDERR "please check for presence of process: $pid\n";
   print STDERR "link:  \'$locklink\'\nand file: \'$linkdata\'\n\n";
   print STDERR "The query interval in your cronjob might be too small if this\n";
   print STDERR "problem happens more often and you cannot find these files\n\n";
   print STDERR "If you find these files:\n";
   print STDERR "Remove them to restart automatic mirroring\n\n";
   print STDERR "Please send a bug report to <ripe-dbm\@ripe.net> with\n";
   print STDERR "all data (\'ls -al $LOCKDIR\' and contents of \'$linkdata\')\n";
   print STDERR "when you cannot find an explanation for this problem\n";
   
   exit 1;
}

#
# and now really start running...

local($returncode)=0;
local($period)=0;
local($failedconnects)=0;

local($from,$to,$version,$line,$delay,$outputfound,$msg,$mtime,$oldmtime); 

if ($opt_u) {
   
   ($period,$delay)=split(/\:/, $opt_u);
   
   $SIG{"ALRM"}='timeouthandler';
   
   alarm $period;
   
   #
   # sleep for a random 0..59 seconds to divert
   # the load from all clients on the server even if cron starts
   # them all at the same time
   
   sleep int(rand(60));
   
}

#
# we want to exit as cleanly as possible when we get a signal to do so
   
$SIG{'HUP'}='stopupdating';
$SIG{'INT'}='stopupdating';
$SIG{'KILL'}='stopupdating';
$SIG{'TERM'}='stopupdating';
   

$from=&getcurrentserial($opt_s)+1;

$to="LAST";

while ((!$TIMEOUT) && (!$STOPUPDATING)) {

   #
   # do the whois query
   
   if (($msg=&initwhoisqry($opt_h?$opt_h:"WHOISHOST",$opt_p?$opt_p:"",
                           "-g $opt_s:$UPDATEVERSION:$from-$to",AF_INETVALUE,SOCKADDRVALUE,SOCK_STREAMVALUE))) {

      &exitsyncdbase($msg) if (!$period);
      
      $failedconnects++;
      
      &exitsyncdbase($msg." ($failedconnects times!)") if ($failedconnects*$delay>$period/20);      
   
   }
   
   #
   # slurp the complete input in order to disconnect as soon as possible.

   open(TMPFILE,"+>".$tmpdata) || &fatalerror("cannot open whois output file");

   while ($line=<WHOIS_S>) {
      print TMPFILE $line;
   }

   seek(TMPFILE,0,0);

   $outputfound=0;

   #
   # find output
   
   while (($line=<TMPFILE>) && ($line!~ /^\%START/)) { 
      print STDERR "skip:", $line if $opt_V;
   
      if (($line=~ /^%/) && ($line=~ /$PROBLEMS/i)) {
   
         close(TMPFILE);
      
         if ($line=~ /ERROR/i) {
            &exitsyncdbase("Error found: ".$line);
            
         }
         else {
            print STDERR "Warning (from raw input data): $line\n"; 
         }
      
      }
   
      next if (($line=~ /^\s*$/) || ($line=~ /^\s*[\*\#]/));

      #
      # for backwards compatibility, can go in next release
      
      last if ($line=~ /^ERROR\:\s+Warning\s+\(1\)/);
      
      $outputfound=1;

   }

   #
   # quit if we found meaningless data, but keep going if we
   # found nothing (usually server timeouts, bad connectivity)
   
   if ($line!~ /^\%START/) {
   
      if ($outputfound) {
         &exitsyncdbase("No valid data found in whois data: $tmpdata");
      }
      else {
         &exitsyncdbase(0) if (!$period);
      }
   
   }
   else {
   
      print STDERR "last:", $line if $opt_V;

      #
      # call the right updating procedure
   
      if ($line=~ /^\%START\s+Version:\s*(\d+)\s+/) {
         print STDERR "found: ", $line if $opt_V;
         $version=$1;
   
         if ($version==1) {
            ($returncode)=&UpdateVersionOne(TMPFILE,$line,$opt_s,*from, $to, $period);
         
         }
         else {
            &exitsyncdbase("got unsupported update version format ($version), we support versions: ".join(" ",@UPDATEVERSIONS));   
            close(TMPFILE);
         }

         &exitsyncdbase($returncode) if ($returncode);

      }
   
   }
   
   last if (!$period);
   
   if ($SYNCFASTMODE) {
   
      $oldmtime=(stat($SYNCFASTMODE))[9];
      $mtime=$oldmtime;
      
      while ($oldmtime==$mtime) {
         sleep $delay;
         $mtime=(stat($SYNCFASTMODE))[9];
         print STDERR "old: $oldmtime mtime: $mtime\n" if ($opt_V);
      }
   
   }
   else {
      sleep $delay;
   }
   
}

&exitsyncdbase($returncode);

# end of program