#charset "us-ascii"
/* 
 *  Copyright (c) 2005 by Kevin Forchione. All rights reserved.
 *   
 *  This file is part of the TADS 3 Services Pack
 *
 *  tsp_inheritance_order.t
 *
 *  Provides a mechanism for analyzing the inheritance 
 *  structure of an object. 
 */

#include "tsp_inheritance_order.h"
#include "tsp_mod_registry.h"

RegisterModule

modify Object
{
    /*----------------------------------------------------------------
     *  STRUCTURE LISTS
     *----------------------------------------------------------------
     */
    inherOrderIter(sVec, func)
    {
        sVec.applyAll(new function(val) {
            local ret;

            // clear any existing instances of self in the vector
            switch(dataType(val))
            {
                case TypeList:
                    val[1] == self ? ret = nil : ret = val;
                    break;

                case TypeObject:
                    val == self ? ret = nil : ret = val;
                    break;

                default:
                    ret = val;
            }

            return ret;
        });
        
        // add self to the vector
        func(self, getPropList());
        
        // insert the superclasses of this element into the queue
        foreach (local o in getSuperclassList())
            o.inherOrderIter(sVec, func);
    }

    /*
     *  Returns a structure list for the object in the format of 
     *  [ [ obj [ prop-list ], [ superclass1 [ prop-list ]], ... ]
     *
     *  Valid argument forms:
     *  
     *      getInherStrucList()
     *      getInherStrucList(func)
     *      getInherStrucList(suppress)
     *      getInherStrucList(func, suppress)
     *
     *  If suppress is true then object/propList sub-elements are 
     *  not added to the list when the propList is empty.
     *
     *  The callback function func should be of the form func(obj, 
     *  prop) that returns either true or nil. If the function 
     *  returns true the object / property sub-element is added to 
     *  the list, otherwise it is not.
     */
    getInherStrucList([cond])
    {
        local sVec, func, suppress;

        func = cond.car();
        sVec = new Vector(40);
        
        if (cond.length() == 1 
            && dataType(cond[1]) is in (TypeNil, TypeTrue))
        {
            suppress = cond[1];
            func = nil;
        }
        else if (cond.length() > 1 
            && dataType(cond[2]) is in (TypeNil, TypeTrue))
            suppress = cond[2];
        
        inherOrderIter(sVec, new function(obj, propList) { 
            if (func == nil)
            {
                if (propList.length() != 0 || !suppress)
                    sVec = sVec.append([obj, propList]);
            }
            else 
            {
                local pVec = new Vector(50);
                
                foreach (local prop in propList)
                    if (func(obj, prop))
                        pVec = pVec.appendUnique([prop]);

                if (pVec.length() != 0 || !suppress)
                    sVec = sVec.append([obj, pVec.toList()]);
            }
        });

        /* remove any empty elements */
        sVec.removeElement(nil);
        
        return sVec.toList();
    }
        
    /*
     *  Returns a structure list.
     *
     *  These property lists only contain properties for the superclass
     *  object from which the property is inherited.
     *
     *  Valid argument forms:
     *  
     *      getInherDefStrucList()
     *      getInherDefStrucList(func)
     *      getInherDefStrucList(suppress)
     *      getInherDefStrucList(func, suppress)
     *
     *  If suppress is true then object/propList sub-elements are 
     *  not added to the list when the propList is empty.
     *
     *  The callback function func should be of the form func(obj, 
     *  prop) that returns either true or nil. If the function 
     *  returns true the object / property sub-element is added to 
     *  the list, otherwise it is not.
     *
     *  If an argument is passed it should be a function of the form 
     *  func(obj, prop) that returns either true or nil. If the function
     *  returns true the object / property is added to the list,
     *  otherwise it is not.
     */
    getInherDefStrucList([cond])
    {
        local func = cond.car();
        local suppress;
        
        if (cond.length() == 1 && dataType(cond[1]) is in (TypeNil,
            TypeTrue))
        {
            suppress = cond[1];
            func = nil;
        }
        else if (cond.length() > 1 && dataType(cond[2]) is in (TypeNil,
            TypeTrue))
            suppress = cond[2];
        
        return getInherStrucList(new function(obj, prop) {
            if (obj != self.propDefined(prop, PropDefGetClass))
                return nil;
            else if (func == nil)
                return true;
            else return func(obj, prop);
        }, suppress);
    }
    
    /*
     *  Returns a structure list that only contains properties that are
     *  mutable by the program (i.e. not TypeDString, TypeCode,
     *  TypeNativeCode.) 
     *
     *  Furthermore, these lists only contain properties for 
     *  the superclass object from which the property is 
     *  inherited.
     *
     *  Valid argument forms:
     *  
     *      getInherStateStrucList()
     *      getInherStateStrucList(func)
     *      getInherStateStrucList(suppress)
     *      getInherStateStrucList(func, suppress)
     *
     *  If suppress is true then object/propList sub-elements are 
     *  not added to the list when the propList is empty.
     *
     *  The callback function func should be of the form func(obj, 
     *  prop) that returns either true or nil. If the function 
     *  returns true the object / property sub-element is added to 
     *  the list, otherwise it is not.
     */
    getInherStateStrucList([cond])
    {
        local func = cond.car();
        local suppress;
        
        if (cond.length() == 1 && dataType(cond[1]) is in (TypeNil,
            TypeTrue))
        {
            suppress = cond[1];
            func = nil;
        }
        else if (cond.length() > 1 && dataType(cond[2]) is in (TypeNil,
            TypeTrue))
            suppress = cond[2];
        
        return getInherStrucList(new function(obj, prop) {
            if (obj.propType(prop) is in (TypeDString, TypeCode,
                    TypeNativeCode))
                return nil;
            else if (obj.propType(prop) is in (TypeNil, nil)
                && dataType(prop) == TypeProp)
                return nil;
            else if (obj != self.propDefined(prop, PropDefGetClass))
                return nil;
            else if (func == nil)
                return true;
            else return func(obj, prop);
        }, suppress);
    }

    /*
     *----------------------------------------------------------------
     *  CLASS LISTS
     *----------------------------------------------------------------
     */

    /*
     *  Method returns a list of the object and the superclasses that make
     *  up this object's structure. 
     *
     *  If an argument is passed it should be a
     *  function of the form func(obj, propList) that returns either
     *  true or nil. If the function returns true the object is added
     *  to the classes list, otherwise it is not.
     *  
     *  Valid argument forms:
     *  
     *      getInherOrderList()
     *      getInherOrderList(func)
     *
     *  The callback function func should be of the form func(obj, 
     *  propList) that returns either true or nil. If the function 
     *  returns true the object is added to the list, otherwise it 
     *  is not.
     */
    getInherOrderList([cond])
    {
        local sVec, func;
        
        func = cond.car();
        sVec = new Vector(40);
                
        inherOrderIter(sVec, new function(obj, propList) {
            if (func == nil || func(obj, propList))
                sVec = sVec.appendUnique([obj]);
        });
        
        /* remove any empty elements */
        sVec.removeElement(nil);
        
        return sVec.toList();
    }

    /*
     *  Returns the next inheritance object based on the
     *  object's inheritance order. 
     *
     *      - If no argument is passed 
     *          then self is assumed as the class with which to 
     *          begin the search. 
     *      - If val is an integer N
     *          then the N+1 class is returned. 
     *      - If val is an object
     *          then the next object in the inheritance order is 
     *          returned. 
     * 
     *  If a second argument is provided it indicates whether to return nil or
     *  throw a runtime exception error when there is no next object in the 
     *  inheritance order. A second argument of true will throw a runtime 
     *  error, while no argument or an argument of nil will return nil.
     */
    getNextInherOrder([args]) 
    {
        local val, exc, lst, f;

        switch (args.length())
        {
            case 0:
                val = self;
                break;

            case 1:
                switch (dataType(args.car()))
                {
                    case TypeTrue:
                    case TypeNil:
                        val = self;
                        exc = args.car();

                    default:
                        val = args.car();
                }
                break;

            default:
                val = args[1];
                exc = args[2];
        }

        lst = getInherOrderList();

        /*
         *  Our inheritance argument is an integer N, which
         *  means that we want the object that is the N+1
         *  member of our inheritance order.
         */
        if (dataType(val) == TypeInt)
        {
            if (val < 0 || val + 1 > lst.length())
            {
                if (exc)
                    throw new TspInherOrderError('no next object');
                else return nil;
            }
            else return lst[val+1];
        }
        else if (dataType(val) == TypeObject)
        {
            f = lst.indexOf(val);
            if (f == nil)
            {
                if (exc)
                    throw new TspInherOrderError('no next object');
                else return nil;
            }

            if (f == lst.length())
            {
                if (exc)
                    throw new TspInherOrderError('no next object');
                else return nil;
            }

            return lst[f+1];
        }
        else 
            throw new TspInherOrderError('invalid argument datatype');
    }

    /*
     *----------------------------------------------------------------
     *  PROPERTY LISTS
     *----------------------------------------------------------------
     */
        
    /*
     *  Method returns a list of the properties that make up this
     *  object's structure. 
     *
     *  Valid argument forms:
     *  
     *      getInherPropList()
     *      getInherPropList(func)
     *
     *  The callback function func should be of the form func(obj, 
     *  prop) that returns either true or nil. If the function 
     *  returns true the property is added to the list, otherwise it 
     *  is not.
     */
    getInherPropList([cond])
    {
        local sVec, func;

        func = cond.car();
        sVec = new Vector(1000);
        
        inherOrderIter(sVec, new function(obj, propList) {
            if (func == nil)
                sVec = sVec.appendUnique(propList); 
            else foreach (local prop in propList)
                    if (func(obj, prop))
                        sVec = sVec.appendUnique([prop]);
        });
        
        /* remove any empty elements */
        sVec.removeElement(nil);
        
        return sVec.toList();
    }
    
    /*
     *  Returns a property list.
     *
     *  The property list only contains properties for the superclass
     *  object from which the property is inherited.
     *
     *  If an argument is passed it should be a function of the form 
     *  func(obj, prop) that returns either true or nil. If the function
     *  returns true the object / property is added to the list,
     *  otherwise it is not.
     */
    getInherDefPropList([cond])
    {
        local func = cond.car();
        
        return getInherPropList(new function(obj, prop) {
            if (obj != self.propDefined(prop, PropDefGetClass))
                return nil;
            else if (func == nil)
                return true;
            else return func(obj, prop);
        });
    } 

    /*
     *  Returns a property list.
     *
     *  The property list only contains state properties for 
     *  the superclass object from which the property is inherited.
     *
     *  If an argument is passed it should be a function of the form 
     *  func(obj, prop) that returns either true or nil. If the function
     *  returns true the object / property is added to the list,
     *  otherwise it is not.
     */
    getInherStatePropList([cond])
    {
        local func = cond.car();

        return getInherPropList(new function(obj, prop) {
            if (obj.propType(prop) is in (TypeDString, TypeCode,
                    TypeNativeCode))
                return nil;
            else if (obj.propType(prop) is in (TypeNil, nil)
                && dataType(prop) == TypeProp)
                return nil;
            else if (obj != self.propDefined(prop, PropDefGetClass))
                return nil;
            else if (func == nil)
                return true;
            else return func(obj, prop);
        });
    }  
}

class TspInherOrderError: RuntimeError
{
    construct(msg)
    {
        exceptionMessage = msg;
        inherited(0);
    }
}

/*
 *  The basic purpose of these methods is to produce a clone
 *  of an object that has a single inheritance model. This 
 *  is especially useful when method inheritance is controlled
 *  with the inheritedobj macro.
 */
modify TadsObject
{
    /* 
     *  TSP Single Inheritance Model Clone
     *
     *  Creates a clone of this object whose superclass
     *  list consists of a single object, such that each
     *  of this object's inheritance order objects is 
     *  derived from the next.
     */
    createTspSimClone()
    {
        local obj, lst, curr, next;

        obj = createTspMimClone();
        lst = obj.getInherOrderList();
        for (local i = 1; i < lst.length(); ++i)
        {
            curr = lst[i];
            next = lst[i+1];
            curr.setSuperclassList([next]);
        }
        return obj;
    }

    /*
     *  TSP Multiple Inheritance Model Clone
     *
     *  Creates a clone of this object whose superclass
     *  list consists of TSP ODDBSM clones representing this
     *  object's inheritance order.
     */
    createTspMimClone()
    {
        local obj, lst, cls, vec;

        lst = getInherOrderList().sublist(2);
        vec = new Vector(lst.length());
        foreach (local obj in lst)
            if (obj.ofKind(TadsObject) && obj != TadsObject)
                vec.append(obj.createTspOddBsmClone());

        cls = vec.toList();
        if (cls.length() == 0)
            cls += TspSimObject;
        obj = createClone();
        obj.setSuperclassList(cls);

        return obj;
    }

    /*
     *  TSP Object Directly-defined Behavior-state Clone
     *
     *  Creates a clone of this object and then set
     *  its superclass to TspSimObject. This creates
     *  a TspSimObject instance that defines the directly-
     *  defined behavior-states of this object.
     */
    createTspOddBsmClone() 
    {
        local obj;

        obj = createClone();
        obj.oddBsmPtr_ = self;
        obj.setSuperclassList([TspSimObject]);

        return obj;
    }
}

/*
 *  TSP Single Inheritance Module Object
 *
 *  This is the superclass that all object directly-defined 
 *  behavior-state clones inherit from. This class is used 
 *  to record information about the object directly-defined 
 *  behavior-state pointer to the original object, so that
 *  the clone isn't simply a mysteriously anonymous instance 
 *  of TadsObject.
 */
class TspSimObject: object
{
    oddBsmPtr_ = nil

    /*
     *  Returns true if this object is inherits from
     *  an object directly-defined behavior-state of 
     *  kind obj; otherwise the method returns nil.
     */
    oddBsmOfKind(obj)
    {
        return (getOddBsmPtrList().indexOf(obj) != nil);
    }

    /*
     *  Returns the list of directly-defined clones
     *  that comprise this object.
     */
    getOddBsmPtrList()
    {
        local vec;

        vec = new Vector(50);
        foreach (local obj in getInherOrderList())
            if (obj.oddBsmPtr_)
                vec += obj.oddBsmPtr_;

        return vec.toList();
    }
}

/*
 *  A class that restructures the object deriving from it into
 *  a single-inheritance clone (TspSimObject) of itself. 
 */
class TspSimObjectCvtr: PreinitObject
{
    execute()
    {
        local lst, obj;

        /* 
         *  Get this object's superclass list and remove
         *  SiClone from the list. This will remove the
         *  class from appearing in its reconfiguration, 
         *  unless the object inherits from SiClone class.
         */
        lst = getSuperclassList();
        lst -= TspSimObjectCvtr;

        /*
         *  If the inheritance class list is empty
         *  we'll add TadsObject to it so that the
         *  object will simply become an instance of 
         *  TadsObject.
         */
        if (lst.length() == 0)
            lst += TadsObject;

        /*
         *  Set this object's superclass list to the 
         *  new inheritance list and create a single-inheritance
         *  clone for it.
         */
        setSuperclassList(lst);
        obj = createTspSimClone();

        /*
         *  Get the clone's superclass inheritance list. This
         *  will be a single object/class. We then set this object's
         *  superclass list to the superclass list of the clone. Now
         *  the object is essentially a single-inheritance clone
         *  of its former self. If the object was defined with a
         *  symbolic name in source code it will be referencable 
         *  using that symbolic name.
         */
        lst = obj.getSuperclassList();
        setSuperclassList(lst);
        oddBsmPtr_ = self;
    }
}