/* TADS Phone class
 *
 * Amir Karger
 * $Revision: 1.17 $
 * $Date: 1999/12/26 22:45:12 $
 *
 * See Manual.txt for descriptions of methods etc.
 *
 * See TODO document for significant things, TODO's in comments for smaller
 * ones
 */

#include "chatter.t"   // Suzanne Britton's module for handling conversation

init_phone: function;

/**********************************************************************/
// Basic phone class
//
class phone: item
    noun = 'phone' 'telephone' 'handset' 'receiver'
    sdesc = "telephone"
    ldesc = "It's a telephone. You can dial on it. It's currently
	<<self.off_hook ? "off" : "on">> the hook. "

    // this phone's phone number
    phone_number = nil
    // is this phone currently being used?
    off_hook = nil 
    // phone object that's trying to connect to this phone, if any
    ringing = nil
    // message when you use this phone to dial a valid number that noone's at
    wrong_number = "There's no answer. "
    // message when someone calls this phone and there's noone there
    no_answer = "There's no answer. "
    // message when someone calls this phone and it's being used
    busy_signal = "The phone's busy. "

    // Actor is using this phone
    set_speaker(actor) = {
	// Note that off_hook could in theory be true already, but it
	// doesn't hurt to reset it
	self.off_hook := true;
	self.speaker := actor; 
	actor.my_phone := self;
	if (actor = Me) notify(self, &dial_tone_daemon, 0);
    }
    // Actor is no longer using this phone
    unset_speaker(actor) = {
	self.speaker := nil; 
	actor.my_phone := nil;
	// Note that off_hook could in theory be nil already, but it
	// doesn't hurt to reset it
	self.off_hook := nil;
	if (actor = Me) unnotify(self, &dial_tone_daemon);
    }
    // Connect this phone to another
    connect(other_phone) = {
        self.partner := other_phone;
	other_phone.partner := self;
    }
    // Disconnect this phone from another
    disconnect(other_phone) = {
        other_phone.partner := nil;
	self.partner := nil;
    }
    dial_tone_daemon = {
        if (self.partner = nil) "\b%You% hear%s% a dial tone. ";
    }

    // Return true if actor is allowed to call this number/string
    // from this phone.
    valid_phone_num(actor, num) = { 
        "Invalid phone number. "; 
	return nil;
    }
    valid_string_str(actor, str) = {
        "Invalid phone number. "; 
	return nil;
    }

    verIoPhoneOn(actor) = { 
	if (self.partner) {
	    "%You're% already on the phone! ";
	}
    }

    // dobj is an actor with a valid phone_number OR numObj
    ioPhoneOn(actor, dobj) =
    {
	// note verDo already made sure it's a valid destination
	local phone_num, found_it, l, p;
	local other_actor, other_phone, loc;

	if (dobj = numObj) {
	    phone_num := dobj.value;

	} else if (dobj = strObj) {
	    // TODO note that we can't just change string into a number,
	    // because a ten-digit integer can't be handled by TADS
	    // We need to handle dashes. E.g. just remove them.
	    "Error! strObj doesn't work yet!";
	    exit;

	} else { // it's an actor; 
	    // test whether the actor calling self knows self's number
	    if (find(actor.phone_known, dobj) = nil) {
		"%You% %do%n't know <<dobj.fmtYour>> number! ";
		exit;
	    }
	    phone_num := dobj.phone_number;
	}

	// Make sure you're allowed to call that phone number from this phone
	switch(datatype(phone_num)) {
	    case DTY_NUMBER:
		if (self.valid_phone_num(actor, phone_num) = nil) exit;
		break;
	    case DTY_DSTRING:
		if (self.valid_phone_str(actor, phone_num) = nil) exit;
		break;
	    default:
	        "Error in phone.t! (Bad datatype for phone number)";
	}

	// Pick up the handset (fixed phone) or "turn on" the (mobile) phone
	// if you haven't yet
	self.start_dialing(actor);

	// Make sure there's a phone with that number
	other_phone := nil;
	l := firstobj(phone);
	while (l <> nil) {
	    if (l.phone_number = phone_num) {
		other_phone := l;
		break;
	    }
	    l := nextobj(l,phone);
	}

	if (other_phone) {
	    if (other_phone.off_hook) {
		// use other_phone's busy signal so you can say something like
		// "Wow, his phone is always busy!" or something
		// Note that other_phone will be self if you try to call
		// yourself. But it should correctly give a busy signal.
	        other_phone.busy_signal;
	    } else {
		// A phone may be a mobile phone located on an actor
		// Otherwise, find the phone's room
		loc := other_phone.location;
		other_actor := nil;
		if (loc = nil) {
		    // Note: this will *probably* only happen because the
		    // programmer forgot to add a location property for the
		    // phone, but there *could* be a phone that was broken,
		    // thrown away, whatever
		    other_actor := nil;
		} else {
		    while (loc.location <> nil and not isclass(loc, Actor)) 
			loc := loc.location;

		    if (isclass(loc, Actor)) {
			if (loc.phone_will_answer) other_actor := loc;
		    } else {
			l := firstobj(Actor);
			while (l <> nil) {
			    // TODO uberlocify actor
			    // TODO what if there's > 1 actor in the room?
			    // TODO what about reachable issues?
			    if (l.location = loc
				and l.phone_will_answer) {
				other_actor := l;
				break;
			    }
			    l := nextobj(l, Actor);
			}
		    }
		}

		if (other_actor = nil) {
		    other_phone.no_answer;

		} else {
		    // have the actor respond, e.g. by picking up the phone
		    other_phone.ringing := self;
		    other_actor.phone_answer(self, other_phone); 
		}
	    }

	} else { // no phone has that phone number
	    self.wrong_number;
	}
    }

    verDoPhoneAnswer(actor) = {
        if (self.partner) {
	    "%You're% already on the phone! ";
	} else if (not self.ringing) {
	    "The phone isn't ringing! ";
	}
    }
    // derived PhoneAnswer and Take methods will call this to do the actual
    // process of answering the phone
    doPhoneAnswer(actor) = {
	self.set_speaker(actor);
	self.connect(self.ringing); // connect the two phones
	self.ringing := nil; // noone's calling any more
    }

    // end phone conversation
    verDoHangup(actor) = {
        // make sure we're using it (Check off_hook, not self.partner,
	// because maybe the phone's off the hook but we're not currently
	// talking to anyone.)
	if (self.off_hook = nil) {
	    "The phone's already hung up! ";
	}
    }
    doHangup(actor) = {
	local other_actor, other_phone;
	other_phone := self.partner;

	if (actor = parserGetMe()) "%You% hang up the phone. ";

	if (self.partner) {
	    self.disconnect(other_phone);

	    other_actor := other_phone.speaker;
	    // when you hang up on someone, they'll hang up the phone
	    // and put it down. But if someone hangs up on you, you should
	    // hear a dial tone.
	    // TODO let Me automatically put the phone down too?
	    if (other_actor <> parserGetMe()) {
	        other_phone.doHangup(other_actor);
	    }
	}
	self.unset_speaker(actor);
    }
;

/**********************************************************************/
// KINDS of phones
//
// A "normal" (non-mobile) phone
// Note that you never actually take this phone, i.e. it's location
// is never changed to Me. This makes handling different meanings of
// "get the phone" etc. easier.
class fixed_phone: phone, fixeditem
    verDoTake(actor) = {
        if (actor.my_phone = self) {
	    "You already have the phone! ";
	}
    }
    doTake(actor) = {
	if (actor = parserGetMe()) "%You% pick%s% up the phone. ";
	if (self.ringing) { // someone's calling us
	    self.doPhoneAnswer(actor);
	} else {
	    // pick up the phone but noone's there
	    self.set_speaker(actor);
	}
    }

    // Pick up the handset if you haven't already
    start_dialing(actor) = {
	// (Stolen from adv.t's doWear)
	if (self.speaker <> actor) {
	    /* try taking it -- don't output "You pick up the phone" */
	    if (execCommand(actor, takeVerb, self, EC_HIDE_SUCCESS) != 0) 
	        exit;

	     //  make certain it ended up where we want it - the command
	     //  might have failed without actually indicating failure 
	    if (self.speaker <> actor) exit;
	}
    }

    // so that "put down phone" works
    doSynonym('Hangup') = 'Drop'
;

// Mobile phone
// Complication here is that 'get the phone' may mean pick up the object to
// put it in your inventory, or it may mean answer the phone to see who's
// calling.
// General rule: if the phone is ringing, 'get phone' means answer it
// (taking it beforehand if necessary). Otherwise it just means take it
// 'answer phone' means answer it, taking it beforehand if necessary.
class mobile_phone: phone
    verDoTake(actor) = {
	// This allows "pick up the phone" to mean "answer it" if you're
	// holding it.
        if (self.location <> actor) inherited.verDoTake(actor);
	// TODO if we're going to pick up *and* answer the phone,
	// do we need to test verDoPhoneAnswer also? If so, make sure
	// we don't print out two error messages! Use outcapture?
	return;
    } 
    doTake(actor) = { 
	if (self.location <> actor) inherited.doTake(actor); //i.e. moveInto(Me)

	// use inherited.PhoneAnswer, because
	// self.PhoneAnswer would recursively call doTake again!
	if (self.ringing) inherited.doPhoneAnswer(actor);
    }

    doPhoneAnswer(actor) = {
        if (self.location <> actor) {
	    "(First taking <<self.thedesc>>)\n";
	}
	// This will call verDoTake, which will see if you can take/answer it
	// Then it will call doTake, which will take it if necessary,
	// and then answer it.
	execCommand(actor, takeVerb, self);
    }

    verIoPhoneOn(actor) = { 
	if (self.location <> actor)
	    "%You're% not holding <<self.thedesc>>! ";
	else
	    pass verIoPhoneOn;
    }

    // "turn on" the phone (take it off the hook) before calling
    start_dialing(actor) =
    {
	self.set_speaker(actor);
    }

    // With a mobile phone, you probably don't want to drop it if you're
    // on it (compare with takeVerb).
    verDoDrop(actor) = {
        if (self.speaker = actor) {
	    "%You% need%s% to hang up before you can put down
	    <<self.thedesc>>. ";
	} else {
	    pass verDoDrop;
	}
    }
;

/**********************************************************************/
// Verbs
//
phoneVerb: deepverb
    sdesc = "dial"
    verb = 'phone' 'telephone' 'call' 'call up' 'ring' 'ring up' 
        'dial' 'dial up'
    prepDefault = onPrep
    ioAction(onPrep) = 'PhoneOn'
    ioAction(withPrep) = 'PhoneOn' // call with phone == call on phone
    validDoList(actor, prep, iobj) = {return global.actorList + numObj;}
    validIoList(actor, prep, dobj) = {return nil;}
    // can only call one room
    validDo(actor, obj, seqno) = { return (seqno = 1); } 
;

// When a phone's ringing, answer it
// (this may involve taking the phone first)
// This verb was necessary (1) because it seemed dangerous to add the
// word 'answer' to the takeVerb's verb wordlist, and (2) it makes it easier
// to different between the acts of taking a phone and answering it.
phoneAnswerVerb: deepverb
    sdesc = "answer"
    verb = 'answer'
    doAction = 'PhoneAnswer'
;

hangupVerb: deepverb
    verb = 'hangup' 'hang up'
    sdesc = "hang up"
    doAction = 'Hangup'
;

// Don't let PC move when they're talking on a fixed_phone
// Sitting in a chair is sitVerb, which isn't a travelVerb, so
// that should be OK. But what about 'enter'? That can probably
// be handled on a room-by-room basis.
modify travelVerb
    verbAction(actor, dobj, prep, iobj) = {
        local ph := actor.my_phone;
	if (ph <> nil and isclass(ph, fixed_phone)) {
	    "%You% can't go anywhere while on the phone. ";
	    exit;
	} else {
	    pass verbAction;
	}
    }
;

/**********************************************************************/
// Call from init() to allow phoning to save time
// Create a list of all phone objects (things you can call)
init_phone: function {
    local o, l;
    // stolen from lamplist
    l := [];
    o := firstobj(Actor);
    while(o <> nil) {
	l := l + o;
        o := nextobj(o, Actor);
    }
    global.actorList := l;
}

/**********************************************************************/
// Changes to Actor class to allow phoning
//
// TODO should this be movableActor?
modify Actor
    // what do I say when I pick up the phone?
    phone_hello = "Someone picks up the phone. \"Hello?\" "
    // what's my (current) phone number, if any
    phone_number = nil
    // what phone am I currently speaking on?
    my_phone = nil
    // who's talking on the phone that my phone is connected to :>
    phone_partner = {
        if (self.my_phone and 
           self.my_phone.partner) {
	   return self.my_phone.partner.speaker;
	} else return nil;
    }
    // will I pick up the phone if it rings?
    phone_will_answer = true
    // Whose phone numbers do I know? (usu. nil for NPCs)
    phone_known = []

    // Let me address an actor on the phone (i.e., let the player type a
    // command like, "Actor, verb object")
    validActor = {
	return (inherited.validActor 
	    // My phone partner is the actor :>
	    or parserGetMe().phone_partner = self);
    }

    // Don't test whether you know phone number here, or "dial joe" when you
    // don't know joe's number will yield "what do you want to dial joe on?",
    // even if you're holding the phone
    verDoPhoneOn(actor, io) = {}

    // What happens when someone calls this actor?
    // call_actor is calling on call_phone
    // self is receiving the call on receive_phone
    //
    // This method can be overridden. E.g., it could give a message
    // and then call doHangup
    phone_answer(call_phone, receive_phone) = {

	receive_phone.doPhoneAnswer(self);
	if (self.phone_partner <> call_phone.speaker) {
	    "Error in phone.t! (phone_answer)";
	    exit;
	}

	// What does the actor say when answering the phone?
	self.phone_hello;
    }

;

/**********************************************************************/
// Changes to deepverb class to allow phoning
//
// I could just modify isReachable, because that's what validActor,
// validDo, and validIo call right now. But in theory, TADS v2.6 might
// have something more complicated for validDo.
modify deepverb
    is_phone_verb = nil // is this a verb involving talking?

    // Direct object situations when Joe is on the phone with me:
    // "ask *Joe* about foo" OR "Joe, tell *me* about foo"
    // It might be wrong to check for is_phone_verb here, because what if you
    // want to be able to do "joe, save me". OTOH, we could require the author
    // to override such situations, because most often if you give an
    // order to an NPC that doesn't involve a is_phone_verb, the DO will be
    // something they can see.
    validDo(actor, obj, seqno) = {
        return (inherited.validDo(actor, obj, seqno) or
	    (self.is_phone_verb and actor.phone_partner = obj));
    }

    validDoList(actor, prep, iobj) = {
	local ret;
	ret :=inherited.validDoList(actor, prep, iobj);
	// actor's phone partner is a valid object only if 
	// this verb is a talking verb
	if (self.is_phone_verb and actor.phone_partner) {
	    ret += actor.phone_partner;
	}
	return ret;
    }

    // Indirect object situations when Joe is on the phone with me:
    // e.g., "ask Joe about me"
    validIo(actor, obj, seqno) =
    {
	return (inherited.validIo(actor, obj, seqno) or
	    (self.is_phone_verb and actor.phone_partner = obj));
    }

    validIoList(actor, prep, dobj) = {
	local ret;
	ret :=inherited.validIoList(actor, prep, dobj);
	// actor's phone partner is a valid object only if 
	// this verb is a talking verb
	if (self.is_phone_verb and actor.phone_partner) {
	    ret += actor.phone_partner;
	}
	return ret;
    }
;

/**********************************************************************/
// Changes to numObj, strObj class to allow phoning
//
modify numObj verDoPhoneOn(actor,io) = {};
modify strObj verDoPhoneOn(actor,io) = {};

/**********************************************************************/
// Allow verbs to be used on the telephone
modify askVerb is_phone_verb = true ;
modify tellVerb is_phone_verb = true ;
modify sayVerb is_phone_verb = true ;
modify talkVerb is_phone_verb = true ;
modify apologizeVerb is_phone_verb = true ;

modify helloVerb is_phone_verb = true ;
modify goodbyeVerb is_phone_verb = true ;
modify helpVerb is_phone_verb = true ;
modify yesVerb is_phone_verb = true ;
modify noVerb is_phone_verb = true ;
modify thankVerb is_phone_verb = true ;
modify whatisVerb is_phone_verb = true ;
modify whoamiVerb is_phone_verb = true ;

