#charset "us-ascii"

/*
 *  TADS 3 Library Extension: AutoSave (Language-independent definitions)
 *  Version: April 11, 2005.
 *
 *  Copyright 2004-2005 by Michel Nizette.
 *
 *  This module implements a mechanism by which the game state can be
 *  restored, upon the player's demand, to an earlier position.  It
 *  distinguishes itself from the usual save/restore mechanism in that the
 *  position to be restored is a specific point determined by the game itself,
 *  rather than an arbitrary point recorded in an external game file on the
 *  player's command.
 *
 *  The idea is *not* to replace the existing save/restore mechanism with a
 *  more restricted one that would only allow the game state to be restored at
 *  pre-determined points.  In my opinion, a game that wouldn't allow the
 *  player to quit, save, and busy themselves with real-life activities
 *  exactly when they want would be rather frustrating, so I don't mean to
 *  disable the usual save/restore mechanism.
 *
 *  Instead, the idea is to supplement it with an additional mechanism
 *  allowing the player to restore an earlier position on the game's choice
 *  rather than on the player's choice, but with the guarantee that the
 *  position restored is a winnable state.
 *
 *  A number of IF authors these days consider that being stuck in an
 *  unwinnable situation (that is, a situation where the story can only end
 *  with the player's demise) is not fun, so that they try to carefully design
 *  their game so that the player's actions can never bring it to an
 *  unwinnable state.  This technique often offers good results, but can
 *  sometimes give the player the impression that the story is overly
 *  contrived just for the sake of keeping the game winnable.  For example,
 *  an enemy chasing the player character can feel unnaturally stupid or
 *  clumsy if the player is always given the possibility of deceiving the
 *  enemy by using one out of a (necessarily finite) number of strategies, no
 *  matter how many times the enemy was fooled before by the same trick.
 *
 *  In such situations, a valid alternative for the game might be to abandon
 *  the pretense of being always winnable, at least in a number of (preferably
 *  short) stages where the player is confronted to events so dangerous that
 *  the game doesn't rule out the possibility of dooming the player character
 *  irrecoverably.  In such situations, though, it would be nice if the game
 *  mitigated partly the discomfort of being in a state of unknown winnability
 *  by offering the option to restore the game state to an earlier position
 *  known to be safe.
 *
 *  Of course, the player can always save the game as many times as they think
 *  is necessary and give themselves a large choice of restorable positions,
 *  or they can UNDO as many times as they want to.  But how do they know
 *  *which* past position is winnable?  They simply don't, if the game doesn't
 *  explicitly tell them.  This is a situation where the player doesn't really
 *  want to decide explicitly which position to restore, or how many times to
 *  undo; all they want is to get back to the latest safe position, no matter
 *  when this position was actually reached.
 *
 *  That's what this module is for.  The game can mark the end of a turn as
 *  "safe", and specify whether subsequent turns are "safe" or "dangerous"
 *  until further notice.  A safe turn is a turn at the beginning of which the
 *  game is guaranteed to be winnable; a dangerous turn is a turn where no
 *  such guarantee exists.  After a dangerous turn, the player can type the
 *  command RETRY to restore the last safe position.  The mechanism works
 *  properly with the regular save/restore mechanism, because the last safe
 *  position is stored in saved game files as part of the game state.
 *
 *  The Library extension defines an object named challengeManager, whose
 *  purpose is to schedule the beginning or end of challenges.  A challenge is
 *  a safe game turn followed by a run of dangerous game turns.  The beginning
 *  of a challenge has to be a safe turn because it's the position restored
 *  via the command RETRY.  As a rule, the challenge should start just when
 *  the danger becomes apparent, but before the player has had any chance to
 *  get themselves irrecoverably stuck.
 *
 *  The turn where a challenge ends is always followed by a run of safe
 *  turns (if no new challange starts immediately) or by a unique safe turn
 *  (if a new challenge starts immediately after the old one ends).  The
 *  command RETRY only works during dangerous turns, as there is no point in
 *  restoring a former position if the current turn is safe.
 *
 *  When the game ends, the game usually prompts the player to choose from a
 *  number of end-of-game options.  For the challenge manager's purpose, this
 *  prompt is treated as a new turn following the turn where the end-of-game
 *  condition was triggered.  So, if an ending occurs within a running
 *  challenge, the option of retrying the challenge can be offered.  See the
 *  source code comments for more information.
 *
 *  A game schedules the beginning or end of a challenge for the end of the
 *  current turn by invoking the method challengeManager.setChallenge or
 *  challengeManager.setChallengeUpdate.  See the source comments about these
 *  methods for more details, but you can get a feeling of how the mechanism
 *  works by creating a dummy game, and invoking
 *  challengeManager.setChallenge(true) or challengeManager.setChallenge(nil),
 *  as part of some game actions or events, to schedule the beginning or
 *  end of a challenge, respectively.
 *
 *  By default, the game doesn't start in a challenge.  If you want the game's
 *  first turn to coincide with the beginning of a challenge, set the
 *  autoSave_challengeInitiallyRunning flag on your gameMain object (more
 *  details are given as part of the source code comments).  Note that the
 *  first turn of a game is always safe, which is consistent with the
 *  definition of a safe turn if the game is winnable at all.
 */

#include <tads.h>
#include <file.h>
#include <adv3.h>


/*
 *  The AutoSave Extension module ID.
 */
ModuleID
{
  name = 'AutoSave Library Extension';
  byline = 'by Michel Nizette';
  htmlByline = 'by <a href="mailto:Michel@Nizette.name">Michel Nizette</a>';
  version = 'April 11, 2005';
}


/*
 *  Default global settings used by the AutoSave module.  You can override
 *  these settings in your gameMain object.
 */
modify GameMainDef

  /*
   *  Flag: unless the player specifies otherwise, does the game report on
   *  changes in challenge conditions?  If this setting is true, the game will
   *  notify the player whenever a challenge begins or ends, so that they know
   *  when the command RETRY is effective or not.  If false, such changes in
   *  challenge conditions won't be reported.
   *
   *  This flag only controls how the game behaves initially: the player can
   *  always enable or disable challenge notifications via the commands
   *  RETRY SHOW and RETRY HIDE.
   */
  autoSave_reportsInitiallyOn = true

  /*
   *  Flag: when a change in challenge conditions is reported the first time,
   *  does the game tell the player about how they can disable these
   *  notifications via the command RETRY HIDE?
   */
  autoSave_reportsOffInitiallySuggested = true

  /*
   *  Flag: does the game start in a challenge?  If yes, the command RETRY is
   *  initially effective and can be used to bring the game back to the
   *  beginning.
   *
   *  In case the game does start in a challenge, this property can also be
   *  set to a single-quoted string or a function pointer, instead of the
   *  boolean value true, which has the same effect as using one of these
   *  datatypes for the argument of the setChallenge method of the
   *  challengeManager object (see the source code comment about this method).
   */
  autoSave_challengeInitiallyRunning = nil

  /*
   *  The name of the temporary file where the game state is saved at the
   *  beginning of challenges.  Note that this file can be erased at any time,
   *  as the saved game file is copied to a ByteArray immediately after its
   *  creation and is not used anymore afterwards.
   */
  autoSave_tempFileName = 'autosave.tmp'
;

/*
 *  The object that controls the beginning or end of challenges and schedules
 *  auto-saves accordingly.
 */
challengeManager: object

  /*
   *  Schedule the beginning or end of a challenge for the end of the current
   *  turn.  This method should be called as part of an action handling,
   *  a daemon execution, an NPC's scripted behavior, or whatever, when the
   *  level of "forgiveness" of the game (i.e., whether the game guarantees to
   *  be always winnable or not) is about to change according to such a game
   *  event.
   *
   *  The argument controls whether a challenge is starting: use true to
   *  start a new challenge, and nil to end a currently running challenge.  A
   *  new challenge can be started regardless of whether there is already a
   *  running challenge or not.  A previously running challenge, if there is
   *  one, is ended and replaced by the new challenge.
   *
   *  Challenges should always begin or end when the story has reached a
   *  winnable position.  The beginning of a challenge means that the player's
   *  *next* actions could cause their demise, but that the *last* action
   *  brought the game to a winnable state.  The position at the beginning of
   *  a challenge is the one that is saved automatically and that can be
   *  later called back with the command RETRY, so it's important that it be
   *  winnable.
   *
   *  The end of a challenge means that the player's last action brought the
   *  game to a winnable position and that their next actions in the
   *  foreseeable future aren't going to make the game unwinnable.
   *
   *  The command RETRY is effective only if there is a running challenge, and
   *  that the challenge hasn't just started at the end of the last turn,
   *  since there is no point in trying to restore a past winnable position if
   *  the current position itself is winnable.
   *
   *  In case a challenge is starting, the argument can be a single-quoted
   *  string or a function pointer instead of the boolean value true, to
   *  further customize the game's behavior when a safe position is restored
   *  via the command RETRY.
   *
   *  If the argument is true, then the game shows the room description
   *  immediately after a RETRY, in order to refresh the player's memory about
   *  the game conditions prevailing at the restored position.  This is
   *  consistent with the behaviors of the UNDO and RESTORE commands.
   *
   *  If the argument is a single-quoted string, then this string is displayed
   *  immediately after the room description, which can be useful to give the
   *  player a brief recapitulation of the nature of the challenge they are
   *  confronted to.
   *
   *  If the argument is a function pointer, then no room description is shown
   *  by default after a RETRY, and the function is executed instead.  This
   *  gives the author full control upon how the player's memory is refreshed
   *  after restoring a safe position.
   */
  setChallenge(restartResponse) { setChallengeUpdate(restartResponse, true); }

  /*
   *  Schedule the beginning or end of a challenge for the end of the current
   *  turn.  This method has essentially the same effect as setChallenge, but
   *  offers an additional level of customization.
   *
   *  The first argument, restartResponse, has exactly the same effect as the
   *  unique argument of setChallenge.  The second argument, updateResponse,
   *  can be used to cuztomize how the player is notified about changes in
   *  challenge conditions.
   *
   *  Use updateResponse = nil if you do not want the challenge change to be
   *  reported at all, irrespectively of the player-specified settings in this
   *  respect.
   *
   *  Use updateResponse = true if you want the Library to handle these
   *  reports itself.  The Library describes the nature of the change (i.e.,
   *  whether a challenge is starting or ending, and when a challenge is
   *  ending, whether it replaces a previously running challenge or not),
   *  then tells the player whether the command RETRY is effective or not
   *  as a consquence of the new conditions.
   *
   *  Use a single-quoted string if you want to customize the description of
   *  the change in challenge conditions; the string is displayed instead of
   *  the default description, after which the Library displays its usual
   *  notification about the new usage of the command RETRY.
   *
   *  You can also use a property pointer instead of a single-quoted string,
   *  in which case the specifed property is invoked on the libMessages
   *  object.  This property should display text that describes the change
   *  in challenge conditions, after which the Library displays its usual
   *  notification about the new usage of the command RETRY.
   */
  setChallengeUpdate(restartResponse, updateResponse)
  {
    /*
     *  Schedule the start of a new challenge, if that's what the caller
     *  wants.
     */
    challengeStartScheduled = (restartResponse != nil);
    /*
     *  If there is a running challenge, schedule its end, no matter what else
     *  the caller wants.
     */
    challengeEndScheduled = challengeRunning;
    /*
     *  Store the arguments for later usage.
     */
    self.updateResponse = updateResponse;
    self.restartResponse = restartResponse;
  }

  /*
   *  This method is executed every time the player character is about to take
   *  a non-idle turn.
   */
  beforePlayerCharNonIdleTurn()
  {
    if (challengeStartScheduled)
    {
      /*
       *  The start of a new challenge was scheduled.  Forget about the last
       *  auto-saved game.
       */
      autoSavedGame = nil;
      /*
       *  Add a new auto-save action at the beginning of the player
       *  character's command queue.
       */
      local pc = libGlobal.playerChar;
      pc.addFirstPendingAction(nil, pc,
          AutoSaveAction.createActionInstance());
    }
    else if (challengeEndScheduled)
    {
      /*
       *  The end of a challenge was scheduled, and no new challenge is to
       *  start.  Since we're not waiting for an auto-save operation to
       *  complete, we can notify the player about this already, and update
       *  the challenge manager according to the new conditions.
       */
      reportChallengeUpdate(true);
      update(nil);
    }
  }

  /*
   *  Flags: is a challenge scheduled to start or to end?
   */
  challengeStartScheduled = nil
  challengeEndScheduled = nil

  /*
   *  This method is executed immediately after an auto-save operation.  The
   *  argument contains the saved game data in case of success, or nil in case
   *  of failure.
   */
  afterAutoSave(bytes)
  {
    /*
     *  Notify the player about the new challenge start.
     */
    reportChallengeUpdate(bytes != nil);
    /*
     *  Update the challenge manager according to the new conditions.
     */
    update(bytes);
  }

  /*
   *  This method is executed immediately when a game finishes as a result of
   *  a call to finishGame or finishGameMsg.
   */
  beforeFinishGame()
  {
    /*
     *  Do not consider anymore the currently running challenge (if there is
     *  one) as having just started, so that the challenge manager can
     *  understand why a player would want to retry it right now.
     */
    clearChallengeStarting();
    /*
     *  Update the state of the challenge manager according to any scheduled
     *  changes in the challenge conditions.
     */
    if (challengeStartScheduled || challengeEndScheduled) update(nil);
    /*
     *  If there is a running challenge that hasn't just started (even now
     *  that we have handled scheduled changes), tell the caller that we think
     *  it's sensible to offer the option of retry the challenge.
     */
    return challengeRunning && !challengeStarting;
  }

  /*
   *  Retry the current challenge.  In case the game state at the beginning of
   *  the challenge is successfully restored, throw a
   *  TerminateCommandException.
   */
  doRetryChallenge()
  {
    /*
     *  If there is no challenge running, then there is no challenge to retry.
     */
    if (!challengeRunning) libMessages.autoSave_retryNoChallenge();
    /*
     *  Otherwise, if the auto-save operation at the beginning of the
     *  challenge failed, tell the player that we can't proceed.
     */
    else if (autoSavedGame == nil) libMessages.autoSave_retryUnavailable();
    /*
     *  Otherwise, if the challenge is just starting, tell the player that
     *  there is no point in retrying it as there is nothing to be undone.
     */
    else if (challengeStarting) libMessages.autoSave_retryChallengeStarting();
    else
    {
      /*
       *  We agree to retry the challenge.  Store the saved game in a local
       *  variable (so that we don't lose it as part of the current game
       *  state), then restore the auto-saved position.
       */
      local bytes = autoSavedGame;
      local succ = RetryAction.performRestore(bytes);
      if (succ)
      {
        /*
         *  Success.  Refresh the player's memory about the conditions
         *  prevailing at the beginning of the challenge.
         */
        reportChallengeRestart();
        /*
         *  Update the challenge manager state to reflect the challenge start,
         *  as that wasn't done yet by the time the position was auto-saved.
         */
        update(bytes);
        /*
         *  Abandon any remaining commands on the command line.
         */
        throw new TerminateCommandException();
      }
    }
  }

  /*
   *  Notify the player about changes in challenge conditions.  The argument
   *  is nil if an auto-save operation failed, true otherwise.
   */
  reportChallengeUpdate(success)
  {
    /*
     *  If either the player or the author don't want these notifications to
     *  be displayed, return immediately.
     */
    if (!reportsOn || updateResponse == nil) return;

    if (updateResponse == true)
    {
      /*
       *  The author wants the Libary to handle the whole notification message
       *  by itself.  Choose the right libMessages property to invoke to
       *  report the changes in challenge conditions.
       */
      if (challengeEndScheduled && challengeStartScheduled)
          updateResponse = &autoSave_challengeChanging;
      else if (challengeStartScheduled)
          updateResponse = &autoSave_challengeStarting;
      else if (challengeEndScheduled)
          updateResponse = &autoSave_challengeEnding;
    }
    /*
     *  Display anything necessary at the beginning of a challenge change
     *  notification.
     */
    libMessages.autoSave_beginChallengeReport();
    /*
     *  If updateResponse is a string describing the challenge change, display
     *  it.
     */
    if (dataType(updateResponse) == TypeSString)
        "<<updateResponse>>";
    /*
     *  Otherwise, if we have a libMessages property to invoke, invoke it.
     */
    else if (dataType(updateResponse) == TypeProp)
        libMessages.(updateResponse);
    /*
     *  Now, choose the right Library message to describe the new usage of the
     *  command RETRY.
     */
    if (challengeStartScheduled)
        libMessages.(success ? &autoSave_retryEnabled
                             : &autoSave_retryBroken);
    else libMessages.autoSave_retryDisabled();
    /*
     *  If so desired, tell the player how to turn off these challenge
     *  notifications.
     */
    if (reportsOffSuggested) libMessages.autoSave_suggestReportsOff();
    /*
     *  Dispay anything necessary at the end of a challenge change
     *  notification.
     */
    libMessages.autoSave_endChallengeReport();
  }

  /*
   *  Flags: does the player want to be notified about challenge changes?
   *  do we want to tell them how to turn these notifications off next time
   *  a notification is displayed?
   */
  reportsOn = nil
  reportsOffSuggested = nil

  /*
   *  This property controls how the player is notified about challenge
   *  changes, in a way that depends on the datatype of the poperty.
   */
  updateResponse = nil

  /*
   *  Refresh the player's memory about the conditions prevailing at the
   *  beginning of a restarting challenge.
   */
  reportChallengeRestart()
  {
    /*
     *  If restartResponse is true or is a single-quoted string, then describe
     *  the room and, in the case of a string, display it.
     */
    if (restartResponse == true || dataType(restartResponse) == TypeSString)
    {
      "<.p>";
      libGlobal.playerChar.lookAround(true);
      if (dataType(restartResponse) == TypeSString) "<.p><<restartResponse>>";
    }
    /*
     *  Otherwise, if restartResponse is a pointer to a function, invoke the
     *  function.
     */
    else if (dataTypeXlat(restartResponse) == TypeFuncPtr)
        (self.restartResponse)();
  }

  /*
   *  This property controls how the player's memory is refreshed about the
   *  conditions prevailing at the beginning of a restarting challenge, in a
   *  way that depends on the datatype of the poperty.
   */
  restartResponse = nil

  /*
   *  Update the challenge manager according to the scheduled challenge
   *  changes.  The argument, if it is not nil, is supposed to contain the
   *  data of the game state at the beginning the challenge that's going to
   *  run from now on.
   */
  update(bytes)
  {
    if (reportsOn && updateResponse != nil)
    {
      /*
       *  If we've reached here, this means that the player has been notified
       *  about the change in the challenge conditions, in which case the game
       *  should also have told the player how to turn off these notifications
       *  if desired.  Therefore, from now on, we don't want to explain that
       *  anymore.
       */
      reportsOffSuggested = nil;
    }
    /*
     *  Remember the auto-saved game data.
     */
    autoSavedGame = bytes;
    /*
     *  Forget about the previously specified custom responses to auto-save
     *  or RETRY events.
     */
    updateResponse = nil;
    restartResponse = nil;
    /*
     *  A new challenge starts and runs if and only if one is scheduled to
     *  start.
     */
    challengeRunning = challengeStartScheduled;
    challengeStarting = challengeStartScheduled;
    /*
     *  Forget about the scheduled changes; we're done with them.
     */
    challengeStartScheduled = nil;
    challengeEndScheduled = nil;
    /*
     *  A challenge is considered starting for only one turn.  Set up a fuse
     *  that will clear the "starting" flag as part of the next turn.
     */
    if (challengeStarting) new Fuse(self, &clearChallengeStarting, 0);
  }

  /*
   *  Clear the "challenge starting" flag.  (The reason for making this a
   *  method is so that we can execute it in a fuse.)
   */
  clearChallengeStarting() { challengeStarting = nil; }

  /*
   *  Turn challenge change notifications on or off on the player's volition,
   *  according to the argument.
   */
  retryToggle(reportsOn)
  {
    /*
     *  Remember the new setting.
     */
    self.reportsOn = reportsOn;
    /*
     *  Display the proper parameterized
     *  acknowledgement message.
     */
    libMessages.autoSave_acknowledgeRetryToggle(reportsOn, challengeRunning,
        challengeStarting, challengeRunning && autoSavedGame == nil);
    /*
     *  If we've reached here, the player obviously knows how to turn
     *  notifications on or off, so there's no point anymore about informing
     *  them about it.
     */
    reportsOffSuggested = nil;
  }

  /*
   *  Flags: is a challenge currently running or starting?
   */
  challengeRunning = nil
  challengeStarting = nil

  /*
   *  Game state data at the beginning of a challenge.
   */
  autoSavedGame = nil
;


/*
 *  Initialize the challenge manager.
 */
PreinitObject

  execute()
  {
    /*
     *  Copy the relevant settings from gameMain to challengeManager.
     */
    challengeManager.reportsOn
        = gameMain.autoSave_reportsInitiallyOn;
    challengeManager.reportsOffSuggested
        = gameMain.autoSave_reportsOffInitiallySuggested;
    /*
     *  If the game is supposed to start in a challenge, schedule the
     *  challenge start, but don't report anything about it.  It's up to the
     *  author to make the challenge state prevailing at the beginning of the
     *  story known to the player.  The report wouldn't come at the
     *  appropriate time anyway, since it would just come before the command
     *  prompt.
     */
    challengeManager.setChallengeUpdate(
        gameMain.autoSave_challengeInitiallyRunning, nil);
  }
;


/*
 *  Add a hook in Actor so that the challenge manager has a chance to do
 *  something just before the PC takes a non-idle turn.
 */
modify Actor

  /*
   *  Add the hook at the beginning of the method executeActorTurn, after
   *  checking that the current actor is actually the player character, that
   *  they aren't waiting for another actor, and that the next action to be
   *  executed is in the form of parsed command tokens, not of a programmatic
   *  pre-resolved action.
   *
   *  This isn't ideal, but it's the best we can do without replacing the
   *  whole method.  Ideally, we'd have liked to add the hook just before the
   *  "if (pendingCommand.length() == 0 && isPlayerChar())" test in the
   *  method as implemented originally in the TADS 3 Library.  Any earlier
   *  position can in principle be preceded by changes in the game conditions,
   *  such as a different actor becoming the new player character, which would
   *  have the undesirable effect that the auto-save wouldn't be executed as
   *  part of the player character's turn after all.
   *
   *  However, such a possibility seems very faint and corresponds to a
   *  very pathological situation, IMO, so I believe it's safe enough to
   *  add the hook at the beginning of the method.
   */
  executeActorTurn()
  {
    if (isPlayerChar() && !checkWaitingForActor()
        && (pendingCommand.length() == 0
            || pendingCommand[1].ofKind(PendingCommandToks)))
        beforePlayerCharNonIdleTurn();
    return inherited();
  }

  /*
   *  Said hook -- defer its execution to the challenge manager.
   */
  beforePlayerCharNonIdleTurn()
  {
    challengeManager.beforePlayerCharNonIdleTurn();
  }
;


/*
 *  The auto-save programmatic action responsible for saving the game state
 *  at the beginning of a challenge.
 */
DefineSystemAction(AutoSave)

  /*
   *  This action is tacit and shouldn't be repeatable or undoable.
   */
  isRepeatable = nil
  includeInUndo = nil

  /*
   *  Carry out the auto-save action.
   */
  execSystemAction()
  {
    /*
     *  Remember the current real time -- we'll want to restore it after the
     *  save in case the operation takes any noticeable time, so that it
     *  doesn't count against the game clock.
     */
    local origElapsedTime = realTimeManager.getElapsedTime();
    /*
     *  Notify any interested observer that we are about to save the game.
     *  For these observers' purpose, there's no need to distinguish between
     *  an auto-save and a regular save action requested by the player.
     */
    PreSaveObject.classExec();
    /*
     *  Try to save the game state to a bytes array; display a failure
     *  message if appropriate.
     */
    local bytes = nil;
    try { bytes = saveGameToByteArray(); }
    catch (FileException exc) { libMessages.autoSave_autoSaveFailed(exc); }
    /*
     *  Restore the real time at the beginning of the save operation.
     */
    realTimeManager.setElapsedTime(origElapsedTime);
    /*
     *  Notify the challenge manager that we've performed an auto-save.
     */
    challengeManager.afterAutoSave(bytes);
  }
;


/*
 *  The RETRY command used to restore the game state at the beginning of a
 *  challenge.
 */
DefineSystemAction(Retry)

  /*
   *  Trying to undo a RETRY is not meaningful anymore than trying to undo
   *  a RESTORE.
   */
  includeInUndo = nil

  /*
   *  Execute the RETRY action.
   */
  execSystemAction()
  {
    /*
     *  Attempt to retry the current challenge.
     */
    doRetryChallenge();
    /*
     *  Whatever happened, forget about any remaining commands on the command
     *  line.
     */
    throw new TerminateCommandException();
  }

  /*
   *  Attempt to retry the current challenge.  Defer the call to the challenge
   *  manager.
   */
  doRetryChallenge() { challengeManager.doRetryChallenge(); }

  /*
   *  Restore the game state contained in the specified byte array.  Return
   *  true in case of success, nil in case of failure.
   */
  performRestore(bytes)
  {
    /*
     *  Remember the current real time -- we'll want to reset the game clock
     *  to it after the restore in case the operation fails but takes any
     *  noticeable time nevertheless, so that it doesn't count against the
     *  game clock.
     */
    local origElapsedTime = realTimeManager.getElapsedTime();
    /*
     *  Ask for confirmation, as we're about to lose the current game state.
     */
    libMessages.autoSave_confirmRetry();
    if (!yesOrNo())
    {
      /*
       *  Report that the operation was cancelled.
       */
      libMessages.autoSave_notRetrying();
      /*
       *  Return immediately with an indication of failure.
       */
      return nil;
    }
    /*
     *  We got the confirmation.  Try to restore the game state.
     */
    try { restoreGameFromByteArray(bytes); }
    catch (FileException exc)
    {
      /*
       *  Failure.  If the problem is a matter of file safety settings, report
       *  that.
       */
      if (exc.ofKind(FileSafetyException))
          libMessages.autoSave_retryFileSafetyTooRestrictive();
      /*
       *  Otherwise, report a genuine I/O error.
       */
      else libMessages.autoSave_retryFileException(exc);
      /*
       *  Reset the game clock to the time at the beginning of the retry.
       */
      realTimeManager.setElapsedTime(origElapsedTime);
      /*
       *  Return immediately with an indication of failure.
       */
      return nil;
    }
    /*
     *  Confirm success.
     */
    libMessages.autoSave_retryOkay();
    /*
     *  Notify any interested observer that we've performed a kind of restore
     *  operation.  Use restore-code 3, which isn't used in the base library,
     *  to identify a RETRY.
     */
    PostRestoreObject.restoreCode = 3;
    PostRestoreObject.classExec();
    /*
     *  Return with an indication of success.
     */
    return true;
  }
;


/*
 *  Define RETRY as an end-of-game option.
 *
 *  If you want to use this option, you don't have to go into the trouble of
 *  figuring out by yourself whether there is a currently running challenge
 *  to be retried or not.  It's safe to always include this option in the
 *  option list passed to finishGame or finishGameMsg, because the challenge
 *  manager is given the opportunity to inspect the option list and to remove
 *  the RETRY option if it thinks that it isn't an appropriate time to offer
 *  the option.
 */
finishOptionRetry: FinishOption

  /*
   *  Carry out the option.
   */
  doOption()
  {
    /*
     *  Attempt to retry the current challenge, if there is one running.  Upon
     *  success, an exception will be thrown, breaking out of the finish-
     *  option loop.
     */
    RetryAction.doRetryChallenge();
    /*
     *  If we've reached here, this means that the RETRY has failed.  Propose
     *  a list of finish options again.
     */
    return true;
  }
;


/*
 *  Modify finishGameMsg so that it notifies the challenge manager of an
 *  end-of-game condition.
 */
modify finishGameMsg(msg, extra)
{
  /*
   *  Notify the challenge manager of an end-of-game condition.
   */
  local result = challengeManager.beforeFinishGame();
  /*
   *  If finishOptionRetry is among the options, and the challenge manager
   *  said it doesn't think it's justified to offer a RETRY option, remove it.
   */
  if (!result && extra != nil && extra.indexOf(finishOptionRetry) != nil)
      extra -= finishOptionRetry;
  /*
   *  Call the replaced function implementation with the modified arguments.
   */
  replaced(msg, extra);
}


/*
 *  Save the game state to a byte array instead of a file.  A FileException is
 *  thrown in case of failure.
 */
function saveGameToByteArray()
{
  /*
   *  Try to save the game in a temporary file.  If a runtime error results,
   *  wrap it in a SaveRestoreException, which is a subclass of FileException.
   *  We'll make the caller's life easier if every exception thrown denoting
   *  an I/O problem is a FileException.
   */
  try { saveGame(gameMain.autoSave_tempFileName); }
  catch (RuntimeError exc) { throw new SaveRestoreException(exc); }
  /*
   *  Success so far.  Copy the saved game from the temporary file to a byte
   *  array.  Don't catch any FileException thrown.
   */
  local file = File.openRawResource(gameMain.autoSave_tempFileName);
  local bytes = new ByteArray(file.getFileSize());
  file.readBytes(bytes);
  file.closeFile();
  /*
   *  Success.  Return the byte array.
   */
  return bytes;
}


/*
 *  Restore a game state saved in a byte array instead of a file.  A
 *  FileException is thrown in case of failure.
 */
function restoreGameFromByteArray(bytes)
{
  /*
   *  Write the byte array to a temporary file.  Don't catch any
   *  FileException.
   */
  local file = File.openRawFile(gameMain.autoSave_tempFileName,
                                FileAccessWrite);
  file.writeBytes(bytes);
  file.closeFile();
  /*
   *  Success so far.  Restore the game state from the temporary file.  If a
   *  runtime error results, wrap it in a SaveRestoreException, which is a
   *  subclass of FileException.  We'll make the caller's life easier if every
   *  exception thrown denoting an I/O problem is a FileException.
   */
  try { restoreGame(gameMain.autoSave_tempFileName); }
  catch (RuntimeError exc) { throw new SaveRestoreException(exc); }
}


/*
 *  A FileException wrapping the RuntimeError thrown as a consequence of a
 *  failure in saveGame or restoreGame.
 */
class SaveRestoreException: FileException

  /*
   *  Construct a new SaveRestoreException.  Remember the RuntimeError
   *  specified as an argument.
   */
  construct(runtimeError) { self.runtimeError = runtimeError; }

  /*
   *  Describe the error.  (Delegate the call to the underlying RuntimeError.)
   */
  displayException() { runtimeError.displayException(); }

  /*
   *  The underlying RuntimeError.
   */
  runtimeError = nil
;


/*
 *  The command RETRY SHOW enables future notifications about changes in
 *  challenge conditions.
 */
DefineSystemAction(RetryShow)

  /*
   *  Defer the execution to the challenge manager.
   */
  execSystemAction() { challengeManager.retryToggle(true); }
;


/*
 *  The command RETRY HIDE disables future notifications about changes in
 *  challenge conditions.
 */
DefineSystemAction(RetryHide)

  /*
   *  Defer the execution to the challenge manager.
   */
  execSystemAction() { challengeManager.retryToggle(nil); }
;
