/* gtimage.c: graphic image display module (allegro)
        for GlkDOS, curses.h/allegro implementation of the Glk API
    Designed by L. Ross Raszewski <lraszewski@justice.loyola.edu>
    
    based upon GlkTerm by Andrew Plotkin <erkyrath@netcom.com>
    http://www.eblong.com/zarf/glk/index.html
*/

#include <allegro.h>
#include "cscr.h"
#include "gtpref.h"
#include "gtoption.h"
#include <stdio.h>
#include <stdlib.h>
#include "glk.h"
#include "glkdos.h"
#include "gtw_grx.h"
#include "gi_blorb.h"
#include "gtw_buf.h"
void gtblorb_describe_error(giblorb_err_t error);
glui32 win_textbuffer_draw(winid_t win, glui32 image, glsi32 val1, glsi32 val2);
BITMAP *JpegtoBITMAP(FILE *);
BITMAP *PNGtoBITMAP(FILE *);

#define CACHE_IMAGES 10
int cached_images=0;
extern BITMAP *Buffer;
// Cached images
struct quick_image {
 BITMAP *bmp;
 glui32 num;
 int time;
} img_bank[CACHE_IMAGES];

typedef void *giblorb_chunkdesc_t;
typedef void *giblorb_resdesc_t;

struct giblorb_map_struct {
    glui32 inited; /* holds giblorb_Inited_Magic if the map structure is 
        valid */
    strid_t file;
    
    int numchunks;
    giblorb_chunkdesc_t *chunks; /* list of chunk descriptors */
    
    int numresources;
    giblorb_resdesc_t *resources; /* list of resource descriptors */
    giblorb_resdesc_t **ressorted; /* list of pointers to descriptors 
        in map->resources -- sorted by usage and resource number. */
};


glui32 gli_graphics_test(glui32 wintype)
{
  return wintype==wintype_Graphics || wintype==wintype_TextBuffer;
}


static int get_cache()
{
 int i, bsf=0;
 if (cached_images < CACHE_IMAGES)
  return cached_images++;
 for(i=1;i<cached_images;i++)
  if (img_bank[i].time>img_bank[bsf].time) bsf=i;
 destroy_bitmap(img_bank[bsf].bmp);
 return bsf;
}
  
static BITMAP *cache_image(glui32 num)
{
  BITMAP *b;
  giblorb_err_t error;
  giblorb_result_t res;
  FILE *ff;
  long t;
  int i;
  BITMAP *(*func)(FILE *);
  error=giblorb_load_resource(giblorb_get_resource_map(),giblorb_method_Memory,&res,giblorb_ID_Pict,num);
  if (error) { gtblorb_describe_error(error); return NULL; }
  if (memcmp(((char *)res.data.ptr)+1,"PNG",3)==0) func=PNGtoBITMAP;
  else func=JpegtoBITMAP;
  free(res.data.ptr);
  giblorb_load_resource(giblorb_get_resource_map(),giblorb_method_FilePos,&res,giblorb_ID_Pict,num);
  ff=giblorb_get_resource_map()->file->file;
  t=ftell(ff);
  fseek(ff,res.data.startpos,SEEK_SET);
  b=(*func)(ff);
  fseek(ff,t,SEEK_SET);
  i=get_cache();
  img_bank[i].bmp=b;
  img_bank[i].num=num;
  img_bank[i].time=0;
  return b;
}

static BITMAP *get_image(glui32 num)
{
 int i;
 for(i=0; i<cached_images; i++)
  if (num == img_bank[i].num) {
        img_bank[i].time=0;
        return img_bank[i].bmp;
        }
   else img_bank[i].time++;
 return cache_image(num);
}

  
glui32 glk_image_draw(winid_t win, glui32 image, glsi32 val1, glsi32 val2)
{
    window_graphics_t *dwin;
    BITMAP *b;
    if (win->type==wintype_TextBuffer)
     return win_textbuffer_draw(win,image,val1,val2);

    if (win->type!=wintype_Graphics)
    { gli_strict_warning("image_draw: graphics supported only in graphics windows.");
      return FALSE;
    }
   dwin=win->data;
   b=get_image(image);
   if (!b) return FALSE;
   blit(b,dwin->data,0,0,val1,val2,b->w,b->h);
   dwin->dirty=1;
   return TRUE;
}

glui32 sglk_image_draw(winid_t win, BITMAP *b, glsi32 val1, glsi32 val2)
{
    window_graphics_t *dwin;
    if (win->type!=wintype_Graphics)
    { gli_strict_warning("image_draw: graphics supported only in graphics windows.");
      return FALSE;
    }
   dwin=win->data;
   if (!b) return FALSE;
   blit(b,dwin->data,0,0,val1,val2,b->w,b->h);
   dwin->dirty=1;
   return TRUE;
}

glui32 glk_image_draw_scaled(winid_t win, glui32 image, 
    glsi32 val1, glsi32 val2, glui32 width, glui32 height)
{
    window_graphics_t *dwin;
    BITMAP *b;
//    if (win->type==wintype_TextBuffer)
//     return win_textbuffer_draw_scaled(win,image,val1,val2,width,height);
    if (win->type!=wintype_Graphics)
    { gli_strict_warning("image_draw_scaled: graphics supported only in graphics windows.");
      return FALSE;
    }
   dwin=win->data;
   b=get_image(image);
   if (!b) return FALSE;
   stretch_blit(b,dwin->data,0,0,b->w,b->h,val1,val2,width,height);
   dwin->dirty=1;
   return TRUE;
}
glui32 sglk_image_draw_scaled(winid_t win, BITMAP *b, 
    glsi32 val1, glsi32 val2, glui32 width, glui32 height)
{
    window_graphics_t *dwin;
    if (win->type!=wintype_Graphics)
    { gli_strict_warning("image_draw_scaled: graphics supported only in graphics windows.");
      return FALSE;
    }
   dwin=win->data;
   if (!b) return FALSE;
   stretch_blit(b,dwin->data,0,0,b->w,b->h,val1,val2,width,height);
   dwin->dirty=1;
   return TRUE;
}

glui32 glk_image_get_info(glui32 image, glui32 *width, glui32 *height)
{
    BITMAP *b=get_image(image);
    if (!b) return FALSE;
    if (width) *width=b->w;
    if (height) *height=b->h;
    return TRUE;
}

void glk_window_flow_break(winid_t win)
{
    gli_strict_warning("window_flow_break: graphics not supported in text buffers.");
}

glui32 win_textbuffer_draw(winid_t win, glui32 image, glsi32 val1, glsi32 val2)
{
 BITMAP *b; int x,y;
 window_textbuffer_t *dwin=win->data;
 tbinline *new;
 if (val1==imagealign_MarginLeft || val1==imagealign_MarginRight)
  if (dwin->chars[dwin->numchars-1]!='\n') return FALSE;
 b=get_image(image);
 if (!b) return FALSE;
 x=text_length(font,"M");
 y=text_height(font);
 new=(tbinline *)malloc(sizeof(tbinline));
 new->next=dwin->inlines;
 dwin->inlines=new;
 new->img=image;
 new->pos=dwin->numchars;
 new->len=b->w / x;
 new->len+=(b->w %x) != 0; 
 new->height=b->h / y;
 new->height+=(b->h %y) != 0; 
 new->sx=0;
 new->sy=0;
 new->align=val1;
 win_textbuffer_putchar(win,GRX_CHAR);
 if (val1==imagealign_MarginRight)
  win_textbuffer_putchar(win,'\n');
 return TRUE;
}

static struct inimage {
 grect_t box;
 int y;
 int x;
 tbinline *i;
 struct inimage *next;
} *inline_cache=NULL;
/* draw an image in a textbuffer (when drawnow happens) */
void win_text_drawat(grect_t box, int y, int x, tbinline *i)
{
 struct inimage *n;
 n=(struct inimage *)malloc(sizeof(struct inimage));
 n->next=inline_cache;
 inline_cache=n;
 n->box=box;
 n->y=y;
 n->x=x;
 n->i=i;
}
/* draw an image in a textbuffer (now) */
static void win_text_drawatnow(grect_t box, int y, int x, tbinline *i)
{
 BITMAP *b;
 int w, h;
 int bu, bl;
 int sy;
 int hh;
 b=get_image(i->img);
 sy=0;
 hh=b->h;
 w= text_length(font,"M");
 h=text_height(font);
 bu=box.top *h;
 bl=box.bottom*h;
 x *= w;
 y *= h;
 if (i->align==imagealign_InlineUp  || i->align==imagealign_MarginLeft || i->align==imagealign_MarginRight)
  y -= b->h -h;
 else if (i->align==imagealign_InlineCenter)
  y -= (b->h)/2;
 if (y < bu)
 {
  sy=bu-y;
  y=bu;
 }
 if (y+b->h > bl)
 {
  hh -= ((hh+y)-bl);
 }
 if (i->align==imagealign_MarginRight)
  x = (box.right * w) - b->w;
 blit(b,Buffer,0,sy,x,y,b->w,hh);
}

/* draw all queued textbuffer images now */
void win_text_drawnow()
{
 struct inimage *n;
 while(inline_cache)
 {
   n=inline_cache;
   inline_cache=n->next;
   win_text_drawatnow(n->box,n->y,n->x,n->i);
   free(n);
 }
}
 
