/* ccurses.c: CSCR curses interface
    This wraps various curses.h functions for use with 
    cscr, and adds a screen capture facility

    By L. Ross Raszewski <lraszewski@justice.loyola.edu>

    Uses curses.h
   

*/

#include <stdarg.h>
#include <unistd.h>
#include <curses.h>
#include "cscr.h"
#include <dpmi.h>
#include "gtpref.h"
#include "mouse.h"
// Fake definitions to keep allegro happy

int VHORIZ_REZ;
int VVERT_REZ;

// Exportable variables
struct mouse_info_s mouse_info;

// Extended character printing table. This is very complex; we print
// The characters that have corresponding MDA characters, translating
// from latin1 to MDA

unsigned char OPT_AO_FF_OUTPUT[]= {
        0,   173, 155, 156, 15 , 157, '|', 21,
        0,     0, 166, 174, 170, '-',   0,   0,
        248, 241, 253,   0, '\'', 230, 20, 249,
        0,     0, 167, 175, 172,  171,  0, 168,
        0,     0,   0,   0, 142, 143,  146, 128,
        0,   144,   0,   0,  0,   0,   0,   0,  
        0,   165,   0,   0,  0,  0,  163,  0,   
        0,   0,   0,   0,154,  0,   0,   225, 
        133, 160, 131,  0, 132, 134, 145, 135,  
        130, 130, 136, 137, 141, 161, 140, 139, 
        0,   164, 149, 161, 147, 0, 148, 246,   
        237, 163, 151, 150, 129, 0, 0, 152,     
         };

// Static local variables
static int mouse_click=0;       // Mouse has pending click
static int mouse_alive=0;       // Mouse is active
static int mouse_exists;        // init_mouse succeeded
static chtype bkgd_col;         // the background
static chtype _bk;              // used by suspend/resume

// Internal helper functions
// Liberally ripped from conio source.
/* Set intensity/blinking bit to INTENSE (bright background).  */
static void ivideo(void)
{
  __dpmi_regs regs;
  regs.h.bl = 0;
  regs.x.ax = 0x1003;
  __dpmi_int(0x10, &regs);
}
// Update the mouse state
static int poll_mouse()
{
 if (!mouse_alive) return 0;
 if (mouse_click) return 1;
 mouse_click=get_mouse(&mouse_info.x,&mouse_info.y);
 if (mouse_click)
 {
   mouse_info.text_x=(mouse_info.x * COLS)/ 640;
   mouse_info.text_y=(mouse_info.y * LINES)/200;
   return 1;
 }
 return 0;
}


// ccurses: Curses implementation of cscr

// Group 1
int cscr_version()
{
 return CSCR_CURSES;
}
void cscr_start()
{   int j,k,i;
    initscr();
    cbreak();
    noecho();
    nonl(); 
    intrflush(stdscr, FALSE); 
    keypad(stdscr, TRUE);
    scrollok(stdscr, FALSE);
    start_color();
    i=1;
    // create a predictable color cube
    for(j=0;j<8;j++)
     for(k=0;k<8;k++)
      init_pair(i++,k,j);
   ivideo();
   bkgd_col=cscr_color(glk_preferences[pref_fg_color], glk_preferences[pref_bg_color]) | ' ';
   attrset(bkgd_col);
   mouse_exists=(init_mouse()!=0);
}
void cscr_exit()
{
    clear();
    refresh();
    endwin();
    cscr_sleep_mouse();
}
void cscr_resume()
{
 bkgd(_bk);
 clear();
 ivideo();
 refresh();
}
void cscr_suspend()
{
 _bk = stdscr->_bkgd;
 bkgd(cscr_color(7,0)|' ');
 clear();
 refresh();
}

// Group 2:
void  cscr_charpos_to_pxl(int *x, int *y)
{

}
unsigned int cscr_getcols()
{
 return COLS;
}
unsigned int cscr_getlines()
{
 return LINES;
}

// Group 3
void cscr_wake_mouse()
{
 if (mouse_alive) return;
 mouse_alive=1;
 show_mouse();
}
void cscr_sleep_mouse()
{
 if (!mouse_alive) return;
 mouse_alive=0;
 hide_mouse();
}
int cscr_query_mouse()
{
 return mouse_exists;
}

// Group 4a
void cscr_clear()
{
 int i,j;
 clear();
 for(i=0;i<LINES;i++)
  for(j=0;j<COLS;j++)
   stdscr->_y[i][j]=bkgd_col;
}
void cscr_clrtoeol()
{
 chtype b=stdscr->_bkgd;
 bkgd(bkgd_col);
 clrtoeol();
 bkgd(b);
}
void cscr_refresh()
{
refresh();
}
void cscr_moveyx(int y, int x)
{
 move(y,x);
}
void cscr_screencap()
{
 int i,j;
 FILE *f;
 char s[13];
 static int n=0;
 do {
  sprintf(s,"scr%05d.txt",n);
  n++;
  }
 while(__file_exists(s) && n < 100000);
 if (n>99999) return;
 f=fopen(s,"w");
 for(i=0;i<LINES;i++)
 {
  for(j=0;j<COLS;j++)
   putc(stdscr->_y[i][j] & A_CHARTEXT,f);
  putc('\n',f);
 }
 fclose(f);
}

// Group 4b
int cscr_addstr(char *s)
{
 return addstr(s);
}
void cscr_addch(chtype c)
{
 addch(c);
}
int cscr_printw(char *fmt, ...)
{
 va_list args;
 int retval=ERR;
 va_start(args,fmt);

	vsprintf(c_printscanbuf, fmt, args);
	va_end(args);
        if (cscr_addstr(c_printscanbuf) == ERR)
		return (retval);
	retval = (strlen(c_printscanbuf));
	return (retval);

}

// Group 4c
chtype cscr_color(int f, int b)
{
 int p,a;
 a=0;
 if (f >= 8) { a = A_BOLD; f -= 8; }
 if (b >= 8) { a |= A_BLINK; b -=8; }

 p=f + (8*b) + 1;
 return COLOR_PAIR(p) | a;

}
unsigned char cscr_color_to_8bit(chtype c)
{
 int cp = c >> 24;
 unsigned char fg = (cp - 1 ) % 8;
 unsigned char bg = (cp - 1 ) / 8;
 if (c & A_BOLD) fg |= 8;
 if (c & A_BGBRITE) bg |= 8;
 return fg | (bg << 4 );
}
void cscr_attron(chtype a)
{
 attron(a);
}
void cscr_attrset(chtype x)
{
 if (x)
 attrset(x);
 else attrset(bkgd_col);
}
chtype cscr_bkgd(chtype i)
{
 bkgd(i & ~A_BLINK);
 return bkgd_col=i;
}

// Group 5
int cscr_kbhit()
{ 
 if (poll_mouse()) return 1;
 else return kbhit();
}
void cscr_blinkey()
{
 // This function doesn't do anything, since curses uses the bios cursor
 // blinkey
}
chtype cscr_getch()
{
 if (poll_mouse()) { mouse_click=0; return MOUSE_CLICK_KEY; } 
 return getch();
}



// Function to keep conio quiet.
int _set_screen_lines(int x)
{
 return x;
}
