/* print.c
 *
 *  ``pinfocom'' -- a portable Infocom Inc. data file interpreter.
 *  Copyright (C) 1987-1992  InfoTaskForce
 *  Modified 2003 by Nils Gesbert
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; see the file COPYING.  If not, write to the
 *  Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.
 */

/*
 * $Header: RCS/print.c,v 3.0 1992/10/21 16:56:19 pds Stab $
 */

#include <string.h>

#include "infocom.h"

#define BUFMIN              80
#define LONG_SCORE_WIDTH    60
#define MAX_MOVES           1599
#define STATLEN             24

int     print_mode;     /* General Printing Mode           */
int     single_mode;    /* Mode for printing the next char */

static word word_bank;  /* There are 3 banks of common     */
                        /* words, each 32 words long.      */

static print_buf_t text, room;

static char score_buf[STATLEN+1];

Bool
print_init()
{
    /*extern print_buf_t  *pbf_p;*/
    /*extern char         *ti_location;*/
    /*extern char         *ti_status;*/

    text.len = 0;
    text.max = BUFMIN * 5;
    text.buf = (byte *)malloc(text.max + 1);
    if (!text.buf) return FALSE;
    pbf_p = &text;

    room.max = BUFMIN;
    room.buf = (byte *)malloc(room.max + 1);
    if (!room.buf) {
      free (text.buf);
      return FALSE;
    }
    room.buf[0] = ' ';
    room.len = 1;

    ti_location = (char *)room.buf;
    ti_status = (char *)score_buf;
    return TRUE;
}

inline void
print_num A1(word, number)
{
    Bool    neg;
    int     num;
    char    buf[15], *cp;

    num = (signed_word)number;
    if (neg = (num < 0))
        num = -num;

    cp = buf;
    do
    {
        *(cp++) = '0' + (num % 10);
        num /= 10;
    }
    while (num > 0);

    if (neg)
        *cp = '-';
    else
        --cp;

    for (; cp >= buf; --cp)
        print_char((word)*cp);
}

inline void
print_str A1(const char *, str)
{
    while (*str != '\0')
        print_char((word)*str++);
}

inline void
print2 A1(word, address)
{
    word    page;
    word    offset;

    /* Ouh les vilains pas beaux qui ne regardent pas la valeur de BLOCK_SIZE...*/
/*     page = address >> 8; */
/*     offset = (address & 0xFF) << 1; */
    page = address >> 11;
    offset = (address << 1) & 0xFFF;

    prt_coded(&page, &offset);
}

inline void
print1 A1(word, address)
{
    word    page;
    word    offset;

    page = address / BLOCK_SIZE;
    offset = address % BLOCK_SIZE;
    prt_coded(&page, &offset);
}

void
p_obj A1(word, obj_num)
{
    object_t    *obj;
    word        address;
    word        page;
    word        offset;

    obj = obj_addr(obj_num);
    address = Z_TO_WORD(obj->data);
    page = address / BLOCK_SIZE;
    offset = address % BLOCK_SIZE;

    /*
     * The first byte at the address is the length of the data: if
     * it's 0 then there's nothing to print, so don't.
     */
    if (get_byte(&page, &offset) > 0)
    {
        prt_coded(&page, &offset);
    }
}

void
wrt()
{
    /*extern word     pc_page;*/
    /*extern word     pc_offset;*/

    prt_coded(&pc_page, &pc_offset);
    fix_pc();
}

inline void
writeln()
{
    wrt();
    new_line();
    rtn(1);
}

void
new_line()
{
    /*extern print_buf_t  *pbf_p;*/

    pbf_p->buf[pbf_p->len] = '\0';
    scr_putline((char *)pbf_p->buf);
    pbf_p->len = 0;
}

void
print_char A1(word, ch)
{
    /*extern print_buf_t *pbf_p;*/

    /*
     * If we're at the end of the buffer then get some more memory...
     */
    if (pbf_p->len == pbf_p->max)
    {
        pbf_p->max += BUFMIN;
        pbf_p->buf = (byte *)realloc(pbf_p->buf, pbf_p->max + 1);
	if (!(pbf_p->buf)) {
	  error ("print_char : plus de mmoire (longueur du buffer : %hu)", pbf_p->len);
	  return;
	}
    }

    pbf_p->buf[pbf_p->len++] = (byte)ch;
}


static inline void
letter A1(char, ch)
{
    /*extern char     table[];*/

    if (ch == 0)
    {
        print_char((word)' ');
        single_mode = print_mode;
        return;
    }

    if (ch <= 3)
    {
        /* Set single_mode to "Common Word" & set word_bank */

        single_mode |= 0x80;
        word_bank = (ch - 1) << 6;
        return;
    }

    if ((ch == 4) || (ch == 5))
    {
        /* Switch printing modes */

        if (single_mode == 0)
            single_mode = ch - 3;
        else
        {
            if (single_mode == ch - 3)
                single_mode = 0;
            print_mode = single_mode;
        }
        return;
    }

    if ((ch == 6) && (single_mode == 2))
    {
        /* Increment printing mode to 3 - ASCII Letter. */

        ++single_mode;
        return;
    }

    if ((ch == 7) && (single_mode == 2))
    {
        /* Print a Carriage Return */

        new_line();
        single_mode = print_mode;
        return;
    }

    /* None of the above, so this must be a single character */

    print_char((word)table[(single_mode * 26) + ch - 6]);
    single_mode = print_mode;
}

inline void
decode A1(word, data)
{
    /*extern byte     *common_word_ptr;*/

    word            page;
    word            offset;
    word            code;
    int             i;
    byte            *ptr;
    char            ch[3];

    /* Reduce word to 3 characters of 5 bits */

    code = data;
    for (i = 0; i <= 2; i++)
    {
        ch[i] = code & 0x1F;
        code >>= 5;
    }

    /* Print each character */

    for (i = 2; i >= 0; i--)
    {
        if (single_mode & 0x80)
        {
            /* Print a Special Word */

            ptr = common_word_ptr + word_bank + (int)(ch[i] << 1);
            page = Z_TO_BYTE_I(ptr);
            offset = Z_TO_BYTE(ptr) << 1;
            prt_coded(&page, &offset);
            single_mode = print_mode;
            continue;
        }
        if (single_mode < 3)
        {
            /* Print a single character */

            letter(ch[i]);
            continue;
        }
        if (single_mode == 3)
        {
            /*
             * Print ASCII character - store the high 3 bits of
             * char in the low 3 bits of the current printing mode.
             */

            single_mode = 0x40 + ch[i];
            continue;
        }
        if (single_mode & 0x40)
        {
            /*
             * Print an ASCII character - consists of the current
             * character as the low 5 bits and the high 3 bits coming
             * from the low 3 bits of the current printing mode.
             */

            ch[i] += (single_mode & 0x03) << 5;
            print_char((word)ch[i]);
            single_mode = print_mode;
        }
    }
}

void
prt_coded A2(word*, page, word*, offset)
{
    word    data;

    /*
     * Print mode = < 0 :   Common Word;
     *              = 0 :   Lower Case Letter;
     *              = 1 :   Upper Case Letter;
     *              = 2 :   Number or Symbol;
     *              = 3 :   ASCII Letter - first byte;
     *              > 3 :   ASCII Letter - second byte;
     */
    print_mode = 0;
    single_mode = 0;

    /* Last word has high bit set */
    do
    {
        data = get_word(page, offset);
        decode(data);
    }
    while ((data & 0x8000) == 0);
}

void
set_score()
{
    /*extern print_buf_t  *pbf_p;*/

    /*
     * Set the description
     */
    pbf_p = &room;
    room.len = 1;
    p_obj(load_var(0x10));
    room.buf[room.len] = '\0';
    pbf_p = &text;

    /*
     * Fill in the score or time fields...
     */
    sprintf (score_buf,
	     F1_IS_SET(B_USE_TIME) ? "Time: %hu:%02hu " : "Score: %-3hd  Moves: %-4hd ",
	     load_var(0x11), load_var(0x12));
}
