package zpplet.misc;

import java.util.*;
import java.io.*;

import zpplet.iff.*;
import zpplet.machine.ZMachine;
import zpplet.ops.ZStackFrame;

public class ZState
	{
	private ZMachine zm;
	private Stack st;
	private int pc;
	private byte[] dynamic;
	private int[] l;

	public ZState(ZMachine zm)
		{
		this.zm = zm;
		snapshot();
		}

	public void snapshot()
		{
		dynamic = new byte[zm.hd.getStaticBase()];
		System.arraycopy(zm.getMem(), 0, dynamic, 0, dynamic.length);
		st = (Stack)zm.st.clone();
		l = (int[])zm.l.clone();
		pc = zm.pc;
		}

	public void restoreSnapshot()
		{
		System.arraycopy(dynamic, 0, zm.getMem(), 0, dynamic.length);
		zm.st = (Stack)st.clone();
		zm.l = (int[])l.clone();
		zm.pc = pc;
		boolean t = zm.hd.getTranscripting();
		zm.setHeaderFlags();
		zm.hd.setTranscripting(t);
		}

	private final static int QUETZAL_PROCEDURE = 0x10;

	public void loadQuetzalFile(String fname)
			throws IOException, ZError
		{
		IFFInput infile = null;
		IFF.ChunkInfo chunkinfo;
		int release;
		int checksum;
		long ifhdend;

		try
			{
			infile = new IFFInput(fname);
			chunkinfo = infile.openChunk();
			if (!chunkinfo.type.equals("FORMIFZS"))
				throw new IOException("Invalid file type (expecting Quetzal format)");

			// find the IFHD
			chunkinfo = infile.skipToChunk("IFhd");
			if (chunkinfo == null)
				throw new IOException("Missing 'IFhd' chunk");
			release = infile.readShort() & 0xFFFF;
			byte[] serial = new byte[6];
			infile.read(serial);
			checksum = infile.readShort() & 0xFFFF;

			// verify the story file type
			if (release != zm.hd.getRelease())
				throw new ZError("Release does not match (game "
						+ zm.hd.getRelease() + ", file " + release + ")");
			if (checksum != zm.hd.getChecksum())
				throw new ZError("Checksum does not match (game 0x"
						+ Integer.toHexString(zm.hd.getChecksum()) + ", file 0x"
						+ Integer.toHexString(checksum) + ")");
			byte[] zmserial = zm.hd.getSerialNumber();
			for (int i = 0; i < serial.length; i++)
				if (serial[i] != zmserial[i])
					throw new ZError("Serial number does not match");

			// Set the program counter
			pc = (infile.readByte() & 0xFF) << 16;
			pc += infile.readShort() & 0xFFFF;
			infile.closeChunk();

			ifhdend = infile.getFilePointer();

			// read the memory chunk
			chunkinfo = infile.skipToChunk("UMem");
			if (chunkinfo != null)
				{
				if (chunkinfo.length != zm.hd.getStaticBase())
					throw new IOException("Dynamic memory area is "
							+ chunkinfo.length + ", expected "
							+ zm.hd.getStaticBase());
				dynamic = new byte[chunkinfo.length];
				infile.read(dynamic);
				}
			else
				{
				boolean runmode;
				byte ch;
				int nbytesout;
				int length;

				infile.seek(ifhdend);
				chunkinfo = infile.skipToChunk("CMem");
				if (chunkinfo == null)
					throw new IOException("Missing 'CMem' or 'UMem' chunk");
				dynamic = new byte[zm.hd.getStaticBase()];
				System.arraycopy(zm.restart.dynamic, 0, dynamic, 0,
						zm.hd.getStaticBase());

				length = chunkinfo.length;
				nbytesout = 0;
				runmode = false;
				while (length-- > 0)
					{
					if (nbytesout >= zm.hd.getStaticBase())
						throw new IOException("'CMem' exceeded dynamic memory size");
					ch = infile.readByte();
					if (runmode)
						{
						runmode = false;
						nbytesout += (ch & 0xFF) + 1;
						}
					else if (ch != 0)
						dynamic[nbytesout++] ^= ch;
					else
						runmode = true;
					}
				}
			infile.closeChunk();

			// read the stacks
			infile.seek(ifhdend);
			chunkinfo = infile.skipToChunk("Stks");
			if (chunkinfo == null)
				throw new IOException("Missing 'Stks' chunk");
			st = new Stack();

			int[] prevlocals = null;

			while (infile.getChunkPosition() < chunkinfo.length)
				{
				// read the frame header
				int framepc = (infile.readByte() & 0xFF) << 16;
				framepc += infile.readShort() & 0xFFFF;
				byte flags = infile.readByte();
				byte resultvar = infile.readByte();
				byte argmask = infile.readByte();
				int evalwords = infile.readShort() & 0xFFFF;

				// read locals
				int numlocals = flags & 0xF;
				int[] framelocals = new int[numlocals];
				for (int i = 0; i < numlocals; i++)
					framelocals[i] = infile.readShort() & 0xFFFF;

				// build internal frame
				if (prevlocals != null) // skip first dummy frame
					{
					ZStackFrame sf = new ZStackFrame();
					sf.interrupt = false;
					if ((flags & QUETZAL_PROCEDURE) != 0)
						sf.store = -1;
					else
						sf.store = resultvar;

					if ((argmask & (argmask + 1)) != 0)
						throw new IOException(
								"This Quetzal implementation does not support noncontiguous arguments");
					int argcount = 0;
					while (argmask > 0)
						{
						argcount++;
						argmask >>= 1;
						}
					
					sf.pc = framepc;
					sf.args = argcount;
					sf.l = prevlocals;
					st.push(sf);
					}
				prevlocals = framelocals;

				// push the evaluation stack
				for (int i = 0; i < evalwords; i++)
					st.push(new Integer(infile.readShort() & 0xFFFF));
				}

			this.l = prevlocals;
			infile.closeChunk();
			}
		finally
			{
			if (infile != null)
				infile.close();
			}
		restoreSnapshot();
		}

	private void writeCMemChunk(IFFOutput outfile)
			throws IOException
		{
		int runsize = 0;

		outfile.openChunk("CMem");
		for (int i = 0; i < zm.hd.getStaticBase(); i++)
			{
			if (dynamic[i] == zm.restart.dynamic[i])
				runsize++;
			else
				{
				while (runsize > 0)
					{
					outfile.writeByte(0);
					if (runsize >= 256)
						{
						outfile.writeByte((byte)255);
						runsize -= 256;
						}
					else
						{
						outfile.writeByte((byte)(runsize - 1));
						runsize = 0;
						}
					}
				outfile.writeByte((byte)(dynamic[i] ^ zm.restart.dynamic[i]));
				}
			}
		outfile.closeChunk();
		}

	// return the locals stored in the next frame;
	// return current locals if no more frames
	int[] nextLocals(Object[] e, int i)
		{
		while (i < e.length)
			{
			if (e[i] instanceof ZStackFrame)
				return ((ZStackFrame)e[i]).l;
			i++;
			}
		return l;
		}

	public void saveQuetzalFile(String fname)
			throws IOException
		{
		IFFOutput outfile = null;
		try
			{
			outfile = new IFFOutput(fname, "FORMIFZS");

			// IFhd chunk (general header info)
			outfile.openChunk("IFhd");
			outfile.writeShort((short)zm.hd.getRelease());
			byte[] serial = zm.hd.getSerialNumber();
			outfile.write(serial);
			outfile.writeShort((short)zm.hd.getChecksum());
			outfile.writeByte((byte)((pc >> 16) & 0xFF));
			outfile.writeShort((short)(pc & 0xFFFF));
			outfile.closeChunk();

			// CMem chunk (memory image)
			writeCMemChunk(outfile);

			// Stks chunk (stack info)
			outfile.openChunk("Stks");

			// write the dummy frame
			outfile.writeByte(0); // PC high
			outfile.writeShort(0); // PC low
			outfile.writeByte(0); // flags
			outfile.writeByte(0); // storevar
			outfile.writeByte(0); // argmask

			Object[] e = st.toArray();
			int iStack = 0;

			long argloc = outfile.getFilePointer();
			outfile.writeShort(0); // n words used for evaluation stack
			while (e[iStack] instanceof Integer)
				outfile.writeShort((short)((Integer)e[iStack++]).intValue());
			outfile.writeShortAt((short)iStack, argloc);

			// write remainder of stack (regular complete frames)
			while (iStack < e.length)
				{
				byte frameflags;
				byte storevar;
				byte argmask;
				boolean framestore;
				int[] locals;

				// frame
				ZStackFrame sf = (ZStackFrame)e[iStack++];
				framestore = (sf.store > -1);
				if (framestore)
					storevar = (byte)sf.store;
				else
					storevar = 0;
				locals = nextLocals(e, iStack);
				// i.e., the locals FOR this frame, not the saved ones
				// from the previous frame

				// eval stack
				int nEval = 0;
				while ((iStack + nEval) < e.length)
					if (e[iStack + nEval] instanceof Integer)
						nEval++;
					else
						break;
				short[] evals = new short[nEval];
				for (int i = 0; i < nEval; i++)
					evals[i] = (short)((Integer)e[iStack++]).intValue();

				frameflags = (byte)locals.length;
				if (!framestore)
					frameflags |= QUETZAL_PROCEDURE;
				argmask = (byte)((1 << sf.args) - 1);

				// write frame info
				outfile.writeByte((byte)((sf.pc >> 16) & 0xFF));
				outfile.writeShort((short)(sf.pc & 0xFFFF));
				outfile.writeByte(frameflags);
				outfile.writeByte(storevar);
				outfile.writeByte(argmask);
				outfile.writeShort((short)evals.length);

				// write locals
				for (int i = 0; i < locals.length; i++)
					outfile.writeShort((short)locals[i]);

				// write eval stack
				for (int i = 0; i < evals.length; i++)
					outfile.writeShort(evals[i]);
				}
			outfile.closeChunk();
			}
		finally
			{
			if (outfile != null)
				outfile.close();
			}
		}
	}