package zpplet.misc;

import java.awt.event.KeyEvent;

import zpplet.machine.ZMachine;

public final class ZChars
	{
	public final static int ABORT = -1; // signal to terminate ASAP
	public final static int TIMEOUT = 0; // signals a timeout on input
	public final static int INVALID = -2; // indicates a bad character; ignore

	public final static int BACKSPACE = 8; // input
	public final static int TAB = 9; // output
	public final static int SENTENCE_SPACE = 11; // output, V6
	public final static int RETURN_CHAR = 13;
	public final static int ESCAPE = 27; // input
	// 32-126 (' ' to '~') are regular ASCII
	public final static int DELETE = 127; // input [NONSTANDARD]
	public final static int CURSOR_UP = 129; // input
	public final static int CURSOR_DOWN = 130; // input
	public final static int CURSOR_LEFT = 131; // input
	public final static int CURSOR_RIGHT = 132; // input
	public final static int FUNCTION_KEYS = 133; // input, F1=133 ... F12=144
	public final static int KEYPAD_KEYS = 145; // input, 0=145 ... 9=154
	// 155-251 = fancy pants foreign letters (mapped via unicode table)
	public final static int MOUSE_MENU = 252; // input
	public final static int MOUSE_DBLCLICK = 253; // input
	public final static int MOUSE_CLICK = 254; // input
	// 255-1023 = '?'
	
	// style/color encapsulation characters
	public final static char FORMAT_ESCAPE = '\uFFFF';
	// followed by 2 characters: style/font, and color

	private static char unicode_defaults[] = { '\u00e4', /* a-umlaut */
	'\u00f6', /* o-umlaut */
	'\u00fc', /* u-umlaut */
	'\u00c4', /* A-umlaut */
	'\u00d6', /* O-umlaut */
	'\u00dc', /* U-umlaut */
	'\u00df', /* sz-ligature */
	'\u00bb', /* right-pointing quote */
	'\u00ab', /* left-pointing quote */
	'\u00eb', /* e-umlaut */
	'\u00ef', /* i-umlaut */
	'\u00ff', /* y-umlaut */
	'\u00cb', /* E-umlaut */
	'\u00cf', /* I-umlaut */
	'\u00e1', /* a-acute */
	'\u00e9', /* e-acute */
	'\u00ed', /* i-acute */
	'\u00f3', /* o-acute */
	'\u00fa', /* u-acute */
	'\u00fd', /* y-acute */
	'\u00c1', /* A-acute */
	'\u00c9', /* E-acute */
	'\u00cd', /* I-acute */
	'\u00d3', /* O-acute */
	'\u00da', /* U-acute */
	'\u00dd', /* Y-acute */
	'\u00e0', /* a-grave */
	'\u00e8', /* e-grave */
	'\u00ec', /* i-grave */
	'\u00f2', /* o-grave */
	'\u00f9', /* u-grave */
	'\u00c0', /* A-grave */
	'\u00c8', /* E-grave */
	'\u00cc', /* I-grave */
	'\u00d2', /* O-grave */
	'\u00d9', /* U-grave */
	'\u00e2', /* a-circumflex */
	'\u00ea', /* e-circumflex */
	'\u00ee', /* i-circumflex */
	'\u00f4', /* o-circumflex */
	'\u00fb', /* u-circumflex */
	'\u00c2', /* A-circumflex */
	'\u00ca', /* E-circumflex */
	'\u00ce', /* I-circumflex */
	'\u00d4', /* O-circumflex */
	'\u00da', /* U-circumflex */
	'\u00e5', /* a-ring */
	'\u00c5', /* A-ring */
	'\u00f8', /* o-slash */
	'\u00d8', /* O-slash */
	'\u00e3', /* a-tilde */
	'\u00f1', /* n-tilde */
	'\u00f5', /* o-tilde */
	'\u00c3', /* A-tilde */
	'\u00d1', /* N-tilde */
	'\u00d5', /* O-tilde */
	'\u00e6', /* ae-ligature */
	'\u00c6', /* AE-ligature */
	'\u00e7', /* c-cedilla */
	'\u00c7', /* C-cedilla */
	'\u00fe', /* Icelandic thorn */
	'\u00f0', /* Icelandic eth */
	'\u00de', /* Icelandic Thorn */
	'\u00d0', /* Icelandic Eth */
	'\u00a3', /* UK pound symbol */
	'\u0153', /* oe ligature */
	'\u0152', /* OE ligature */
	'\u00a1', /* inverse-! */
	'\u00bf', /* inverse-? */
	};

	private ZMachine zm;
	private int alphabase;
	private int terminatorbase;
	private int unicodebase;

	public ZChars(ZMachine zm)
		{
		this.zm = zm;
		}

	public void setCustomAddresses(int alphabase, int terminatorbase, int unicodebase)
		{
		this.alphabase = alphabase;
		this.terminatorbase = terminatorbase;
		this.unicodebase = unicodebase;
		}

	static public int fromActionKeyToZAscii(int key)
		{
		switch (key)
			{
			case KeyEvent.VK_ESCAPE:
				return ESCAPE;
			case KeyEvent.VK_UP:
				return CURSOR_UP;
			case KeyEvent.VK_DOWN:
				return CURSOR_DOWN;
			case KeyEvent.VK_LEFT:
				return CURSOR_LEFT;
			case KeyEvent.VK_RIGHT:
				return CURSOR_RIGHT;
			case KeyEvent.VK_HOME:
				return KEYPAD_KEYS + 7;
			case KeyEvent.VK_END:
				return KEYPAD_KEYS + 1;
			}

		if ((key >= KeyEvent.VK_F1) && (key <= KeyEvent.VK_F12))
			return key - KeyEvent.VK_F1 + FUNCTION_KEYS;

		if ((key >= KeyEvent.VK_NUMPAD0) && (key <= KeyEvent.VK_NUMPAD9))
			return key - KeyEvent.VK_NUMPAD0 + KEYPAD_KEYS;

		return INVALID;
		}

	public int fromUnicodeToZAscii(char unicode)
		{
		if (isTerminator(unicode))
			return RETURN_CHAR;

		if (unicode == '\b')
			return BACKSPACE;
		if (unicode == '\uu001b')
			return ESCAPE;
		if (unicode < 0x20)
			return INVALID;

		if (unicode <= DELETE) // normal ascii, including DELETE
			return unicode;

		if (unicodebase == 0)
			{
			for (int i = 0; i < unicode_defaults.length; i++)
				if (unicode_defaults[i] == unicode)
					return 155 + i;
			return INVALID;
			}

		int n = zm.getByte(unicodebase);
		for (int i = 0; i < n; i++)
			if (zm.getWord(unicodebase + 1 + i * 2) == unicode)
				return 155 + i;
		return INVALID;
		}

	public boolean isStorable(int zch)
		{ // true if can be stored to a string in memory
		return ((zch >= 32) && (zch <= 126)) || (zch == RETURN_CHAR) || ((zch >= 155) && (zch <= 251));  
		}
	
	public char toOutput(int zascii)
		{
		if ((zascii >= 32) && (zascii <= 126)) // normal ASCII
			return (char)zascii;

		// unicode
		if ((zascii >= 155) && (zascii <= 251))
			{
			if (unicodebase == 0)
				{
				if ((zascii - 155) < unicode_defaults.length)
					return unicode_defaults[zascii - 155];
				return 0;
				}

			int n = zm.getByte(unicodebase);
			if ((zascii - 155) < n)
				return (char)zm.getWord(unicodebase + 1 + (zascii - 155) * 2);
			return 0;
			}

		return 0;
		}
	
/* // fake the graphics characters
	switch (zascii)
		{
		case 179:
			return '|';
		case 186:
			return '#';
		case 196:
			return '-';
		case 205:
			return '=';
		}
	if ((zascii >= 179) && (zascii <= 218))
		return '+';
*/

	public char fromTable(int zchar, int alpha)
			throws ZError
		{
		if (alphabase == 0)
			switch (alpha)
				{
				case 0:
					return (char)('a' + zchar - 6);
				case 1:
					return (char)('A' + zchar - 6);
				case 2:
					return A2DEFAULT.charAt(zchar - 6);
				default:
					throw new ZError("Bad alphabet " + alpha);
				}

		return (char)zm.getByte(alphabase + alpha * 26 + zchar - 6);
		}

	public boolean isFunctionKey(int ch)
		{
		if ((ch >= 129) && (ch <= 154))
			return true;
		if ((ch >= 252) && (ch <= 254))
			return true;
		return false;
		}

	public boolean isTerminator(int ch)
		{
		if ((ch == 13) || (ch == 10))
			return true;
		if (terminatorbase != 0)
			{
			int i = terminatorbase;
			while (zm.getByte(i) != 0)
				{
				if (zm.getByte(i) == 255)
					return isFunctionKey(ch);
				if (zm.getByte(i++) == ch)
					return true;
				}
			}
		return false;
		}

	private final static String A2DEFAULT = " \n0123456789.,!?_#\'\"/\\-:()";

	private int getAlphaTranslation(int ch)
	// low byte = character; high byte = alphabet
	// alphabet = 3 means use ASCII
		{
		if (alphabase == 0)
			{
			if ((ch >= 'a') && (ch <= 'z'))
				return ch - 'a'; // A0
			if ((ch >= 'A') && (ch <= 'Z'))
				return ch - 'A' + 0x100; // A1
			int index = A2DEFAULT.indexOf((char)ch);
			if (index >= 0)
				return index + 0x200; // A2
			return ch + 0x300;
			}

		for (int i = 0; i < 26 * 3; i++)
			if (zm.getByte(alphabase + i) == ch)
				return (i % 26) + (i / 26) * 0x100;
		return ch + 0x300;
		}

/*	private void debugShowWord(short[] words)
		{
		byte[] mem = zm.getMem();
		int addr = mem.length - words.length * 2;
		for (int i = 0; i < words.length; i++)
			zm.setShort(addr + i * 2, words[i]);
		try
			{
			String s = zm.getStringAt(addr);
			System.out.println("encoded '" + s + "'");
			}
		catch (Exception e)
			{}
		} */

	public int[] encode(int addr, int len, int nwords)
		{
		int[] result = new int[nwords];
		int[] zchars = new int[nwords * 3];
		int zi = 0;

		try
			{
			for (int i = 0; i < len; i++)
				{
				int ch = getAlphaTranslation(zm.getByte(addr + i));
				switch (ch >> 8)
					{
					case 0:
					case 1: // encode A1 as A0 (lower case)
						zchars[zi++] = (ch & 0xFF) + 6;
						break;
					case 2:
						zchars[zi++] = 5;
						zchars[zi++] = (ch & 0xFF) + 6;
						break;
					default: // must encode as ASCII
						ch &= 0xFF;
						zchars[zi++] = 5;
						zchars[zi++] = 6;
						zchars[zi++] = (ch >> 5);
						zchars[zi++] = (ch & 0x1F);
					}
				}
			}
		catch (ArrayIndexOutOfBoundsException e)
			{}

		while (zi < zchars.length)
			zchars[zi++] = 5;

		zi = 0;
		for (int i = 0; i < nwords; i++)
			result[i] = (zchars[zi++] << 10) + (zchars[zi++] << 5) + zchars[zi++];
		result[nwords - 1] |= 0x8000;
		return result;
		}
	}