package zpplet.machine;

import java.util.*;
import java.io.*;

import javax.swing.JOptionPane;

import zpplet.data.*;
import zpplet.header.ZHeader;
import zpplet.misc.*;
import zpplet.ops.ZInstruction;
import zpplet.system.*;

public abstract class ZMachine
		extends Thread
	{
	protected final static int STANDARDS_VERSION_MAJOR = 0;
	protected final static int STANDARDS_VERSION_MINOR = 0;
	protected final static int INTERPRETER_VERSION = 'J';

	public final static int OSTREAM_SCREEN = 1;
	public final static int OSTREAM_TRANSCRIPT = 2; // to a file, for printing
	public final static int OSTREAM_MEMORY = 3;
	public final static int OSTREAM_LOG = 4; // to a file, for input

	public final static int ISTREAM_KEYBOARD = 0;
	public final static int ISTREAM_LOG = 1; // from a file

	public boolean running;
	public int pc; // program counter
	public ZWindow[] w;
	public ZWindow curw;
	public ZHeader hd; // image header
	public ZScreen s;
	public ZObjectTree objs;
	public ZDictionary zd;

	public Random random;
	protected byte[] m; // virtual memory image
	public Stack st;
	public int[] l;
	protected int g; // base address of globals
	public ZState restart;

	public int checksum;

	public boolean[] outputs; // output stream selection
	public BufferedWriter transcript;
	public Stack stream3; // contexts for stream 3 outputs
	public BufferedWriter log;
	public BufferedReader input;

	public ZInstruction zi;
	public ZChars zc;
	public ZMedia media;

	public ZMachine(byte[] m, ZScreen s)
		{
		// universal initialization here
		this.m = m;
		this.s = s;
		s.attach(this);
		setDaemon(true);
		running = false;

		zc = new ZChars(this);

		l = new int[0];
		st = new Stack();
		random = new Random();
		outputs = new boolean[5]; // 0 is not used
		outputs[OSTREAM_SCREEN] = true;
		input = null;
		stream3 = new Stack();
		}
	
	public boolean wantsMedia()
		{
		// true if sound/pictures bits set; works even before initStructures()
		ZHeader header = new ZHeader(m);
		return (media == null) && header.wantsMedia();
		}

	public static ZMachine NewZMachine(String fname, ZScreen screen)
		{
		byte[] data;
		try
			{
			File f = new File(fname);
			data = new byte[(int)f.length()];
			FileInputStream fs = new FileInputStream(fname);
			fs.read(data);
			fs.close();
			}
		catch (IOException e)
			{
			return null;
			}
		switch (ZHeader.getImageVersion(data))
			{
			case 2:
				return new ZMachine2(data, screen);
			case 3:
				return new ZMachine3(data, screen);
			case 4:
				return new ZMachine4(data, screen);
			case 5:
				return new ZMachine5(data, screen);
			case 6:
				return new ZMachine6(data, screen);
			case 7:
				return new ZMachine7(data, screen);
			case 8:
				return new ZMachine8(data, screen);
			case 'F': // blorb file assumed
				data = null;
				ZMedia m = new ZMedia();
				if (m.readBlorbFile(fname)) return m.getZM(screen);
				break;
			}
		return null;
		}

	// create appropriate ZMachine instance from byte data
	public static ZMachine NewZMachine(byte[] data, ZScreen screen)
		{
		switch (ZHeader.getImageVersion(data))
			{
			case 2:
				return new ZMachine2(data, screen);
			case 3:
				return new ZMachine3(data, screen);
			case 4:
				return new ZMachine4(data, screen);
			case 5:
				return new ZMachine5(data, screen);
			case 6:
				return new ZMachine6(data, screen);
			case 7:
				return new ZMachine7(data, screen);
			case 8:
				return new ZMachine8(data, screen);
			default:
				return null;
			}
		}

	public void loadMedia(String fname)
		// TODO: should load media from a stream?
		// would require IFF rewrite from RandomAccessFile-based to DataInputStream
		// on a byte[]...
		{
		try
			{
			media = new ZMedia();
			if (!media.readBlorbFile(fname)) media = null;
			}
		catch (Exception e)
			{
			media = null;
			}
		}

	final public synchronized void kill()
		{
		running = false;
		s.detach();
		for (int i = 0; i < w.length; i++)
			w[i] = null;
		interrupt();
		boolean joined = false;
		while (!joined)
			try
				{
				join();
				joined = true;
				}
			catch (InterruptedException e)
				{}
		}

	// version-specific initializers
	abstract protected void initStructures();

	abstract protected void initWindows();

	abstract public void setHeaderFlags(); // initial flag setting

	// common initialization, final stage prior to run
	final public void start()
		{
		initStructures();

		s.redoMetrics();
		setHeaderFlags();
		initWindows();

		g = hd.getGlobalTableAddr();
		checksum = calculateChecksum();
		zi.callMain();
		hd.setTranscripting(false);
		restart = new ZState(this);

		running = true;
		s.clear();
		s.requestFocus();

		super.start();
		}

	final public void run()
		{
		while (running)
			{
			try
				{
				zi.decode();
				zi.execute();
				}
			catch (Exception e)
				{
				if (e.getMessage() != null)
					{
					e.printStackTrace();
					JOptionPane.showMessageDialog(s, e.getMessage(), null, JOptionPane.WARNING_MESSAGE);
					}
				running = false;
				}
			}
		}

	public void restart()
		{
		restart.restoreSnapshot();
		}

	public void updateStatusLine()
		{
		ZWindow current = curw;
		try
			{
			String left = getStringAt(objs.getShortNameAddr(getVariable(16)));
			w[1].clear();
			w[1].moveCursor(2, 1);
			curw = w[1];
			printAsciiString(left);

			String right;
			if (hd.isTimeGame())
				{
				int hours = getVariable(17);
				int minutes = getVariable(18);
				String meridiem;
				if (hours < 12)
					meridiem = "AM";
				else
					meridiem = "PM";
				hours %= 12;
				if (hours == 0) hours = 12;
				right = Integer.toString(hours) + (minutes < 10 ? ":0" : ":") + Integer.toString(minutes) + meridiem;
				}
			else
				{
				int score = getVariable(17);
				int turns = getVariable(18);
				right = Integer.toString(score) + "/" + Integer.toString(turns);
				}
			w[1].moveCursor(s.getChars() - right.length() - 1, 1);
			printAsciiString(right);
			}
		catch (ZError e)
			{}

		curw = current;
		}

	abstract public int unpackSAddr(int addr); // string addresses

	abstract public int unpackRAddr(int addr); // routine addresses

	final public void printAsciiChar(int ch)
		{
		if (outputs[OSTREAM_MEMORY])
			{
			printToStream3(ch);
			return;
			}

		if (outputs[OSTREAM_SCREEN])
			{
			if (zc.isTerminator(ch))
				curw.newLine();
			else
				curw.printZAscii(ch);
			}

		if (outputs[OSTREAM_TRANSCRIPT]) try
			{
			transcript.write(ch);
			}
		catch (IOException e)
			{
			System.err.println("Transcript write FAILED");
			}
		}

	abstract public String getStringAt(int addr)
			throws ZError;

	final protected int calculateChecksum()
		{
		int filesize = hd.getFileLength();
		int result = 0;
		if (filesize <= m.length) for (int i = 0x40; i < filesize; i++)
			result += getByte(i);
		return result & 0xFFFF;
		}

	final public int getVariable(int varnum)
		{
		if (varnum == 0) // stack
			return ((Integer)st.pop()).intValue();
		else if (varnum >= 0x10) // globals
			return getWord(g + (varnum - 0x10) * 2);
		else
			// locals
			return l[varnum - 1];
		}

	final public void setVariable(int varnum, int val)
		{
		val &= 0xFFFF;
		if (varnum == 0) // stack
			st.push(new Integer(val));

		else if (varnum < 0x10) // locals
			l[varnum - 1] = val;

		else
			// globals
			setWord(g + ((varnum - 0x10) * 2), val);
		}

	final public int getCodeByte()
		{
		return m[pc++] & 0xFF;
		}

	final public int getCodeWord()
		{
		return ((m[pc++] << 8) & 0xFF00) | (m[pc++] & 0xFF);
		}

	final public int getByte(int addr)
		{
		return m[addr] & 0xFF;
		}

	final public void setByte(int addr, int val)
		{
		m[addr] = (byte)(val & 0xFF);
		}

	final public int getWord(int addr) // unsigned
		{
		return ((m[addr] << 8) & 0xFF00) | (m[addr + 1] & 0xFF);
		}

	final public void setWord(int addr, int val)
		{
		m[addr] = (byte)((val >> 8) & 0xFF);
		m[addr + 1] = (byte)(val & 0xFF);
		}

	final public int getInput(boolean buffered, int timeout, int timeoutroutine)
		{
		int result = 0;
		if (buffered)
			result = s.readBufferedCode(timeout, timeoutroutine);
		else
			result = s.readKeyCode(timeout, timeoutroutine);

		if (outputs[OSTREAM_LOG] && (result >= 0)) try
			{
			log.write(result);
			}
		catch (IOException e)
			{
			System.err.println("Log write FAILED");
			}
		return result;
		}

	public void resize0(int height)
		{
		//w[0].resize0(height);
		w[0].clear();
		}

	public void resize()
		{
		w[0].resize(s.getChars(), s.getLines() - w[1].getLines());
		w[1].resize(s.getChars(), w[1].getLines());
		updateStatusLine();
		}

	final public byte[] getMem()
		{
		return m;
		}

	final public boolean verifyChecksum()
		{
		return (hd.getFileLength() <= m.length) && (hd.getChecksum() == checksum);
		}

	protected class Stream3Data
		{
		public Stream3Data(int addr, int width)
			{
			this.addr = addr;
			this.width = width;
			pixelwidth = 0;
			}

		public int addr;
		public int width;
		public int pixelwidth; // total pixel width printed (V6)
		}

	public void openStream3(int addr, int width)
		{
		stream3.push(new Stream3Data(addr, width));
		outputs[OSTREAM_MEMORY] = true;
		setWord(addr, 0);
		}

	public void closeStream3()
		{
		stream3.pop();
		outputs[OSTREAM_MEMORY] = !stream3.isEmpty();
		}

	public void printToStream3(int ch)
		{
		// "print" to memory location; not used until V4
		int printmemory = ((Stream3Data)stream3.peek()).addr;
		int nchars = getWord(printmemory);
		setByte(printmemory + nchars + 2, ch);
		setWord(printmemory, ++nchars);
		}

	public void printAsciiString(String s)
		{
		for (int i = 0; i < s.length(); i++)
			printAsciiChar(s.charAt(i));
		}

	public int printStringAt(int addr)
			throws ZError
		{
		String s = getStringAt(addr);
		printAsciiString(s);
		return s.length();
		}

	public void debugDumpHeader()
		{
		for (int i = 0; i < 0x38; i++)
			System.out.println("0x" + Integer.toHexString(i) + ": " + getByte(i));
		}

	public void setOutputStream(int stream, boolean value, int addr, int width)
		{
		switch (stream)
			{
			case OSTREAM_SCREEN:
				outputs[stream] = value;
				break;

			case OSTREAM_MEMORY:
				if (value)
					openStream3(addr, width);
				else
					closeStream3();
				break;

			case OSTREAM_TRANSCRIPT:
				if (value)
					{
					String fn = s.getFileName("Create Transcript", true);
					value = false;
					if (fn != null) try
						{
						transcript = new BufferedWriter(new FileWriter(fn));
						value = true;
						}
					catch (IOException e)
						{}
					}
				else
					{
					try
						{
						transcript.close();
						}
					catch (IOException e)
						{}
					transcript = null;
					}
				hd.setTranscripting(value);
				outputs[stream] = value;
				break;

			case OSTREAM_LOG:
				if (value)
					{
					String fn = s.getFileName("Create Log", true);
					value = false;
					if (fn != null) try
						{
						log = new BufferedWriter(new FileWriter(fn));
						value = true;
						}
					catch (IOException e)
						{}
					}
				else
					{
					try
						{
						log.close();
						}
					catch (IOException e)
						{}
					log = null;
					}
				outputs[stream] = value;
				break;

			default:
				System.err.println("Attempted to set stream " + stream + " to " + value);
			}
		}

	public synchronized void setInputStream(int n)
		{
		if (n == ISTREAM_KEYBOARD)
			{
			if (input == null) return;
			try
				{
				input.close();
				}
			catch (IOException e)
				{}
			input = null;
			return;
			}

		else if (n == ISTREAM_LOG)
			{
			try
				{
				if (input != null) input.close();
				}
			catch (IOException e)
				{}
			input = null;
			String fn = s.getFileName("Open Log File", false);
			if (fn != null) try
				{
				input = new BufferedReader(new FileReader(fn));
				s.primeInput();
				}
			catch (IOException e)
				{
				System.err.println(e.getMessage());
				}
			return;
			}

		System.err.println("Tried to set input stream to " + n);
		}
	
	public void notifyConfigChange()
		{
		w[0].font.touch();
		s.setSize(s.getWidth(), s.getHeight());
		s.repaint();
		}
	}