/*
 * $Id: emx.c 1.18 1996/09/26 04:36:49 Madsen Exp $
 *
 * IO interface for OS/2 and EMX 0.9b/gcc
 *
 */

#include <ctype.h>
#include <getopt.h>
#include <io.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/video.h>
#include <sys/kbdscan.h>
#include <time.h>
#include "frotz.h"

#define INCL_KBD
#define INCL_DOSPROCESS
#include <os2emx.h>

#define INFORMATION "\
\n\
FROTZ V2.01 - an interpreter for all Infocom games. Complies with standard 0.2\n\
for all games except V6 (graphic) games. Written by Stefan Jokisch in 1995-96\n\
Frotz/OS2 v2.01 beta 7 ported by Christopher J. Madsen <ac608@yfn.ysu.edu>\n\
\n\
\t-f # foreground colour\n\
\t-b # background colour\n\
\t-F # reverse mode foreground colour\n\
\t-B # reverse mode background colour\n\
\t-e # emphasis colour\n\
\n\
\t-w # screen width\n\
\t-h # screen height\n\
\t-l # left margin\n\
\t-r # right margin\n\
\t-c # context lines\n\
\n\
See the documentation for the more obscure command line switches."

#define byte0(v)	((unsigned char *)&v)[0]
#define byte1(v)	((unsigned char *)&v)[1]
#define byte2(v)	((unsigned char *)&v)[2]
#define byte3(v)	((unsigned char *)&v)[3]
#define word0(v)	((unsigned short*)&v)[0]
#define word1(v)	((unsigned short*)&v)[1]

static int user_foreground = -1;
static int user_background = -1;
static int user_reverse_fg = -1;
static int user_reverse_bg = -1;
static int user_emphasis = F_YELLOW|INTENSITY;
static int user_screen_width = -1;
static int user_screen_height = -1;
static int user_tandy_bit = 0;

static int last_row    = -1;
static int last_column = -1;

static char *stripped_story_name = NULL;

static int dump_message = 0;

static int current_style = 0;
static int current_fg = DEFAULT_COLOUR;
static int current_bg = DEFAULT_COLOUR;

#define MAX_SCREEN_WIDTH 134

static unsigned char current_attrib = F_WHITE|B_BLUE;
static unsigned char current_line[MAX_SCREEN_WIDTH*2];
static int cursor_x = 0;        /* Current cursor position */
static int cursor_y = 0;        /* Top left is 0,0 */

static unsigned char euro_map[] = {
    0x84, 0x94, 0x81, 0x8e, 0x99, 0x9a, 0xe1, 0xaf, 0xae, 0x89,
    0x8b, 0x98, 0xd3, 0xd8, 0xa0, 0x82, 0xa1, 0xa2, 0xa3, 0xec,
    0xb5, 0x90, 0xd6, 0xe0, 0xe9, 0xed, 0x85, 0x8a, 0x8d, 0x95,
    0x97, 0xb7, 0xd4, 0xde, 0xe3, 0xeb, 0x83, 0x88, 0x8c, 0x93,
    0x96, 0xb6, 0xd2, 0xd7, 0xe2, 0xea, 0x86, 0x8f, 0x9b, 0x9d,
    0xc6, 0xa4, 0xe4, 0xc7, 0xa5, 0xe5, 0x91, 0x92, 0x87, 0x80,
    0xd0, 0xe7, 0xd1, 0xe8, 0x9c, 0x00, 0x00, 0xad, 0xa8
};

static unsigned char pc_colour[] = {
    F_BLACK,
    F_BLACK,
    F_BLACK,
    F_RED,
    F_GREEN,
    F_YELLOW|INTENSITY,
    F_BLUE,
    F_MAGENTA,
    F_CYAN,
    F_WHITE,
    F_BLACK|INTENSITY
};

static unsigned char zcolour[] = {
    BLACK_COLOUR,
    BLUE_COLOUR,
    GREEN_COLOUR,
    CYAN_COLOUR,
    RED_COLOUR,
    MAGENTA_COLOUR,
    F_YELLOW + 16,
    F_WHITE  + 16,
    GREY_COLOUR,
    (F_BLUE|INTENSITY) + 16,
    (F_GREEN|INTENSITY) + 16,
    (F_CYAN|INTENSITY) + 16,
    (F_RED|INTENSITY) + 16,
    (F_MAGENTA|INTENSITY) + 16,
    YELLOW_COLOUR,
    WHITE_COLOUR
};


/* ========================================================================= *\


			  S O U N D   R O U T I N E S


\* ========================================================================= */

/*
 * os_beep
 *
 * Play a beep sound. Ideally, the sound should be high- (number == 1)
 * or low-pitched (number == 2).
 *
 */

void os_beep(int number)
{
  DosBeep((number == 1) ? 1000 : 500, 75);
}/* os_beep */

/*
 * os_prepare_sample
 *
 * Load the given sample from the disk.
 *
 */

void os_prepare_sample(int number)
{
}/* os_prepare_sample */

/*
 * os_start_sample
 *
 * Play the given sample at the given volume (ranging from 1 to 8 and
 * 255 meaning a default volume). The sound is played once or several
 * times in the background (255 meaning forever). In Z-code 3, the
 * repeats value is always 0 and the number of repeats is taken from
 * the sound file itself. The end_of_sound function is called as soon
 * as the sound finishes.
 *
 */

void os_start_sample(int number, int volume, int repeats)
{
}/* os_start_sample */

/*
 * os_stop_sample
 *
 * Turn off the current sample.
 *
 */

void os_stop_sample(void)
{
}/* os_stop_sample */

/*
 * os_finish_with_sample
 *
 * Remove the current sample from memory (if any).
 *
 */

void os_finish_with_sample(void)
{
}/* os_finish_with_sample */

/*
 * os_wait_sample
 *
 * Stop repeating the current sample and wait until it finishes.
 *
 */

void os_wait_sample(void)
{
}/* os_wait_sample */



/* ========================================================================= *\


			P I C T U R E   R O U T I N E S


\* ========================================================================= */

/*
 * os_picture_data
 *
 * Return true if the given picture is available. If so, store the
 * picture width and height in the appropriate variables. Picture
 * number 0 is a special case: Write the number of pictures available
 * and the picture file release number into the height and width
 * variables respectively when this picture number is asked for.
 *
 */

int os_picture_data(int picture, int *height, int *width)
{
  return 0;
}/* os_picture_data */

/*
 * os_draw_picture
 *
 * Display a picture at the given coordinates. Top left is (1,1).
 *
 */

void os_draw_picture(int picture, int y, int x)
{
}/* os_draw_picture */

/*
 * os_peek_colour
 *
 * Return the colour of the screen unit below the cursor. (If the
 * interface uses a text mode, it may return the background colour
 * of the character at the cursor position instead.) This is used
 * when text is printed on top of pictures. Note that this coulor
 * need not be in the standard set of Z-machine colours. To handle
 * this situation, Frotz extends the colour scheme: Colours above
 * 15 (and below 256) may be used by the interface to refer to non
 * standard colours. Of course, os_set_colour must be able to deal
 * with these colours.
 *
 */

int os_peek_colour(void)
{
  return (current_line[cursor_x*2 + 1] >> 4) + 16;
}/* os_peek_colour */


/* ========================================================================= *\


			  M O U S E   R O U T I N E S


\* ========================================================================= */

/*
 * os_mouse_area
 *
 * Restrict the movement of the mouse cursor to the given area. Top
 * left coordinates are (1,1).
 *
 */

void os_mouse_area(int top, int left, int bottom, int right)
{
}/* os_mouse_area */


/* ========================================================================= *\


			   T E X T   R O U T I N E S


\* ========================================================================= */

extern char *euro_substitute;

/*
 * os_font_available
 *
 * Return true if the given font is supported by this interface. The
 * font can be any of
 *
 *    TEXT_FONT
 *    PICTURE_FONT
 *    GRAPHICS_FONT
 *    FIXED_WIDTH_FONT
 *
 * The fonts are defined in the Specification of the Z-machine.
 *
 */

int os_font_available(int font)
{
  if ((font == TEXT_FONT) || (font == FIXED_WIDTH_FONT))
    return 1;

  return 0;
}/* os_font_available */

/*
 * adjust_style
 *
 * Set the current colours. This combines the current colour selection
 * and the current text style.
 *
 */

static void adjust_style(void)
{
  unsigned char fg,bg,text_fg,text_bg;

  /* Convert Z-machine colours to IBM PC colours */

  if (current_fg == DEFAULT_COLOUR)
    fg = user_foreground;
  else if (current_fg < 16)
    fg = pc_colour[current_fg];
  else
    fg = current_fg - 16;

  if (current_bg == DEFAULT_COLOUR)
    bg = user_background;
  else if (current_bg < 16)
    bg = pc_colour[current_bg];
  else
    bg = current_bg - 16;

  /* Set colours for text output */

  text_fg = fg;
  text_bg = bg;

  /* Handle reverse text style */

  if (current_style & REVERSE_STYLE) {
    text_fg = bg;
    text_bg = fg;

    if (current_bg == DEFAULT_COLOUR && user_reverse_fg != -1)
      text_fg = user_reverse_fg;
    if (current_fg == DEFAULT_COLOUR && user_reverse_bg != -1)
      text_bg = user_reverse_bg;
  }

  /* Handle emphasis style */

  if (current_style & EMPHASIS_STYLE) {
    text_fg = user_emphasis;
  }

  /* Handle boldface style */

  if (current_style & BOLDFACE_STYLE) {
    text_fg ^= 8;
  }

  current_attrib = ((text_bg << 4) | text_fg);
}/* adjust_style */

/*
 * os_set_colour
 *
 * Set the foreground and background colours which can be:
 *
 *     DEFAULT_COLOUR
 *     BLACK_COLOUR
 *     RED_COLOUR
 *     GREEN_COLOUR
 *     YELLOW_COLOUR
 *     BLUE_COLOUR
 *     MAGENTA_COLOUR
 *     CYAN_COLOUR
 *     WHITE_COLOUR
 *
 *     MS-DOS 320 columns MCGA mode only:
 *
 *     GREY_COLOUR
 *
 *     Amiga only:
 *
 *     LIGHTGREY_COLOUR
 *     MEDIUMGREY_COLOUR
 *     DARKGREY_COLOUR
 *
 * There may be more colours in the range from 16 to 255; see the
 * remarks about os_peek_colour.
 *
 */

void os_set_colour(int new_foreground, int new_background)
{
  /* Save colours in global variables */
  current_fg = new_foreground;
  current_bg = new_background;

  /* Apply changes */
  adjust_style();
}/* os_set_colour */

/*
 * os_set_font
 *
 * Set the font for text output. The interpreter takes care not to
 * choose fonts which aren't supported by the interface.
 *
 */

void os_set_font(int new_font, int *height, int *width)
{
  /* Return font dimensions */

  *height = h_font_height;
  *width  = h_font_width;
}/* os_set_font */

/*
 * os_set_text_style
 *
 * Set the current text style. Following flags can be set:
 *
 *     REVERSE_STYLE
 *     BOLDFACE_STYLE
 *     EMPHASIS_STYLE (aka underline aka italics)
 *     FIXED_WIDTH_STYLE
 *
 */

void os_set_text_style(int new_style)
{
  /* Save style in global variable */
  current_style = new_style;

  /* Apply changes */
  adjust_style();
}/* os_set_text_style */

/*
 * redraw
 *
 * Draw the contents of the current line buffer on the screen.
 *
 */

static void redraw(void)
{
  v_attrib(current_attrib);
  v_gotoxy(cursor_x, cursor_y);
  v_putline(current_line, 0, cursor_y, h_screen_cols);
} /* end redraw */

/*
 * os_display_char
 *
 * Display a character of the current font using the current colours
 * and text style. The cursor moves to the next position. Printable
 * codes are all ASCII values from 32 to 126, European characters from
 * EURO_MIN to EURO_MAX (as defined in the Specification of the
 * Z-machine), character 9 (paragraph indentation) and character 11
 * (gap between two sentences). The screen should not be scrolled after
 * printing to the bottom right corner.
 *
 */

void os_display_char(int c)
{
  /* Return if we are discarding a debugging message */

  if (dump_message != 0)
    return;

  /* Handle newline */
  if (c == '\n') {
    display_new_line();
    return;
  }

  /* Handle European characters */
  if (c >= EURO_MIN && c <= EURO_MAX && beyond_zork_flag == 0)
    if (euro_map[c - EURO_MIN])
      c = euro_map[c - EURO_MIN];
    else {
      os_display_char(euro_substitute[2 * (c - EURO_MIN)]);
      c = euro_substitute[2 * (c - EURO_MIN) + 1];
      if (c == ' ')
        return;
    } /* end else must use substitute */

  /* Handle special indentations */
  if (c == 9) {                 /* FIXME only 1 space if not bol */
    os_display_char(' ');
    os_display_char(' ');
    c = ' ';
  }
  if (c == 11) {
    os_display_char(' ');
    c = ' ';
  }

  /* Print character */
  current_line[cursor_x*2]   = c;
  current_line[cursor_x*2+1] = current_attrib;
  ++cursor_x;
}/* os_display_char */

/*
 * os_display_string
 *
 * Pass a string of characters to os_display_char.
 *
 */

void os_display_string(const char *s)
{
    int i;

    for (i = 0; s[i] != 0; i++)
	os_display_char((unsigned char) s[i]);

}/* os_display_string */

/*
 * os_char_width
 *
 * Return the length of the character in screen units.
 *
 */

int os_char_width(int c)
{
  /* Handle European characters */
  if (c >= EURO_MIN && c <= EURO_MAX && beyond_zork_flag == 0
      && (euro_map[c - EURO_MIN] == 0)
      && (euro_substitute[2 * (c - EURO_MIN) + 1] != ' '))
    return 2; /* This char has a 2 character substitute */

  /* Handle special indentations */
  if (c == 9)
    return 3;
  if (c == 11)
    return 2;

  return 1;
}/* os_char_width */

/*
 * os_string_width
 *
 * Calculate the length of a word in screen units. Apart from letters,
 * the word may contain special codes:
 *
 *    NEW_STYLE - next character is a new text style
 *    NEW_FONT  - next character is a new font
 *
 */

int os_string_width(const char *s)
{
  int width;
  int c;
  int i;

  width = 0;
  i = 0;

  while (s[i] != 0) {
    c = (unsigned char) s[i++];

    if (c == NEW_STYLE || c == NEW_FONT)
      i++;
    else
      width += os_char_width(c);
  }

  return width;
}/* os_string_width */

/*
 * os_set_cursor
 *
 * Place the text cursor at the given coordinates. Top left is (1,1).
 *
 */

void os_set_cursor(int y, int x)
{
  if (--y != cursor_y) {
    redraw();
    cursor_y = y;
    v_getline(current_line, 0, cursor_y, h_screen_cols);
  }
  cursor_x = x-1;
}/* os_set_cursor */


/* ========================================================================= *\


			  I N P U T   R O U T I N E S


\* ========================================================================= */

#define ARROW_UP    129
#define ARROW_DOWN  130
#define ARROW_LEFT  131
#define ARROW_RIGHT 132

#define FUNCTION_KEY_F1  133
#define FUNCTION_KEY_F11 143
#define FUNCTION_KEY_F12 144

#define KEYPAD_0 145
#define KEYPAD_1 146

#define SPECIAL_KEY_MIN 256

#define SPECIAL_KEY_HOME 256
#define SPECIAL_KEY_END 257
#define SPECIAL_KEY_WORD_LEFT 258
#define SPECIAL_KEY_WORD_RIGHT 259
#define SPECIAL_KEY_DELETE 260
#define SPECIAL_KEY_INSERT 261
#define SPECIAL_KEY_PAGE_UP 262
#define SPECIAL_KEY_PAGE_DOWN 263
#define SPECIAL_KEY_DEL_WORD_LEFT 264
#define SPECIAL_KEY_DEL_WORD_RIGHT 265

#define SPECIAL_KEY_MAX 265

#define HISTORY_LINES 128

static char history[HISTORY_LINES][80];

static int history_valid = 0;
static int history_count = 0;
static int history_entry = 0;
static int history_prefix_len = -1;

static int cursor = 1;          /* Cursor visible */
static int cursor_bot = 0;
static int cursor_ovr = 0;
static int cursor_ins = 0;

static int overwrite = 0;

static char *input_buffer = 0;

static int input_len = 0;
static int input_pos = 0;
static int input_max = 0;
static int input_lft = 0;

static clock_t time_limit = 0;

static int emacs_keys[] = {
  0,
  SPECIAL_KEY_HOME,
  ARROW_LEFT,
  3,
  SPECIAL_KEY_DELETE,
  SPECIAL_KEY_END,
  ARROW_RIGHT,
  7,
  8,
  9,
  10,
  11,
  12,
  13,
  ARROW_DOWN,
  15,
  ARROW_UP,
  17,
  18,
  19,
  20,
  27,                   /* C-u, erase input */
  22,
  23,
  24,
  25,
  26
}; /* end emacs_keys */

/*
 * os_cursor_on
 *
 * Make the cursor visible.
 *
 */

void os_cursor_on(void)
{
  cursor = 1;
  v_ctype((overwrite ? cursor_ovr : cursor_ins), cursor_bot);
}/* os_cursor_on */

/*
 * os_cursor_off
 *
 * Turn cursor off.
 *
 */

void os_cursor_off(void)
{
  cursor = 0;
  v_hidecursor();
}/* os_cursor_off */

/*
 * set_time_limit
 *
 * Calculate the target time for a given delay of 1/10 seconds and
 * store it in a global structure.
 *
 */

static void set_time_limit (int timeout)
{
  if (timeout)
    time_limit = clock() + timeout * (CLOCKS_PER_SEC / 10);
  else
    time_limit = 0;
}/* set_time_limit */

/*
 * get_key
 *
 * Read a key (a standard ASCII character, a European letter, a function
 * key, an arrow key, a keypad key, a special key or a hot key), wait for
 * a mouse click (single or double click) or return when the time limit
 * is exceeded.
 *
 */

static int get_key(void)
{
  KBDINFO     info;
  USHORT      info_rc, rc;
  USHORT      info_mask = 0;
  KBDKEYINFO  keyRec;
  int  key = -1;
  int  i;

  info.cb = sizeof(info);
  info_rc = KbdGetStatus(&info, 0);
  if (info_rc == 0) {
    info_mask = info.fsMask;
    info.fsMask &= ~0x7f;
    info.fsMask |= KEYBOARD_BINARY_MODE;
    info.cb = sizeof(info);
    KbdSetStatus(&info, 0);
  }

  do {
    if (time_limit && (clock() >= time_limit)) {
      rc = 1;                   /* Signal timeout */
      break;
    }
    rc = KbdCharIn(&keyRec, time_limit ? IO_NOWAIT : IO_WAIT, 0);
  } while ((rc != 0) || !(keyRec.fbStatus & KBDTRF_FINAL_CHAR_IN));

  if (info_rc == 0) {
    info.cb = sizeof(info);
    info.fsMask = info_mask;
    KbdSetStatus(&info, 0);
  }

  if (rc) return 0;             /* Timed out */

  if ((keyRec.chChar == 0 || keyRec.chChar == 0xe0)
      && (keyRec.fbStatus & KBDTRF_EXTENDED_CODE)) {
    key = keyRec.chScan;
    switch (key) {
     case K_DOWN:
      key = ARROW_DOWN;
      break;
     case K_UP:
      key = ARROW_UP;
      break;
     case K_LEFT:
      key = ARROW_LEFT;
      break;
     case K_RIGHT:
      key = ARROW_RIGHT;
      break;
     case K_HOME:
      key = SPECIAL_KEY_HOME;
      break;
     case K_END:
      key = SPECIAL_KEY_END;
      break;
     case K_ALT_LEFT:
     case K_CTRL_LEFT:
     case K_ALT_B:
      key = SPECIAL_KEY_WORD_LEFT;
      break;
     case K_ALT_RIGHT:
     case K_CTRL_RIGHT:
     case K_ALT_F:
      key = SPECIAL_KEY_WORD_RIGHT;
      break;
     case K_DEL:
      key = SPECIAL_KEY_DELETE;
      break;
     case K_ALT_BACKSPACE:
      key = SPECIAL_KEY_DEL_WORD_LEFT;
      break;
     case K_ALT_D:
     case K_ALT_DEL:
     case K_CTRL_DEL:
      key = SPECIAL_KEY_DEL_WORD_RIGHT;
      break;
     case K_INS:
      key = SPECIAL_KEY_INSERT;
      break;
     case K_PAGEUP:
      key = SPECIAL_KEY_PAGE_UP;
      break;
     case K_PAGEDOWN:
      key = SPECIAL_KEY_PAGE_DOWN;
      break;
     case K_ALT_A:
      key = HOT_KEY_ALIAS;
      break;
     case K_ALT_P:
      key = HOT_KEY_PLAYBACK;
      break;
     case K_ALT_R:
      key = HOT_KEY_RECORDING;
      break;
     case K_ALT_S:
      key = HOT_KEY_SEED;
      break;
     case K_ALT_U:
      key = HOT_KEY_UNDO;
      break;
     case K_ALT_N:
      key = HOT_KEY_RESTART;
      break;
     case K_ALT_X:
      key = HOT_KEY_QUIT;
      break;
     case K_F11:
      key = FUNCTION_KEY_F11;
      break;
     case K_F12:
      key = FUNCTION_KEY_F12;
      break;
     default:
      if ((key >= K_F1) && (key <= K_F10))
        key = FUNCTION_KEY_F1 - K_F1 + key;
      else
        key = -1;
      break;
    } /* end switch scan code */
  } /* end if extended scan code */
  else {
    key = keyRec.chChar;

    if (key >= '0' && key <= '9' && keyRec.chScan > 0x10)
      key += KEYPAD_0 - '0';    /* Numeric keypad */
    else
      /* Handle European characters */
      for (i = 0; i < sizeof(euro_map); i++)
        if (key == euro_map[i]) {
          key = i + EURO_MIN;
          break;
        }
  } /* end else normal character */

  return key;
}/* get_key */

/*
 * cursor_left
 *
 * Move the cursor one character to the left.
 *
 */

static void cursor_left(void)
{
  if (input_pos != 0) {
    input_pos--;
    cursor_x -= os_char_width((unsigned char) input_buffer[input_pos]);
  }
}/* cursor_left */

/*
 * cursor_right
 *
 * Move the cursor one character to the right.
 *
 */

static void cursor_right(void)
{
  if (input_pos != input_len) {
    cursor_x += os_char_width((unsigned char)input_buffer[input_pos]);
    input_pos++;
  }
}/* cursor_right */

/*
 * first_char
 *
 * Move the cursor to the beginning of the input line.
 *
 */

static void first_char(void)
{
  while (input_pos != 0)
    cursor_left();
}/* first_char */

/*
 * last_char
 *
 * Move the cursor to the end of the input line.
 *
 */

static void last_char(void)
{
  while (input_pos != input_len)
    cursor_right();
}/* last_char */

/*
 * prev_word
 *
 * Move the cursor to the start of the previous word.
 *
 */

static void prev_word(void)
{
  do {
    cursor_left();

    if (input_pos == 0)
      return;
  } while (input_buffer[input_pos] == ' ' || input_buffer[input_pos-1] != ' ');
}/* prev_word */

/*
 * next_word
 *
 * Move the cursor to the start of the next word.
 *
 */

static void next_word(void)
{

  do {

    cursor_right();

    if (input_pos == input_len)
      return;

  } while (input_buffer[input_pos] == ' ' || input_buffer[input_pos - 1] != ' ');

}/* next_word */

/*
 * delete_char
 *
 * Delete the character below the cursor.
 *
 */

static void delete_char(void)
{
  int char_width, string_width;

  if (input_pos == input_len)
    return;

  char_width   = os_char_width((unsigned char) input_buffer[input_pos]);
  string_width = os_string_width(input_buffer+input_pos+1);

  input_lft += char_width;

  memmove(current_line+cursor_x*2,
          current_line+(cursor_x+char_width)*2,
          string_width*2);
  current_line[(cursor_x+string_width)*2]   = ' ';
  current_line[(cursor_x+string_width)*2+1] = current_attrib;

  memmove (input_buffer + input_pos,
           input_buffer + input_pos + 1,
           input_len - input_pos);

  input_len--;
}/* delete_char */

/*
 * delete_left
 *
 * Delete the character to the left of the cursor.
 *
 */

static void delete_left(void)
{

  if (input_pos == 0)
    return;

  cursor_left();
  delete_char();

}/* delete_left */

/*
 * delete_word_left
 *
 * Delete the word to the left of the cursor.
 *
 */

static void delete_word_left(void)
{
  while (input_pos > 0 && input_buffer[input_pos-1] == ' ')
    delete_left();

  while (input_pos > 0 && input_buffer[input_pos-1] != ' ')
    delete_left();
}/* delete_word_left */

/*
 * delete_word_right
 *
 * Delete the word to the right of the cursor.
 *
 */

static void delete_word_right(void)
{
  while (input_pos < input_len && input_buffer[input_pos] == ' ')
    delete_char();

  while (input_pos < input_len && input_buffer[input_pos] != ' ')
    delete_char();
}/* delete_word_right */

/*
 * delete_to_eol
 *
 * Delete everything to the right of the cursor.
 *
 */

static void delete_to_eol(void)
{
  while (input_pos < input_len)
    delete_char();
}/* delete_to_eol */

/*
 * erase_input
 *
 * Clear the input line.
 *
 */

static void erase_input(void)
{

  last_char();

  while (input_len != 0)
    delete_left();

}/* erase_input */

/*
 * insert_char
 *
 * Insert a character into the input buffer.
 *
 */

static void insert_char(int c)
{
  int saved_x;

  /* Delete the character below the cursor if overwrite is on */

  if (overwrite != 0)
    delete_char();

  /* See if the character fits */

  if (input_len == input_max || input_lft < os_char_width(c))
    return;

  input_lft -= os_char_width(c);

  /* Insert new character */

  input_len++;

  memmove(input_buffer + input_pos + 1,
          input_buffer + input_pos,
          input_len - input_pos);

  input_buffer[input_pos] = c;

  saved_x = cursor_x;

  os_display_string(input_buffer + input_pos);

  cursor_x = saved_x;

  /* Cursor moves to next position */

  cursor_right();

}/* insert_char */

/*
 * transpose_chars
 *
 * Transpose character under cursor with previous character (except at eol)
 *
 */

static void transpose_chars(void)
{
  int   at_end,was_over;
  char  c1,c2;

  if (input_pos < 1 || input_len < 2)
    return;

  at_end = (input_pos == input_len);
  was_over = overwrite;
  overwrite = 0;

  c1 = input_buffer[input_pos-1 - at_end];
  c2 = input_buffer[input_pos   - at_end];

  if (at_end)
    delete_left();
  else
    delete_char();
  delete_left();

  insert_char(c2);
  insert_char(c1);

  overwrite = was_over;
} /* end transpose_chars */

/*
 * store_input
 *
 * Copy the current input line to the history buffer.
 *
 */

static void store_input(void)
{
  if (input_buffer[0] == 0)
    return;

  last_char();

  /* Add line to history unless it's less than 4 characters or
     the same as the previous line */
  if (input_len > 3
      && (!history_valid
          || strcmp(history[history_count ? history_count-1 : HISTORY_LINES-1],
                    input_buffer))) {
    memmove(history[history_count], input_buffer, 80);

    if (++history_count == HISTORY_LINES)
      history_count = 0;
    if (++history_valid > HISTORY_LINES)
      history_valid = HISTORY_LINES;
  } /* end if adding line to history */
}/* store_input */

/*
 * get_history_entry
 *
 * Search for a matching history entry and display it on the input line.
 *
 * DIRECTION is 1 for previous entry or -1 for next entry.
 *
 */

static void get_history_entry(int direction)
{
  int i,index;

  if (!history_valid)
    return;

  if (history_prefix_len < 0) {
    history_prefix_len = input_len;
    history_entry = 0;
  }

  for (;;) {
    history_entry += direction;
    if (history_entry > history_valid)
      history_entry = 0;
    else if (history_entry < 0)
      history_entry = history_valid;

    if (history_entry == 0) {
      first_char();
      for (i = history_prefix_len; i; --i)
        cursor_right();
      delete_to_eol();
      return;
    }

    index = history_count - history_entry;
    if (index < 0)
      index += HISTORY_LINES;

    if (strncmp(history[index],input_buffer,history_prefix_len) == 0) {
      erase_input();
      for (i = 0; history[index][i] != 0; i++)
        insert_char((unsigned char) history[index][i]);
      return;
    }
  }
}/* get_history_entry */

/*
 * os_read
 *
 * Read a line of input from the keyboard into a buffer. The buffer
 * may already be primed with some text. In this case, the "initial"
 * text is already displayed on the screen. After the input action
 * is complete, the function returns with the terminating key value.
 * The length of the input should not exceed "max" characters plus
 * an extra 0 terminator.
 *
 * Terminating keys are the return key (13) and all function keys
 * (see the Specification of the Z-machine) which are accepted by
 * the is_terminator function. Mouse clicks behave like function
 * keys except that the mouse position is stored in global variables
 * "mouse_x" and "mouse_y" (top left coordinates are (1,1)).
 *
 * Furthermore, Frotz introduces some special terminating keys:
 *
 *     HOT_KEY_PLAYBACK (Alt-P)
 *     HOT_KEY_RECORDING (Alt-R)
 *     HOT_KEY_SEED (Alt-S)
 *     HOT_KEY_UNDO (Alt-U)
 *     HOT_KEY_RESTART (Alt-N, "new game")
 *     HOT_KEY_QUIT (Alt-X, "exit game")
 *
 * If the timeout argument is not zero, the input gets interrupted
 * after timeout/10 seconds (and the return value is 0).
 *
 * The complete input line including the cursor must fit in "width"
 * screen units.
 *
 * The screen is not scrolled after the return key was pressed, and
 * the cursor should be placed at the end of the input line when the
 * function returns.
 *
 */

int os_read(int max, char *buffer, int timeout, int width)
{
  int c;

  input_buffer = buffer;

  /* Initialise input variables */

  input_len = strlen(buffer);
  input_pos = input_len;
  input_max = max;
  input_lft = width - os_string_width(buffer) - os_char_width(' ');

  history_prefix_len = -1;

  if (max > 79)
    input_max = 79;

  if (cursor) os_cursor_on();   /* Make sure we have the right cursor */

  set_time_limit(timeout);      /* Calculate time limit */

  /* Loop until input is finished */

  do {
    redraw();
    c = get_key();

    if (c > 0 && c < 27)
      c = emacs_keys[c];

    /* Backspace, return and escape keys */

    if (c == 8)
      delete_left();
    if (c == 13)
      store_input();
    if (c == 27)
      erase_input();

    /* Plain characters (ASCII or European letters) */

    if ((c >= ' ' && c <= 126) || (c >= EURO_MIN && c <= EURO_MAX))
      insert_char(c);

    /* Editing keys */

    if (cwin == 0) {
      if (c == ARROW_UP)
        get_history_entry(1);
      else if (c == ARROW_DOWN)
        get_history_entry(-1);
      else
        history_prefix_len = -1;

      if (!timeout) {
        /* Moving the cursor left locks up Border Zone */
        if (c == ARROW_LEFT)
          cursor_left();
        else if (c == ARROW_RIGHT)
          cursor_right();
        else if (c == SPECIAL_KEY_HOME)
          first_char();
        else if (c == SPECIAL_KEY_END)
          last_char();
        else if (c == SPECIAL_KEY_WORD_LEFT)
          prev_word();
        else if (c == SPECIAL_KEY_WORD_RIGHT)
          next_word();
      } /* end if no time limit */

      if (c >= ARROW_UP && c <= ARROW_RIGHT)
        c = -1;

      if (c == 11) /* C-k */
        delete_to_eol();
      if (c == 20) /* C-t */
        transpose_chars();
      if (c == SPECIAL_KEY_DELETE)
        delete_char();
      if (c == 127 || c == SPECIAL_KEY_DEL_WORD_LEFT) /* C-backspace */
        delete_word_left();
      if (c == SPECIAL_KEY_DEL_WORD_RIGHT)
        delete_word_right();
      if (c == SPECIAL_KEY_INSERT) {
        overwrite ^= 1;
        if (cursor) os_cursor_on(); /* Update cursor shape */
      }
    } /* end if cwin == 0 */

    /* Page up/down are replacements for cursor up/down */

    if (c == SPECIAL_KEY_PAGE_UP)
      c = ARROW_UP;
    if (c == SPECIAL_KEY_PAGE_DOWN)
      c = ARROW_DOWN;

  } while (input_pos < input_len || is_terminator(c) == 0);

  /* Return terminating key */

  return c;
}/* os_read */

/*
 * os_read_char
 *
 * Read a single character from the keyboard (or a mouse click) and
 * return it. Input aborts after timeout/10 seconds.
 *
 */

int os_read_char(int timeout)
{
  int  key;

  redraw();
  set_time_limit(timeout);      /* Calculate time limit */

  do {
    key = get_key();
  } while (key != 0 && key != 8 && key != 13 && key != 27
           && ((key < ' ') || key==128 || (key >= 255)));

  return key;
}/* os_read_char */


/* ========================================================================= *\


	    S C R E E N   M A N I P U L A T I O N   R O U T I N E S


\* ========================================================================= */

/*
 * os_erase_area
 *
 * Fill a rectangular area of the screen with the current background
 * colour. Top left coordinates are (1,1). The cursor does not move.
 *
 */

void os_erase_area(int top, int left, int bottom, int right)
{
  redraw();
  v_scroll(left-1,top-1, right-1,bottom-1, 1,V_SCROLL_CLEAR);
  v_getline(current_line, 0, cursor_y, h_screen_cols);
}/* os_erase_area */

/*
 * os_scroll_area
 *
 * Scroll a rectangular area of the screen up (units > 0) or down
 * (units < 0) and fill the empty space with the current background
 * colour. Top left coordinates are (1,1). The cursor stays put.
 *
 */

void os_scroll_area(int top, int left, int bottom, int right, int units)
{
  int flag = V_SCROLL_UP;

  if (units < 0) {
    flag = V_SCROLL_DOWN;
    units = -units;
  }

  redraw();
  v_scroll(left-1,top-1, right-1,bottom-1, units,flag);
  v_getline(current_line, 0, cursor_y, h_screen_cols);
}/* os_scroll_area */


/* ========================================================================= *\


		 I N I T I A L I S A T I O N   R O U T I N E S


\* ========================================================================= */

extern char script_name[];
extern char command_name[];
extern char save_name[];
extern char auxilary_name[];
extern char init_name[];

/*
 * valid_color
 *
 * Make sure a user-specified color is valid (0-15).
 * Returns the color, or -1 if the color was not valid.
 *
 */

static int valid_color(int color)
{
  if ((color < 0) || (color > 15))
    return -1;
  else
    return color;
} /* end valid_color */

/*
 * os_process_arguments
 *
 * Handle command line switches. Some variables may be set to activate
 * special features of Frotz:
 *
 *     option_attribute_assignment
 *     option_attribute_testing
 *     option_context_lines
 *     option_object_locating
 *     option_object_movement
 *     option_left_margin
 *     option_right_margin
 *     option_piracy
 *     option_undo_slots
 *
 * The name of the story file is stored in "story_name".
 *
 */

void os_process_arguments(int argc, char *argv[])
{
  int num = 0;
  int c;
  int i;

  /* Get more arguments from environment variable and/or response files */

  _envargs(&argc, &argv, "FROTZOPT");
  _response(&argc, &argv);

  /* Parse the options */

  optmode = GETOPT_ANY;         /* Accept options after story file */

  do {
    c = getopt(argc, argv, "aAb:B:c:e:f:F:h:l:oOpr:tu:w:x");

    if (optarg != NULL)
      num = atoi(optarg);

    switch (c) {
     case 'a':
      option_attribute_assignment = 1;		break;
     case 'A':
      option_attribute_testing = 1;		break;
     case 'b':
      user_background = valid_color(num);	break;
     case 'B':
      user_reverse_bg = valid_color(num);	break;
     case 'c':
      option_context_lines = num;		break;
     case 'e':
      user_emphasis = valid_color(num);		break;
     case 'f':
      user_foreground = valid_color(num);	break;
     case 'F':
      user_reverse_fg = valid_color(num);	break;
     case 'h':
      user_screen_height = num;			break;
     case 'l':
      option_left_margin = num;			break;
     case 'o':
      option_object_movement = 1;		break;
     case 'O':
      option_object_locating = 1;		break;
     case 'p':
      option_piracy = 1;			break;
     case 'r':
      option_right_margin = num;		break;
     case 't':
      user_tandy_bit = 1;			break;
     case 'u':
      option_undo_slots = num;			break;
     case 'w':
      user_screen_width = num;			break;
     case 'x':
      add_alias("g","again");
      add_alias("x","examine");
      add_alias("z","wait");
      break;
    } /* end switch */
  } while (c != EOF);

  /* Display usage and exit with error level 1 */

  if (optind != argc - 1) {
    puts(INFORMATION);
    exit(1);
  }

  /* Save the story file name */

  story_name = argv[optind];

  /* Strip path off the story file name */

  stripped_story_name = story_name;

  for (i = 0; story_name[i] != 0; i++)
    if (story_name[i] == '\\' || story_name[i] == '/' || story_name[i] == ':')
      stripped_story_name = story_name + i + 1;

  /* Create nice default file names */

  for (i=0; stripped_story_name[i] != 0 && stripped_story_name[i] != '.'; i++)
    script_name[i] =
      command_name[i] =
      save_name[i] =
      init_name[i] =
      auxilary_name[i] = stripped_story_name[i];

  /* Don't forget the extensions */

  strcpy(script_name + i, ".scr");
  strcpy(command_name + i, ".rec");
  strcpy(save_name + i, ".sav");
  strcpy(init_name + i, ".frc");
  strcpy(auxilary_name + i, ".aux");
}/* os_process_arguments */

/*
 * os_init_screen
 *
 * Initialise the IO interface. Prepare the screen and other devices
 * (mouse, sound card). Set various OS depending story file header
 * entries:
 *
 *     h_config (aka flags 1)
 *     h_flags (aka flags 2)
 *     h_screen_cols (aka screen width in characters)
 *     h_screen_rows (aka screen height in lines)
 *     h_screen_width
 *     h_screen_height
 *     h_font_height (defaults to 1)
 *     h_font_width (defaults to 1)
 *     h_default_foreground
 *     h_default_background
 *     h_interpreter_number
 *     h_interpreter_version
 *     h_user_name (optional; not used by any game)
 *
 * Finally, set reserve_mem to the amount of memory (in bytes) that
 * should not be used for multiple undo and reserved for later use.
 *
 */

void os_init_screen(void)
{
  int   errors;
  char  filename[_MAX_PATH];

  v_init();
  cursor_x = cursor_y = 0;

  /* Calculate cursor sizes */

  if (v_hardware() == V_COLOR_12)
    cursor_bot = 14;
  else
    cursor_bot = 10;

  cursor_ovr = 0;
  cursor_ins = cursor_bot - 1;

  /* Set various bits in the configuration byte. These bits tell
     the game which features are supported by the interpreter. */

  if (h_version == V3 && user_tandy_bit != 0)
    h_config |= CONFIG_TANDY;

  if (h_version == V3)
    h_config |= CONFIG_SPLITSCREEN;

  if (h_version >= V4)
    h_config |= CONFIG_BOLDFACE;

  if (h_version >= V4)
    h_config |= CONFIG_EMPHASIS | CONFIG_FIXED | CONFIG_TIMEDINPUT;

  if (h_version >= V5)
    h_config |= CONFIG_COLOUR;

  /* Handle various game flags. These flags are set if the game wants
     to use certain features. The flags must be cleared if the feature
     is not available. */

  if (h_flags & GRAPHICS_FLAG)
    h_flags &= ~GRAPHICS_FLAG;

  if (h_version == V3 && (h_flags & OLD_SOUND_FLAG))
    h_flags &= ~OLD_SOUND_FLAG;

  if (h_version >= V5 && (h_flags & UNDO_FLAG))
    if (option_undo_slots == 0)
      h_flags &= ~UNDO_FLAG;

  if (h_flags & SOUND_FLAG)
    h_flags &= ~SOUND_FLAG;

  if (h_flags & MOUSE_FLAG)
    h_flags &= ~MOUSE_FLAG;

  h_flags &= ~MENU_FLAG;

  /* Provide default colours if these have not already been set by
     the user. */

  if (user_foreground == -1)
    user_foreground = F_WHITE;
  if (user_background == -1)
    user_background = F_BLUE;

  /* Convert IBM PC colours to Z-machine colours. */

  h_default_foreground = zcolour[user_foreground];
  h_default_background = zcolour[user_background];

  /* Set the screen dimensions and the font size in screen units. */

  v_dimen(&last_column, &last_row);
  h_screen_width = last_column--;
  h_screen_height = last_row--;
  h_font_height = 1;
  h_font_width = 1;

  /* The user may override the default screen size by command line
     options. This might be useful for palmtop users, or for people
     who want to run Frotz under Microsoft Windows. */

  if (user_screen_width > 0 && user_screen_width < h_screen_width)
    h_screen_width = user_screen_width;
  if (user_screen_height > 0 && user_screen_height < h_screen_height)
    h_screen_height = user_screen_height;

  /* Set the screen dimensions in character grid positions. */

  h_screen_rows = h_screen_height / h_font_height;
  h_screen_cols = h_screen_width / h_font_width;

  /* Set the interpreter number (a constant telling the game which
     operating system it runs on) and the interpreter version. The
     interpreter number has effect on all V6 games and Beyond Zork. */

  h_interpreter_number = INTERP_MSDOS;
  h_interpreter_version = 'E';

  adjust_style();
  v_attrib(current_attrib);
  v_clear();
  v_getline(current_line, 0, cursor_y, h_screen_cols);

  /* Load aliases from init files */

  restart_screen();             /* Prepare for error messages  */

  errors = 0;
  _makepath(filename, NULL, getenv("HOME"), "frotz.frc", NULL);
  if (access(filename, 4) == 0)
    errors = load_aliases(filename);

  if (access(init_name, 4) == 0)
    errors += load_aliases(init_name);

  if (errors)
    os_more_prompt();
}/* os_init_screen */

/*
 * os_reset_screen
 *
 * Reset the screen before the program ends.
 *
 */

void os_reset_screen(void)
{
  /* Print message */

  os_display_string("[Hit any key to exit.]");

  /* Wait for a key */

  os_read_char(0);

}/* os_reset_screen */

/*
 * os_fatal
 *
 * Display error message and stop interpreter.
 *
 */

void os_fatal(const char *s)
{
  /* Display error message */

  fputs("\nFatal error: ", stderr);
  fputs(s, stderr);
  fputs("\n", stderr);

  /* Exit with error level 1 */

  exit(1);
}/* os_fatal */



/* ========================================================================= *\


			V A R I O U S   R O U T I N E S


\* ========================================================================= */

/*
 * os_more_prompt
 *
 * Display a MORE prompt, wait for a keypress and remove the MORE
 * prompt from the screen.
 *
 */

void os_more_prompt(void)
{
  int saved_x = cursor_x;

  /* Prepare MORE prompt by selecting text font and plain style */

  int saved_style = current_style;
  current_style = 0;
  adjust_style();

  /* Print MORE message and wait for a key */

  os_display_string("[MORE]");
  os_read_char(0);

  /* Remove MORE prompt from the screen */

  cursor_x = saved_x;
  v_gotoxy(cursor_x,cursor_y);
  v_clreol();
  v_getline(current_line, 0, cursor_y, h_screen_cols);

  /* Restore text font and style */

  current_style = saved_style;

  adjust_style();
}/* os_more_prompt */

/*
 * os_get_file_name
 *
 * Return the name of a file. Flag can be one of:
 *
 *    FILE_SAVE     - Save file
 *    FILE_RESTORE  - Restore file
 *    FILE_SCRIPT   - Transscript file
 *    FILE_RECORD   - Command file for recording
 *    FILE_PLAYBACK - Command file for playback
 *    FILE_SAVE_AUX - Save auxilary ("preferred settings") file
 *    FILE_LOAD_AUX - Load auxilary ("preferred settings") file
 *    FILE_SAVE_INIT- Save Frotz alias file
 *    FILE_LOAD_INIT- Load Frotz alias file
 *
 * The length of the file name is limited by MAX_FILE_NAME. Ideally
 * an interpreter should open a file requester to ask for the file
 * name. If it is unable to do that then it should use diplay_string
 * and display_new_line to prompt for a file name.
 *
 */

int os_get_file_name(char *file_name, char *default_name, int flag)
{
    char *extension;
    int key;

    /* Select appropriate extension */

    extension = ".aux";

    if (flag == FILE_SAVE || flag == FILE_RESTORE)
	extension = ".sav";
    if (flag == FILE_SCRIPT)
	extension = ".scr";
    if (flag == FILE_RECORD || flag == FILE_PLAYBACK)
	extension = ".rec";
    if (flag == FILE_SAVE_INIT || flag == FILE_LOAD_INIT)
	extension = ".frc";

    /* Input file name but reserve four bytes for a file name extension */

    display_string("Enter file name (extension ");
    display_string(extension);
    display_string(" will be added).  (Enter . to cancel.)\nDefault is \"");
    display_string(default_name);
    display_string("\": ");

    file_name[0] = 0;

    while (os_read(MAX_FILE_NAME - 4, file_name, 0, 9999) != 13);
    display_new_line();

    if (strcmp(file_name,".") == 0)
      return 0;                 /* Enter '.' to cancel */

    /* Use the default name if nothing was typed */

    if (file_name[0] == 0)
	strcpy(file_name, default_name);

    /* Add the extension */

    if (strchr(file_name, '.') == 0)
	strcat(file_name, extension);

    /* Return true if the file should be opened for reading */

    if (flag == FILE_RESTORE  || flag == FILE_PLAYBACK ||
        flag == FILE_LOAD_AUX || flag == FILE_LOAD_INIT)
	return 1;

    /* Return true if the file does not exist */

    if (access(file_name, 0) != 0)
	return 1;

    /* Ask user for permission to overwrite */

    display_string("Overwrite existing file? (y/n) >");

    key = os_read_char(0);
    if (isalpha(key))
      os_display_char(key);
    display_new_line();

    /* Return true if user agrees */

    if (key == 'y' || key == 'Y')
	return 1;

    /* Return false otherwise */

    return 0;
}/* os_get_file_name */

/*
 * os_message_start
 *
 * Prepare for printing a "debugging" message. The text of the message
 * will be passed to os_display_char. Possibly, the interface may want
 * to direct the message to an additional "messages" window. Otherwise,
 * it should display the message on the game screen -- or discard it if
 * the upper window is selected (ie. if cwin != 0).
 *
 */

void os_message_start(void)
{
  if (cwin == 0) {
    /* Start message in a new line */
    if (get_line_width(cwin) != get_units_left())
      display_new_line();

    display_string("  [");

  } else dump_message = 1;
}/* os_message_start */

/*
 * os_message_end
 *
 * Stop printing a debugging message.
 *
 */

void os_message_end(void)
{
  if (cwin == 0) {

    /* Finish message */

    display_string("]\n");

  } else dump_message = 0;
}/* os_message_end */
