/* GLE - The GTK+ Layout Engine
 * Copyright (C) 1998, 1999 Tim Janik
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	 See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307, USA.
 */
#include	<gle/gleconfig.h>

#include	"gleted.h"
#include	"gleeditor.h"
#include	"gleprivate.h"
#include	"glegtoplevel.h"
#include	"gleetils.h"
#include	"gledumper.h"
#include	"glercargs.h"
#include	"gleselector.h"
#include	<string.h>
#include	<stdio.h>



/* --- macros --- */
#define GWIDGET_SET_TED(gw, ted)      gle_gobject_set_qdata (GLE_GOBJECT (gw), quark_ted, (ted))


/* --- signals --- */
enum
{
  SIGNAL_LAST
};
typedef struct
{
  GleTed	*ted;
  GleGWidget	*gwidget;
} GleTedPopupData;


/* --- prototypes --- */
static void		gle_ted_class_init		(GleTedClass	*class);
static void		gle_ted_init			(GleTed		*ted);
static void		gle_ted_construct		(GleTed		*ted,
							 GleGWidget	*gwidget);
static void		gle_ted_destroy			(GtkObject	*object);
static gboolean		gle_ted_ptree_button		(GleTed		*ted,
							 GleGWidget	*gwidget,
							 guint		 root_x,
							 guint		 root_y,
							 guint		 button,
							 guint32	 time);
static void		gle_ted_find_widget		(GleTed		*ted);
static void		gle_ted_popup_operation		(gpointer	 callback_data,
							 guint		 popup_op,
							 GtkWidget	*item);
static GleTedPopupData*	gle_ted_popup_start		(GleTed		*ted,
							 GtkItemFactory	*popup_factory,
							 GleGWidget	*gwidget);
static void		gle_ted_popup_done		(gpointer	 data);


/* --- menu operations --- */
enum
{
  POPUP_OP_NONE,
  POPUP_OP_P_EDITOR,
  POPUP_OP_P_TED,
  POPUP_OP_P_DUMP,
  POPUP_OP_P_SOLE_DUMP,
  POPUP_OP_P_PP,
  POPUP_OP_P_POS,
  POPUP_OP_TEST_REALIZE,
  POPUP_OP_TEST_UNREALIZE,
  POPUP_OP_FLAGS,
  POPUP_OP_QUEUE_REDRAW,
  POPUP_OP_QUEUE_CLEAR,
  POPUP_OP_QUEUE_RESIZE,
  POPUP_OP_REALLOCATE,
  POPUP_OP_COMPOSITE_NAME,
  POPUP_OP_TOGGLE_SENSITIVITY,
  POPUP_OP_W_HIDE,
  POPUP_OP_W_SHOW,
  POPUP_OP_W_DESTROY,
  POPUP_OP_P_DESTROY,
  POPUP_OP_LAST
};
static GtkItemFactoryEntry popup_entries[] =
{
#define POP(popup_op)	(gle_ted_popup_operation), (POPUP_OP_ ## popup_op)
  { "/Proxy Operations", "",		POP (NONE),			"<Title>" },
  { "/---",		  "",		NULL, 0,			"<Separator>" },
  { "/Widget Editor...", "<ctrl>E",	POP (P_EDITOR),			NULL },
  { "/Tree Editor...",	"<ctrl>T",	POP (P_TED),			NULL },
  { "/Test Tree Dump",	NULL,		POP (P_DUMP),			NULL },
  { "/Test Sole Dump",	NULL,		POP (P_SOLE_DUMP),		NULL },
  { "/PP",		NULL,		POP (P_PP),			NULL },
  { "/Window Position",	NULL,		POP (P_POS),			NULL },
  { "/Destroy",		NULL,		POP (P_DESTROY),		NULL },
  { "/---",		  "",		NULL, 0,			"<Separator>" },
  { "/Widget Operations", "",		POP (NONE),			"<Title>" },
  { "/---",		NULL,		NULL, 0,			"<Separator>" },
  { "/Show Flags...",	"<ctrl>F",	POP (FLAGS),			NULL },
  { "/Dump Composite Name", NULL,	POP (COMPOSITE_NAME),		NULL },
  { "/Toggle Sensitivity", NULL,	POP (TOGGLE_SENSITIVITY),	NULL },
  { "/Queue Redraw",	NULL,		POP (QUEUE_REDRAW),		NULL },
  { "/Queue Clear",	NULL,		POP (QUEUE_CLEAR),		NULL },
  { "/Queue Resize",	NULL,		POP (QUEUE_RESIZE),		NULL },
  { "/Reallocate",	NULL,		POP (REALLOCATE),		NULL },
  { "/Test Unrealize",	NULL,		POP (TEST_UNREALIZE),		NULL },
  { "/Test Realize",	NULL,		POP (TEST_REALIZE),		NULL },
  { "/Hide Widget",	"<ctrl>H",	POP (W_HIDE),		NULL },
  { "/Show Widget",	"<ctrl>S",	POP (W_SHOW),		NULL },
  { "/Destroy Widget",	NULL,		POP (W_DESTROY),		NULL },
#undef	POP
};
static guint n_popup_entries = (sizeof (popup_entries) / sizeof (popup_entries[0]));


/* --- variables --- */
static GtkWindowClass	*parent_class = NULL;
static GleTedClass	*gle_ted_class = NULL;
static GQuark		 quark_ted = 0;


/* --- menus --- */
static gchar	*gle_ted_factories_path = "<GLE-GleTed>";
static GtkItemFactoryEntry menubar_entries[] =
{
#define TED_OP(ted_op)	(gle_ted_operation), (GLE_TED_OP_ ## ted_op)
  { "/File/Test",	"",		TED_OP (NONE),	"<Item>" },
  { "/File/-----",	"",		NULL, 0,		"<Separator>" },
  { "/File/-----",	"",		NULL, 0,		"<Separator>" },
  { "/File/Close",	"<ctrl>W",	TED_OP (DELETE),	"<Item>" },
#undef	TED_OP
};
static guint n_menubar_entries = sizeof (menubar_entries) / sizeof (menubar_entries[0]);


/* --- functions --- */
GtkType
gle_ted_get_type (void)
{
  static GtkType ted_type = 0;
  
  if (!ted_type)
    {
      GtkTypeInfo ted_info =
      {
	"GleTed",
	sizeof (GleTed),
	sizeof (GleTedClass),
	(GtkClassInitFunc) gle_ted_class_init,
	(GtkObjectInitFunc) gle_ted_init,
	/* reserved_1 */ NULL,
	/* reserved_2 */ NULL,
	(GtkClassInitFunc) NULL,
      };
      
      ted_type = gtk_type_unique (GTK_TYPE_WINDOW, &ted_info);
    }
  
  return ted_type;
}

static gint
gle_ted_delete_event (GtkWidget		*widget,
		      GdkEventAny	*event)
{
  gle_ted_operation (GLE_TED (widget), GLE_TED_OP_DELETE);
  
  return TRUE;
}

static void
gle_ted_class_init (GleTedClass *class)
{
  GtkObjectClass *object_class;
  GtkWidgetClass *widget_class;
  GtkContainerClass *container_class;
  GtkWindowClass *window_class;
  
  gle_ted_class = class;
  object_class = (GtkObjectClass*) class;
  widget_class = (GtkWidgetClass*) class;
  container_class = (GtkContainerClass*) class;
  window_class = (GtkWindowClass*) class;
  parent_class = gtk_type_class (GTK_TYPE_WINDOW);
  
  quark_ted = g_quark_from_static_string (GLE_PRIVATE_KEY (ted));
  
  object_class->destroy = gle_ted_destroy;
  
  widget_class->delete_event = gle_ted_delete_event;
  
  class->factories_path = gle_ted_factories_path;
  class->popup_factory = NULL;
}

static void
gle_ted_init (GleTed *ted)
{
  GleTedClass *class;
  GtkWidget *ted_vbox;
  GtkWidget *main_vbox;
  GtkWidget *status_hbox;
  GtkWidget *hbox;
  GtkWidget *any;
  GtkWidget *button;
  
  GLE_TAG (ted, "GLE-Ted");
  class = GLE_TED_CLASS (GTK_OBJECT (ted)->klass);
  
  ted->menubar_factory = NULL;
  ted->window_label = NULL;
  ted->ptree = NULL;
  ted->rebuild_button = NULL;
  ted->find_button = NULL;
  
  /* create/reference the popup factory
   */
  if (!gle_ted_class->popup_factory)
    {
      gle_ted_class->popup_factory =
	gtk_item_factory_new (GTK_TYPE_MENU,
			      gle_ted_class->factories_path,
			      NULL);
      GLE_TAG_WEAK (gle_ted_class->popup_factory);
      gtk_signal_connect (GTK_OBJECT (gle_ted_class->popup_factory),
			  "destroy",
			  GTK_SIGNAL_FUNC (gtk_widget_destroyed),
			  &gle_ted_class->popup_factory);
      gtk_item_factory_create_items (gle_ted_class->popup_factory,
				     n_popup_entries,
				     popup_entries,
				     gle_ted_class);
    }
  else
    gtk_object_ref (GTK_OBJECT (gle_ted_class->popup_factory));
  gtk_window_add_accel_group (GTK_WINDOW (ted), gle_ted_class->popup_factory->accel_group);
  
  /* create GUI
   */
  gtk_widget_set (GTK_WIDGET (ted),
		  "title", "GLE-Ted",
		  "GtkWindow::type", GTK_WINDOW_TOPLEVEL,
		  "GtkWindow::allow_shrink", TRUE,
		  "GtkWindow::allow_grow", TRUE,
		  "GtkWindow::auto_shrink", FALSE,
		  NULL);
  ted_vbox =
    gtk_widget_new (GTK_TYPE_VBOX,
		    "homogeneous", FALSE,
		    "spacing", 0,
		    "border_width", 0,
		    "parent", ted,
		    "visible", TRUE,
		    NULL);
  main_vbox =
    gtk_widget_new (GTK_TYPE_VBOX,
		    "homogeneous", FALSE,
		    "spacing", 5,
		    "border_width", 5,
		    "visible", TRUE,
		    NULL);
  gtk_container_add_with_args (GTK_CONTAINER (ted_vbox), main_vbox,
			       "expand", TRUE,
			       "fill", TRUE,
			       NULL);
  status_hbox =
    gtk_widget_new (GTK_TYPE_HBOX,
		    "homogeneous", FALSE,
		    "spacing", 5,
		    "border_width", 5,
		    "visible", TRUE,
		    NULL);
  gtk_container_add_with_args (GTK_CONTAINER (main_vbox), status_hbox,
			       "expand", FALSE,
			       "fill", TRUE,
			       NULL);
  ted->window_label =
    gtk_widget_new (GTK_TYPE_LABEL,
		    "label", "<None>",
		    "visible", TRUE,
		    NULL);
  gtk_container_add_with_args (GTK_CONTAINER (main_vbox), ted->window_label,
			       "expand", FALSE,
			       "fill", TRUE,
			       NULL);
  
  /* crete the widget tree
   */
  ted->ptree = (GlePTree*)
    gtk_widget_new (GLE_TYPE_PTREE,
		    "visible", TRUE,
		    "object_signal::button_click", gle_ted_ptree_button, ted,
		    NULL);
  any = gtk_widget_new (GTK_TYPE_SCROLLED_WINDOW,
			"visible", TRUE,
			"hscrollbar_policy", GTK_POLICY_AUTOMATIC,
			"vscrollbar_policy", GTK_POLICY_AUTOMATIC,
			"parent", main_vbox,
			NULL);
  gtk_container_add (GTK_CONTAINER (any), GTK_WIDGET (ted->ptree));
  
  /* create tree buttons
   */
  hbox =
    gtk_widget_new (GTK_TYPE_HBOX,
		    "homogeneous", TRUE,
		    "spacing", 5,
		    "border_width", 0,
		    "visible", TRUE,
		    NULL);
  gtk_container_add_with_args (GTK_CONTAINER (main_vbox), hbox,
			       "expand", FALSE,
			       "fill", TRUE,
			       NULL);
  ted->rebuild_button =
    gtk_widget_new (GTK_TYPE_BUTTON,
		    "label", "Rebuild Tree",
		    "object_signal::clicked", gle_ptree_rebuild, ted->ptree,
		    "visible", TRUE,
		    NULL);
  gtk_container_add_with_args (GTK_CONTAINER (hbox), ted->rebuild_button,
			       "expand", FALSE,
			       "fill", TRUE,
			       NULL);
  ted->find_button =
    gtk_widget_new (GTK_TYPE_BUTTON,
		    "label", "Find Widget",
		    "object_signal::clicked", gle_ted_find_widget, ted,
		    "visible", TRUE,
		    NULL);
  gtk_container_add_with_args (GTK_CONTAINER (hbox), ted->find_button,
			       "expand", FALSE,
			       "fill", TRUE,
			       NULL);
  any =
    gtk_widget_new (gtk_hseparator_get_type (),
		    "GtkWidget::visible", TRUE,
		    NULL);
  gtk_box_pack_start (GTK_BOX (ted_vbox), any, FALSE, TRUE, 0);
  button =
    gtk_widget_new (GTK_TYPE_BUTTON,
		    "visible", TRUE,
		    "label", "Close",
		    "border_width", 5,
		    "parent", ted_vbox,
		    "can_default", TRUE,
		    "has_default", TRUE,
		    "object_signal::clicked", gtk_widget_destroy, ted,
		    NULL);
  gtk_box_set_child_packing (GTK_BOX (ted_vbox), button, FALSE, TRUE, 0, GTK_PACK_START);
  
  /* create the menu bar
   */
  ted->menubar_factory = gtk_item_factory_new (GTK_TYPE_MENU_BAR,
					       gle_ted_factories_path,
					       NULL);
  GLE_TAG_WEAK (ted->menubar_factory);
  gtk_window_add_accel_group (GTK_WINDOW (ted),
			      ted->menubar_factory->accel_group);
  gtk_item_factory_create_items (ted->menubar_factory,
				 n_menubar_entries,
				 menubar_entries,
				 ted);
  gtk_container_add_with_args (GTK_CONTAINER (ted_vbox), ted->menubar_factory->widget,
			       "expand", FALSE,
			       "fill", TRUE,
			       "position", 0,
			       NULL);
  gtk_widget_show (ted->menubar_factory->widget);
}

GtkWidget*
gle_ted_new (GleGWidget *gwidget)
{
  GleTed *ted;
  
  g_return_val_if_fail (gwidget != NULL, NULL);
  g_return_val_if_fail (GLE_IS_GWIDGET (gwidget), NULL);
  
  ted = gtk_type_new (GLE_TYPE_TED);
  
  gle_ted_construct (ted, gwidget);
  
  return GTK_WIDGET (ted);
}

static void
gle_ted_construct (GleTed	  *ted,
		   GleGWidget	  *gwidget)
{
  g_return_if_fail (ted != NULL);
  g_return_if_fail (GLE_IS_TED (ted));
  g_return_if_fail (gwidget != NULL);
  g_return_if_fail (GLE_IS_GWIDGET (gwidget));
  g_return_if_fail (ted->gwidget == NULL);
  g_return_if_fail (gle_ted_from_gwidget (gwidget) == NULL);
  
  ted->gwidget = gwidget;
  GWIDGET_SET_TED (gwidget, GTK_WIDGET (ted));
  
  GLE_NOTIFIER_INSTALL_DATA (gwidget, "pre_destroy", gtk_widget_destroy, ted);
  
  gle_ptree_set_gwidget (ted->ptree, gwidget);
}

static void
gle_ted_destroy (GtkObject *object)
{
  GleTed *ted;
  
  g_return_if_fail (object != NULL);
  g_return_if_fail (GLE_IS_TED (object));
  
  ted = GLE_TED (object);
  
  GLE_NOTIFIER_REMOVE_FD (ted->gwidget, gtk_widget_destroy, ted);
  
  GWIDGET_SET_TED (ted->gwidget, NULL);
  ted->gwidget = NULL;
  
  /* unreference the popup factory */
  gtk_object_unref (GTK_OBJECT (gle_ted_class->popup_factory));
  
  /* get rid of the menu bar factory */
  gtk_object_unref (GTK_OBJECT (ted->menubar_factory));
  ted->menubar_factory = NULL;
  
  GTK_OBJECT_CLASS (parent_class)->destroy (object);
}

static gboolean
gle_ted_ptree_button (GleTed	 *ted,
		      GleGWidget *gwidget,
		      guint	  root_x,
		      guint	  root_y,
		      guint	  button,
		      guint32	  time)
{
  gboolean handled = FALSE;
  
  g_return_val_if_fail (ted != NULL, FALSE);
  g_return_val_if_fail (GLE_IS_TED (ted), FALSE);
  
  if (gwidget)
    {
      if (button == 3)
	{
	  GleTedPopupData *popup_data;
	  
	  popup_data = gle_ted_popup_start (ted, gle_ted_class->popup_factory, gwidget);
	  if (popup_data)
	    {
	      gtk_item_factory_popup_with_data (gle_ted_class->popup_factory,
						popup_data,
						gle_ted_popup_done,
						root_x,
						root_y,
						button,
						time);
	      handled = TRUE;
	    }
	}
    }
  
  return handled;
}

static void
gle_ted_candidate_check (GleTed	    *ted,
			 GtkWidget **new_candidate,
			 gint	    *candidate_ok)
{
  
  g_return_if_fail (ted != NULL);
  g_return_if_fail (GLE_IS_TED (ted));

  if (*candidate_ok)
    {
      GleGWidget *gwidget;

      gwidget = gle_widget_get_gwidget (*new_candidate);
      if (!gle_ptree_mark_gwidget (ted->ptree, gwidget))
	*candidate_ok = FALSE;
    }
  if (!*candidate_ok && *new_candidate)
    *new_candidate = gtk_widget_get_toplevel (*new_candidate);
}

void
gle_ted_mark_gwidget (GleTed	     *ted,
		      GleGWidget     *gwidget)
{
  g_return_if_fail (ted != NULL);
  g_return_if_fail (GLE_IS_TED (ted));
  
  gle_ptree_mark_gwidget (ted->ptree, gwidget);
}

GleTed*
gle_ted_from_gwidget (GleGWidget *gwidget)
{
  g_return_val_if_fail (gwidget != NULL, NULL);
  g_return_val_if_fail (GLE_IS_GWIDGET (gwidget), NULL);
  
  return gle_gobject_get_qdata (GLE_GOBJECT (gwidget), quark_ted);
}

static void
gle_ted_find_widget (GleTed *ted)
{
  GtkWidget *selector;
  
  g_return_if_fail (ted != NULL);
  g_return_if_fail (GLE_IS_TED (ted));
  
  gle_ptree_mark_gwidget (ted->ptree, NULL);
  if (gle_selector_in_selection (NULL) ||
      gdk_pointer_is_grabbed ())
    return;
  
  selector = gle_selector_new ("GLE Find Widget", "Select a widget");
  GLE_TAG (selector, "GLE-Ted-Selector");
  GLE_TAG_WEAK (GLE_SELECTOR (selector)->warning_window);
  gle_selector_set_cursor (GLE_SELECTOR (selector), gdk_cursor_new (GDK_HAND2));
  gtk_signal_connect_object (GTK_OBJECT (selector),
			     "candidate_check",
			     gle_ted_candidate_check,
			     (GtkObject*) ted);
  
  gtk_widget_show (GTK_WIDGET (ted));
  gdk_window_raise (GTK_WIDGET (ted)->window);
  
  gtk_object_ref (GTK_OBJECT (ted));
  gtk_object_ref (GTK_OBJECT (selector));
  gle_selector_make_selection (GLE_SELECTOR (selector));
  gtk_widget_destroy (GTK_WIDGET (selector));
  gtk_object_unref (GTK_OBJECT (selector));
  gtk_object_unref (GTK_OBJECT (ted));
}

void
gle_ted_operation (GleTed	*ted,
		   GleTedOps	 ted_op)
{
  g_return_if_fail (ted != NULL);
  g_return_if_fail (GLE_IS_TED (ted));
  g_return_if_fail (ted_op < GLE_TED_OP_LAST);
  
  gtk_widget_ref (GTK_WIDGET (ted));
  
  switch (ted_op)
    {
    case    GLE_TED_OP_NONE:
      break;
    case  GLE_TED_OP_DELETE:
      gle_set_flash_widget (NULL);
      gtk_widget_destroy (GTK_WIDGET (ted));
      break;
    case  GLE_TED_OP_TREE_UPDATE:
      gle_ptree_update (ted->ptree);
      break;
    case  GLE_TED_OP_TREE_REBUILD:
      gle_ptree_rebuild (ted->ptree);
      break;
    default:
      fprintf (stderr, "GleTedOps: invlid op-code %u\n", ted_op);
      break;
    }
  
  gtk_widget_unref (GTK_WIDGET (ted));
}

static GleTed*
gle_ted_get_current (void)
{
  GdkEvent *event;
  GtkWidget *widget;
  
  event = gtk_get_current_event ();
  if (!event)
    return NULL;
  widget = gtk_get_event_widget (event);
  gdk_event_free (event);
  if (!widget)
    return NULL;
  widget = gtk_widget_get_toplevel (widget);
  if (widget && GLE_IS_TED (widget))
    return GLE_TED (widget);
  
  return NULL;
}

static void
gle_ted_popup_operation (gpointer	 callback_data,
			 guint		 popup_op,
			 GtkWidget	*item)
{
  GleTed *ted;
  GleTedPopupData *popup_data;
  GSList *gwidget_list, *slist;
  
  g_return_if_fail (item != NULL);
  g_return_if_fail (GTK_IS_MENU_ITEM (item));
  g_return_if_fail (popup_op < POPUP_OP_LAST);
  
  popup_data = gtk_item_factory_popup_data_from_widget (item);
  if (popup_data)
    {
      /* we are invoked from a popup */
      ted = popup_data->ted;
      gle_gwidget_ref (popup_data->gwidget);
      gwidget_list = g_slist_prepend (NULL, popup_data->gwidget);
    }
  else
    {
      /* we are invoked via accelerator, in this case
       * we try to get the Ted from the current event and
       * then group-operate on all selected gwidgets
       */
      
      ted = gle_ted_get_current ();
      if (!ted)
	return;
      
      gwidget_list = gle_ptree_selected_gwidgets (ted->ptree);
      for (slist = gwidget_list; slist; slist = slist->next)
	gle_gwidget_ref (slist->data);
    }
  
  gtk_widget_ref (GTK_WIDGET (ted));
  
  for (slist = gwidget_list; slist; slist = slist->next)
    {
      GleGWidget *gwidget;
      GtkWidget *widget;
      
      gwidget = slist->data;
      if (GLE_GOBJECT_DESTROYED (gwidget))
	{
	  gle_gwidget_unref (gwidget);
	  continue;
	}
      
      widget = GLE_GOBJECT_IS_INSTANTIATED (gwidget) ? GLE_GWIDGET_WIDGET (gwidget) : NULL;
      
      switch (popup_op)
	{
	  GtkAllocation allocation;
	  GtkWidget *dialog;
	  
	case POPUP_OP_NONE:
	  break;
	case POPUP_OP_P_TED:
	  if (GLE_IS_GCONTAINER (gwidget))
	    {
	      dialog = (GtkWidget*) gle_ted_from_gwidget (gwidget);
	      if (!dialog)
		dialog = gle_ted_new (gwidget);
	      gtk_widget_show (dialog);
	      gdk_window_raise (dialog->window);
	    }
	  break;
	case POPUP_OP_P_EDITOR:
	  dialog = (GtkWidget*) gle_editor_from_gobject (GLE_GOBJECT (gwidget));
	  if (!dialog)
	    dialog = gle_editor_new (GLE_GOBJECT (gwidget));
	  gtk_widget_show (dialog);
	  gdk_window_raise (dialog->window);
	  break;
	case POPUP_OP_P_DESTROY:
	  gle_gwidget_destroy (gwidget);
	  break;
	case POPUP_OP_P_PP:
	  g_print ("Adresses: proxy(%p) widget (%p)\n", gwidget, widget);
	  break;
	case POPUP_OP_P_POS:
	  {
	    gint wx, wy;

	    gdk_window_get_position (widget->window, &wx, &wy);
	    g_print ("widget->window%s get_position: x=%d y=%d\n",
		     GTK_WIDGET_NO_WINDOW (widget) ? " (no-window widget)" : "",
		     wx, wy);
	  }
	  break;
	case POPUP_OP_TEST_REALIZE:
	  if (widget)
	    gtk_widget_realize (widget);
	  break;
	case POPUP_OP_TEST_UNREALIZE:
	  if (widget)
	    gtk_widget_unrealize (widget);
	  break;
	case POPUP_OP_FLAGS:
	  if (widget)
	    {
	      dialog = gle_object_popup_flag_list (GTK_OBJECT (widget));
	      GLE_TAG (dialog, "junk");
	    }
	  break;
	case POPUP_OP_P_SOLE_DUMP:
	  gle_dump_gwidget (stdout, "", FALSE,
			    gwidget,
			    GLE_DUMP_ALL_ARGS);
	  fputc ('\n', stdout);
	  break;
	case POPUP_OP_P_DUMP:
	  gle_dump_gwidget (stdout, "", FALSE,
			    gwidget,
			    GLE_DUMP_ALL_ARGS | GLE_DUMP_RECURSIVE);
	  fputc ('\n', stdout);
	  break;
	case POPUP_OP_QUEUE_REDRAW:
	  if (widget)
	    gtk_widget_queue_draw (widget);
	  break;
	case POPUP_OP_QUEUE_CLEAR:
	  if (widget)
	    gtk_widget_queue_clear (widget);
	  break;
	case POPUP_OP_QUEUE_RESIZE:
	  if (widget)
	    gtk_widget_queue_resize (widget);
	  break;
	case POPUP_OP_REALLOCATE:
	  if (widget)
	    {
	      allocation.x = widget->allocation.x;
	      allocation.y = widget->allocation.y;
	      allocation.width = widget->allocation.width;
	      allocation.height = widget->allocation.height;
	      gtk_widget_size_allocate (widget, &allocation);
	    }
	  break;
	case POPUP_OP_COMPOSITE_NAME:
	  if (widget)
	    {
	      if (!GTK_WIDGET_COMPOSITE_CHILD (widget))
		g_message ("Not a composite widget");
	      else
		{
		  gchar *name;

		  name = gtk_widget_get_composite_name (widget);
		  if (name)
		    g_message ("Composite Name: \"%s\"", name);
		  else
		    g_message ("Composite Name: NULL");
		  g_free (name);
		}
	    }
	  break;
	case POPUP_OP_TOGGLE_SENSITIVITY:
	  if (widget)
	    gtk_widget_set_sensitive (widget, ! GTK_WIDGET_SENSITIVE (widget));
	  break;
	case POPUP_OP_W_HIDE:
	  if (widget)
	    gtk_widget_hide (widget);
	  break;
	case POPUP_OP_W_SHOW:
	  if (widget)
	    gtk_widget_show (widget);
	  break;
	case POPUP_OP_W_DESTROY:
	  if (widget)
	    gtk_widget_destroy (widget);
	  break;
	default:
	  fprintf (stderr, "GleTed: invalid popup operation: %u\n", popup_op);
	  break;
	}
      
      gle_gwidget_unref (gwidget);
    }
  g_slist_free (gwidget_list);
  
  gtk_widget_unref (GTK_WIDGET (ted));
}

static GleTedPopupData*
gle_ted_popup_start (GleTed	    *ted,
		     GtkItemFactory *popup_factory,
		     GleGWidget	    *gwidget)
{
  GleTedPopupData *popup_data;
  GtkWidget *widget;
  
  if (!gwidget || GLE_GOBJECT_DESTROYED (gwidget))
    return NULL;
  
  popup_data = g_new (GleTedPopupData, 1);
  popup_data->ted = ted;
  popup_data->gwidget = gwidget;
  gle_gwidget_ref (popup_data->gwidget);
  
  widget = GLE_GOBJECT_IS_INSTANTIATED (gwidget) ? GLE_GWIDGET_WIDGET (gwidget) : NULL;
  
  gle_item_factory_adjust (popup_factory,
			   POPUP_OP_P_EDITOR,
			   TRUE,
			   NULL);
  gle_item_factory_adjust (popup_factory,
			   POPUP_OP_P_TED,
			   GLE_IS_GCONTAINER (gwidget),
			   NULL);
  gle_item_factory_adjust (popup_factory,
			   POPUP_OP_P_DESTROY,
			   TRUE,
			   NULL);
  gle_item_factory_adjust (popup_factory,
			   POPUP_OP_P_PP,
			   TRUE,
			   NULL);
  gle_item_factory_adjust (popup_factory,
			   POPUP_OP_P_POS,
			   widget && widget->window != NULL,
			   NULL);
  gle_item_factory_adjust (popup_factory,
			   POPUP_OP_TEST_REALIZE,
			   widget && !GTK_WIDGET_REALIZED (widget),
			   NULL);
  gle_item_factory_adjust (popup_factory,
			   POPUP_OP_TEST_UNREALIZE,
			   widget && GTK_WIDGET_REALIZED (widget),
			   NULL);
  gle_item_factory_adjust (popup_factory,
			   POPUP_OP_FLAGS,
			   widget != NULL,
			   NULL);
  gle_item_factory_adjust (popup_factory,
			   POPUP_OP_P_SOLE_DUMP,
			   TRUE,
			   NULL);
  gle_item_factory_adjust (popup_factory,
			   POPUP_OP_P_DUMP,
			   TRUE,
			   NULL);
  gle_item_factory_adjust (popup_factory,
			   POPUP_OP_QUEUE_REDRAW,
			   widget && GTK_WIDGET_REALIZED (widget),
			   NULL);
  gle_item_factory_adjust (popup_factory,
			   POPUP_OP_QUEUE_CLEAR,
			   widget && GTK_WIDGET_REALIZED (widget),
			   NULL);
  gle_item_factory_adjust (popup_factory,
			   POPUP_OP_QUEUE_RESIZE,
			   widget && GTK_WIDGET_REALIZED (widget),
			   NULL);
  gle_item_factory_adjust (popup_factory,
			   POPUP_OP_REALLOCATE,
			   widget && GTK_WIDGET_REALIZED (widget),
			   NULL);
  gle_item_factory_adjust (popup_factory,
			   POPUP_OP_COMPOSITE_NAME,
			   widget != NULL,
			   NULL);
  gle_item_factory_adjust (popup_factory,
			   POPUP_OP_TOGGLE_SENSITIVITY,
			   widget && GTK_WIDGET_REALIZED (widget),
			   NULL);
  gle_item_factory_adjust (popup_factory,
			   POPUP_OP_W_HIDE,
			   widget && GTK_WIDGET_VISIBLE (widget),
			   NULL);
  gle_item_factory_adjust (popup_factory,
			   POPUP_OP_W_SHOW,
			   widget && !GTK_WIDGET_VISIBLE (widget),
			   NULL);
  gle_item_factory_adjust (popup_factory,
			   POPUP_OP_W_DESTROY,
			   widget != NULL,
			   NULL);
  
  return popup_data;
}

static void
gle_ted_popup_done (gpointer data)
{
  GleTedPopupData *popup_data = data;
  
  g_return_if_fail (popup_data != NULL);
  
  gle_gwidget_unref (popup_data->gwidget);
  g_free (popup_data);
}
