/*  Test of the gtk editor widget.
 *  Copyright (C) 1998 Thomas Mailund <mailund@daimi.aau.dk>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*/

#include <stdlib.h>
#include <stdio.h>
#include <gdk/gdkkeysyms.h>
#include <gtk/gtk.h>
#include "gtkeditor.h"

#define BUF_MAX_SIZE 128	/* seems fair for one line */

/* global vars */
GtkEditor *mainbuffer;
GtkEntry *minibuffer;
void (*minibuffer_callback)(char *) = NULL;


/* minibuffer stuff */
static gint
minibuf_key_press (GtkWidget *widget, GdkEventKey *event)
{
  g_return_val_if_fail (widget != NULL, FALSE);
  g_return_val_if_fail (event != NULL, FALSE);

  if (event->keyval == GDK_Return) {
    gtk_widget_grab_focus (GTK_WIDGET (mainbuffer));
    if (minibuffer_callback) {
      /* The callback is called with a copy of the text. It needs to
       * take care of freeing the text, should that be done. */
      minibuffer_callback (g_strdup(gtk_entry_get_text (minibuffer)));
    }
    gtk_entry_set_text (minibuffer, "");
  }

  return FALSE;
}

static void
get_text_from_minibuffer (GtkSignalFunc callback)
{
  gtk_entry_set_text (minibuffer, "");
  minibuffer_callback = callback;
  gtk_widget_grab_focus (GTK_WIDGET (minibuffer));
}

/* callbacks */

static void
search_from_point_cb (char *string)
{
#ifdef DEBUG
  g_print ("search_from_point_cb (%s)\n", string);
#endif
  gtk_editor_search_from_point (mainbuffer, string, FALSE);
  g_free (string);
}

static void
search_back_from_point_cb (char *string)
{
#ifdef DEBUG
  g_print ("search_back_from_point_cb (%s)\n", string);
#endif
  gtk_editor_search_back_from_point (mainbuffer, string, FALSE);
  g_free (string);
}

static void
load_file_cb (char *filename)
{
  int i;
  char buf[BUF_MAX_SIZE];
  FILE *fp = fopen (filename, "r");

#ifdef DEBUG
  g_print ("load_file_cb (%s)\n", filename);
#endif

  if (!fp) return;

  /* clear old text */
  i = gtk_text_get_length (GTK_TEXT(mainbuffer));
  if (i>0) {
    gtk_text_set_point (GTK_TEXT(mainbuffer), i);
    gtk_text_backward_delete (GTK_TEXT(mainbuffer), i);
  }

  /* load new */
  gtk_text_freeze (GTK_TEXT (mainbuffer));
  
  do {
    if (fgets (buf, BUF_MAX_SIZE, fp) == NULL) {
      break;
    } else {
      gtk_text_insert (GTK_TEXT (mainbuffer), NULL, NULL, NULL,
		       buf, strlen (buf));
    }
  } while (!feof (fp));

  gtk_text_thaw (GTK_TEXT (mainbuffer));

  /* clean up and hilite */
  fclose (fp);
  g_free (filename);
  /* FIXME: couldn't this be done while inserting? */
  gtk_editor_hilite_buffer (mainbuffer);
}


static void
save_file_cb (char *filename)
{
  int i, len;
  char *buf;
  FILE *fp = fopen (filename, "w");

#ifdef DEBUG
  g_print ("save_file_cb (%s)\n", filename);
#endif

  if (!fp) return;

  gtk_text_freeze (GTK_TEXT (mainbuffer));

  /* get_chars allocs a new buffer each time, so I have to do all this
   * g_free stuff */  
  len = gtk_text_get_length (GTK_TEXT (mainbuffer));
  for (i = 0; i < len - BUF_MAX_SIZE;) {
    buf = gtk_editable_get_chars (GTK_EDITABLE (mainbuffer), i, i + BUF_MAX_SIZE);
    fputs (buf, fp);
    i += BUF_MAX_SIZE;
    g_free (buf);
  }
  /* and the leftovers */
  buf = gtk_editable_get_chars (GTK_EDITABLE (mainbuffer), i, len);
  fputs (buf, fp);
  g_free (buf);

  gtk_text_thaw (GTK_TEXT (mainbuffer));

  fclose (fp);
}


/* FIXME: this should be done better! */
/* FIXME: these functions should be bound to keys when the keybinding
 * parser is done. */
static void
search_from_point (void)
{
  get_text_from_minibuffer (search_from_point_cb);
}

static void
search_back_from_point (void)
{
  get_text_from_minibuffer (search_back_from_point_cb);
}

static void
clear_buffer (void)
{
  gtk_editable_delete_text (GTK_EDITABLE (mainbuffer), 0, -1);
}

static void
load_file (void)
{
  get_text_from_minibuffer (load_file_cb);
}

static void
save_file (void)
{
  get_text_from_minibuffer (save_file_cb);
}

/* key events */
static gint
mainbuf_key_press (GtkWidget *widget, GdkEventKey *event)
{
  g_return_val_if_fail (widget != NULL, FALSE);
  g_return_val_if_fail (event != NULL, FALSE);

  switch (event->keyval) {
  case GDK_F1:
    clear_buffer ();
    break;
  case GDK_F2:
    load_file ();
    break;
  case GDK_F3:
    save_file ();
    break;
  case GDK_F4:
    search_from_point ();
    break;
  case GDK_F5:
    search_back_from_point ();
    break;
  }

  return FALSE;
}

/* main program */

static GtkWidget*
create_window ()
{
  GtkWidget *window;
  GtkWidget *vbox;
  GtkWidget *editor;
  GtkWidget *entry;

  window = gtk_window_new (GTK_WINDOW_TOPLEVEL);
  gtk_window_set_title (GTK_WINDOW (window), "Editor Test");

  vbox = gtk_vbox_new (FALSE, 0);
  gtk_widget_show (vbox);
  gtk_container_add (GTK_CONTAINER (window), vbox);

  editor = gtk_editor_new(NULL, NULL);
  gtk_widget_show (editor);

  gtk_text_set_editable (GTK_TEXT (editor), TRUE);
  gtk_box_pack_start (GTK_BOX (vbox), editor, TRUE, TRUE, 0);

  entry = gtk_entry_new ();
  gtk_widget_show (entry);
  gtk_box_pack_start (GTK_BOX (vbox), entry, FALSE, FALSE, 0);

  minibuffer = GTK_ENTRY (entry);
  gtk_signal_connect (GTK_OBJECT (minibuffer), "key_press_event",
		      GTK_SIGNAL_FUNC (minibuf_key_press),
		      NULL);
  mainbuffer = GTK_EDITOR (editor);
  gtk_signal_connect (GTK_OBJECT (mainbuffer), "key_press_event",
		      GTK_SIGNAL_FUNC (mainbuf_key_press),
		      NULL);

  return window;
}

int
main (int argc, char *argv[]) 
{
  GtkWidget *main_window;

  /* initialize */
  gtk_init (&argc, &argv);
  gtk_rc_parse ("editorrc");

  main_window = create_window ();

  /* show */
  gtk_widget_show (main_window);

  /* args? */
  if (argc > 1) {
    load_file_cb (g_strdup (argv[1])); /* we need a copy 'cause load
					* frees the string */
  }
  
  /* main loop */
  gtk_main ();

  return EXIT_SUCCESS;
}
