/* GLE - The GTK+ Layout Engine
 * Copyright (C) 1998, 1999 Tim Janik
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	 See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307, USA.
 */
#include	<gle/gleconfig.h>

#include	"glegobject.h"
#include	"gleprivate.h"
#include	"gleproxy.h"
#include	"glehandler.h"

#include 	<stdlib.h>
#include 	<stdio.h>

/* --- typedefs --- */
typedef void (*GleSignalGleGNameChanged)	(GtkObject	*object,
						 const gchar	*old_gname,
						 gpointer	 func_data);


/* --- prototypes --- */
static void  gle_gobject_base_class_init	(GleGObjectClass*class);
static void  gle_gobject_class_init		(GleGObjectClass*class);
static void  gle_gobject_init			(GleGObject	*gobject);
static void  gle_gobject_do_initialize		(GleGObject	*gobject);
static void  gle_gobject_do_pre_destroy		(GleGObject	*gobject);
static void  gle_gobject_do_instantiate		(GleGObject	*gobject);
static void  gle_gobject_do_associate		(GleGObject	*gobject,
						 GtkObject	*object);
static void  gle_gobject_do_foreign_associated	(GleGObject	*gobject);
static void  gle_gobject_do_pre_disassociate	(GleGObject	*gobject);
static void  gle_gobject_do_disassociate	(GleGObject	*gobject);
static void  gle_gobject_do_apply_all_args	(GleGObject	*gobject);
static void  gle_gobject_do_reset_all_args	(GleGObject	*gobject);
static void  gle_gobject_do_restore_all_args	(GleGObject	*gobject);
static void  gle_gobject_do_save_all_args	(GleGObject	*gobject);
static void  gle_gobject_do_update_all_args	(GleGObject	*gobject);
static void  gle_gobject_do_check_args		(GleGObject	*gobject);
static void  gle_gobject_do_rebuilt_args	(GleGObject	*gobject);
static void  gle_gobject_do_set_gname		(GleGObject	*gobject,
						 const gchar	*gname);
static void  gle_gobject_setup_args		(GleGObject	*gobject);


/* --- variables --- */
GHashTable		*gle_gname_hash_table = NULL;
static GMemChunk	*gle_cb_node_mem_chunk = NULL;
static GQuark		 quark_gle_gobject = 0;
static GQuark		 quark_cb_nodes = 0;
static guint		 signal_gle_gname_changed = 0;


/* --- Callback Nodes --- */
#define	gle_gobject_set_cb_nodes(gobject, nodes)	\
    g_datalist_id_set_data (&(gobject)->datalist, quark_cb_nodes, nodes)
#define	gle_gobject_get_cb_nodes(gobject)		\
    g_datalist_id_get_data (&(gobject)->datalist, quark_cb_nodes)
#define	gle_cb_node_ref(node)				\
    (node->ref_count++)
struct _GleCbNode
{
  guint		 ref_count : 16;
  guint		 method_id : 10;
  guint		 notify_type : 4;
  GleFunc	 func;
  gpointer	 data;
  GleCbNode	*next;
  GleCbNode	*prev;
};
static inline void
gle_cb_node_unref (GleGObject *gobject,
		   GleCbNode  *node)
{
  if (node->ref_count)
    {
      node->ref_count--;
      if (!node->ref_count)
	{
	  if (node->prev)
	    node->prev->next = node->next;
	  else
	    gle_gobject_set_cb_nodes (gobject, node->next);
	  if (node->next)
	    node->next->prev = node->prev;
	  g_chunk_free (node, gle_cb_node_mem_chunk);
	}
    }
  else
    g_warning ("GleCbNode with 0 ref_count"); /* FIXME: remove */
}
static inline GleCbNode*
gle_cb_nodes_last (GleCbNode *node)
{
  if (node)
    while (node->next)
      node = node->next;
  return node;
}


/* --- functions --- */
GtkType
gle_gobject_get_type (void)
{
  static GtkType gobject_type = 0;
  
  if (!gobject_type)
    {
      GtkTypeInfo gobject_info =
      {
	"GleGObject",
	sizeof (GleGObject),
	sizeof (GleGObjectClass),
	(GtkClassInitFunc) gle_gobject_class_init,
	(GtkObjectInitFunc) gle_gobject_init,
	/* reserved_1 */ NULL,
	/* reserved_2 */ NULL,
	(GtkClassInitFunc) gle_gobject_base_class_init,
      };
      
      gobject_type = gtk_type_unique (0, &gobject_info);
      if (gobject_type)
	g_message ("registering fundamental type `GleGObject' with %d (GtkObject=%d, limit=%d)",
		   gobject_type, GTK_TYPE_OBJECT, GTK_TYPE_FUNDAMENTAL_MAX);
      else
	g_assert_not_reached ();
      gtk_type_set_chunk_alloc (gobject_type, GLE_GOBJECTS_PREALLOC);
      /* set the rgument collector type for an external fundamental */
      gtk_type_set_varargs_type (gobject_type, GTK_TYPE_POINTER);
    }
  
  return gobject_type;
}

static void
gle_marshaller_gle_gname_changed (GtkObject	*object,
				  GtkSignalFunc func,
				  gpointer	func_data,
				  GtkArg	*args)
{
  GleSignalGleGNameChanged rfunc = (GleSignalGleGNameChanged) func;
  (* rfunc) (object, GTK_VALUE_STRING (args[0]), func_data);
}

static void
gle_gobject_base_class_init (GleGObjectClass *class)
{
  class->method_ids = NULL;
  class->n_method_ids = 0;
  class->banned_args = NULL;
}

static void
gle_gobject_class_init (GleGObjectClass *class)
{
  quark_gle_gobject = g_quark_from_static_string (GLE_PRIVATE_KEY (gle-gobject));
  quark_cb_nodes = g_quark_from_static_string ("gle-cb-nodes");
  
  class->initialize = gle_gobject_do_initialize;
  class->instantiate = gle_gobject_do_instantiate;
  class->foreign_associated = gle_gobject_do_foreign_associated;

  class->pre_destroy = gle_gobject_do_pre_destroy;
  class->associate = gle_gobject_do_associate;
  class->pre_disassociate = gle_gobject_do_pre_disassociate;
  class->disassociate = gle_gobject_do_disassociate;
  class->apply_all_args = gle_gobject_do_apply_all_args;
  class->reset_all_args = gle_gobject_do_reset_all_args;
  class->restore_all_args = gle_gobject_do_restore_all_args;
  class->save_all_args = gle_gobject_do_save_all_args;
  class->update_all_args = gle_gobject_do_update_all_args;
  class->check_args = gle_gobject_do_check_args;
  class->rebuilt_args = gle_gobject_do_rebuilt_args;
  class->set_gname = gle_gobject_do_set_gname;

  /* install notification methods
   */
  GLE_CLASS_ADD_NOTIFY_METHOD (class, GleGObjectClass, pre_destroy);
  GLE_CLASS_ADD_NOTIFY_METHOD (class, GleGObjectClass, associate);
  GLE_CLASS_ADD_NOTIFY_METHOD (class, GleGObjectClass, pre_disassociate);
  GLE_CLASS_ADD_NOTIFY_METHOD (class, GleGObjectClass, disassociate);
  GLE_CLASS_ADD_NOTIFY_METHOD (class, GleGObjectClass, apply_all_args);
  GLE_CLASS_ADD_NOTIFY_METHOD (class, GleGObjectClass, reset_all_args);
  GLE_CLASS_ADD_NOTIFY_METHOD (class, GleGObjectClass, restore_all_args);
  GLE_CLASS_ADD_NOTIFY_METHOD (class, GleGObjectClass, save_all_args);
  GLE_CLASS_ADD_NOTIFY_METHOD (class, GleGObjectClass, update_all_args);
  GLE_CLASS_ADD_NOTIFY_METHOD (class, GleGObjectClass, check_args);
  GLE_CLASS_ADD_NOTIFY_METHOD (class, GleGObjectClass, rebuilt_args);
  GLE_CLASS_ADD_NOTIFY_METHOD (class, GleGObjectClass, set_gname);
  
  /* install additional signals for gtk objects */
  signal_gle_gname_changed =
    gtk_object_class_user_signal_new (gtk_type_class (GTK_TYPE_OBJECT),
				      "gle-gname-changed",
				      0,
				      gle_marshaller_gle_gname_changed,
				      GTK_TYPE_NONE,
				      1, GTK_TYPE_STRING);
}

static void
gle_gobject_init (GleGObject	 *gobject)
{
  gobject->object_type = 0;
  gobject->iref_count = 1;
  gobject->flags = 0;
  gobject->queue_flags = 0;
  gobject->gname = NULL;
  g_datalist_init (&gobject->datalist);
  gobject->object = NULL;
  gobject->n_garg_groups = 0;
  gobject->garg_groups = NULL;
  gobject->connections = NULL;
}

GleGObject*
gle_gobject_new (GtkType type)
{
  GtkType proxy_type;
  GleGObject *gobject;
  
  g_return_val_if_fail (gtk_type_is_a (type, GTK_TYPE_OBJECT), NULL);
  
  proxy_type = gle_proxy_for_gtktype (type);
  
  if (!gtk_type_is_a (proxy_type, GLE_TYPE_GOBJECT))
    {
      g_warning ("No Gle proxy type registered for `%s'",
		 gtk_type_name (type));
      return NULL;
    }
  
  gobject = gtk_type_new (proxy_type);
  
  gobject->object_type = type;
  gobject->gname = gle_gname_new (type);
  gobject->flags = GLE_GENERIC_GNAME;
  
  if (!gle_gname_hash_table)
    gle_gname_hash_table = g_hash_table_new (g_str_hash, g_str_equal);
  g_hash_table_insert (gle_gname_hash_table, gobject->gname, gobject);
  
  GLE_GOBJECT_GET_CLASS (gobject)->initialize (gobject);
  
  return gobject;
}

static void
gle_gobject_do_initialize (GleGObject	  *gobject)
{
  gle_gobject_setup_args (gobject);
}

void
gle_gobject_ref (GleGObject	*gobject)
{
  g_return_if_fail (gobject != NULL);
  g_return_if_fail (GLE_IS_GOBJECT (gobject));
  g_return_if_fail (gobject->iref_count > 0);
  
  gobject->iref_count += 1;
}

void
gle_gobject_unref (GleGObject	  *gobject)
{
  g_return_if_fail (gobject != NULL);
  g_return_if_fail (GLE_IS_GOBJECT (gobject));
  g_return_if_fail (gobject->iref_count > 0);
  
  if (gobject->iref_count == 1)
    {
      gle_gobject_destroy (gobject);
      
      g_return_if_fail (gobject->iref_count > 0);
    }
  
  gobject->iref_count -= 1;
  
  if (!gobject->iref_count)
    {
      GtkType proxy_type;
      
      proxy_type = GLE_GOBJECT_PROXY_TYPE (gobject);
      
      if (gobject->object)
	g_warning ("GtkObject cleanup is messed up for `%s' proxy %s \"%s\"",
		   gtk_type_name (GLE_GOBJECT_OTYPE (gobject)),
		   gtk_type_name (GLE_GOBJECT_PROXY_TYPE (gobject)),
		   gobject->gname);

      gle_connections_destroy (gobject);
      
      gobject->iref_count = 0;
      gobject->type_object.klass = NULL;
      gobject->object_type = 0;
      
      gtk_type_free (proxy_type, gobject);
    }
}

void
gle_gobject_destroy (GleGObject	*gobject)
{
  g_return_if_fail (gobject != NULL);
  g_return_if_fail (GLE_IS_GOBJECT (gobject));

  if (!GLE_GOBJECT_DESTROYED (gobject))
    {
      gle_gobject_ref (gobject);

      GLE_GOBJECT_SET_FLAG (gobject, GLE_GOBJECT_DESTROYED);
      
      if (gobject->object)
	gtk_object_destroy (gobject->object);
      
      GLE_GOBJECT_NOTIFY (gobject, pre_destroy, NULL, NULL);
      
      GLE_GOBJECT_GET_CLASS (gobject)->pre_destroy (gobject);
      
      gle_gobject_unref (gobject);
    }
}

static void
gle_gobject_do_pre_destroy (GleGObject *gobject)
{
  GleGArgGroup *group;
  GleCbNode *cb_node;
  
  /* free object argument list
   */
  for (group = gobject->garg_groups + gobject->n_garg_groups - 1;
       group >= gobject->garg_groups;
       group--)
    {
      GSList *slist;
      
      for (slist = group->gargs; slist; slist = slist->next)
	gle_garg_free (slist->data);
      g_slist_free (group->gargs);
      group->gargs = NULL;
    }
  g_free (gobject->garg_groups);
  gobject->garg_groups = NULL;
  gobject->n_garg_groups = 0;
  
  /* remove callbacks
   */
  cb_node = gle_gobject_get_cb_nodes (gobject);
  gle_gobject_set_cb_nodes (gobject, NULL);
  while (cb_node)
    {
      GleCbNode *old_node;
      
      old_node = cb_node;
      cb_node = cb_node->next;
      gle_cb_node_unref (gobject, old_node);
    }
  
  /* destroy any associated data
   */
  g_datalist_clear (&gobject->datalist);
  
  /* remove from global hash table
   */
  g_hash_table_remove (gle_gname_hash_table, gobject->gname);
  if (GLE_GOBJECT_GENERIC_GNAME (gobject))
    gle_gname_free (gobject->gname);
  else
    g_free (gobject->gname);
  gobject->gname = NULL;
}

void
gle_gobject_instantiate (GleGObject	*gobject)
{
  GtkObject *object;
  
  g_return_if_fail (gobject != NULL);
  g_return_if_fail (GLE_IS_GOBJECT (gobject));
  g_return_if_fail (!GLE_GOBJECT_IS_INSTANTIATED (gobject));
  
  gle_gobject_ref (gobject);
  
  gle_gobject_check_args (gobject);
  
  GLE_GOBJECT_GET_CLASS (gobject)->instantiate (gobject);
  object = gobject->object;
  gobject->object = NULL;
  GLE_GOBJECT_GET_CLASS (gobject)->associate (gobject, object);
  
  GLE_GOBJECT_NOTIFY (gobject, associate, object, NULL);

  gle_gobject_connect (gobject);
  
  gle_gobject_unref (gobject);
}

static void
gle_gobject_do_instantiate (GleGObject	   *gobject)
{
  gobject->object = gtk_type_new (GLE_GOBJECT_OTYPE (gobject));
  
  gle_gobject_apply_construct_args (gobject);
  gtk_object_default_construct (gobject->object);
}

void
gle_gobject_associate (GleGObject     *gobject,
		       GtkObject      *object)
{
  g_return_if_fail (gobject != NULL);
  g_return_if_fail (GLE_IS_GOBJECT (gobject));
  g_return_if_fail (object != NULL);
  g_return_if_fail (GTK_IS_OBJECT (object));
  g_return_if_fail (!GTK_OBJECT_DESTROYED (object));
  g_return_if_fail (!GLE_OBJECT_HAS_GOBJECT (object));
  g_return_if_fail (!GLE_HAS_TAG (object));
  g_return_if_fail (!GLE_GOBJECT_IS_INSTANTIATED (gobject));
  
  gle_gobject_ref (gobject);
  gtk_object_ref (object);
  
  GLE_GOBJECT_GET_CLASS (gobject)->associate (gobject, object);
  
  GLE_GOBJECT_GET_CLASS (gobject)->foreign_associated (gobject);
  
  GLE_GOBJECT_NOTIFY (gobject, associate, object, NULL);
  
  gtk_object_unref (object);
  gle_gobject_unref (gobject);
}

static void
gle_gobject_do_associate (GleGObject *gobject,
			  GtkObject  *object)
{
  gobject->object = object;
  
  gtk_signal_connect_object (gobject->object,
			     "destroy",
			     gle_gobject_disassociate,
			     (GtkObject*) gobject);
  gtk_object_set_data_by_id (gobject->object, quark_gle_gobject, gobject);
}

static void
gle_gobject_do_foreign_associated (GleGObject *gobject)
{
  gle_gobject_update_all_args (gobject);
  gle_gobject_save_all_args (gobject);
}

void
gle_gobject_disassociate (GleGObject	 *gobject)
{
  g_return_if_fail (gobject != NULL);
  g_return_if_fail (GLE_IS_GOBJECT (gobject));
  g_return_if_fail (GLE_GOBJECT_IS_INSTANTIATED (gobject));
  
  gle_gobject_ref (gobject);
  
  GLE_GOBJECT_GET_CLASS (gobject)->pre_disassociate (gobject);
  
  GLE_GOBJECT_NOTIFY (gobject, pre_disassociate, NULL, NULL);
  
  GLE_GOBJECT_GET_CLASS (gobject)->disassociate (gobject);
  
  GLE_GOBJECT_NOTIFY (gobject, disassociate, NULL, NULL);
  
  gle_gobject_unref (gobject);
}

static void
gle_gobject_do_pre_disassociate (GleGObject *gobject)
{
  GleConnection *connection;

  for (connection = gobject->connections; connection; connection = connection->next)
    connection->gtk_id = 0;
}

static void
gle_gobject_do_disassociate (GleGObject	    *gobject)
{
  gtk_signal_disconnect_by_func (gobject->object,
				 gle_gobject_disassociate,
				 gobject);
  gtk_object_remove_data_by_id (gobject->object, quark_gle_gobject);
  gobject->object = NULL;
}

void
gle_gobject_apply_object_args (GleGObject *gobject)
{
  GleGArgGroup *group;
  
  g_return_if_fail (gobject != NULL);
  g_return_if_fail (GLE_IS_GOBJECT (gobject));
  g_return_if_fail (GLE_GOBJECT_IS_INSTANTIATED (gobject));
  
  if (!GTK_OBJECT_DESTROYED (gobject->object))
    {
      for (group = gobject->garg_groups;
	   group < gobject->garg_groups + gobject->n_garg_groups;
	   group++)
	{
	  GSList *slist;
	  
	  if (group->proxy_type != GLE_TYPE_GOBJECT)
	    continue;
	  
	  for (slist = group->gargs; slist; slist = slist->next)
	    {
	      GleGArg *garg;
	      
	      garg = slist->data;
	      if (GLE_GARG_IS_WRITABLE (garg) &&
		  GLE_GARG_NEEDS_SET (garg) &&
		  !GLE_GARG_IS_CONSTRUCT_ONLY (garg))
		gle_garg_set (garg, gobject->object, NULL);
	    }
	}
    }
}

void
gle_gobject_apply_construct_args (GleGObject	*gobject)
{
  GleGArgGroup *group;
  
  g_return_if_fail (gobject != NULL);
  g_return_if_fail (GLE_IS_GOBJECT (gobject));
  g_return_if_fail (GLE_GOBJECT_IS_INSTANTIATED (gobject));
  
  if (!GTK_OBJECT_DESTROYED (gobject->object))
    {
      for (group = gobject->garg_groups;
	   group < gobject->garg_groups + gobject->n_garg_groups;
	   group++)
	{
	  GSList *slist;
	  
	  if (group->proxy_type != GLE_TYPE_GOBJECT)
	    continue;
	  
	  for (slist = group->gargs; slist; slist = slist->next)
	    {
	      GleGArg *garg;
	      
	      garg = slist->data;
	      if (GLE_GARG_IS_WRITABLE (garg) &&
                  GLE_GARG_NEEDS_SET (garg) &&
		  GLE_GARG_IS_CONSTRUCT (garg))
		gle_garg_set (garg, gobject->object, NULL);
	    }
	}
    }
}

void
gle_gobject_reset_object_args (GleGObject *gobject)
{
  GleGArgGroup *group;
  
  g_return_if_fail (gobject != NULL);
  g_return_if_fail (GLE_IS_GOBJECT (gobject));
  
  for (group = gobject->garg_groups;
       group < gobject->garg_groups + gobject->n_garg_groups;
       group++)
    {
      GSList *slist;
      
      if (group->proxy_type != GLE_TYPE_GOBJECT)
	continue;
      
      for (slist = group->gargs; slist; slist = slist->next)
	{
	  GleGArg *garg;
	  
	  garg = slist->data;
	  gle_garg_reset (garg);
	}
    }
}

void
gle_gobject_restore_object_args (GleGObject *gobject)
{
  GleGArgGroup *group;
  
  g_return_if_fail (gobject != NULL);
  g_return_if_fail (GLE_IS_GOBJECT (gobject));
  
  for (group = gobject->garg_groups;
       group < gobject->garg_groups + gobject->n_garg_groups;
       group++)
    {
      GSList *slist;
      
      if (group->proxy_type != GLE_TYPE_GOBJECT)
	continue;
      
      for (slist = group->gargs; slist; slist = slist->next)
	{
	  GleGArg *garg;
	  
	  garg = slist->data;
	  gle_garg_restore (garg);
	}
    }
}

void
gle_gobject_save_object_args (GleGObject *gobject)
{
  GleGArgGroup *group;
  
  g_return_if_fail (gobject != NULL);
  g_return_if_fail (GLE_IS_GOBJECT (gobject));
  
  for (group = gobject->garg_groups;
       group < gobject->garg_groups + gobject->n_garg_groups;
       group++)
    {
      GSList *slist;
      
      if (group->proxy_type != GLE_TYPE_GOBJECT)
	continue;
      
      for (slist = group->gargs; slist; slist = slist->next)
	{
	  GleGArg *garg;
	  
	  garg = slist->data;
	  gle_garg_save (garg);
	}
    }
}

void
gle_gobject_update_object_args (GleGObject *gobject)
{
  GleGArgGroup *group;
  
  g_return_if_fail (gobject != NULL);
  g_return_if_fail (GLE_IS_GOBJECT (gobject));
  
  if (GLE_GOBJECT_IS_INSTANTIATED (gobject) &&
      !GTK_OBJECT_DESTROYED (gobject->object))
    {
      for (group = gobject->garg_groups;
	   group < gobject->garg_groups + gobject->n_garg_groups;
	   group++)
	{
	  GSList *slist;
	  
	  if (group->proxy_type != GLE_TYPE_GOBJECT)
	    continue;
	  
	  for (slist = group->gargs; slist; slist = slist->next)
	    {
	      GleGArg *garg;
	      
	      garg = slist->data;
	      if (GLE_GARG_IS_READABLE (garg))
		{
		  GLE_GARG_NEEDS_SET (garg) = FALSE;
		  
		  gle_garg_get (garg, gobject->object, NULL);
		}
	    }
	}
    }
}

void
gle_gobject_apply_all_args (GleGObject *gobject)
{
  g_return_if_fail (gobject != NULL);
  g_return_if_fail (GLE_IS_GOBJECT (gobject));
  g_return_if_fail (GLE_GOBJECT_IS_INSTANTIATED (gobject));
  
  gle_gobject_ref (gobject);
  if (!GTK_OBJECT_DESTROYED (gobject->object))
    {
      GLE_GOBJECT_GET_CLASS (gobject)->apply_all_args (gobject);
      
      GLE_GOBJECT_NOTIFY (gobject, apply_all_args, NULL, NULL);
    }
  
  gle_gobject_unref (gobject);
}

static void
gle_gobject_do_apply_all_args (GleGObject *gobject)
{
  gle_gobject_apply_object_args (gobject);
}

void
gle_gobject_reset_all_args (GleGObject *gobject)
{
  g_return_if_fail (gobject != NULL);
  g_return_if_fail (GLE_IS_GOBJECT (gobject));
  
  gle_gobject_ref (gobject);
  
  GLE_GOBJECT_GET_CLASS (gobject)->reset_all_args (gobject);
  
  GLE_GOBJECT_NOTIFY (gobject, reset_all_args, NULL, NULL);
  
  gle_gobject_unref (gobject);
}

static void
gle_gobject_do_reset_all_args (GleGObject *gobject)
{
  gle_gobject_reset_object_args (gobject);
}

void
gle_gobject_restore_all_args (GleGObject *gobject)
{
  g_return_if_fail (gobject != NULL);
  g_return_if_fail (GLE_IS_GOBJECT (gobject));
  
  gle_gobject_ref (gobject);
  
  GLE_GOBJECT_GET_CLASS (gobject)->restore_all_args (gobject);
  
  GLE_GOBJECT_NOTIFY (gobject, restore_all_args, NULL, NULL);
  
  gle_gobject_unref (gobject);
}

static void
gle_gobject_do_restore_all_args (GleGObject *gobject)
{
  gle_gobject_restore_object_args (gobject);
}

void
gle_gobject_save_all_args (GleGObject *gobject)
{
  g_return_if_fail (gobject != NULL);
  g_return_if_fail (GLE_IS_GOBJECT (gobject));
  
  gle_gobject_ref (gobject);
  
  GLE_GOBJECT_GET_CLASS (gobject)->save_all_args (gobject);
  
  GLE_GOBJECT_NOTIFY (gobject, save_all_args, NULL, NULL);
  
  gle_gobject_unref (gobject);
}

static void
gle_gobject_do_save_all_args (GleGObject *gobject)
{
  gle_gobject_save_object_args (gobject);
}

void
gle_gobject_update_all_args (GleGObject *gobject)
{
  g_return_if_fail (gobject != NULL);
  g_return_if_fail (GLE_IS_GOBJECT (gobject));
  
  gle_gobject_ref (gobject);
  
  if (GLE_GOBJECT_IS_INSTANTIATED (gobject) &&
      !GTK_OBJECT_DESTROYED (gobject->object))
    {
      GLE_GOBJECT_GET_CLASS (gobject)->update_all_args (gobject);
      
      GLE_GOBJECT_NOTIFY (gobject, update_all_args, NULL, NULL);
    }
  
  gle_gobject_unref (gobject);
}

static void
gle_gobject_do_update_all_args (GleGObject *gobject)
{
  gle_gobject_update_object_args (gobject);
}

void
gle_gobject_check_args (GleGObject *gobject)
{
  g_return_if_fail (gobject != NULL);
  g_return_if_fail (GLE_IS_GOBJECT (gobject));
  
  gle_gobject_ref (gobject);
  
  GLE_GOBJECT_GET_CLASS (gobject)->check_args (gobject);
  
  GLE_GOBJECT_NOTIFY (gobject, check_args, NULL, NULL);
  
  gle_gobject_unref (gobject);
}

static void
gle_gobject_do_check_args (GleGObject *gobject)
{
  GleGArgGroup *group;
  gboolean need_construct_args;

  need_construct_args = (!GLE_GOBJECT_IS_INSTANTIATED (gobject) ||
			 !GTK_OBJECT_CONSTRUCTED (gobject->object));
  
  for (group = gobject->garg_groups;
       group < gobject->garg_groups + gobject->n_garg_groups;
       group++)
    {
      GSList *slist;
      
      /* we check _all_ arguments so derived classes need not to
       * implement this method
       *
       * if (group->proxy_type != GLE_TYPE_GOBJECT)
       *   continue;
       */
      
      for (slist = group->gargs; slist; slist = slist->next)
	{
	  GleGArg *garg;
	  
	  garg = slist->data;
	  if (GLE_GARG_IS_WRITABLE (garg))
	    {
	      GtkType arg_type;
	      
	      arg_type = GTK_FUNDAMENTAL_TYPE (GLE_GARG_ARG_TYPE (garg));
	      if (arg_type > GTK_TYPE_FLAT_LAST ||
		  arg_type == GTK_TYPE_BOXED ||
		  arg_type == GTK_TYPE_POINTER)
		{
		  gle_garg_reset (garg);
		  garg->needs_set = FALSE;
		}
	      else if (need_construct_args &&
		       GLE_GARG_IS_CONSTRUCT_ONLY (garg) &&
		       garg->info->class_type == GLE_GOBJECT_OTYPE (gobject))
		garg->needs_set = !gle_garg_defaults (garg);
	      else
		garg->needs_set = (!gle_garg_defaults (garg) &&
				   !GLE_GARG_IS_CONSTRUCT_ONLY (garg));
	    }
	}
    }
}

void
gle_gobject_rebuilt_args (GleGObject	 *gobject)
{
  g_return_if_fail (gobject != NULL);
  g_return_if_fail (GLE_IS_GOBJECT (gobject));
  
  gle_gobject_ref (gobject);
  
  GLE_GOBJECT_GET_CLASS (gobject)->rebuilt_args (gobject);
  
  GLE_GOBJECT_NOTIFY (gobject, rebuilt_args, NULL, NULL);
  
  gle_gobject_unref (gobject);
}

static void
gle_gobject_do_rebuilt_args (GleGObject	    *gobject)
{
}

void
gle_gobject_set_gname (GleGObject	 *gobject,
		       const gchar	 *gname)
{
  gchar *name;
  
  g_return_if_fail (gobject != NULL);
  g_return_if_fail (GLE_IS_GOBJECT (gobject));
  g_return_if_fail (gname != NULL);
  g_return_if_fail (gname[0] != 0);
  
  name = gle_gname_canonicalize (g_strdup (gname));
  
  if (gle_gname_lookup (name))
    {
      g_warning ("gle_gobject_set_gname(): GleGObject \"%s\" already exists", name);
      g_free (name);
      return;
    }
  
  gle_gobject_ref (gobject);
  
  GLE_GOBJECT_GET_CLASS (gobject)->set_gname (gobject, name);
  
  GLE_GOBJECT_NOTIFY (gobject, set_gname, name, NULL);
  
  g_free (name);
  
  gle_gobject_unref (gobject);
}

static void
gle_gobject_do_set_gname (GleGObject	 *gobject,
			  const gchar	 *gname)
{
  gchar *old_gname;
  gboolean is_gname;

  is_gname = GLE_GOBJECT_GENERIC_GNAME (gobject);

  GLE_GOBJECT_UNSET_FLAG (gobject, GLE_GENERIC_GNAME);
  
  old_gname = gobject->gname;
  g_hash_table_remove (gle_gname_hash_table, old_gname);
  gobject->gname = g_strdup (gname);
  
  g_hash_table_insert (gle_gname_hash_table, gobject->gname, gobject);
  
  if (GLE_GOBJECT_IS_INSTANTIATED (gobject))
    gtk_signal_emit (gobject->object, signal_gle_gname_changed, old_gname);

  if (is_gname)
    gle_gname_free (old_gname);
  else
    g_free (old_gname);
}

static void
gle_gobject_setup_args (GleGObject     *gobject)
{
  GtkType type;
  
  g_return_if_fail (gobject != NULL);
  g_return_if_fail (gobject->garg_groups == NULL);
  
  type = GLE_GOBJECT_OTYPE (gobject);
  
  while (GTK_FUNDAMENTAL_TYPE (type) == GTK_TYPE_OBJECT)
    {
      GtkArg  *args;
      GleGArg *garg;
      guint    nargs;
      guint    i;
      GSList  *slist;
      GSList  *c_slist;
      
      slist = NULL;
      c_slist = NULL;
      args = gtk_object_query_args (type, NULL, &nargs);
      for (i = 0; i < nargs; i++)
	{
	  const GleArgInfo *garg_info;
	  
	  garg_info = gle_arg_get_info (args[i].name, FALSE);
	  if (!garg_info)
	    {
	      g_warning ("unresolved object arg: \"%s\"", args[i].name);
	      continue;
	    }
	  if (gle_class_arg_info_banned (GLE_GOBJECT_GET_CLASS (gobject), garg_info))
	    continue;
	  
	  garg = gle_garg_new (garg_info);
	  
	  if (!(garg_info->arg_flags & GTK_ARG_CONSTRUCT_ONLY))
	    slist = g_slist_prepend (slist, garg);
	  else if (garg_info->class_type == GLE_GOBJECT_OTYPE (gobject))
	    c_slist = g_slist_prepend (c_slist, garg);
	}
      g_free (args);
      
      slist = g_slist_reverse (slist);
      if (c_slist)
	{
	  GSList  *tmp;
	  
	  tmp = c_slist;
	  c_slist = g_slist_reverse (c_slist);
	  tmp->next = slist;
	  slist = c_slist;
	}
      gle_gobject_add_garg_group (gobject,
				  g_quark_from_static_string (gtk_type_name (type)),
				  GLE_TYPE_GOBJECT,
				  slist);
      
      type = gtk_type_parent (type);
    }
}

void
gle_gobject_connect (GleGObject	    *gobject)
{
  g_return_if_fail (gobject != NULL);
  g_return_if_fail (GLE_IS_GOBJECT (gobject));

  if (GLE_GOBJECT_IS_INSTANTIATED (gobject) && !GTK_OBJECT_DESTROYED (gobject->object))
    {
      GleConnection *connection;

      for (connection = gobject->connections; connection; connection = connection->next)
	gle_connection_connect (connection);
    }
}

void
gle_gobject_disconnect (GleGObject     *gobject)
{
  g_return_if_fail (gobject != NULL);
  g_return_if_fail (GLE_IS_GOBJECT (gobject));

  if (GLE_GOBJECT_IS_INSTANTIATED (gobject) && !GTK_OBJECT_DESTROYED (gobject->object))
    {
      GleConnection *connection;

      for (connection = gobject->connections; connection; connection = connection->next)
	gle_connection_disconnect (connection);
    }
}

void
gle_gobject_set_qdata (GleGObject *gobject,
		       GQuark	   id,
		       gpointer	   data)
{
  g_return_if_fail (gobject != NULL);
  g_return_if_fail (GLE_IS_GOBJECT (gobject));
  
  g_datalist_id_set_data (&gobject->datalist, id, data);
}

void
gle_gobject_set_qdata_full (GleGObject    *gobject,
			    GQuark         id,
			    gpointer       data,
			    GDestroyNotify destroy)
{
  g_return_if_fail (gobject != NULL);
  g_return_if_fail (GLE_IS_GOBJECT (gobject));
  
  g_datalist_id_set_data_full (&gobject->datalist, id, data, destroy);
}

gpointer
gle_gobject_get_qdata (GleGObject *gobject,
		       GQuark	   id)
{
  g_return_val_if_fail (gobject != NULL, NULL);
  g_return_val_if_fail (GLE_IS_GOBJECT (gobject), NULL);
  
  return g_datalist_id_get_data (&gobject->datalist, id);
}

GSList*
gle_gobject_get_gargs (GleGObject   *gobject,
		       GQuark	    garg_group)
{
  GleGArgGroup *group;
  
  g_return_val_if_fail (gobject != NULL, NULL);
  g_return_val_if_fail (GLE_IS_GOBJECT (gobject), NULL);
  g_return_val_if_fail (garg_group > 0, NULL);
  
  for (group = gobject->garg_groups;
       group < gobject->garg_groups + gobject->n_garg_groups;
       group++)
    if (group->garg_group == garg_group)
      return group->gargs;
  
  return NULL;
}

void
gle_gobject_add_garg_group (GleGObject	 *gobject,
			    GQuark	  garg_group,
			    GtkType	  proxy_type,
			    GSList	 *gargs)
{
  GleGArgGroup *group;
  
  g_return_if_fail (gobject != NULL);
  g_return_if_fail (GLE_IS_GOBJECT (gobject));
  g_return_if_fail (garg_group > 0);
  g_return_if_fail (gtk_type_is_a (proxy_type, GLE_TYPE_GOBJECT));
  
  for (group = gobject->garg_groups;
       group < gobject->garg_groups + gobject->n_garg_groups;
       group++)
    if (group->garg_group == garg_group)
      {
	g_warning ("%s \"%s\" already contains an arg group \"%s\"",
		   gtk_type_name (GLE_GOBJECT_PROXY_TYPE (gobject)),
		   gobject->gname,
		   g_quark_to_string (garg_group));
	return;
      }
  
  gobject->n_garg_groups += 1;
  gobject->garg_groups = g_renew (GleGArgGroup, gobject->garg_groups, gobject->n_garg_groups);
  group = gobject->garg_groups + gobject->n_garg_groups - 1;
  group->garg_group = garg_group;
  group->proxy_type = proxy_type;
  group->gargs = gargs;
}

void
gle_gobject_set_garg_group (GleGObject	 *gobject,
			    GQuark	  garg_group,
			    GSList	 *gargs)
{
  GleGArgGroup *group;
  
  g_return_if_fail (gobject != NULL);
  g_return_if_fail (GLE_IS_GOBJECT (gobject));
  g_return_if_fail (garg_group > 0);
  g_return_if_fail (gargs != NULL);
  
  for (group = gobject->garg_groups;
       group < gobject->garg_groups + gobject->n_garg_groups;
       group++)
    if (group->garg_group == garg_group)
      break;
  
  if (group->garg_group != garg_group)
    {
      g_warning ("%s \"%s\" contains no arg group \"%s\"",
		 gtk_type_name (GLE_GOBJECT_PROXY_TYPE (gobject)),
		 gobject->gname,
		 g_quark_to_string (garg_group));
      return;
    }
  
  group->gargs = gargs;
}

GleGArg*
gle_gobject_get_object_garg (GleGObject	    *gobject,
			     const gchar    *arg_name)
{
  GleGArgGroup *group;
  const GleArgInfo *arg_info;
  
  g_return_val_if_fail (gobject != NULL, NULL);
  g_return_val_if_fail (GLE_IS_GOBJECT (gobject), NULL);
  g_return_val_if_fail (arg_name != NULL, NULL);
  
  arg_info = gle_arg_get_info (arg_name, FALSE);
  if (!arg_info)
    return NULL;
  
  for (group = gobject->garg_groups;
       group < gobject->garg_groups + gobject->n_garg_groups;
       group++)
    {
      GSList *slist;
      
      if (group->proxy_type != GLE_TYPE_GOBJECT)
	continue;
      
      for (slist = group->gargs; slist; slist = slist->next)
	{
	  GleGArg *garg;
	  
	  garg = slist->data;
	  if (garg->info == arg_info)
	    return garg;
	}
    }
  
  return NULL;
}

GleGObject*
gle_object_get_gobject (GtkObject      *object)
{
  GleGObject *gobject;
  
  g_return_val_if_fail (object != NULL, NULL);
  g_return_val_if_fail (GTK_IS_OBJECT (object), NULL);
  
  if (!GTK_OBJECT_DESTROYED (object))
    {
      gobject = gtk_object_get_data_by_id (object, quark_gle_gobject);
    }
  else
    gobject = NULL;
  
  return gobject;
}

GleGObject*
gle_object_force_gobject (GtkObject *object)
{
  GleGObject *gobject;
  
  g_return_val_if_fail (object != NULL, NULL);
  g_return_val_if_fail (GTK_IS_OBJECT (object), NULL);
  g_return_val_if_fail (!GLE_HAS_TAG (object), NULL);
  
  gobject = gtk_object_get_data_by_id (object, quark_gle_gobject);
  if (!gobject)
    {
      gobject = gle_gobject_new (GTK_OBJECT_TYPE (object));
      gle_gobject_associate (gobject, object);
    }
  
  return gobject;
}

gchar*
gle_object_get_gname (GtkObject	*object)
{
  GleGObject *gobject;
  
  g_return_val_if_fail (object != NULL, NULL);
  g_return_val_if_fail (GTK_IS_OBJECT (object), NULL);
  
  if (!GTK_OBJECT_DESTROYED (object))
    {
      gobject = gtk_object_get_data_by_id (object, quark_gle_gobject);
    }
  else
    gobject = NULL;
  
  if (gobject)
    return gobject->gname;
  
  return NULL;
}

GtkObject*
gle_object_from_gname (const gchar *gname)
{
  GleGObject *gobject;

  g_return_val_if_fail (gname != NULL, NULL);

  gobject = gle_gname_lookup (gname);

  if (gobject && GLE_GOBJECT_IS_INSTANTIATED (gobject))
    return gobject->object;
  else
    return NULL;
}

GleGObject*
gle_gname_lookup (const gchar *gname)
{
  g_return_val_if_fail (gname != NULL, NULL);
  
  return gle_gname_hash_table ? g_hash_table_lookup (gle_gname_hash_table, gname) : NULL;
}

void
gle_gobject_notify_add (GleGObject     *gobject,
			const gchar    *method,
			GleNotifyType	notify_type,
			GleFunc		func,
			gpointer	data)
{
  GleCbNode *node, *last;
  GQuark method_id;
  
  g_return_if_fail (gobject != NULL);
  g_return_if_fail (GLE_IS_GOBJECT (gobject));
  g_return_if_fail (method != NULL);
  g_return_if_fail (notify_type > 0);
  g_return_if_fail (func != NULL);

  method_id = gle_class_find_notify_method (GLE_GOBJECT_GET_CLASS (gobject), method);
  if (!method_id)
    {
      g_warning ("Can not find method \"%s\" for `%s' proxy %s \"%s\"",
		 method,
		 gtk_type_name (GLE_GOBJECT_OTYPE (gobject)),
		 gtk_type_name (GLE_GOBJECT_PROXY_TYPE (gobject)),
		 gobject->gname);
      return;
    }
  
  if (!gle_cb_node_mem_chunk)
    gle_cb_node_mem_chunk = g_mem_chunk_create (GleCbNode, GLE_CB_NODES_PREALLOC, G_ALLOC_AND_FREE);
  
  node = g_chunk_new0 (GleCbNode, gle_cb_node_mem_chunk);
  node->ref_count = 1;
  node->method_id = method_id;
  node->notify_type = notify_type;
  node->func = func;
  node->data = data;
  
  /* append at the end, why did we do that? */
  node->next = NULL;
  last = gle_cb_nodes_last (gle_gobject_get_cb_nodes (gobject));
  if (last)
    {
      last->next = node;
      node->prev = last;
    }
  else
    {
      gle_gobject_set_cb_nodes (gobject, node);
      node->prev = NULL;
    }
}

void
gle_gobject_notify_del (GleGObject     *gobject,
			const gchar    *method,
			GleNotifyMask	mask,
			GleFunc		func,
			gpointer	data)
{
  GleCbNode *node;
  gboolean found_one;
  GleNotifyMask skip;
  GQuark method_id;
  
  g_return_if_fail (gobject != NULL);
  g_return_if_fail (GLE_IS_GOBJECT (gobject));
  if (mask & GLE_NOTIFY_FUNC_MASK)
    g_return_if_fail (func != NULL);
  if (mask & GLE_NOTIFY_METHOD_MASK)
    {
      g_return_if_fail (method != NULL);

      method_id = gle_class_find_notify_method (GLE_GOBJECT_GET_CLASS (gobject), method);
      if (!method_id)
	{
	  g_warning ("Can not find method \"%s\" for `%s' proxy %s \"%s\"",
		     method,
		     gtk_type_name (GLE_GOBJECT_OTYPE (gobject)),
		     gtk_type_name (GLE_GOBJECT_PROXY_TYPE (gobject)),
		     gobject->gname);
	  return;
	}
    }
  else
    method_id = 0;
  
  found_one = FALSE;
  node = gle_gobject_get_cb_nodes (gobject);
  skip = GLE_NOTIFY_MASK & ~mask;
  while (node)
    {
      if ((skip & GLE_NOTIFY_FUNC_MASK || node->func == func) &&
	  (skip & GLE_NOTIFY_DATA_MASK || node->data == data) &&
	  (skip & GLE_NOTIFY_METHOD_MASK || node->method_id == method_id) &&
	  node->notify_type)
	{
	  GleCbNode *old;
	  
	  old = node;
	  node = node->next;
	  
	  old->notify_type = 0;
	  old->method_id = 0;
	  
	  gle_cb_node_unref (gobject, old);
	  found_one = TRUE;
	}
      else
	node = node->next;
    }
  if (!found_one)
    g_warning ("Can not delete notifier from `%s' proxy %s \"%s\"",
	       gtk_type_name (GLE_GOBJECT_OTYPE (gobject)),
	       gtk_type_name (GLE_GOBJECT_PROXY_TYPE (gobject)),
	       gobject->gname);
}

gboolean
gle_gobject_notify_check (GleGObject	*gobject,
			  const gchar   *method,
			  GleNotifyMask	 mask,
			  GleFunc	 func,
			  gpointer	 data)
{
  GleCbNode *node;
  GleNotifyMask skip;
  GQuark method_id;
  
  g_return_val_if_fail (gobject != NULL, FALSE);
  g_return_val_if_fail (GLE_IS_GOBJECT (gobject), FALSE);
  if (mask & GLE_NOTIFY_FUNC_MASK)
    g_return_val_if_fail (func != NULL, FALSE);
  if (mask & GLE_NOTIFY_METHOD_MASK)
    {
      g_return_val_if_fail (method != NULL, FALSE);

      method_id = gle_class_find_notify_method (GLE_GOBJECT_GET_CLASS (gobject), method);
      if (!method_id)
	{
	  g_warning ("Can not find method \"%s\" for `%s' proxy %s \"%s\"",
		     method,
		     gtk_type_name (GLE_GOBJECT_OTYPE (gobject)),
		     gtk_type_name (GLE_GOBJECT_PROXY_TYPE (gobject)),
		     gobject->gname);
	  return FALSE;
	}
    }
  else
    method_id = 0;
  
  node = gle_gobject_get_cb_nodes (gobject);
  skip = GLE_NOTIFY_MASK & ~mask;
  while (node)
    {
      if ((skip & GLE_NOTIFY_FUNC_MASK || node->func == func) &&
	  (skip & GLE_NOTIFY_DATA_MASK || node->data == data) &&
	  (skip & GLE_NOTIFY_METHOD_MASK || node->method_id == method_id) &&
	  node->notify_type)
	return TRUE;
      node = node->next;
    }
  return FALSE;
}

void
gle_gobject_notify_do (GleGObject     *gobject,
		       GQuark	       method_id,
		       gpointer	       arg1,
		       gpointer	       arg2)
{
  GleCbNode *node;
  
  g_return_if_fail (gobject != NULL);
  g_return_if_fail (GLE_IS_GOBJECT (gobject));
  g_return_if_fail (method_id > 0);
  
  node = gle_gobject_get_cb_nodes (gobject);
  while (node)
    {
      if (node->method_id == method_id)
	{
	  GleCbNode *old;
	  
	  gle_cb_node_ref (node);
	  switch (node->notify_type)
	    {
	    case GLE_NOTIFY_GO_D:
	    {
	      register GleNotifyFunc_go_d notify;
	      
	      notify = node->func;
	      notify (gobject, node->data, arg1, arg2);
	    }
	    break;
	    
	    case GLE_NOTIFY_D_GO:
	    {
	      register GleNotifyFunc_d_go notify;
	      
	      notify = node->func;
	      notify (node->data, gobject, arg1, arg2);
	    }
	    break;
	    
	    default:
	      break;
	    }
	  old = node;
	  node = old->next;
	  gle_cb_node_unref (gobject, old);
	}
      else
	node = node->next;
    }
}

void
gle_class_add_notify_method (GleGObjectClass *class,
			     const gchar     *method_name,
			     guint            method_offs)
{
  g_return_if_fail (class != NULL);
  g_return_if_fail (GLE_IS_GOBJECT_CLASS (class));
  g_return_if_fail (method_name != NULL);
  g_return_if_fail (method_offs > 0);

  class->n_method_ids += 1;
  class->method_ids = g_renew (GQuark, class->method_ids, class->n_method_ids);
  class->method_ids[class->n_method_ids - 1] = g_quark_from_static_string (method_name);
}

GQuark
gle_class_find_notify_method (GleGObjectClass *class,
			      const gchar     *method_name)
{
  GQuark method_id;

  g_return_val_if_fail (class != NULL, 0);
  g_return_val_if_fail (GLE_IS_GOBJECT_CLASS (class), 0);
  g_return_val_if_fail (method_name != NULL, 0);

  method_id = g_quark_try_string (method_name);
  if (!method_id)
    return 0;

  do
    {
      GQuark *last;
      GQuark *work;
      
      last = class->method_ids + class->n_method_ids;
      for (work = class->method_ids; work < last; work++)
	if (*work == method_id)
	  return method_id;
      class = gtk_type_parent_class (class->type_class.type);
    }
  while (class);

  return 0;
}

gboolean
gle_class_arg_info_banned (GleGObjectClass  *class,
			   const GleArgInfo *arg_info)
{
  g_return_val_if_fail (class != NULL, 0);
  g_return_val_if_fail (GLE_IS_GOBJECT_CLASS (class), 0);
  g_return_val_if_fail (arg_info != NULL, 0);

  do
    {
      GSList *slist;

      for (slist = class->banned_args; slist; slist = slist->next)
	if (slist->data == arg_info)
	  return TRUE;
      class = gtk_type_parent_class (class->type_class.type);
    }
  while (class);

  return FALSE;
}

static GSList *gobject_queue = NULL;

static gboolean
gle_gobject_queue_handler (gpointer data)
{
  GSList *free_slist, *slist;
  
  free_slist = gobject_queue;
  gobject_queue = NULL;
  slist = free_slist;
  while (slist)
    {
      GleGObject *gobject;
      guint queue_flags;
      
      gobject = GLE_GOBJECT (slist->data);
      slist = slist->next;
      queue_flags = gobject->queue_flags;
      gobject->queue_flags = 0;
      
      if (!GLE_GOBJECT_DESTROYED (gobject))
	{
	  if (queue_flags & GLE_QUEUE_APPLY_ARGS)
	    gle_gobject_apply_all_args (gobject);
	  if (queue_flags & GLE_QUEUE_UPDATE_ARGS)
	    gle_gobject_update_all_args (gobject);
	  if (queue_flags & GLE_QUEUE_SAVE_ARGS)
	    gle_gobject_save_all_args (gobject);
	}
      
      gle_gobject_unref (gobject);
    }
  g_slist_free (free_slist);
  
  return FALSE;
}

void
gle_gobject_queue (GleGObject	  *gobject,
		   GleQueueFlags   flags)
{
  g_return_if_fail (gobject != NULL);
  g_return_if_fail (GLE_IS_GOBJECT (gobject));
  
  if (GLE_GOBJECT_DESTROYED (gobject))
    return;

  if (flags)
    {
      if (!gobject->queue_flags)
	{
	  if (!gobject_queue)
	    gtk_idle_add_priority ((GTK_PRIORITY_INTERNAL + GTK_PRIORITY_HIGH) / 2,
				   gle_gobject_queue_handler,
				   NULL);
	  gle_gobject_ref (gobject);
	  gobject_queue = g_slist_prepend (gobject_queue, gobject);
	}
      gobject->queue_flags |= flags;
    }
}

static void
gobjects_collect (gpointer	 key,
		  gpointer	 value,
		  gpointer	 user_data)
{
  GSList **slist_p = user_data;
  GleGObject *gobject = value;
  
  *slist_p = g_slist_prepend (*slist_p, gobject);
}

GSList*
gle_gobjects_list (void)
{
  GSList *slist;
  
  slist = NULL;
  if (gle_gname_hash_table)
    g_hash_table_foreach (gle_gname_hash_table, gobjects_collect, &slist);
  
  return slist;
}

static guint *type_counters = NULL;
static guint  n_type_counters = 0;

void
gle_gname_free (gchar *gname)
{
  gchar *s;
  GtkType type;
  guint counter = 0;
  guint seqn;

  g_return_if_fail (gname != NULL);

  s = gname;
  while (*s)
    if (*s == '-')
      {
	*s = 0;
	s++;
	counter = strtol (s, NULL, 10);
	break;
      }
    else
      s++;

  type = gtk_type_from_name (gname);
  seqn = GTK_TYPE_SEQNO (type);
  if (!seqn || !counter || seqn > n_type_counters)
    g_warning ("attempt to free invalid gname");
  else
    {
      seqn--;
      counter--;
      type_counters[seqn] = MIN (type_counters[seqn], counter);
      g_free (gname);
    }
}

gchar*
gle_gname_new (GtkType gtk_type)
{
  gchar *gname;
  gchar *type_name;
  gchar *s;
  guint len;
  guint seqn;
  
  g_return_val_if_fail (GTK_FUNDAMENTAL_TYPE (gtk_type) == GTK_TYPE_OBJECT, NULL);

  seqn = GTK_TYPE_SEQNO (gtk_type);

  if (n_type_counters < seqn)
    {
      type_counters = g_realloc (type_counters, sizeof (guint) * seqn);
      memset (type_counters + n_type_counters, 0, sizeof (guint) * (seqn - n_type_counters));
      n_type_counters = seqn;
    }
  seqn--;
  
  type_name = gtk_type_name (gtk_type);
  len = strlen (type_name);
  gname = g_new (gchar, len + 1 + 10 + 1);
  *gname = 0;
  strcpy (gname, type_name);
  s = gname + len;
  *s = '-';
  s++;
  do
    {
      g_snprintf (s, 10, "%u", ++type_counters[seqn]);
    }
  while (gle_gname_lookup (gname));
  
  return gname;
}

gchar*
gle_gname_canonicalize (gchar *gname)
{
  g_return_val_if_fail (gname != NULL, NULL);
  
  g_str_canonicalize (gname, '_', G_CSET_A_2_Z G_CSET_a_2_z ".-_0123456789", NULL);
  
  return gname;
}
