;;; Delete and expunge commands for VM.
;;; Copyright (C) 1989 Kyle E. Jones
;;;
;;; This program is free software; you can redistribute it and/or modify
;;; it under the terms of the GNU General Public License as published by
;;; the Free Software Foundation; either version 1, or (at your option)
;;; any later version.
;;;
;;; This program is distributed in the hope that it will be useful,
;;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;;; GNU General Public License for more details.
;;;
;;; You should have received a copy of the GNU General Public License
;;; along with this program; if not, write to the Free Software
;;; Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
;;;
;;; Send bug reports to kyle@cs.odu.edu.

(require 'vm)

(defun vm-delete-message (count)
  "Mark the current message for deletion.
With a prefix arg mark the next COUNT messages for deletion.  A negative
arg means the previous COUNT messages are marked."
  (interactive "p")
  (if (interactive-p)
      (vm-follow-summary-cursor))
  (if vm-mail-buffer
      (set-buffer vm-mail-buffer))
  (vm-error-if-folder-empty)
  (if (not (eq vm-circular-folders t))
      (vm-check-count count))
  (let ((direction (if (< count 0) 'backward 'forward))
	(count (vm-abs count))
	(oldmp vm-message-pointer)
	(vm-message-pointer vm-message-pointer))
    (while (not (zerop count))
      (if (not (vm-deleted-flag (car vm-message-pointer)))
	  (vm-set-deleted-flag (car vm-message-pointer) t))
      (vm-decrement count)
      (if (not (zerop count))
	  (vm-move-message-pointer direction))))
  (vm-update-summary-and-mode-line)
  (if vm-move-after-deleting
      (vm-next-message count t)))

(defun vm-undelete-message (count)
  "Remove the deletion mark from the current message.
With a prefix arg unmark the next COUNT messages.  A negative arg means
the previous COUNT messages are unmarked."
  (interactive "p")
  (if (interactive-p)
      (vm-follow-summary-cursor))
  (if vm-mail-buffer
      (set-buffer vm-mail-buffer))
  (vm-error-if-folder-empty)
  (if (not (eq vm-circular-folders t))
      (vm-check-count count))
  (let ((direction (if (< count 0) 'backward 'forward))
	(count (vm-abs count))
	(oldmp vm-message-pointer)
	(vm-message-pointer vm-message-pointer))
    (while (not (zerop count))
      (if (vm-deleted-flag (car vm-message-pointer))
	  (vm-set-deleted-flag (car vm-message-pointer) nil))
      (vm-decrement count)
      (if (not (zerop count))
	  (vm-move-message-pointer direction))))
  (vm-update-summary-and-mode-line))

(defun vm-kill-subject ()
  "Mark all messages with the same subject as the current message
\(ignoring re:'s) for deletion."
  (interactive)
  (vm-follow-summary-cursor)
  (if vm-mail-buffer
      (set-buffer vm-mail-buffer))
  (vm-error-if-folder-empty)
  (let ((subject (vm-su-subject (car vm-message-pointer)))
	(mp vm-message-list))
    (if (string-match "^\\(re: *\\)+" subject)
	(setq subject (substring subject (match-end 0))))
    (setq subject (concat "^\\(re: *\\)*" (regexp-quote subject) " *$"))
    (while mp
      (if (and (not (vm-deleted-flag (car mp)))
	       (string-match subject (vm-su-subject (car mp))))
	  (vm-set-deleted-flag (car mp) t))
      (setq mp (cdr mp))))
  (vm-update-summary-and-mode-line))

(defun vm-expunge-folder (&optional quitting shaddap)
  "Expunge deleted messages, but don't save folder to disk or exit VM."
  (interactive)
  (if vm-mail-buffer
      (set-buffer vm-mail-buffer))
  (let ((inhibit-quit t))
    (if (vm-gobble-deleted-messages)
	(if (not quitting)
	    (progn
	      (if (not shaddap)
		  (message "Deleted messages expunged."))
	      (vm-number-messages)
	      (if vm-summary-buffer
		  (vm-do-summary))
	      (if (and vm-message-pointer vm-summary-buffer)
		  (vm-set-summary-pointer (car vm-message-pointer)))
	      (if (null vm-message-pointer)
		  (vm-next-message)
		(if (null vm-system-state)
		    (vm-preview-current-message)
		  (vm-update-summary-and-mode-line)))))
      (error "No messages are marked for deletion."))))
