/*
 * Copyright (C) 1992, Board of Trustees of the University of Illinois.
 *
 * Permission is granted to copy and distribute source with out fee.
 * Commercialization of this product requires prior licensing
 * from the National Center for Supercomputing Applications of the
 * University of Illinois.  Commercialization includes the integration of this 
 * code in part or whole into a product for resale.  Free distribution of 
 * unmodified source and use of NCSA software is not considered 
 * commercialization.
 *
 */


/*
*   magnify.c
*
*   This routines magnifies an eight-bit raster image by an
*       arbitrary X and Y magnification factor.
*
*  Quincey Koziol                       Dec. 1991
*
*/

#include <stdlib.h>
#include <string.h>
#include "portable.h"

/**********************************************************************
*  Function :   magnify
*  Purpose  :   Magnify an image by independant X and Y magnification
*                   factors.  Note that the coordinates which specify
*                   the area in the "from" image are _inclusive_, i.e.
*                   if you would like to magnify the entire image, the
*                   "from" coordinates should be specified as:
*                   (0,0) and ((width-1),(height-1)), where "width"
*                   and "height" are the width and height of the image
*                   (respectively) in pixels.
*  Parameters   :
*           from_buffer - pointer to the buffer which contains the image
*                           to magnify
*           to_buffer - pointer to the buffer in which to place the
*                           magnified image
*           from_x0, from_y0 - Upper Left Corner (ULC) of the rectangular
*                           area in the image to magnify
*           from_x1, from_y1 - Lower Right Corner (LRC) of the rectangular
*                           area in the image to magnify
*           from_width, from_height - the width and height of the image
*                           to magnify (or "from" image)
*           to_width, to_height - the width and height of the magnified
*                           image (or "to" image)
*  Returns  :   (TRUE) for success, (FALSE) for error
*  Calls    :
*  Called by    :
**********************************************************************/
#ifdef __STDC__
boolean magnify(uint8 *from_buffer,uint8 *to_buffer,uint16 from_x0,
    uint16 from_y0,uint16 from_x1,uint16 from_y1,uint16 from_width,
    uint16 from_height,uint16 to_width,uint16 to_height)
#else
boolean magnify(from_buffer,to_buffer,from_x0,from_y0,from_x1,from_y1,
    from_width,from_height,to_width,to_height)
uint8 *from_buffer,
    *to_buffer;
uint16 from_x0,from_y0,from_x1,from_y1,
    from_width,from_height,
    to_width,to_height;
#endif
{
    uint8 *buf_off,     /* the current offset into the magnified data */
        *last_buf,      /* pointer to the last useful magnified line */
        *y_off,         /* the y offset into the data block */
        *last_y_coor,   /* pointer to the last line copied */
        **y_coor;       /* pointers to image data */
    float64 temp_val,   /* temporary value for holding double results */
        wind_width,     /* the width of the window to magnify */
        wind_height;    /* the height of the window to magnify */
    uint16 u,v;         /* local unsigned counting variables */
    uint16 *x_coor,     /* the X coor. lookup table */
        *x_coor_temp;   /* temporary pointer to the x lookup table */

    if(from_width==0 || from_height==0) /* check for bad image dimensions */
        return(FALSE);
    if(to_width==0 || to_height==0)     /* check for bad image dimensions */
        return(FALSE);
    if(from_x0>from_x1 || from_y0>from_y1)  /* check for an invalid window */
        return(FALSE);

/* find width and height of the window to magnify */
    wind_width=(from_x1-from_x0)+1;  
    wind_height=(from_y1-from_y0)+1;

/* allocate room for the x coordinate lookup table */
    x_coor=(uint16 *)vmalloc((uint16)(to_width*sizeof(uint16)));     
    EXCHECK(x_coor==NULL,XCoorFailed);      /* check if malloc() failed */
    temp_val=wind_width/(float64)to_width;
    for(u=0; u<to_width; u++)    /* calculate the x coordinate lookup table */
        x_coor[u]=((uint16)((float64)u*temp_val)+from_x0);

/* allocate room for the array of pointers */
    y_coor=(uint8 **)vmalloc((uint16)(to_height*sizeof(uint8 *)));  
    EXCHECK(y_coor==NULL,YCoorFailed);      /* check if malloc() failed */
    temp_val=wind_height/(float64)to_height;
    for(u=0; u<to_height; u++)      /* calculate the y coordinates */
        y_coor[u]=from_buffer+((uint32)((float64)u*temp_val)+from_y0)*from_width;

    last_buf=to_buffer;     /* set the previous line pointer */
    buf_off=to_buffer;      /* set the pointer to the "to" image */
    last_y_coor=NULL;       /* force to calculate the first line */

    for(u=0; u<to_height; u++) {  /* go through each magnified line */
/* if this line is not the same as the previous one, then make it again */
        if(y_coor[u]!=last_y_coor) {  
            last_y_coor=y_off=y_coor[u];
            x_coor_temp=x_coor;     /* assign the temporary pointer */
            last_buf=buf_off;       /* set the pointer to the previous line */
            for(v=0; v<to_width; v++)   /* go through each line magnifying it */
                *buf_off++=y_off[*x_coor_temp++];
		  }	/* end if */
/* this line is the same as the previous one, just copy it */
        else {  
            memcpy(buf_off,last_buf,to_width); /* copy the previous line */
            buf_off+=to_width;      /* advance the buffer offset pointer */
          } /* end else */
	  }	/* end for */
    vfree((char *)y_coor);
    vfree((char *)x_coor);
    return(TRUE);

YCoorFailed:    /* Failed to allocate memory for the Y coor. lookup table */
    vfree(x_coor);
XCoorFailed:    /* Failed to allocate memory for the X coor. lookup table */
    return(FALSE);
}   /* end magnify() */

