/*
  This file is part of TALER
  Copyright (C) 2020, 2023, 2025 Taler Systems SA

  Challenger is free software; you can redistribute it and/or modify it under the
  terms of the GNU General Public License as published by the Free Software
  Foundation; either version 3, or (at your option) any later version.

  Challenger is distributed in the hope that it will be useful, but WITHOUT ANY
  WARRANTY; without even the implied warranty of EXCHANGEABILITY or FITNESS FOR
  A PARTICULAR PURPOSE.  See the GNU General Public License for more details.

  You should have received a copy of the GNU General Public License along with
  Challenger; see the file COPYING.  If not, see <http://www.gnu.org/licenses/>
*/
/**
 * @file challenger-httpd_spa.c
 * @brief logic to load the single page app (/)
 * @author Sebastian Marchano
 * @author Christian Grothoff
 */
#include "platform.h"
#include <gnunet/gnunet_util_lib.h>
#include <taler/taler_util.h>
#include <taler/taler_mhd_lib.h>
#include <gnunet/gnunet_mhd_compat.h>
#include "challenger_util.h"
#include "challenger-httpd_spa.h"

/**
 * Resources of the Challenger SPA.
 */
static struct TALER_MHD_Spa *spa;


MHD_RESULT
CH_handler_spa (struct CH_HandlerContext *hc,
                const char *upload_data,
                size_t *upload_data_size)
{
  const char *path = hc->path;

  if ( (NULL == path) ||
       (0 == strcmp (path,
                     "")) )
    path = "index.html";
  return TALER_MHD_spa_handler (spa,
                                hc->connection,
                                path);
}


enum GNUNET_GenericReturnValue
CH_spa_init ()
{
  spa = TALER_MHD_spa_load (CHALLENGER_project_data (),
                            "spa/");
  if (NULL == spa)
  {
    GNUNET_break (0);
    return GNUNET_SYSERR;
  }
  return GNUNET_OK;
}


/**
 * Nicely shut down.
 */
void __attribute__ ((destructor))
get_spa_fini (void);

/* Declaration to avoid compiler warning */
void __attribute__ ((destructor))
get_spa_fini ()
{
  if (NULL != spa)
  {
    TALER_MHD_spa_free (spa);
    spa = NULL;
  }
}


/**
 * Generates the response for "/", redirecting the
 * client to the "/webui/" from where we serve the SPA.
 *
 * @param rh request handler
 * @param connection MHD connection
 * @param hc handler context
 * @return MHD result code
 */
MHD_RESULT
CH_spa_redirect (struct CH_HandlerContext *hc,
                 const char *upload_data,
                 size_t *upload_data_size)
{
  const char *text = "Redirecting to /webui/";
  struct MHD_Response *response;
  char *dst;
  const char *rparams;

  response = MHD_create_response_from_buffer (strlen (text),
                                              (void *) text,
                                              MHD_RESPMEM_PERSISTENT);
  if (NULL == response)
  {
    GNUNET_break (0);
    return MHD_NO;
  }
  TALER_MHD_add_global_headers (response,
                                true);
  GNUNET_break (MHD_YES ==
                MHD_add_response_header (response,
                                         MHD_HTTP_HEADER_CONTENT_TYPE,
                                         "text/plain"));

  rparams = strchr (hc->full_url, '?');
  if (NULL == rparams)
    dst = GNUNET_strdup ("/webui/");
  else
    GNUNET_asprintf (&dst,
                     "/webui/%s",
                     rparams);

  if (MHD_NO ==
      MHD_add_response_header (response,
                               MHD_HTTP_HEADER_LOCATION,
                               dst))
  {
    GNUNET_break (0);
    MHD_destroy_response (response);
    GNUNET_free (dst);
    return MHD_NO;
  }

  {
    MHD_RESULT ret;

    ret = MHD_queue_response (hc->connection,
                              MHD_HTTP_FOUND,
                              response);
    MHD_destroy_response (response);
    GNUNET_free (dst);
    return ret;
  }
}
