/*  GUBI - Gtk+ User Interface Builder
 *  Copyright (C) 1997	Tim Janik	<timj@psynet.net>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/
#include	"rcs.h"
RCS_ID("$Id: wdtree.c,v 1.12 1997/08/19 01:35:26 timj Exp $")


#define		__wdtree_c__

#include	"wdtree.h"
#include	"gsi.h"
#include	"gtkext.h"
#include	"treelist.h"
#include	"browser.h"
#include	"editor.h"
#include	"widdata.h"
#include	"defines.h"


/* --- external global variables --- */
GList		*UD_tree_list=NULL;



/* --- prototypes --- */
static	void	widget_data_parent_add		(gb_wdat_base_S	*Parent,
						 gb_wdat_base_S	*widget_data,
						 guint		position);
static	void	tree_widget_data_insert_foreach	(gpointer	data,
						 gpointer	user_data);
static	void	tree_widget_data_unlink_foreach	(gpointer	data,
						 gpointer	user_data);
static	void	SigH_tree_window_destroy	(GtkWidget	*window);

static	gb_wdat_base_S*
		tree_widget_data_find_prev	(gb_wdat_base_S	*widget_data);
static	gint	current_widget_timeout		(gpointer	data);



/* --- variables --- */
static	gboolean	wdtree_initialized=FALSE;
static	gb_wdat_base_S	*gb_current_widget_data=NULL;
static	tree_S		*current_tree=NULL;


/* --- functions --- */
void
trees_initialize	(void)
{
	wdtree_initialized=TRUE;
}

tree_S*
tree_new	(gboolean	user_window,
		 gboolean	visible)
{
	register tree_S	*tree;
	
	g_assert(user_window || !visible);
	
	
	/* create and fill in current tree_S
	*/
	tree=g_new(tree_S, 1);
	tree->type=GB_STRUCT_NONE;
	tree->visible=visible;
	tree->auto_resize=TRUE;
	tree->width=-1;
	tree->height=-1;
	tree->current=NULL;
	tree->widget_data_list=NULL;
	
	
	/* append to global UD_tree_list
	*/
	if (user_window)
		UD_tree_list=g_list_append(UD_tree_list, tree);
	
	return tree;
}


void
tree_delete	(tree_S	*tree)
{
	g_assert(tree);
	
	if (g_list_length(tree->widget_data_list)) {
		register gb_wdat_base_S	*window_data;
		
		window_data=tree->widget_data_list->data;
		g_assert(GB_IS_WIDDAT_WINDOW(window_data));
		
		GUBI_DATA(window_data)->tree=NULL;
		
		if (GUBI_DATA(window_data)->editor)
			gtk_widget_destroy(GUBI_DATA(window_data)->editor);
		if (GUBI_DATA(window_data)->browser)
			gtk_widget_destroy(GUBI_DATA(window_data)->browser);
		if (window_data->widget)
			gtk_widget_destroy(window_data->widget);
		
		widget_data_unregister_symbol_name(window_data);
		
		widget_data_delete_R(window_data);
	}

	UD_tree_list=g_list_remove(UD_tree_list, tree);
	
	g_free(tree);
}


void
widget_data_parent_add	(gb_wdat_base_S	*Parent,
			 gb_wdat_base_S	*widget_data,
			 guint		position)
{
	g_assert(GB_IS_WIDDAT_CONTAINER(Parent));
	g_assert(GB_IS_WIDDAT(widget_data));
	g_assert(!widget_data->parent);
	g_assert(!GB_IS_WIDDAT_WINDOW(widget_data));
	g_assert(!(GUBI_DATA(widget_data)->category & WID_CAT_AUXILLARY));
	
	if (HAS_MAX_CHILD_COUNT(Parent))
		g_error("%s\n add <%s> to <%s>: max_child_count (%d) reached: child_count=%d",
			FUNCNAME,
			widget_data_get_symbol_name(widget_data),
			widget_data_get_symbol_name(Parent),
			GUBI_DATA(Parent)->max_child_count,
			GUBI_DATA(Parent)->child_count);
	
	position=CLAMP(position, 0, GUBI_DATA(Parent)->max_child_count);
	
	
	/* add to parent
	*/
	widget_data->parent=Parent;
	GUBI_DATA(Parent)->children=g_list_insert(GUBI_DATA(Parent)->children,
						  widget_data,
						  position);
	
	UPDATE_CHILD_COUNT(Parent);
}


void
tree_widget_data_insert_foreach	(gpointer	data,
				 gpointer	user_data)
{
	register tree_S		*tree;
	register gb_wdat_base_S	*widget_data;
	register gb_wdat_base_S	*prev_data;
	register GList		*list;
	register GList		*new_list;
	
	tree=user_data;
	widget_data=data;
	
	g_assert(tree);
	g_assert(GB_IS_WIDDAT(widget_data));
	g_assert(!GB_IS_WIDDAT_WINDOW(widget_data));
	g_assert(GB_IS_WIDDAT_CONTAINER(widget_data->parent));
	g_assert(!widget_data->next);
	g_assert(!GUBI_DATA(widget_data)->tree);
	

	/* append to list
	*/
	GUBI_DATA(widget_data)->tree=tree;
	prev_data=tree_widget_data_find_prev(widget_data);
	g_assert(GB_IS_WIDDAT(prev_data));
	widget_data->next=prev_data->next;
	prev_data->next=widget_data;
	list=g_list_find(tree->widget_data_list, prev_data);
	g_assert(list);
	new_list=g_list_alloc();
	new_list->data=widget_data;
	new_list->next=list->next;
	if (list->next)
		list->next->prev=new_list;
	new_list->prev=list;
	list->next=new_list;
	
	
	/* register our symbol_name
	*/
	g_assert(!widget_data_lookup(widget_data_get_symbol_name(widget_data)));
	widget_data_register_symbol_name(widget_data);
	
	
	/* link children to end of list
	*/
	g_list_foreach(GUBI_DATA(widget_data)->children,
			tree_widget_data_insert_foreach,
			tree);
}


void
tree_insert_widget_data_R	(tree_S		*tree,
				 gb_wdat_base_S	*parent_data,
				 gb_wdat_base_S	*widget_data,
				 guint	position)
{
	g_assert(tree);
	g_assert(GB_IS_WIDDAT(widget_data));
	g_assert( (GB_IS_WIDDAT_WINDOW(widget_data) && !parent_data) || 
		  (!GB_IS_WIDDAT_WINDOW(widget_data) && GB_IS_WIDDAT_CONTAINER(parent_data)) );
	g_assert(!widget_data->next);
	g_assert(!GUBI_DATA(widget_data)->tree);
	g_assert(!GUBI_DATA(widget_data)->editor);
	g_assert(!GUBI_DATA(widget_data)->browser);
	g_assert(!(GUBI_DATA(widget_data)->category & WID_CAT_AUXILLARY));
	
	if (GB_TYPE_IS_WIDDAT_WINDOW(widget_data->type)) {
		
		g_assert(!widget_data->parent);
		g_assert(!tree->widget_data_list);
		g_assert(!GUBI_DATA(widget_data)->children);
		
		
		/* link window into tree_S
		*/
		GUBI_DATA(widget_data)->tree=tree;
		tree->widget_data_list=g_list_append(tree->widget_data_list, widget_data);
		
		
		/* register our symbol_name
		*/
		g_assert(!widget_data_lookup(widget_data_get_symbol_name(widget_data)));
		widget_data_register_symbol_name(widget_data);
	
	} else if (!HAS_MAX_CHILD_COUNT(parent_data)) {
		
		g_assert(tree->widget_data_list);
		
		widget_data_parent_add(parent_data, widget_data, position);
		
		
		/* link us to end of list
		 * including children
		*/
		tree_widget_data_insert_foreach(widget_data, tree);
		
		
		/* build widget if parent exists
		*/
		if (widget_data->parent->widget) {
			gb_widget_create(widget_data);
			g_list_foreach(GUBI_DATA(widget_data)->children,
				       (GFunc)gb_widget_create,
				       NULL);
		}
	} else {
		g_error("%s\nmaximum child count reached in `%s'",
			FUNCNAME,
			widget_data_get_symbol_name(parent_data));
	}
}


void
tree_widget_data_unlink_foreach	(gpointer	data,
				 gpointer	user_data)
{
	register tree_S		*tree;
	register gb_wdat_base_S	*widget_data;	/* recursion! */
	register gb_wdat_base_S	*temp_data;
	register GList		*list;
	
	
	widget_data=data;
	tree=GUBI_DATA(widget_data)->tree;
	
	g_assert(tree);
	g_assert(GB_IS_WIDDAT(widget_data));
	g_assert(GUBI_DATA(widget_data)->tree);
	g_assert(!GB_IS_WIDDAT_WINDOW(widget_data));
	
	
	/* delete references from other widgets in widget data
	*/
	while (GUBI_DATA(widget_data)->link_refs) {
		register gb_wdat_base_S	*reference_data;
		
		reference_data=GUBI_DATA(widget_data)->link_refs->data;
		g_assert(GB_IS_WIDDAT(reference_data));
		
		gsi_struct_unlink((gsi_base_S*)widget_data, (gsi_base_S*)reference_data);
		
		if (GUBI_DATA(reference_data)->editor)
			editor_refresh_window(reference_data);
	}
	
	
	/* if we have an editor/browser then destroy it
	*/
	if (GUBI_DATA(widget_data)->editor)
		gtk_widget_destroy(GUBI_DATA(widget_data)->editor);
	if (GUBI_DATA(widget_data)->browser)
		gtk_widget_destroy(GUBI_DATA(widget_data)->browser);
	
	
	/* unregister our symbol_name
	*/
	widget_data_unregister_symbol_name(widget_data);
	
	
	/* unlink from tree
	*/
	GUBI_DATA(widget_data)->tree=NULL;
	list=g_list_find(tree->widget_data_list, widget_data);
	g_assert(list && list->prev);
	if (!widget_data->next) {
		g_assert(!list->next);
	} else {
		g_assert(list->next);
		g_assert(widget_data->next==list->next->data);
	}
	temp_data=list->prev->data;
	g_assert(temp_data->next==widget_data);
	if (list->next) {
		temp_data->next=list->next->data;
	} else
		temp_data->next=NULL;
	widget_data->next=NULL;
	tree->widget_data_list=g_list_remove(tree->widget_data_list, widget_data);
	
	
	/* unlink children from tree
	*/
	g_list_foreach(GUBI_DATA(widget_data)->children,
			tree_widget_data_unlink_foreach,
			NULL);
}


void
tree_unlink_widget_data_R	(tree_S		*tree,
				 gb_wdat_base_S	*widget_data)
{
	g_assert(tree);
	g_assert(GB_IS_WIDDAT(widget_data));
	g_assert(!GB_IS_WIDDAT_WINDOW(widget_data));
	g_assert(GUBI_DATA(widget_data)->tree==tree);
	g_assert(GB_IS_WIDDAT_CONTAINER(widget_data->parent));
	g_assert(!(GUBI_DATA(widget_data)->category & WID_CAT_AUXILLARY));
	
	
	/* destroy GtkWidget
	 * (gtk destroys our children widgets as well!)
	*/
	if (widget_data->widget) {
		gtk_widget_hide(widget_data->widget);
	}
	/* if (widget_data->parent->widget && widget_data->widget) {
	 *	gtk_container_remove(GTK_CONTAINER(widget_data->parent->widget), widget_data->widget);
	 * }
	 * FIXME: this fails on a GtkFileSelection
	*/
	if (widget_data->widget)
		gtk_widget_destroy(widget_data->widget);
	
	
	/* unlink from tree recursively
	*/
	tree_widget_data_unlink_foreach(widget_data, NULL);
	
	
	/* unlink from parent
	*/
	GUBI_DATA(widget_data->parent)->children=g_list_remove(
		GUBI_DATA(widget_data->parent)->children,
		widget_data);

	UPDATE_CHILD_COUNT(widget_data->parent);
	widget_data->parent=NULL;
}


void
tree_rebuild	(tree_S	*tree)
{
	register gb_wdat_window_S	*window_data;

	g_return_if_fail(wdtree_initialized);
	
	g_assert(tree);
	g_assert(tree->widget_data_list);
	window_data=GB_wCAST(window, tree->widget_data_list->data);
	g_assert(GB_IS_WIDDAT_WINDOW(window_data));
	
	
	/* destroy old window if we have a GtkWidget
	*/
	if (window_data->widget && GTK_WIDGET_VISIBLE(window_data->widget)) {
		tree->width=window_data->widget->allocation.width;
		tree->height=window_data->widget->allocation.height;
	}
	if (window_data->widget) {
		register tree_S		*save_tree;
		
		save_tree=GUBI_DATA(window_data)->tree;
		GUBI_DATA(window_data)->tree=NULL;
		gtk_widget_destroy(window_data->widget);
		GUBI_DATA(window_data)->tree=save_tree;
	}
	
	
	/* build window and its widget tree
	*/
	gb_window_build(window_data);
}


void
tree_show	(tree_S	*tree)
{
	register gb_wdat_window_S	*window_data;

	if (!wdtree_initialized)
		return;
	
	g_assert(tree);
	g_assert(tree->widget_data_list);
	window_data=GB_wCAST(window, tree->widget_data_list->data);
	g_assert(GB_IS_WIDDAT_WINDOW(window_data));
	
	/* rebuild if we haven't a GtkWidget
	*/
	if (!window_data->widget)
		tree_rebuild(tree);
	else
		gtk_widget_hide(window_data->widget);
	
	
	/* we need an apropriate postion override!
	*/
	gtk_window_position(GTK_WINDOW(window_data->widget), GTK_WIN_POS_NONE);
	if (window_data->x<0 || window_data->y<0)
		gtk_widget_set_uposition(window_data->widget, 1, 1);
	
	
	/* we need an apropriate size override!
	*/
	if (tree->auto_resize) {
		tree->width=-1;
		tree->height=-1;
	} else if (tree->width>=0 && tree->height>=0) {
		gtk_widget_set_usize(window_data->widget,
				     tree->width,
				     tree->height);
		gtk_window_set_policy(GTK_WINDOW(window_data->widget),
				      TRUE,
				      TRUE,
				      (window_data->resize_policy&GB_AUTO_SHRINK)!=0);
	}
	
	
	/* show if visible	gdk_window_get_origin
	*/
	if (tree->visible) {
		gtk_signal_connect_object(GTK_OBJECT(window_data->widget),
					  "destroy",
					  GTK_SIGNAL_FUNC(SigH_tree_window_destroy),
					  GTK_OBJECT(window_data->widget));
		gtk_widget_show(window_data->widget);
	}
}


void
tree_set_hints		(gboolean	flash_current)
{
	static	 gint			current_widget_timer=0;
	
	
	/* let the current widget blink?
	*/
	if (!current_widget_timer && flash_current)
		current_widget_timer=gtk_timeout_add(SHOW_CURRENT_DELAY, current_widget_timeout, NULL);
	if (current_widget_timer && !flash_current) {
		gtk_timeout_remove(current_widget_timer);
		current_widget_timer=0;
	}
	if (gb_current_widget_data &&
	    gb_current_widget_data->widget &&
	    GTK_WIDGET_DRAWABLE(gb_current_widget_data->widget) &&
	    gb_current_widget_data->widget->window)
		_gdk_window_clear_expose(gb_current_widget_data->widget->window);
}


void
SigH_tree_window_destroy(GtkWidget	*window)
{
	register gubi_data_S	*gubi_data;
	
	g_assert(gubi_data=gtk_object_get_user_data(GTK_OBJECT(window)));
	g_assert(GB_IS_WIDDAT_WINDOW(gubi_data->widget_data));
	
	if (gubi_data->tree) {
		gubi_data->tree->visible=FALSE;
		treelist_hints_update(gubi_data->tree);
	}
}


gb_wdat_base_S*
tree_widget_data_find_prev	(gb_wdat_base_S	*widget_data)
{
	register GList		*list;
	
	g_assert(GB_IS_WIDDAT(widget_data));
	
	
	/* we return NULL if we don't have a parent
	*/
	if (!widget_data->parent)
		return NULL;
	
	
	/* find our previous sibling
	*/
	list=GUBI_DATA(widget_data->parent)->children;
	list=g_list_find(list, widget_data);
	g_assert(list);
	
	list=list->prev;
	
	
	/* if there is no previous sibling, our parent is
	 * the previous widget
	*/
	if (!list)
		return widget_data->parent;
	
	
	/* we found a sibling, lets get the last child in it's tree
	*/
	widget_data=widget_data_last_child(list->data);
	
	return widget_data;
}


gb_wdat_base_S*
widget_data_last_child	(gb_wdat_base_S	*widget_data)
{
	g_assert(GB_IS_WIDDAT(widget_data));
	
	while (GUBI_DATA(widget_data)->children)
		widget_data=g_list_last(GUBI_DATA(widget_data)->children)->data;
	
	return widget_data;
}


tree_S*
tree_get_current	(void)
{
	return current_tree;
}


gb_wdat_window_S*
window_data_get_current	(void)
{
	if (current_tree) {
		register gb_wdat_window_S       *window_data;
		
		g_assert(current_tree->widget_data_list);
		window_data=current_tree->widget_data_list->data;
		g_assert(GB_IS_WIDDAT_WINDOW(window_data));
		
		return window_data;
	} else
		return NULL;
}


void
tree_set_current	(tree_S		*tree)
{
	gb_current_widget_data=NULL;

	if (tree!=current_tree) {
		if (current_tree) {
			browser_select_current_tree(current_tree, FALSE);
			if (current_tree &&
			    current_tree->current &&
			    current_tree->current->widget &&
	/* FIXME: see gtk_object_check_cast */ current_tree->current->widget->object.klass &&
			    GTK_WIDGET_DRAWABLE(current_tree->current->widget) &&
			    current_tree->current->widget->window)
				_gdk_window_clear_expose(current_tree->current->widget->window);
		}
		if (tree)
			browser_select_current_tree(tree, TRUE);
	}
	current_tree=tree;

	if (current_tree)
		gb_current_widget_data=current_tree->current;
}


void
tree_set_current_widget_data	(tree_S		*tree,
				 gb_wdat_base_S	*widget_data)
{
	g_assert(tree);
	
	if (widget_data) {
		g_assert(GB_IS_WIDDAT(widget_data));
		g_assert(g_list_find(tree->widget_data_list, widget_data));
	}
	
	if (tree->current==gb_current_widget_data) {
		gb_current_widget_data=NULL;
		
		if (tree->current &&
		    tree->current->widget &&
		    GTK_WIDGET_DRAWABLE(tree->current->widget) &&
		    tree->current->widget->window)
			_gdk_window_clear_expose(tree->current->widget->window);
	}
	
	tree->current=widget_data;
	if (tree==current_tree)
		gb_current_widget_data=widget_data;
}


gint
current_widget_timeout	(gpointer	data)
{
	static	 gboolean	current_toggle=FALSE;
	register gb_wdat_base_S	*widget_data=gb_current_widget_data;
	register GdkGC		*GC=NULL;
	register GtkWidget	*widget;
	register GdkWindow	*window;
	
	if (!widget_data || !widget_data->widget) {
		current_toggle=FALSE;
		return TRUE;
	}
	
	switch (widget_data->type) {
	
	case	GB_WIDGET_BUTTON:
	case	GB_WIDGET_TOGGLE_BUTTON:
	case	GB_WIDGET_CHECK_BUTTON:
	case	GB_WIDGET_RADIO_BUTTON:
		widget=widget_data->widget;
		break;
	
	default:
		widget=widget_data->widget;
		break;
	}
	
	if ((current_toggle^=TRUE))
		GC=gtk_widget_get_default_style()->black_gc;
	else
		GC=gtk_widget_get_default_style()->white_gc;
	
	if (!widget || !GTK_WIDGET_DRAWABLE(widget)) {
		current_toggle=FALSE;
		return TRUE;
	}
	
	if (GTK_IS_BUTTON(widget))
		window=widget->parent->window;
	else
		window=widget->window;
	
	gdk_draw_rectangle	(window,
				 GC,
				 FALSE,
				 widget->allocation.x+1,
				 widget->allocation.y+1,
				 widget->allocation.width-1,
				 widget->allocation.height-1);
	gdk_draw_rectangle	(window,
				 GC,
				 FALSE,
				 widget->allocation.x,
				 widget->allocation.y,
				 widget->allocation.width,
				 widget->allocation.height);
	gdk_draw_line	(window,
			 GC,
			 widget->allocation.x,
			 widget->allocation.y,
			 widget->allocation.x +
			   widget->allocation.width,
			 widget->allocation.y +
			   widget->allocation.height);
	gdk_draw_line	(window,
			 GC,
			 widget->allocation.x +
			   widget->allocation.width,
			 widget->allocation.y,
			 widget->allocation.x,
			 widget->allocation.y +
			   widget->allocation.height);
	
	return TRUE;
}
