/* Low level interface to ptrace, for GDB when running under Unix.
   Copyright (C) 1988 Free Software Foundation, Inc.

GDB is distributed in the hope that it will be useful, but WITHOUT ANY
WARRANTY.  No author or distributor accepts responsibility to anyone
for the consequences of using it or for whether it serves any
particular purpose or works at all, unless he says so in writing.
Refer to the GDB General Public License for full details.

Everyone is granted permission to copy, modify and redistribute GDB,
but only under the conditions described in the GDB General Public
License.  A copy of this license is supposed to have been given to you
along with GDB so you can know your rights and responsibilities.  It
should be in a file named COPYING.  Among other things, the copyright
notice and this notice must be preserved on all copies.

In other words, go ahead and share GDB, but don't try to stop
anyone else from sharing it farther.  Help stamp out software hoarding!
*/

#include "defs.h"
#include "param.h"
#include "frame.h"
#include "inferior.h"

#ifdef USG
#include <sys/types.h>
#endif

#include <stdio.h>
#include <sys/param.h>
#include <sys/dir.h>
#include <signal.h>
#include <sys/user.h>
#include <sys/ioctl.h>
#include <fcntl.h>

#ifdef UnicomPBB
#include <sys/sysmacros.h>
#include <sys/kdump.h>
union u_khead {
	char fill[KDHSIZE];
	struct khead kh;
} u_khead;
static int current_process = 0;

#define khead u_khead.kh

#endif

#ifdef COFF_FORMAT
# ifdef COFF_ENCAPSULATE
#   include "a.out.encap.h"
# else
#   include "coff-dbx.h"
#   include "stab.gnu.h"
# endif
int
printprocess();
caddr_t
Funct_Address ();
int 
find_kframe_start();
int
xfer_kcore_file ();
static void process_info();
static void process_command();
int
read_file_hdr ();
int
read_section_hdr ();


#else
# include <a.out.h>
#endif

#ifndef READ_DBX_FORMAT
# ifndef N_SET_MAGIC
#  define N_SET_MAGIC(exec, val) ((exec).a_magic = (val))
# endif
#endif

#include <sys/file.h>
#include <sys/stat.h>
#include "value.h"
#include "expression.h"

extern int errno;

/* This function simply calls ptrace with the given arguments.	
   It exists so that all calls to ptrace are isolated in this 
   machine-dependent file. */
int
call_ptrace (request, pid, arg3, arg4)
     int request, pid, arg3, arg4;
{
  return ptrace (request, pid, arg3, arg4);
}

kill_inferior ()
{
  if (remote_debugging)
    return;
  if (inferior_pid == 0)
    return;
  ptrace (8, inferior_pid, 0, 0);
  wait (0);
  inferior_died ();
}

/* This is used when GDB is exiting.  It gives less chance of error.*/

kill_inferior_fast ()
{
  if (remote_debugging)
    return;
  if (inferior_pid == 0)
    return;
  ptrace (8, inferior_pid, 0, 0);
  wait (0);
}

/* Resume execution of the inferior process.
   If STEP is nonzero, single-step it.
   If SIGNAL is nonzero, give it that signal.  */

void
resume (step, signal)
     int step;
     int signal;
{
  errno = 0;
  if (remote_debugging)
    remote_resume (step, signal);
  else
    {
      ptrace (step ? 9 : 7, inferior_pid, 1, signal);
      if (errno)
	perror_with_name ("ptrace");
    }
}

void
fetch_inferior_registers ()
{
  register int regno;
  register unsigned int regaddr;
  char buf[MAX_REGISTER_RAW_SIZE];
  register int i;

  struct user u;
  unsigned int offset = (char *) &u.u_ar0 - (char *) &u;
  offset = ptrace (3, inferior_pid, offset, 0) - KERNEL_U_ADDR;

  for (regno = 0; regno < NUM_REGS; regno++)
    {
      regaddr = register_addr (regno, offset);
      for (i = 0; i < REGISTER_RAW_SIZE (regno); i += sizeof (int))
	{
	  *(int *) &buf[i] = ptrace (3, inferior_pid, regaddr, 0);
	  regaddr += sizeof (int);
	}
      supply_register (regno, buf);
    }
}

/* Store our register values back into the inferior.
   If REGNO is -1, do this for all registers.
   Otherwise, REGNO specifies which register (so we can save time).  */

store_inferior_registers (regno)
     int regno;
{
  register unsigned int regaddr;
  char buf[80];

  struct user u;
  unsigned int offset = (char *) &u.u_ar0 - (char *) &u;
#ifdef	PT_CASE_6_ADDR_ONLY
  offset = ptrace (3, inferior_pid, offset, 0);
#else
  offset = ptrace (3, inferior_pid, offset, 0) - KERNEL_U_ADDR;
#endif	/* PT_CASE_6_ADDR_ONLY */

  if (regno >= 0)
    {
      regaddr = register_addr (regno, offset);
      errno = 0;
      ptrace (6, inferior_pid, regaddr, read_register (regno));
      if (errno != 0)
	{
	  sprintf (buf, "writing register number %d", regno);
	  perror_with_name (buf);
	}
    }
  else for (regno = 0; regno < NUM_REGS; regno++)
    {
      regaddr = register_addr (regno, offset);
      errno = 0;
      ptrace (6, inferior_pid, regaddr, read_register (regno));
      if (errno != 0)
	{
	  sprintf (buf, "writing register number %d", regno);
	  perror_with_name (buf);
	}
    }
}

/* Copy LEN bytes from inferior's memory starting at MEMADDR
   to debugger memory starting at MYADDR. 
   On failure (cannot read from inferior, usually because address is out
   of bounds) returns the value of errno. */

int
read_inferior_memory (memaddr, myaddr, len)
     CORE_ADDR memaddr;
     char *myaddr;
     int len;
{
  register int i;
  /* Round starting address down to longword boundary.	*/
  register CORE_ADDR addr = memaddr & - sizeof (int);
  /* Round ending address up; get number of longwords that makes.  */
  register int count
    = (((memaddr + len) - addr) + sizeof (int) - 1) / sizeof (int);
  /* Allocate buffer of that many longwords.  */
  register int *buffer = (int *) alloca (count * sizeof (int));
  extern int errno;

  /* Read all the longwords */
  for (i = 0; i < count; i++, addr += sizeof (int))
    {
      errno = 0;
      if (remote_debugging)
	buffer[i] = remote_fetch_word (addr);
      else
	buffer[i] = ptrace (1, inferior_pid, addr, 0);
      if (errno)
	return errno;
    }

  /* Copy appropriate bytes out of the buffer.	*/
  bcopy ((char *) buffer + (memaddr & (sizeof (int) - 1)), myaddr, len);
  return 0;
}

/* Copy LEN bytes of data from debugger memory at MYADDR
   to inferior's memory at MEMADDR.
   On failure (cannot write the inferior)
   returns the value of errno.	*/

int
write_inferior_memory (memaddr, myaddr, len)
     CORE_ADDR memaddr;
     char *myaddr;
     int len;
{
  register int i;
  /* Round starting address down to longword boundary.	*/
  register CORE_ADDR addr = memaddr & - sizeof (int);
  /* Round ending address up; get number of longwords that makes.  */
  register int count
    = (((memaddr + len) - addr) + sizeof (int) - 1) / sizeof (int);
  /* Allocate buffer of that many longwords.  */
  register int *buffer = (int *) alloca (count * sizeof (int));
  extern int errno;

#ifdef	PT_CASES_4_AND_5_DIFF
  int	pt_write;				/* 1st arg to "ptrace()" */
  extern	CORE_ADDR	text_start;	/* "core.c" */
  extern	CORE_ADDR	text_end;	/* "core.c" */
#endif	/* PT_CASES_4_AND_5_DIFF */

  /* Fill start and end extra bytes of buffer with existing memory data.  */

  if (remote_debugging)
    buffer[0] = remote_fetch_word (addr);
  else
    buffer[0] = ptrace (1, inferior_pid, addr, 0);

  if (count > 1)
    {
      if (remote_debugging)
	buffer[count - 1]
	  = remote_fetch_word (addr + (count - 1) * sizeof (int));
      else
	buffer[count - 1]
	  = ptrace (1, inferior_pid,
		    addr + (count - 1) * sizeof (int), 0);
    }

  /* Copy data to be written over corresponding part of buffer */

  bcopy (myaddr, (char *) buffer + (memaddr & (sizeof (int) - 1)), len);

  /* Write the entire buffer.  */

#ifdef	PT_CASES_4_AND_5_DIFF
  pt_write = addr >= text_start && addr < text_end ? 4 : 5 ;
#endif	/* PT_CASES_4_AND_5_DIFF */

  for (i = 0; i < count; i++, addr += sizeof (int))
    {
      errno = 0;
      if (remote_debugging)
	remote_store_word (addr, buffer[i]);
      else
#ifdef	PT_CASES_4_AND_5_DIFF
	ptrace (pt_write, inferior_pid, addr, buffer[i]);
#else
	ptrace (4, inferior_pid, addr, buffer[i]);
#endif	/* PT_CASES_4_AND_5_DIFF */
      if (errno)
	return errno;
    }

  return 0;
}

/* Work with core dump and executable files, for GDB. 
   This code would be in core.c if it weren't machine-dependent. */

/* Recognize COFF format systems because a.out.h defines AOUTHDR.  */
#ifdef AOUTHDR
#define COFF_FORMAT
#endif

#ifndef N_TXTADDR
#define N_TXTADDR(hdr) 0
#endif /* no N_TXTADDR */

#ifndef N_DATADDR
#define N_DATADDR(hdr) hdr.a_text
#endif /* no N_DATADDR */

/* Make COFF and non-COFF names for things a little more compatible
   to reduce conditionals later.  */

#ifndef sun386
#ifdef COFF_FORMAT
#define a_magic magic
#endif
#endif

#ifndef COFF_FORMAT
#define AOUTHDR struct exec
#endif


extern char *sys_siglist[];


/* Hook for `exec_file_command' command to call.  */

extern void (*exec_file_display_hook) ();
   
/* File names of core file and executable file.	 */

extern char *corefile;
extern char *execfile;

/* Descriptors on which core file and executable file are open.
   Note that the execchan is closed when an inferior is created
   and reopened if the inferior dies or is killed.  */

extern int corechan;
extern int execchan;

/* Last modification time of executable file.
   Also used in source.c to compare against mtime of a source file.  */

extern int exec_mtime;

/* Virtual addresses of bounds of the two areas of memory in the core file.  */

extern CORE_ADDR data_start;
extern CORE_ADDR data_end;
extern CORE_ADDR stack_start;
extern CORE_ADDR stack_end;

/* Virtual addresses of bounds of two areas of memory in the exec file.
   Note that the data area in the exec file is used only when there is no core file.  */

extern CORE_ADDR text_start;
extern CORE_ADDR text_end;

extern CORE_ADDR exec_data_start;
extern CORE_ADDR exec_data_end;

/* Address in executable file of start of text area data.  */

extern int text_offset;

/* Address in executable file of start of data area data.  */

extern int exec_data_offset;

/* Address in core file of start of data area data.  */

extern int data_offset;

/* Address in core file of start of stack area data.  */

extern int stack_offset;

#ifdef COFF_FORMAT
/* various coff data structures */

extern FILHDR file_hdr;
extern SCNHDR text_hdr;
extern SCNHDR data_hdr;

#endif /* not COFF_FORMAT */

/* a.out header saved in core file.  */
  
extern AOUTHDR core_aouthdr;

/* a.out header of exec file.  */

extern AOUTHDR exec_aouthdr;

extern void validate_files ();

#ifdef UnicomPBB
 int kernel_dump;
#endif

int dev_kmem;

core_file_command (filename, from_tty)
     char *filename;
     int from_tty;
{
  int val;
  unsigned int core_magic ;
  extern char registers[];

  /* Discard all vestiges of any previous core file
     and mark data and stack spaces as empty.  */

  if (corefile)
    free (corefile);
  corefile = 0;

  if (corechan >= 0)
    close (corechan);
  corechan = -1;

  data_start = 0;
  data_end = 0;
  stack_start = STACK_END_ADDR;
  stack_end = STACK_END_ADDR;

  /* Now, if a new core file was specified, open it and digest it.  */

  if (filename)
    {
      filename = tilde_expand (filename);
      make_cleanup (free, filename);

      dev_kmem = ! strcmp(filename,"/dev/kmem");

      if (have_inferior_p ())
	error ("To look at a core file, you must kill the inferior with \"kill\".");
      corechan = open (filename, O_RDONLY, 0);
      if (corechan < 0)
	perror_with_name (filename);

      core_magic = 0;
      myread(corechan,&core_magic,sizeof(core_magic));
      lseek(corechan,0,0);

      kernel_dump = ( 0xbcdefabc == core_magic);
      
      if (kernel_dump)
	setup_kernel_core(filename,-1);
      else if (! dev_kmem)
      {	     /* 4.2-style (and perhaps also sysV-style) core dump file.	 */
	struct user u;

	int reg_offset;

	val = myread (corechan, &u, sizeof u);
	if (val < 0)
	  perror_with_name (filename);
	data_start = exec_data_start;
#ifdef UnicomPBB
	data_end = data_start + ctob(u.u_dsize);
	stack_start = stack_end - ctob(u.u_ssize);
	data_offset = ctob(USIZE);
	stack_offset = data_offset + ctob(u.u_dsize);
#else
	data_end = data_start + NBPG * u.u_dsize;
	stack_start = stack_end - NBPG * u.u_ssize;
	data_offset = NBPG * UPAGES;
	stack_offset = NBPG * (UPAGES + u.u_dsize);
#endif
	reg_offset = (int) u.u_ar0 - KERNEL_U_ADDR;

	/* I don't know where to find this info.
	   So, for now, mark it as not available.  */
	N_SET_MAGIC (core_aouthdr, 0);

	/* Read the register values out of the core file and store
	   them where `read_register' will find them.  */

	{
	  register int regno;

	  for (regno = 0; regno < NUM_REGS; regno++)
	    {
	      char buf[MAX_REGISTER_RAW_SIZE];

	      val = lseek (corechan, register_addr (regno, reg_offset), 0);
	      if (val < 0)
		perror_with_name (filename);

	      val = myread (corechan, buf, sizeof buf);
	      if (val < 0)
		perror_with_name (filename);
	      supply_register (regno, buf);
	    }
	}
      }
      if (filename[0] == '/')
	corefile = savestring (filename, strlen (filename));
      else
	{
	  corefile = concat (current_directory, "/", filename);
	}
      if (!dev_kmem) 
      {
	      set_current_frame ( create_new_frame (read_register (FP_REGNUM),
					    read_pc ()));
	      select_frame (get_current_frame (), 0);
	      validate_files ();
      }
    }
  else if (from_tty)
    printf ("No core file now.\n");
}

struct mem_map
{
	int type;
	unsigned long addr;
	int next_valid;

};

#define MEM_NOTFOUND 0
#define MEM_GAP 1
#define MEM_STACK    2
#define			STACK_IND 1
#define MEM_SYSREG   3
#define			SYSSEG_IND 3
#define MEM_TEXT     4
#define			TEXT_IND 5
#define MEM_DATA     5
#define			DATA_IND 7
#define MEM_END	     6
#define			END_IND	 8

#define MEM_KMEM     9
struct mem_map mem_map[] = 
{
/*0*/	{ MEM_GAP,	0,	  1 },
/*1*/	{ MEM_STACK,	0x10000,  0 }, /* actually stack and ublock */
/*2*/	{ MEM_GAP,	0x11800,  3 },
/*3*/	{ MEM_SYSREG,	0x80000,  0 },
/*4*/	{ MEM_GAP,	0x100000, 5 },
/*5*/	{ MEM_TEXT,	0x0,	  0},	/* filled in from dump header */
/*6*/	{ MEM_GAP,	0x0,	  7},	/* filled in from dump header */
/*7*/	{ MEM_DATA,	0x0,	  0 },	/* filled in from dump header */
/*8*/	{ MEM_END,	0x0,	  0 },
	{	0}
};

#ifdef OLD
extern CORE_ADDR first_object_file_end;	/* From blockframe.c */
#else
extern CORE_ADDR startup_file_start;	/* From blockframe.c */
extern CORE_ADDR startup_file_end;	/* From blockframe.c */
#endif

setup_kernel_core(filename,pid)
char * filename;
{
  int i,j;
  int val;
  extern char registers[];
  int ub_offset;

  if (pid < 0)
    {
      lseek(corechan,0,0);
      read(corechan,&u_khead,sizeof(u_khead));
      pid = khead.cpid;
    }
  for (i = 0;i< khead.ubcount;i++)
    {
      if (khead.upid[i] == pid)
	{
	  j = i;
	  break;
	}  
    }
  if (i==khead.ubcount)
    error("process not found!");
  current_process=pid;

  lseek(corechan,ub_offset = KDHSIZE+(khead.sysregsize+j*3)*2048,0);

  {
    static union {
      struct  user u_u;
      char u_s[3*2048];
    } u;
    int reg_offset;
    unsigned int sp,fp,pc;
    unsigned int * u_sp;

    val = myread (corechan, &u, sizeof u);
    if (val < 0)
      perror_with_name (filename);
    data_start = mem_map[DATA_IND].addr =  (CORE_ADDR) khead.skdata;

    data_end = mem_map[DATA_IND+1].addr = data_start + khead.kdsize*2048;

    data_offset = KDHSIZE+(khead.sysregsize+3*khead.ubcount)*2048;
    {
      caddr_t trap_addr,next_addr,db_addr;

      stack_end = mem_map[STACK_IND+1].addr =  KERNEL_U_ADDR +3*2048;
      stack_start = mem_map[STACK_IND].addr = KERNEL_U_ADDR ;
      stack_offset = ub_offset	/* + sizeof(struct user)*/;
      /*	  if (trap_addr == 0)*/
      trap_addr= Funct_Address ("&panic");
      next_addr= Funct_Address ("&prdev");
      db_addr  = Funct_Address ("&dbinit");
#ifdef OLD
      if (db_addr)		/* we have no crt0.o and we want to see locore! */
	first_object_file_end = db_addr; 
#else
      startup_file_start = 0;
      startup_file_end = 0;

#endif
      reg_offset = (int) u.u_u.u_ar0 - KERNEL_U_ADDR;

      /* I don't know where to find this info.
	 So, for now, mark it as not available.  */
      N_SET_MAGIC (core_aouthdr, 0);

      /* Read the register values out of the core file and store
	 them where `read_register' will find them.	 */

      {				/* TEMPORARY KLUGE- REGS NOT SAVED BY KDUMP!!! */
	register int regno;

	if (khead.cpid == pid)
	  {	 
	    for (regno = 0; regno < NUM_REGS; regno++)
	      {
		char buf[MAX_REGISTER_RAW_SIZE];
		val = lseek (corechan, register_addr (regno, 
						      ub_offset + reg_offset),
			     0);
		if (val < 0)
		  perror_with_name (filename);
		val = myread (corechan, buf, sizeof buf);
		if (val < 0)
		  perror_with_name (filename);

		supply_register (regno, buf);
	      }
	  }
	else
	  {	 
	    for (regno = 2; regno < NUM_REGS; regno++)
	      {
		int n;
		switch (regno)
		  {
		  case 8:	/* a0 */ /* not saved */
		  case PS_REGNUM:
		    continue;
		  case PC_REGNUM:

		    n = 6; break;
		  default:
		    if (regno <= 7)
		      n = regno-2; /*  d reg */
		    else
		      n = regno -3; /*  areg */
		  }
		supply_register (regno, &u.u_u.u_ssav[n]);
	      }
	  }
      }
    }
  }
  printprocess(j,pid);

}

printprocess(ind,pid)
{
  struct user u;
  int val;

  char def = ' ';
  char cur = ' ';
  if (pid == khead.cpid)
    def = 'D';
  if (pid == current_process)
    cur = 'C';

  lseek(corechan, KDHSIZE+(khead.sysregsize+ind*3)*2048,0);
  val = myread (corechan, &u, sizeof u);
  u.u_psargs[PSARGSZ] =0;
  printf("%5d %c%c %6x %-60s\n",pid,cur,def,u.u_procp,u.u_psargs);
}

  

exec_file_command (filename, from_tty)
     char *filename;
     int from_tty;
{
  int val;

  /* Eliminate all traces of old exec file.
     Mark text segment as empty.  */

  if (execfile)
    free (execfile);
  execfile = 0;
  data_start = 0;
  data_end -= exec_data_start;
  text_start = 0;
  text_end = 0;
  exec_data_start = 0;
  exec_data_end = 0;
  if (execchan >= 0)
    close (execchan);
  execchan = -1;

  /* Now open and digest the file the user requested, if any.  */

  if (filename)
    {
      filename = tilde_expand (filename);
      make_cleanup (free, filename);
      
      execchan = openp (getenv ("PATH"), 1, filename, O_RDONLY, 0,
			&execfile);
      if (execchan < 0)
	perror_with_name (filename);

#ifdef COFF_FORMAT
      {
	int aout_hdrsize;
	int num_sections;

	if (read_file_hdr (execchan, &file_hdr) < 0)
	  error ("\"%s\": not in executable format.", execfile);

	aout_hdrsize = file_hdr.f_opthdr;
	num_sections = file_hdr.f_nscns;

	if ( ! aout_hdrsize )
	  printf ("Warning:  no optional header--File may be relocatable.\n");

	if (read_section_hdr (execchan, _TEXT, &text_hdr, num_sections,
			      aout_hdrsize) < 0)
	  error ("\"%s\": can't read text section header", execfile);

	if (read_section_hdr (execchan, _DATA, &data_hdr, num_sections,
			      aout_hdrsize) < 0)
	  error ("\"%s\": can't read data section header", execfile);

	text_start = mem_map[TEXT_IND].addr = text_hdr.s_vaddr;
	text_end = mem_map[TEXT_IND+1].addr = text_start + text_hdr.s_size;
	text_offset = text_hdr.s_scnptr;
	exec_data_start = data_hdr.s_vaddr;
	exec_data_end = exec_data_start + data_hdr.s_size;
	exec_data_offset = data_hdr.s_scnptr;
	data_start = mem_map[DATA_IND].addr = exec_data_start;
	data_end += exec_data_start;
	mem_map[DATA_IND+1].addr = data_end;
	exec_mtime = file_hdr.f_timdat;
      }
#else				/* not COFF_FORMAT */
      {
	struct stat st_exec;

#ifdef HEADER_SEEK_FD
	HEADER_SEEK_FD (execchan);
#endif
	
	val = myread (execchan, &exec_aouthdr, sizeof (AOUTHDR));

	if (val < 0)
	  perror_with_name (filename);

	text_start = N_TXTADDR (exec_aouthdr);
	exec_data_start = N_DATADDR (exec_aouthdr);

	text_offset = N_TXTOFF (exec_aouthdr);
	exec_data_offset = N_TXTOFF (exec_aouthdr) + exec_aouthdr.a_text;

	text_end = text_start + exec_aouthdr.a_text;
	exec_data_end = exec_data_start + exec_aouthdr.a_data;
	data_start = exec_data_start;
	data_end += exec_data_start;

	fstat (execchan, &st_exec);
	exec_mtime = st_exec.st_mtime;
      }
#endif				/* not COFF_FORMAT */

      validate_files ();
    }
  else if (from_tty)
    printf ("No exec file now.\n");

  /* Tell display code (if any) about the changed file name.  */
  if (exec_file_display_hook)
    (*exec_file_display_hook) (filename);
}
caddr_t 
Funct_Address (exp)
     char *exp;
{
  struct expression *expr;
  register struct cleanup *old_chain = 0;
  register value val;
  int cleanup = 0;

  caddr_t retval = 0;

  if (exp && *exp)
    {
      expr = parse_c_expression (exp);
      old_chain = make_cleanup (free_current_contents, &expr);
      cleanup = 1;
      val = evaluate_expression (expr);
      if (val)
	retval = (caddr_t) val->contents[0];
    }

  if (cleanup)
    do_cleanups (old_chain);
  return retval;
}

int 
find_kframe_start(ks,func,next_func)
	unsigned int * ks,func,next_func;
{
  caddr_t sp = (caddr_t) KERNEL_U_ADDR+(sizeof(struct user) & ~3)+4;
  int offset = (int) ks-KERNEL_U_ADDR;

  while (sp< (caddr_t) KERNEL_U_ADDR+3*2048)
    {
      unsigned register val = * (int *) (sp+offset);

      if (val>= func&&val <next_func)
	return (int) sp-4;
      sp+=2;
    }
  return 0;
}



/* Return 0 if address could be read, 1 if not. */

int
xfer_kcore_file (memaddr, myaddr, len)
     CORE_ADDR memaddr;
     char *myaddr;
     int len;
{
  register int i;
  register int val;
  int xferchan;
  char **xferfile;
  int fileptr;
  int returnval = 0;
  int nelem = sizeof (mem_map)/sizeof (struct mem_map);
  int type;
  CORE_ADDR base_addr, end_addr;

  while (len > 0)
    {
      int seg;
      type = MEM_NOTFOUND;

      xferfile = 0;
      xferchan = 0;
      /* locate segment */
      if (dev_kmem)
	{
	  type = MEM_KMEM;
	}
      else
	for (seg = 0;seg < nelem;seg++)
	  {
	    if (memaddr >= mem_map[seg].addr
		&& memaddr < mem_map[seg+1].addr)
	      {
		type = mem_map[seg].type;
		base_addr = mem_map[seg].addr;
		end_addr	= mem_map[seg+1].addr;
		break;
	      }
	  }
      /* Determine which file the next bunch of addresses reside in,
	 and where in the file.	 Set the file's read/write pointer
	 to point at the proper place for the desired address
	 and set xferfile and xferchan for the correct file.
	 If desired address is nonexistent, leave them zero.
	 i is set to the number of bytes that can be handled
	 along with the next address.  */

      switch (type)
	{
	case MEM_KMEM:
	  i  = len;
	  fileptr = memaddr;
	  xferfile = &corefile;
	  xferchan = corechan;
	  break;
		
	case MEM_NOTFOUND:
	case MEM_END:
	  i = min (len, end_addr - memaddr);
	  break;
	case MEM_SYSREG:	/* not handled yet */
	case MEM_GAP:
	  i = min (len, end_addr - memaddr);
	  break;

	case MEM_TEXT:
	  i = min (len, end_addr - memaddr);
	  fileptr = memaddr - text_start + text_offset;
	  xferfile = &execfile;
	  xferchan = execchan;
	  break;
	case MEM_STACK:
	case MEM_DATA:
	  i = min (len, end_addr - memaddr);
	  if (corechan < 0)
	    {
	      if (type == MEM_DATA)
		{
		  fileptr = memaddr - base_addr + exec_data_offset;
		  xferfile = &execfile;
		  xferchan = execchan;
		}
	    }
	  else
	    {
	      if (type == MEM_STACK)
		fileptr = memaddr - base_addr + stack_offset;
	      else
		fileptr = memaddr - base_addr + data_offset;
	      xferfile = &corefile;
	      xferchan = corechan;
	    }
	}
      /* Now we know which file to use.
	 Set up its pointer and transfer the data.  */
      if (xferfile)
	{
	  if (*xferfile == 0)
	    if (xferfile == &execfile)
	      error ("No program file to examine.");
	    else
	      error ("No core dump file or running program to examine.");
	  val = lseek (xferchan, fileptr, 0);
	  if (val < 0)
	    perror_with_name (*xferfile);
	  val = myread (xferchan, myaddr, i);
	  if (val < 0)
	    perror_with_name (*xferfile);
	}
      /* If this address is for nonexistent memory,
	 read zeros if reading, or do nothing if writing.  */
      else
	{
	  bzero (myaddr, i);
	  returnval = 1;
	}

      memaddr += i;
      myaddr += i;
      len -= i;
    }
  return returnval;
}
static void process_info(arg)
     char * arg;
{
  int i;
  int pid = -1;
  if (!kernel_dump || have_inferior_p ())
    error("Not debugging a kernel.");

  if (arg&&*arg)
    pid = parse_and_eval_address (arg);

  for (i = 0;i< khead.ubcount;i++)
    {
      if (pid <0 || khead.upid[i] == pid)
	printprocess(i,khead.upid[i]);
    }
}
static void process_command(arg)
     char * arg;
{
  int pid = -1;
  if (!kernel_dump || have_inferior_p ())
    error("Not debugging a kernel.");

  if (!corefile)
    error("No core file!");

  if (arg&&*arg)
    pid = parse_and_eval_address (arg);

  setup_kernel_core(corefile,pid);

  set_current_frame ( create_new_frame (read_register (FP_REGNUM),
					read_pc ()));
  select_frame (get_current_frame (), 0);
  validate_files ();
}

_initialize_dep()
{
  add_info("process", process_info, "Info about processes.\n\
With argument PID, just displays that process.");
  add_com("process", class_support, process_command, "Select current process\n\
Specify argument PID. With no argument, reselects default process.");
}
