/* Auxilliary support for dbxread.c in GDB, for systems using stab
   debug symbols in coff.
   Copyright (C) 1989, 1990 Free Software Foundation, Inc.

This file is part of GDB.

GDB is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 1, or (at your option)
any later version.

GDB is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with GDB; see the file COPYING.  If not, write to
the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.  */

#include "defs.h"
#include "param.h"
#include <stdio.h>
#ifdef USG
#include <sys/types.h>
#endif
#include <sys/param.h>
#include <fcntl.h>
#include <sys/file.h>

#ifndef L_INCR
#define L_INCR 1
#endif

#include <obstack.h>

#include "coff-dbx.h"
#include <linenum.h>
#include "stab.gnu.h"

void init_coff_str();
void coffstr_free();
#define obstack_chunk_alloc xmalloc
#define obstack_chunk_free free

int text_scn = 0;
int data_scn = 0;
int bss_scn  = 0;

/* Support for line number handling */
static char *linetab = NULL;
static long linetab_offset;
static int linetab_count;

/* Support for symbol "normalization" */
extern int symnum_limit;
extern int conv_symnum; 
/*extern int symbuf_end;*/

static char *add_coff_string();

#if defined(sun386) || defined(sparc)
#ifndef SUN_STYLE_COFF
#define SUN_STYLE_COFF
#endif
#endif

#ifdef MACHDEP_SET_TEXT_ADDRESS
unsigned long * text_address_needed = 0;
#endif

#define ABS_SNUM  (-1)

#ifndef SYMBUFSIZE
#define SYMBUFSIZE 4096
#endif

#define SHOW_BAD_CONVERSION \
	    printf( \
		   "bad conversion, type: 0x%2x, sc: %d, sn: %d, val: 0x%8x, text = %d, data = %d, bss = %d\n", \
		   type, storage_class,section,p->n_value,text_scn,data_scn,bss_scn)  

/* decoding macros:
 IS_DEBUG_SYMBOL(sym, type) 
 	Yields non-zero if symbol sym is a debugging symbol. 
 TYPE_OF_DEBUG_SYMBOL(sym,type) 
 	Yields debug type of debugging symbol. Call only when IS_DEBUG_SYMBOL
	is true.
 HAS_DESC(sym)
 	Yields non-zero if debugging symbol sym has a usable desc. Call only
	when IS_DEBUG_SYMBOL is true.
 */

#ifdef SUN_STYLE_COFF
#define IS_DEBUG_SYMBOL(sym, type) (sym->n_zeroes == 0 && sym->n_dbx_type != 0)
#define	TYPE_OF_DEBUG_SYMBOL(sym,type)  (sym->n_dbx_type)
#define HAS_DESC(sym) (1)	/* always true for sun style IS_DEBUG_SYMBOL */
#else
#define IS_DEBUG_SYMBOL(sym, type) ( type && ( (type & 0xff00) == type))
#define	TYPE_OF_DEBUG_SYMBOL(sym,type)  (type>>8) 
#define HAS_DESC(sym) (sym->n_name[1] == 0)
#endif

#define MAX_SECTS 64

unsigned int section_to_type_map[MAX_SECTS] = { 0, N_ABS, N_EXT, 0 };
unsigned int *type_map = & section_to_type_map[-(N_DEBUG)];


/* Routines for reading headers and symbols from executable.  */

/* Read COFF file header, check magic number,
   and return number of symbols. */
int
read_file_hdr (chan, file_hdr)
    int chan;
    FILHDR *file_hdr;
{
  lseek (chan, 0L, 0);
  if (myread (chan, (char *)file_hdr, FILHSZ) < 0)
    return -1;

  switch (file_hdr->f_magic)
    {
#ifdef MC68MAGIC
    case MC68MAGIC:
#endif
#ifdef SPARCMAGIC
    case SPARCMAGIC:
#endif
#ifdef NS32GMAGIC
      case NS32GMAGIC:
      case NS32SMAGIC:
#endif
#ifdef I386MAGIC
    case I386MAGIC:
#endif
#ifdef CLIPPERMAGIC
    case CLIPPERMAGIC:
#endif      
	return file_hdr->f_nsyms;

      default:
#ifdef BADMAG
	if (BADMAG(file_hdr))
	  return -1;
	else
	  return file_hdr->f_nsyms;
#else
	return -1;
#endif
    }
}

int
read_section_hdr (chan, section_name, section_hdr, nsects,
                  optional_header_size)
    register int chan;
    register char *section_name;
    SCNHDR *section_hdr;
    register int nsects;
    int optional_header_size;
{
  register int i;

  if (lseek (chan, FILHSZ + optional_header_size, 0) < 0)
    return -1;

  for (i = 0; i < nsects; i++)
    {
      if (myread (chan, (char *)section_hdr, SCNHSZ) < 0)
	return -1;
      if (strncmp (section_hdr->s_name, section_name, 8) == 0)
	return 0;
    }
    return -1;
}

/* Support for line number handling */
int
init_lineno (chan, offset, count)
    int chan;
    long offset;
    int count;
{
  int val;

  if (lseek (chan, offset, 0) < 0)
    return -1;
  
  if (linetab)
    free (linetab);
  linetab = (char *) xmalloc ((count+1) * LINESZ);

  val = myread (chan, linetab, count * LINESZ);
  if (val != count * LINESZ)
    return -1;

  linetab_offset = offset;
  linetab_count = count;
  return 0;
}
static int file_offset = 0;

void
sun_enter_linenos (count)
    long count;
{
  register char *rawptr = &linetab[file_offset - linetab_offset];
  register char *base = &linetab[0];
  struct lineno lptr;

  bcopy (rawptr, &lptr, LINESZ);
  /* skip line group header */
  if (lptr.l_lnno == 0)
    rawptr += LINESZ;

  /* Bcopy since occaisionally rawptr isn't pointing at long
     boundaries.  */
  for (bcopy (rawptr, &lptr, LINESZ);
       (lptr.l_lnno && count--
	&& (rawptr-base)/LINESZ <linetab_count);
       rawptr += LINESZ, bcopy (rawptr, &lptr, LINESZ))
    {
      record_line (lptr.l_lnno, lptr.l_addr.l_paddr);
      file_offset += LINESZ;
    }

}

void
sun_mark_linenos (this_file_offset,desc)
    long this_file_offset;
{
  file_offset = this_file_offset;

  /* The following lets code built with as or an older gas work even though
     the line table is not segmented by main/include files.
   */
  if (desc != N_FUN)
    sun_enter_linenos(linetab_count);
}

clear_coff_sections()
{
  int i;

  for (i = 1; i < ( MAX_SECTS + (N_DEBUG)) ; i++)
    type_map[i]  = 0;
}

map_coff_section(scn_index,scnp)
SCNHDR *scnp;
{
  int sec_type = scnp->s_flags &( STYP_TEXT | STYP_DATA | STYP_BSS );
  int type = 0;

  if (strncmp (scnp->s_name, _TEXT, 8) == 0) {
    text_scn = scn_index; 
    type = N_TEXT;
  }
  else if (strncmp (scnp->s_name, _DATA, 8) == 0) 
    {
      data_scn = scn_index; 
      type = N_DATA;
    }
  else if (strncmp (scnp->s_name, _BSS, 8) == 0) 
    {
      bss_scn = scn_index;
      type = N_BSS;
    }
  else switch(sec_type)
    {
    case STYP_TEXT:
      type = N_TEXT;
      break;
    case STYP_DATA:
      /* following can occur when one moves text into data with ld */
    case (STYP_DATA |STYP_TEXT): 
      type = N_DATA;
      break;
    case STYP_BSS:
      type = N_BSS;
      break;
    default:
      /* this is semi-arbitrary, but seems to work alright */
      if (scnp->s_flags &( STYP_DSECT | STYP_NOLOAD))
	type = N_DATA;
      break;
    }
  type_map[scn_index] = type;
}

struct syment syment_inbuf[SYMBUFSIZE];

convert_symbuf(symtab_input_desc,outbuf,stringtab)
     int symtab_input_desc;
     struct nlist *outbuf;
     char *stringtab;
{
  struct syment *p;		/* pointer into coff symbols */
  struct nlist  *nlp;		/* pointer into aout symbols */
  unsigned char aux_type_prev = 0;
  unsigned short aux_desc_prev = 0;
  int i;
  int count;
  register unsigned int type;
  short desc;

  /* read coff symbols */
  int nbytes = myread (symtab_input_desc, syment_inbuf, SYMBUFSIZE * SYMESZ); 
  if (nbytes <= 0)							
    error ("error or end of file reading symbol table");		
  count = nbytes / SYMESZ; 

  /* convert buffer */
  for (i = 0; i < count; i++)
    {
      p = &SYMENT_INDEX(syment_inbuf,i);
      nlp = &outbuf[i];
      nlp->n_type = IGNORED_SYMBOL_TYPE;
				/* if past limit, return */
      if (conv_symnum++ >= symnum_limit)
	{
	  aux_type_prev = 0;
	  return ( i );	/* xxx  */
	}
#ifdef SUN_STYLE_COFF
      type = 0;
#else
      type = p->n_type;		/* altered if debug symbol */
#endif
      desc = 0;
      if (aux_type_prev != 0)	/* map type based upon previous entry */
        {
	  type = aux_type_prev;
	  desc = aux_desc_prev;
	  if (aux_type_prev ==  IGNORED_SYMBOL_TYPE)
	    nlp->n_un.n_strx = 0;
	  aux_type_prev =  0;
	  aux_desc_prev = 0;
        }
      else if ( IS_DEBUG_SYMBOL(p,type) )
	{
	  type = TYPE_OF_DEBUG_SYMBOL(p,type); /* encoded debugging type */
	  if (HAS_DESC(p))
	    desc = p->n_dbx_desc;
	}
      else
	{  /* not debug type - have to derive it. */
	  int storage_class = p->n_sclass;
	  int section = p->n_scnum;

	  type = type_map[section];
	  if (type)
	    {
	      if (storage_class == C_EXT )
		type |= N_EXT;
	    }
	  else if (section == N_DEBUG)
	    {
#if defined(sun386)		/* kludge for sun */
	      if ( p->n_type == ( N_LINE_COUNT_TYPE << 8 ) )
		{
		  type = N_LINE_COUNT_TYPE;
		  desc = p->n_dbx_desc;
		}
	      else if (p->n_numaux && storage_class == C_FILE)
		type = N_TEXT;
	      else
#endif
		type =  IGNORED_SYMBOL_TYPE;
	    }
	  else if (p->n_numaux)	/* handles .comment - Will do better later */
	    type =  IGNORED_SYMBOL_TYPE;
	  else
	  {
		  SHOW_BAD_CONVERSION;
		  type = N_TEXT; /* shouldnt happen, but set to valid type */
	  }
	}

      /* OK, have type. Now store type, value, desc, and strx  */

      nlp->n_type = type;
      nlp->n_value = p->n_value;
      nlp->n_desc = desc;

      if (type ==  IGNORED_SYMBOL_TYPE)	/* unwanted entry ... */
	{
	  bzero(nlp, sizeof(*nlp));
	  nlp->n_type = type;
	}
      else if (p->n_name[0] == 0) 	/* already in string table */
	{
	  nlp->n_un.n_strx = p->_n._n_n._n_offset;
	}
      else
	{
	  char *name;
	  p->n_value = 0;	/* provides null termination on 8 char name */
	  name = add_coff_string(p->n_name);
	  /* Warning: this may be negative! */
	  nlp->n_un.n_strx = name - stringtab ;
	}

      if (p->n_numaux)
	{
	  switch (type)
	    {			/* eliminate most of these. they dont occur */
	    case N_EINCL:	/* anymore! */
	    case N_BINCL:
	    case N_EXCL:
	    case N_FUN:		/* sun only allows stab auxents on N_FUN */
	    case N_SOL:
	    case N_SO:		/* will we ever see aux on N_SO? */
	      aux_desc_prev = p->n_dbx_desc;
	      aux_type_prev =  N_MULTI_LINE_TYPE;
	      break;
	    default:
	      aux_type_prev =   IGNORED_SYMBOL_TYPE;
	      break;
	    }
	}
    }
  if (aux_type_prev != 0 )	/* dont stop between symbol and its aux */
    {
      lseek(symtab_input_desc,-(SYMESZ),L_INCR);
      count -- ;
      conv_symnum--;
      aux_type_prev = 0;
    }
  return count;
}
/* xxxxxxxxxx */

struct string_object {
  char *first_object;
  struct obstack objs;
};

static struct string_object symtab_extra_strings;
static struct string_object added_symtab_extra_strings;
static struct string_object *current_symtab_extra_strings;

struct obstack * current_extra_strings_obstack;
#ifndef UNSURE
static char *
add_coff_string(name)
     register char *name;
{
  register int size = strlen(name);
  register char *p = (char *) obstack_alloc (current_extra_strings_obstack,
					     size + 1);
  register char *p2 = p;
  char *end = name + size;

  /* Open-coded bcopy--saves function call time.
     These strings are always  short ( <= 8 chars ).  */
  while (name != end)
      *p2++ = *name++;
  
  p[size] = 0;
  return p;
}
#else
static char *
add_coff_string(name)
     char *name;
{
  int name_length = strlen(name) + 1; 	/* avoid side effect in arg to
					   obstack_grow macro. */
  char * fin;
  obstack_grow( (current_extra_strings_obstack), name, name_length);

  fin =  obstack_finish(current_extra_strings_obstack);

  return fin;
}
#endif

reinitialize_extra_strings(type)
{
  if (type == 0)
    {
      current_symtab_extra_strings = &symtab_extra_strings;
    }
  else if ( type == 1)
    {
      current_symtab_extra_strings = &added_symtab_extra_strings;
    }
  else
    fatal("bad argument to reinitialize_coff_strings\n");

  current_extra_strings_obstack = &current_symtab_extra_strings->objs;

  if (current_symtab_extra_strings->first_object)
    obstack_free(current_extra_strings_obstack,
		 current_symtab_extra_strings->first_object);


  current_symtab_extra_strings->first_object
    = obstack_alloc (current_extra_strings_obstack, 0);
}

_initialize_coff_dbx()
{
  obstack_init ( &symtab_extra_strings.objs);
  obstack_init ( &added_symtab_extra_strings.objs);
}
