/* NS32000 Assembler
 * Util.c
 * Various handy utility routines
 */

#include <stdio.h>
#include <ctype.h>
#ifdef MSDOS
#  include "a_out.h"
#else
#  include "a.out.h"
#endif
#include "glob.h"

/* Hash function.  Given a string, returns a number between 0 and
 * HASHLEN - 1.
 */
int
hashfn (id)
char *id;
{
  U16 ret;

  for (ret = 0; *id; id++)
    ret = (ret<<1 | ret>>15) ^ *id;
  return ret & (HASHLEN - 1);
}

/* Passed a string and a bitmap, tries to find a hash entry 1) whose id
 * matches the string and 2) whose typ is in the bitmap.  Returns a pointer
 * to the entry if found, NULL otherwise.
 */
hashptr
find (id, mask)
char *id;
int mask;
{
  hashptr p;

  for (p = hash [hashfn (id)]; p != NULL; p = p->link)
    if ((p->typ & mask) && !strcmp (id, p->id)) break;
  return p;
}

/* Passed a pointer to a hashnode with id and typ filled in, inserts
 * the node into the hash table.
 */
insert (p)
hashptr p;
{
  int i;

  i = hashfn (p->id);
  p->link = hash[i];
  hash[i] = p;
}

/* Insert a label into the hash table.  First allocate lblnode and string
 * space for id.
 */
lblptr
insert_lbl (id, type)
char *id;
int type;
{
  lblptr lp;

  lp = (lblptr) myalloc (sizeof (struct lblnode));
  lp->id_len = strlen (id) + 1;
  lp->id = myalloc (lp->id_len);
  strcpy (lp->id, id);
  lp->typ = type;
  insert ((hashptr)lp);
  return lp;
}

/* Initialize the hash table.  Set all entries to NULL, then insert
 * all ops.
 */
hashinit ()
{
  hashptr *p;
  opptr q;

  for (p = hash; p < hash + HASHLEN;) *p++ = NULL;
  for (q = ops; q->id != NULL; ++q) insert ((hashptr)q);
}

/* Navigate the hash table, passing each entry whose typ is in mask to
 * the function f.
 */
hashnav (mask, f)
int mask;
int (*f)();
{
  hashptr p, *q;

  for (q = hash; q < hash + HASHLEN; q++)
    for (p = *q; p != NULL; p = p->link)
      if (p->typ & mask) (*f)(p);
}

/* Print error message.  Message may have # or ! in the first two positions:
 * # means print file line, ! means die.
 */
/*VARARGS1*/
error (fmt, arg1, arg2, arg3, arg4)
int arg1, arg2, arg3, arg4;
char *fmt;
{
  static old_line;
  int line, die;

  if (*fmt == '#' || *(fmt+1) == '#') line = 1;
  else line = 0;
  if (*fmt == '!' || *(fmt+1) == '!') die = 1;
  else die = 0;
  if (debug || !line || lnnum != old_line) {
    if (fmt != usage)
      fprintf (stderr, line? "as: line %d, ": "as: ", lnnum);
    fprintf (stderr, fmt + line + die, arg1, arg2, arg3, arg4);
    fputs ("\n", stderr);
    ++num_errors;
  }
  if (line) old_line = lnnum;		/* avoid multiple error msgs */
  if (die) quit();
}

/* Allocate heap, report error on failure.
 */
char *
myalloc (size)
int size;
{
  char *p;

  if (NULL == (p = malloc ((unsigned)size)))
    error ("!out of memory");
  return p;
}

/* Write to stream, report errors if any.
 */
myfwrite (cptr, size, fp)
char *cptr;
int size;
FILE *fp;
{
  if (size != fwrite (cptr, 1, size, fp))
    error ("!fwrite failed during phase %d\n", phase);
}

myputc (c, fp)
int c;
FILE *fp;
{
  if (EOF == putc (c, fp))
    error ("!putc failed during phase %d\n", phase);
}

/* Read from stream, report error if any.
 */
myfread (cptr, size, fp)
char *cptr;
int size;
FILE *fp;
{
  if (size != fread (cptr, 1, size, fp))
    error ("!fread failed during phase %d\n", phase);
}

int
mygetc (fp)
FILE *fp;
{
  int c;

  if (EOF == (c = getc (fp)))
    error ("!getc failed during phase %d\n", phase);
  return c;
}

/* Perform fseek, check for errors.
 */
myfseek (fp, offset)
FILE *fp;
long offset;
{
  if (0 != fseek (fp, offset, 0))
    error ("!fseek failed during phase %d\n", phase);
}

/* Call fopen, report errors
 */
FILE *
myfopen (name, type)
char *name, *type;
{
  FILE *fp;

  if (NULL == (fp = fopen (name, type)))
    error ("!could not open %s", name);
  return fp;
}
