/*
 * Here's all the code for handling I/O to FILE structures.
 * Author:	Jordan K. Hubbard
 * Date:	April 27th, 1990
 */

#include "stdio.h"

FILE _stdin = {
     _FILE_READ | _FILE_CBRK | _FILE_DOEOF | 0x04 /* ^D */,
     EOF, 0, 0, 0, 0, NULL,
};

FILE _stdout = {
     _FILE_WRITE | _FILE_MAPCR,
     EOF, 0, 0, 0, 0, NULL,
};

FILE _stderr = {
     _FILE_WRITE | _FILE_MAPCR,
     EOF, 0, 0, 0, 0, NULL,
};

FILE *stdin = &_stdin, *stdout = &_stdout, *stderr = &_stderr;

/* Default uart, scsi address and logical unit number */
static int _default_uart = DEFAULT_UART;
static int _default_sadr = DEFAULT_SCSI_ADR;
static int _default_slun = DEFAULT_SCSI_LUN;

Inline void set_scsi_lun(lun)
int lun;
{
     _default_slun = lun;
}

Inline void set_scsi_addr(addr)
int addr;
{
     _default_sadr = addr;
}

Inline void set_uart(uart)
int uart;
{
     _default_uart = uart;
}

/*
 * Here we define a whole bunch of routines that would probably be much more
 * efficiently expressed as macros, but aren't. This is to avoid the unpleasant
 * side-effects associated with trying to use the address of one of these
 * puppies. We'll let gcc's inlining do the optimization for us.
 */

Inline int fgetc(fp)
register FILE *fp;
{
     char ch;

     fread((char *)&ch, 1, 1, fp);
     return(ch);
}

Inline int getc(fp)
register FILE *fp;
{
     return(fgetc(fp));
}

Inline int getchar()
{
     return(fgetc(stdin));
}

Inline int fputc(i, fp)
register int i;
register FILE *fp;
{
     char ch = (char)i;
     fwrite((char *)&ch, 1, 1, fp);
     return i;
}

Inline int putc(ch, fp)
register int ch;
register FILE *fp;
{
     fputc(ch, fp);
     return ch;
}

Inline int putchar(ch)
int ch;
{
     fputc(ch, stdout);
     return ch;
}

Inline int putw(i, fp)
int i;
FILE *fp;
{
     fwrite((char *)&i, sizeof(i), 1, fp);
     return i;
}

Inline int putl(l, fp)
long l;
FILE *fp;
{
     fwrite((char *)&l, sizeof(l), 1, fp);
     return l;
}

Inline int puth(i, fp)
int i;
FILE *fp;
{
     short h = (short)i;
     fwrite((char *)&h, sizeof(h), 1, fp);
     return i;
}

int ungetc(ch, fp)
int ch;
FILE *fp;
{
     if (!fp || !_fp_type(fp))
	  return(EOF);
     else {
	  if (_fp_type(fp) & _FILE_DOEOF && ch == _eofchar(fp))
	       return EOF;
	  if (_fp_unget(fp) != EOF)
	       warn("ungetc() discarding previous ungetc()");
	  else
	       _fp_unget(fp) = ch;
     }
}

char *fgets(s, n, fp)
char *s;
unsigned int n;
FILE *fp;
{
     register unsigned int i = 0;
     register char ch;

     while ((ch = fgetc(fp)) != '\n' && ch != EOF && i < n)
	  s[i++] = ch;
     if (i < n)
	  s[i] = '\0';
     return(s);
}

Inline char *gets(s)
register char *s;
{
     return(fgets(s, -1, stdin));
}

Inline void fputs(s, fp)
register char *s;
FILE *fp;
{
     while (*s)
	  fputc(*(s++), fp);
}

Inline void puts(s)
register char *s;
{
     fputs(s, stdout);
}

Inline void setbuf(fp, buf)
FILE *fp;
char *buf;
{
     _fp_buffer(fp) = buf;
}

/* open a file, fname is now a starting block */
FILE *fopen(fname, type)
long fname;
char *type;
{
     FILE *ret;

     if (fname < 0 || fname >= INSANE_BADDR)
	  fatal("Demented block size %d passed to fopen()", fname);
     ret = (FILE *)malloc(sizeof(FILE));
     bzero(ret, sizeof(FILE));
     if (!type)
	  _fp_type(ret) = _FILE_READ | _FILE_WRITE;
     else {
	  if (strchr(type, 'r'))
	       _fp_type(ret) = _FILE_READ;
	  if (strchr(type, 'w'))
	       _fp_type(ret) |= _FILE_WRITE;
	  if (strchr(type, 'a'))
	       warn("fopen() currently does not support append");
     }
     _fp_start(ret) = _fp_block(ret) = fname;
     return(ret);
}

/* close a file */
int fclose(fp)
FILE *fp;
{
     if (!fp)
	  return -1;
     else {
	  if (_fp_type(fp) & _FILE_DOEOF)
	       fputc(_eofchar(fp), fp);
	  fflush(fp);
     }
     /* this can be NULL if file was never read/written */
     if (_fp_buffer(fp))
	  free(_fp_buffer(fp));

     /*
      * not sure if I should really allow this, but what the heck.. If the
      * user Really Wants To for some reason..
      */
     if (fp == stdin)
	  stdin = (FILE *)NULL;
     else if (fp == stdout)
	  stdout = (FILE *)NULL;
     else if (fp == stderr)
	  stderr = (FILE *)NULL;
     else
	  free(fp);
     return 0;
}

/* flush a file buffer */
void fflush(fp)
FILE *fp;
{
     if (fp && _fp_type(fp) & _FILE_WRITE) {
	  /* file has a dirty buffer */
	  if (_fp_length(fp) && _fp_offset(fp))
	       block_write(_fp_block(fp), _fp_buffer(fp));
	  _fp_length(fp) = _fp_offset(fp) = 0;
     }
}

/* seek to some spot */
int fseek(fp, offset, type)
FILE *fp;
long offset, type;
{
     long newpos;

     if (!fp)
	  return -1;
     fflush(fp);
     switch(type) {
     case 0:	/* beginning of file */
	  newpos = offset;
	  break;

     case 1:	/* current position */
	  newpos = (_fp_block(fp) * DBLKSIZE) + _fp_offset(fp) + offset;
	  break;

     case 2:	/* EOF */
	  warn("fseek() from EOF not supported");
	  newpos = (_fp_block(fp) * DBLKSIZE) + _fp_offset(fp);
	  break;
     }
     _fp_block(fp) = newpos / DBLKSIZE;
     _fp_offset(fp) = newpos % DBLKSIZE;
     return 0;
}
  
/* set a file option */
Inline void fsetopt(opt, fp)
register int opt;
register FILE *fp;
{
     if (fp)
	  _fp_type(fp) |= opt;
}

/* clear a file option */
Inline void fclropt(opt, fp)
register int opt;
register FILE *fp;
{
     if (fp)
	  _fp_type(fp) ^= opt;
}

/*
 * Read a file. Since files are more-or-less "endless" under this I/O
 * implementation, we don't have to worry about partial blocks. This
 * is a VERY simplistic interpretation!
 */
int fread(ptr, size, nitems, fp)
char *ptr;
int nitems, size;
FILE *fp;
{
     register int i, nbytes = size * nitems;
     char *target;
     static int eol_reached = 0;

     if (!fp || _fp_type(fp) == 0)
	  return 0;
     else if (!_fp_type(fp) & _FILE_READ) {
	  warn("Read attempted on write-only file");
	  return -1;
     }
     /* No buffer yet allocated, allocate one */
     else if (!_fp_buffer(fp))
	  _fp_buffer(fp) = malloc(DBLKSIZE);
     i = 0;
     while (i < nbytes) {
	  if (_fp_unget(fp) != EOF) {
	       ptr[i++] = _fp_unget(fp);
	       _fp_unget(fp) = EOF;
	  }
	  else if (fp == stdin) {
	       /*
		* line mode? This loses, BTW, if we type more than 512 chars
		* without a carriage return, but I don't really care.
		*/
	       if (_fp_type(fp) & _FILE_LINE) {
		    /* have some characters already? */
		    if (_fp_length(fp) && eol_reached) {
			 if (_fp_offset(fp) < _fp_length(fp)) {
			      ptr[i++] = _fp_buffer(fp)[_fp_offset(fp)++];
			      continue;
			 }
			 else
			      _fp_offset(fp) = _fp_length(fp) = 0;
		    }
		    target = _fp_buffer(fp) + _fp_length(fp);
		    eol_reached = 0;
	       }
	       else
		    target = ptr + i;

	       /* stick it somewhere */
	       *target = db_fgetc(_default_uart);

	       /* if EOF handling is desired, do it */
	       if (_fp_type(fp) & _FILE_DOEOF && *target == _eofchar(fp)) {
		    *target = EOF;
		    eol_reached = 1; /* fake an eol */
	       }
	       /* if CR/NL mapping is desired, do it */
	       else if (*target == '\r') {
		    eol_reached = 1;
		    if (_fp_type(fp) & _FILE_MAPCR)
			 *target = '\n';
	       }
	       /* do we need simple line editing? */
	       if (_fp_type(fp) & _FILE_LINE) {
		    /* ^H or DEL? */
		    if (*target == '\b' || *target == 127) {
			 if (_fp_length(fp)) {
			      puts("\b \b");
			      --_fp_length(fp);
			 }
			 else
			      putchar('\007');
		    }
		    /* ^X or ^U? */
		    else if (*target == '\030' || *target == '\024') {
			 while (_fp_length(fp)) {
			      puts("\b \b");
			      --_fp_length(fp);
			 }
		    }
		    else {
			 ++_fp_length(fp);
			 if (_fp_type(fp) & _FILE_ECHO)
			      putchar(*target);
		    }
	       }
	       else {
		    /* if echoing is desired, do so */
		    if (_fp_type(fp) & _FILE_ECHO)
			 putchar(*target);
		    i++;
	       }
	  }
	  /* start fetching blocks as necessary */
	  else {
	       int x_amt = nbytes - i; /* amount left to xfer */
	       
	       /*
		* There's some data in the buffer, use up as much as we can
		* (possibly all of it) before going on.
		*/
	       if (_fp_length(fp)) {
		    int residue = _fp_length(fp) - _fp_offset(fp);
		    if (x_amt < residue) {
			 memcpy(ptr + i, _fp_buffer(fp) + _fp_offset(fp),
				x_amt);
			 _fp_offset(fp) += x_amt;
			 i += x_amt;
		    }
		    else {
			 memcpy(ptr + i, _fp_buffer(fp) + _fp_offset(fp),
				residue);
			 _fp_length(fp) = _fp_offset(fp) = 0;
			 i += residue;
		    }
	       }
	       else {
		    /*
		     * If we can, xfer a block directly to the user area and
		     * simply bypass the FILE buffer.
		     */
		    if (nbytes - i > DBLKSIZE) {
			 block_read(_fp_block(fp)++, ptr + i);
			 i += DBLKSIZE;
			 continue;
		    }
		    /* Otherwise, fill the buffer */
		    else {
			 block_read(_fp_block(fp)++, _fp_buffer(fp));
			 _fp_length(fp) = DBLKSIZE;
			 _fp_offset(fp) = 0;
		    }
	       }
	  }
     }
     return( i );			/* budd -- per jkh */
}

/* Write a file */
int fwrite(ptr, size, nitems, fp)
char *ptr;
int nitems, size;
FILE *fp;
{
     int i, nbytes;

     i = 0;
     nbytes = size * nitems;
     if (!fp || _fp_type(fp) == 0)
	  return 0;
     else if (!_fp_type(fp) & _FILE_WRITE) {
	  warn("Write attempted on read-only file");
	  return -1;
     }
     while (i < nbytes) {
	  if (fp == stdout || fp == stderr) {
	       db_fputc(ptr[i], _default_uart);
	       if (_fp_type(fp) & _FILE_MAPCR && ptr[i] == '\n')
		    db_fputc('\r', _default_uart);
	       ++i;
	  }
	  else {
	       /* can we bypass the buffer? */
	       if (nbytes - i >= DBLKSIZE) {
		    block_write(_fp_block(fp)++, ptr + i);
		    i += DBLKSIZE;
	       }
	       else {
		    /* No buffer yet allocated, allocate one */
		    if (!_fp_buffer(fp))
			 _fp_buffer(fp) = malloc(DBLKSIZE);
		    /* Read old file contents for this block */
		    if (!_fp_length(fp)) {
			 block_read(_fp_block(fp), _fp_buffer(fp));
			 _fp_length(fp) = DBLKSIZE;
			 _fp_offset(fp) = 0;
		    }
		    /* do we need to flush a block? */
		    else if (_fp_offset(fp) == _fp_length(fp)) {
			 block_write(_fp_block(fp)++, _fp_buffer(fp));
			 _fp_length(fp) = _fp_offset(fp) = 0;
		    }
		    /* else we can update a byte in a valid block */
		    else
			 _fp_buffer(fp)[_fp_offset(fp)++] = ptr[i++];
	       }
	  }
     }
     return i;
}
		    
/* low level routines */
int block_write(bn, buf)
long bn;
char *buf;
{
     return(sc_rdwt(DISK_WRITE, bn, buf, 1, _default_sadr, _default_slun));
}

int block_read(bn, buf)
long bn;
char *buf;
{
     return(sc_rdwt(DISK_READ, bn, buf, 1, _default_sadr, _default_slun));
}
