/*
 *	Talk to ROM blaster (or other slow serial device)
 *	Copyright (C) 1988 Philip L. Budne
 *
 *	May be freely distributed and used for non-profit use only,
 *	provided this, and all other Copyright notices remain intact.
 *
 *	Commercial use is strictly forbidden without explicit, written
 *	permission of the author.
 */

/*
 *	niu talk version 3
 *	Phil Budne @ BU / DSG
 *
 *	Sends a command file to an NIU waiting
 *	for a prompt (>>) before sending the next line.
 */

# include <errno.h>
# include <stdio.h>
# include <ctype.h>
# include <signal.h>
# include <sgtty.h>
# include <sys/time.h>

# define TMODE RAW			/* should be RAW, CBREAK if debug */
# define BUFL 200

# define CHAR1 '>'			/* first char in prompt */
/* # define CHAR2 '>'			/* second char in prompt */

char *dev = "/dev/ttyb";		/* device name for modem */

extern int errno;
int breakc = ']' - 0100;		/* break char to leave this mess */
struct sgttyb oldm, newm, oldt, newt;	/* tty settings {modem,tty}x{old,new}*/
int input = 0;				/* terminal input fd */
int output = 1;

int ready = 1;				/* magic flag */

static char wakeup[] = "    \r";

/* defines for command input file */
FILE *ifile;
# define ISOPEN (ifile != NULL)
# define CLOSEIT {fclose(ifile);ifile=NULL;}
# define GETCHAR getc(ifile)

/* defines for output of text from connection */
# define OUTPUT(b,c) {write(output,b,c);}

int tflag;			/* accept terminal input */
int debug;			/* debug level */

main(argc, argv)
int argc;
char *argv[];
{
    int c;
    int errs;
    char *fname = NULL;

    extern char *optarg;
    extern int optind;

    errs = 0;
    while( (c = getopt(argc, argv, "tf:T:d:b:")) != EOF ) {
	switch( c ) {
	case 'd':
	    debug = atoi( optarg );
	    break;
	case 'f':
	    if( fname != NULL )
		errs++;
	    fname = optarg;
	    break;
	case 't':
	    tflag++;
	    break;
	case 'T':
	    dev = optarg;
	    break;
	case 'b':
	    if( optarg[0] == '^' ) {
		if(  optarg[1] >= ' '  && optarg[1] <= '_' )
		    breakc = optarg[1] - 0100;
		else if( islower( optarg[1] ) )
		    breakc = optarg[1] - 0140;
		else if( optarg[1] == '@' )
		    breakc = 0;
		else {
		    fprintf( stderr, "-b ^X | -b decimal-number\n" );
		    exit( 1 );
		}
	    }
	    else if( isdigit( *optarg ) )
		breakc = atoi( optarg );
	    else {
		fprintf( stderr, "-b ^X | -b decimal-number\n" );
		exit( 1 );
	    }
	    break;
	case '?':
	    errs++;
	    break;
	} /* switch */
    } /* while */

    if( tflag == 0 && fname == NULL ) {
	fprintf(stderr, "Must have one of -t or -f\n");
	errs++;
    }

    if( errs > 0 ) {
	fprintf(stderr,
		"Usage: %s [-t] [-f file] [-T dev] [-d deblvl] [-b breakc]\n",
		argv[0] );
	exit( 1 );
    }

    if( fname != NULL ) {
	if( strcmp(fname, "-") == 0 ) {
	    if( tflag > 0 ) {
		fprintf(stderr, "Cannot use stdin as file with -t\n");
		exit( 1 );
	    }
	    ifile = stdin;
	}
	else {
	    ifile = fopen(fname, "r");
	    if( ifile == NULL ) {
		perror( fname );
		exit( 1 );
	    }
	}
    }
    else
	ifile = NULL;

    if( tflag > 0 ) {
	tflag = 1;
	if( ! isatty( input ) ) {
	    fprintf(stderr, "stdin must be a tty for -t\n" );
	    exit( 1 );
	}
	fprintf( stderr, "break character is %#o (%d.)\n", breakc, breakc );
    }

    doit();
} /* main */

doit() {
    char c;
    int modem, rbits;
    int lastc = 0;			/* last from modem (prompt search) */
    int otflag;
    int exiting;			/* 0 normal, 1 type cmd */

    otflag = tflag;

    fprintf(stderr, "Trying %s ... ", dev );
    fflush( stderr );
    if( (modem = open(dev, 2)) < 0 ) {
      perror("could not open modem device");
      exit( 1 );
    }
    else {
	fprintf(stderr, "Open\n" );
	fflush( stderr );
    }

    if( ioctl(modem, TIOCEXCL) < 0 ) {	/* set exclusive access */
	perror("could not set exclusive access");
	close( modem );
	exit( 1 );
    }

    gtty(modem, &oldm);
    gtty(modem, &newm);
    newm.sg_ispeed = newm.sg_ospeed = B2400;
    newm.sg_flags |= (TANDEM|RAW);
    newm.sg_flags &= ~(CRMOD|ECHO);
    stty(modem, &newm);

    write(modem, wakeup, sizeof(wakeup)-1 );

    if( tflag ) {
      gtty(input, &oldt);
      gtty(input, &newt);
      newt.sg_flags |= (TANDEM|TMODE);
      newt.sg_flags &= ~(CRMOD|ECHO);
      stty(input, &newt);
      fprintf(stderr, "Open\r\n");
      fflush( stderr );
    }

    ready = 0;

    while( ISOPEN || tflag || exiting != 0 ) {
	struct timeval tv;
	register int i;
	register char *cp;
	int cc;
	char buf[BUFL];

	rbits = (1<<modem);
	if( tflag )
	    rbits |= (1<<input);		/* listen to tty */

	if( ISOPEN ) {
	    tv.tv_sec = 0;
	    tv.tv_usec = 111111;		/* about 100 chars at 9600bd */

	    select(32, &rbits, 0, 0, &tv);
	    if( rbits == 0 && !tflag && !ready ) {
		static char cr[] = "\r";

		/* nothing to read, no tty, and not ready.
		/* send a cr and wait */
		write(modem, cr, (sizeof(cr))-1 );
		tv.tv_sec = 1;
		tv.tv_usec = 0;
		rbits = (1<<modem);

		select(32, &rbits, 0, 0, &tv);

		/* if still nothing, wait more, then blast out */
		if( rbits == 0 ) {
		    tv.tv_sec = 90;
		    tv.tv_usec = 0;
		    rbits = (1<<modem);
		    select(32, &rbits, 0, 0, &tv);
		    if( rbits == 0 ) {
			printf("\r\n *** TIMEOUT ***\r\n" );
			exiting = 1; /* *** */
		    } /* third select timed out */
		} /* second select timed out */
	    } /* first select timed out, nothing to do */
	} /* ISOPEN */
	else if( exiting != 0 ) {	/* if exiting don't tarry */
	    tv.tv_sec = 1;
	    tv.tv_usec = 0;
	    select(32, &rbits, 0, 0, &tv);
	}
	else				/* tflag and not exiting */
	    select(32, &rbits, 0, 0, 0); /* wait forever */

	if( (rbits & (1<<input)) != 0 ) {	/* data from tty? */
	  if( (cc = read(input, buf, BUFL)) < 0 ) {
	      tflag = 0;
	      if( !ISOPEN )
		  exiting = 1;
	  }
	  else {				/* tty read ok */
	    cp = buf;
	    for( i = 0; i < cc; i++ )
		if( *cp++ == breakc )
		    exiting = 1;
		else
		    write(modem, buf, cc);
	  } /* tty read ok */
	} /* data ready from tty */

	if( (rbits & (1<<modem)) != 0 ) {	/* data from modem? */
	  if( (cc = read(modem, buf, BUFL)) < 0 )
	      break;
	  else {				/* modem read ok */
	    cp = buf;
	    for( i = 0; i < cc; i++ ) {
	      register char c;
	      c = *cp++;
# ifdef CHAR2
	      if( lastc == CHAR1 && c == CHAR2 )
		ready = 1;
	      lastc = c;
# else
	      if( c == CHAR1 )
		  ready = 1;
# endif
	    } /* looking for prompt */
	    OUTPUT(buf,cc);
	    if( ready )
		continue;
	  } /* modem read ok */
	} /* data ready on modem */

	if( ISOPEN ) {
	  if( ready ) {
	      register c;
	      cp = buf;
	      while( (c = GETCHAR) != EOF ) {
		  if( debug > 1 )
		      printf("sending: (%d.)\r\n", c);
		  if( c == '\n' )
		      *cp++ = '\r';
		  *cp++ = c;
		  if( c == '\n' )
		      *cp++ = '\r';
		  if( c == '\n' ) {
		      ready = 0; 	/* just sent a line, wait for prompt */
		      break;
		  } /* sent CR */
	      } /* while */
	      write(modem, buf, cp - buf );

	      if( c == EOF ) {
		  if( debug > 0 )
		      fputs("*EOF*\n", stderr);
		  CLOSEIT;
		  if( !tflag )
		      exiting = 1;
	      } /* broken by EOF, close file */
	  } /* ready for new line */
	} /* have text file */

	if( exiting == 2 )
	    break;

	if( exiting == 1 ) {
/*	    static char quit[] = "\rQUIT\r";
/*	    write( modem, quit, sizeof(quit)-1 );
*/
	    exiting = 2;
	}
    }  /* while ISOPEN || tflag */
      
    if( otflag ) {
	stty(input, &oldt);		/* restore tty */
	fprintf(stderr, "exiting...\n" );
    }

    stty(modem, &oldm);			/* restore modem */
    close( modem );			/* close modem */
    sleep( 1 );				/* wait?? */

} /* doit */
