#include <stdio.h>
#include <sgtty.h>
#include <signal.h>
#include <sys/types.h>
#include <sys/stat.h>
/************************************************************************/
/*									*/
/*  program:								*/
/*	odt -- talk to an LSI 11 from a VAX UNIX terminal		*/
/*									*/
/*  synopsis:								*/
/*	odt [-lx]							*/
/*									*/
/*  description:							*/
/*	  lsi attaches a user's terminal to a tty line that connects	*/
/*	to an LSI 11 so that keystrokes typed at the terminal are	*/
/*	sent to the LSI 11, and characters that the LSI 11 emits	*/
/*	are sent to the terminal display (without any processing).	*/
/*	The standard input, standard output, and standard error		*/
/*	files remain attached to the terminal.  File descriptor		*/
/*	lsifd is attached to the LSI 11 tty line.			*/
/*	  Input from the keyboard is processed as follows: lines	*/
/*	beginning with ! are passed to a subshell.  A backslash /	*/
/*	causes lsi to change the keyboard mode to RAW for the next	*/
/*	character, making it possible to send special characters	*/
/*	like interrupt or kill characters.  Echo is turned off when	*/
/*	lsi begins.  Thus, any echoing of keystrokes is coming from	*/
/*	the LSI 11.							*/
/*									*/
/*	  Normally, odt looks for any available LSI 11.  The argument	*/
/*	-lx restricts the search to the LSI 11 with "number" x.		*/
/*									*/
/* Author/status:							*/
/*	Doug Comer, based on a program by Bob Brown			*/
/*	CS Dept., Purdue University					*/
/*									*/
/************************************************************************/

#define	STDOUT	1
#define	STDIN	0

int	lsifd;				/* LSI 11 file descriptor	*/
int	quit();				/* procedure called to quit	*/
int	quit2();			/* safe interim quit procedure  */
struct	sgttyb	lttymode;		/* LSI tty mode for gtty/stty	*/
struct	sgttyb	kttymode;		/* keyboard mode upon entry	*/
struct	sgttyb	cttymode;		/* changed keyboard mode	*/
int	pid;				/* process id of this process	*/
char	*device = "/dev/ttyb";

/*
 *========================
 *   M a i n   E n t r y
 *========================
 */
main(argc, argv)
int argc;
char *argv[];
{
	int	lsiarg;			/* value of -lddd argument	*/

	signal(SIGINT, quit2);		/* safe handler for parent and child
					   to execute until fork has completed
					   and both have reset the handler */

	while( argc > 1 && argv[0][0] == '-' ) {
	    switch( argv[0][1] ) {
	    case 'l':
		if( argc < 2 )
		    usage();
		device = argv[1];
		argc--;
		argv++;
		break;
	    default:
		usage();
	    }
	    argc--;
	    argv++;
	}

	if( (lsifd = open(device, 2)) < 0 ) {
	    perror( device );
	    exit(1);
	}
	if (ioctl(lsifd, TIOCGETP, &lttymode) < 0) {
		perror( device );
		exit(1);
	} else {
		lttymode.sg_flags |= RAW;
		lttymode.sg_flags &= ~ECHO;
		lttymode.sg_ospeed = lttymode.sg_ispeed = B9600;
		ioctl (lsifd, TIOCSETP, &lttymode);
	}

	/*
         * set up local terminal in cbreak mode
	 */
	if (ioctl(STDIN, TIOCGETP, &kttymode) < 0 ) {
		perror("Cannot set terminal modes");
		exit(1);
	} else {
		cttymode = kttymode;
		cttymode.sg_flags |= CBREAK;
                cttymode.sg_flags &= ~ECHO;
                cttymode.sg_flags &= ~CRMOD;
		if(ioctl(STDIN, TIOCSETP, &cttymode) < 0) {
			perror("Cannot set modes on LSI line");
			exit(1);
		}
	}
	/*
	 * start up the keyboard monitor and line monitor tasks
	 */
	printf("connected\r\n");
	if ((pid=fork()))
		keymon();
	else
		linemon();
	exit(0);
}


int	idle = 0;

/*
 *------------------------------------------------------------------
 *
 * keymon() - terminal keyboard monitor process
 *
 *------------------------------------------------------------------
 */
keymon()
{
	int	col, i;
        int	esc;
        char	c, buf[BUFSIZ];
	int	sigsusp(), sigcont(), sigalrm();

	/*
	 * catch interrupt, suspend and continue signals
	 */
	signal(SIGINT,  quit);
	signal(SIGTSTP, sigsusp);
	signal(SIGCONT, sigcont);

	/*
	 * loop forever over the following: process each keystroke
	 * on the keyboard.
	 */
	col = 0;
        esc = 0;
	while( (i=read(STDIN,&c,1)) != 0) {
	    if (i == 1) {
		idle = 0;
		if ( col==0 && c == '!' ) {
                        while ( c != '\n' && c != '\r' ) {
				read(STDIN, &c, 1);
                                buf[col++] = c;
                        }
                        buf[--col] = '\0';
			ex(buf);
			col = 0;
		} else if ( c=='\\' && esc==0 ) {
			cttymode.sg_flags |= RAW;
			stty(STDIN,&cttymode);
                        esc = 1;
		} else {
                        if ( esc && (c&0177)=='\0' ) {
                                ioctl(lsifd, TIOCSBRK, NULL);
                                sleep(2);
                                ioctl(lsifd, TIOCCBRK, NULL);
                        } else {
                                write(lsifd, &c, 1);
                                if ( (c&0177)=='\n' || (c&0177)=='\r' )
                                        col =  0;
                                else
                                        col++;
                        }
                        if ( esc ) {
                                cttymode.sg_flags &= ~RAW;
                                stty(STDIN, &cttymode);
                        }
                        esc = 0;
		}
	    }
	}
        esc = 0;
	ioctl(STDIN, TIOCSETP, &kttymode);
	quit();
}

/*
 *---------------------------------------------------------------------
 *
 * ex -- execute input following a ! and return to normal processing
 *
 *---------------------------------------------------------------------
 */
ex(bp)
char *bp;
{
	int (*oldint)(), (*oldquit)(), (*oldsusp)(), (*oldcont)();

	oldint = signal(SIGINT, SIG_DFL);
	oldquit = signal(SIGQUIT, SIG_DFL);
	oldsusp = signal(SIGTSTP, SIG_DFL);
	oldcont = signal(SIGCONT, SIG_DFL);
	ioctl(STDIN, TIOCSETP, &kttymode);

	system(bp);
	signal(SIGINT, oldint);
	signal(SIGQUIT, oldquit);
	signal(SIGTSTP, oldsusp);
	signal(SIGCONT, oldcont);
	printf("!\n");
	ioctl(STDIN, TIOCSETP, &cttymode);
}
system(s)
char *s;
{
	int status, pid, w;
	register int (*istat)(), (*qstat)();

	if ((pid = vfork()) == 0) {
		execl("/bin/sh", "sh", "-c", s, 0);
		_exit(127);
	}
	istat = signal(SIGINT, SIG_IGN);
	qstat = signal(SIGQUIT, SIG_IGN);
	while ((w = wait(&status)) != pid && w != -1)
		;
	if (w == -1)
		status = -1;
	signal(SIGINT, istat);
	signal(SIGQUIT, qstat);
	return(status);
}

quit()
{
	kill(pid,9);
	ioctl(STDIN, TIOCSETP, &kttymode);
	printf("\n");
	exit(0);
}

quit2()					/* safe interim quit procedure */
{
	ioctl(STDIN, TIOCSETP, &kttymode);
	printf("\n");
	exit(0);
}

sigsusp()
{
	ioctl(STDIN, TIOCSETP, &kttymode);
        fflush(stdout);
	signal(SIGTSTP, SIG_DFL);
	kill(getpid(), SIGTSTP);
}

sigcont()
{
	signal(SIGCONT, sigcont);
	signal(SIGTSTP, sigsusp);
	ioctl(STDIN, TIOCSETP, &cttymode);
}


/*
 *------------------------------------------------------------------
 *
 *  linemon  -- monitor LSI line and print characters that arrive
 *
 *------------------------------------------------------------------
 */
linemon()
{
	char buf[512];
	int cc;

	/*
	 * ignore interrupt signal -- keyboard process kills us
	 */
        signal(SIGINT, SIG_IGN);
	signal(SIGQUIT, SIG_IGN);
        signal(SIGTSTP,SIG_DFL);
	signal(SIGCONT, SIG_DFL);

	for (;;) {
		if( (cc = read(lsifd, buf, sizeof(buf) )) > 0)
			write(STDOUT, buf, cc);
	}
}

usage() {
    fprintf( stderr, "usage: odt [-l device]\n" );
    exit( 1 );
}
