/*
 *	MC68010 Tiny Monitor -- command loop for new 'C' monitor
 *	Copyright (C) 1988 Philip L. Budne
 *
 *	May be freely distributed and used for non-profit use only,
 *	provided this, and all other Copyright notices remain intact.
 *
 *	Commercial use is strictly forbidden without explicit, written
 *	permission of the author.
 */

#include "sa_io.h"

long regblk[ 15*4 ];			/* all but sp */

newmon() {
    int last;
    last = -1;
    for( ; ; ) {
	putchar('>');
	switch( getchar() ) {
	case ' ':
	    putchar('\n');
	case '\r':
	case '\n':
	    break;

	case 'd':
	case 'D':
	    printf("ump ");
	    do_dump(&last);
	    break;

	case 'f':
	case 'F':
	    printf("rame trace ");
	    do_frame();
	    break;

	case 'g':
	case 'G':
	    printf("o ");
	    do_go();
	    break;

	case 'm':
	case 'M':
	    printf("odify ");
	    do_modify(&last);
	    break;

# ifdef LOADER
	case 'L':
	case 'l':
	    printf("oader! (type 10 <CR>s to escape)\n");
	    do_loader();
	    break;
# endif

	case 'I':
	case 'i':
	    printf("nterrupt test\n");
	    itest();
	    break;

	case 'V':
	case 'v':
	    printf("ector test\n");
	    vtest();
	    break;

	case 'P':
	case 'p':
	    printf("oke ");
	    do_poke();
	    break;

	case 'R':
	case 'r':
	    printf("egisters\n");
	    do_regs();
	    break;

	case 'Q':
	case 'q':
	    printf("uit\n");
	    return;
	    break;			/* !! */

	case 'S':
	case 's':
	    printf("tore ");
	    do_store();
	    break;

	case 'Z':
	case 'z':
	    printf("ero ");
	    do_zero();
	    break;

	case 'h':
	case 'H':
	    printf("elp");
	case '?':
	    putchar('\n');
	    printf("D[<addr>[-<addr>]]  Dump\n");
	    printf("F<addr>             Frame trace\n");
	    printf("G<addr>             Go\n");
	    printf("I                   DUART interrupt test\n");
#ifdef LOADER
	    printf("L                   Loader\n");
#endif
	    printf("M[<addr>]           Modify\n");
	    printf("P<addr>,<byte>      Store one byte\n");
	    printf("Q                   Quit\n");
	    printf("R                   Modify registers\n");
	    printf("S<addr>,<len>       Store raw binary\n");
	    printf("V                   vector test\n");
	    printf("Z<addr>,<len>       Zero bytes\n");
	    break;

	default:
	    printf("XXX\n");
	    break;
	}
    }
}

getarg( tc, eol )
    char *tc;
    int eol;
{
    char c;
    int arg, count;

    arg = count = 0;
    for( ; ; ) {
	c = getchar() & 0177;
	if( c == 0177 || c == '\b' ) {
	    count = 0;			/* force bad return */
	    break;
	}
	if( c == ' ' && count == 0 )
	    continue;
	if( c >= '0' && c <= '9' ) {
	    count++;
	    arg = arg * 16 + c - '0';
	}
	else if( c >= 'A' && c <= 'F' ) {
	    count++;
	    arg = arg * 16 + c - 'A' + 10;
	}
	else if( c >= 'a' && c <= 'f' ) {
	    count++;
	    arg = arg * 16 + c - 'a' + 10;
	}
	else
	    break;
    }
    if( c == '\n' )
	c = '\r';
    *tc = c;
    if( count == 0 ) {
	if( c != '\r' )
	    putchar('\n');
	return( -1 );
    }
    if( eol && c != '\r' )
	putchar('\n');
    return( arg );
}

do_modify(last)
long *last;
{
    int a;
    char tc;
    register char *cp;

    a = getarg( &tc, 1);
    if( a == -1 ) {
	if( tc != '\r' || *last == -1 ) {
	    putchar('\007');
	    return;
	}
	a = *last;
    }
    cp = (char *) a;
    for( ; ; ) {
	printf("%06X/ %02X ? ", (int)cp, (int)(0xff & *cp) );
	a = getarg( &tc, 1 );
	if( a == -1 ) {
	    if( tc == '\r' )
		cp++;
	    else if( tc == '\b' ) {	/* BS? */
		cp--;
	    }
	    else {
		*last = (long) cp;
		return;
	    }
	}
	else
	    *cp++ = a & 0xff;
    }
}

do_frame() {
    int a;
    char tc;
    long *fp;

    if( (a = getarg( &tc, 1 )) == -1 ) {
	putchar('\007');
	return;
    }

    fp = (long *)a;
    while( ((int)fp & 1) == 0 && fp != 0 &&
	  (unsigned)fp >= 0x200000 && (unsigned)fp <= 0x20ffff ) {
	printf("%#x\n", *(fp-4) );
	fp = (long *) *fp;
    }
}

do_go() {
    int a;
    char tc;

    if( (a = getarg( &tc, 1 )) != -1 )
	do_jmp( a, regblk );
    putchar('\007');
}

do_store() {
    int a;
    char tc;
    register unsigned char *cp;
    register int l;

    if( (a = getarg( &tc, 0 )) != -1 && tc == ',' )
	if( (l = getarg( &tc, 1 )) != -1 ) {
	    putchar('=');
	    cp = (unsigned char *) a;
	    while( l-- > 0 )
		*cp++ = rawgetc();
	    return;
	}
    putchar('\n');
    putchar('\007');
}
	    
do_zero() {
    int a;
    char tc;
    register unsigned char *cp;
    register int l;

    if( (a = getarg( &tc, 0 )) != -1 && tc == ',' )
	if( (l = getarg( &tc, 1 )) != -1 ) {
	    cp = (unsigned char *) a;
	    while( l-- > 0 )
		*cp++ = 0;
	    return;
	}
    putchar('\n');
    putchar('\007');
}
	    
do_poke() {
    int a;
    char tc;
    register unsigned char *cp;
    register int l;

    if( (a = getarg( &tc, 0 )) != -1 && tc == ',' )
	if( (l = getarg( &tc, 1 )) != -1 ) {
	    cp = (unsigned char *) a;
	    *cp = l & 0xff;
	    return;
	}
    putchar('\n');
    putchar('\007');
}

static char *regnames[] = {
    "d0", "d1", "d2", "d3", "d4", "d5", "d6", "d7",
    "a0", "a1", "a2", "a3", "a4", "a5", "a6", "sp" };

# define MAXREG 13
do_regs() {
    register int i, a;
    char tc;

    for( i = 0; i < 8; i++ ) {
	int j;
	printf("%s/ %8x", regnames[i], regblk[i] );
	j = i + 8;
	if( j <= MAXREG )
	    printf("   %s/ %8x", regnames[j], regblk[j] );
	putchar('\n');
    }
    putchar('\n');

    i = 0;
    for( ; ; ) {
	if( i < 0 ) i = MAXREG;
	if( i > MAXREG ) i = 0;

	printf("%s/ %8x ? ", regnames[i], regblk[i] );
	if( (a = getarg( &tc, 1 )) != -1 )
	    regblk[i++] = a;
	else {
	    if( tc == '\b' )
		i--;
	    else if( tc == '\r' )
		i++;
	    else
		return;
	}
    }
} /* do_regs */

