/*
 * Copyright (c) 2007, Dennis M. Sosnoski. All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without modification, are permitted provided that the
 * following conditions are met:
 * 
 * Redistributions of source code must retain the above copyright notice, this list of conditions and the following
 * disclaimer. Redistributions in binary form must reproduce the above copyright notice, this list of conditions and the
 * following disclaimer in the documentation and/or other materials provided with the distribution. Neither the name of
 * JiBX nor the names of its contributors may be used to endorse or promote products derived from this software without
 * specific prior written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

package org.jibx.v2;

/**
 * Binding factory interface definition. This interface is implemented by the binding factory class generated by each
 * binding definition. All binding factory instances are guaranteed to be threadsafe and reusable.
 * 
 * @author Dennis M. Sosnoski
 */
public interface BindingFactory
{
    /**
     * Current binary version number. This is a byte-ordered value, allowing for two levels of major and two levels of
     * minor version.
     */
    static final int CURRENT_VERSION_NUMBER = 0x02000000;
    
    /** Current distribution file name. This is filled in by the Ant build process to match the current distribution. */
    static final String CURRENT_VERSION_NAME = "@distrib@";
    
    /** Mask for portions of version number that effect compatibility. */
    static final int COMPATIBLE_VERSION_MASK = 0xFFFF0000;
    
    /**
     * Creat instance of class for element name. This implements substitution group handling, by checking the current
     * element start tag name against the expected element name, and if they're not the same finding the appropriate
     * class based on the substitution group rooted on the expected element name (which must be a global element name).
     *
     * @param root global root element name, including namespace URI, in "lname{uri}" form
     * @param rdr reader
     * @param inst supplied instance of root element class or subclass (<code>null</code> if none)
     * @return instance of appropriate class to use for unmarshalling (may be the same as the provided instance)
     */
    Object createElementInstance(String root, XmlReader rdr, Object inst);
    
    /**
     * Validate instance of class for type name. This implements type substitution handling, by checking for an override
     * xsi:type specification on the current element start tag, and if the type is different from the default finding
     * the appropriate class and returning an instance.
     *
     * @param dflt global default complexType name, including namespace URI, in "lname{uri}" form
     * @param rdr reader
     * @param inst supplied instance of default type class or subclass (<code>null</code> if none)
     * @return instance of appropriate class to use for unmarshalling (may be the same as the provided instance)
     */
    Object createTypeInstance(String dflt, XmlReader rdr, Object inst);
    
    /**
     * Get namespaces defined in mapping. The returned array is indexed by the namespace index number used when
     * marshalling.
     * 
     * @return array of namespaces defined in binding (<code>null</code> if not an output binding)
     */
    String[] getNamespaces();
    
    /**
     * Get initial prefixes for namespaces defined in mapping. The returned array is indexed by the namespace index
     * number used when marshalling. Note that these are only the first prefixes associated with each namespace; it's
     * possible to reuse the namespace in the binding with a different prefix.
     * 
     * @return array of prefixes for namespaces defined in binding (<code>null</code> if not an output binding)
     */
    String[] getPrefixes();
    
    /**
     * Get mapped class names (or type names, in the case of abstract mappings). Returns array of fully-qualified class
     * and/or type names, ordered by index number of the class.
     * 
     * @return array of class names
     */
    String[] getMappedClasses();
    
    /**
     * Get namespaces of elements corresponding to mapped classes. The returned array uses the same ordering as the
     * result of the {@link #getMappedClasses} call. Entries in the array are <code>null</code> if there is no element
     * for a class or the element is in the default namespace.
     * 
     * @return array of element namespaces
     */
    String[] getElementNamespaces();
    
    /**
     * Get names of elements corresponding to mapped classes. The returned array uses the same ordering as the result of
     * the {@link #getMappedClasses} call. Entries in the array are <code>null</code> if there is no element for a
     * class.
     * 
     * @return array of element names
     */
    String[] getElementNames();
}