/* -*- Mode: C++; tab-width: 8; indent-tabs-mode: nil; c-basic-offset: 2 -*- */
/* vim: set ts=8 sts=2 et sw=2 tw=80: */
/* This Source Code Form is subject to the terms of the Mozilla Public
 * License, v. 2.0. If a copy of the MPL was not distributed with this
 * file, You can obtain one at http://mozilla.org/MPL/2.0/. */

#include "nsNSSIOLayer.h"
#include "sslproto.h"
#include "sslerr.h"

#include "gtest/gtest.h"

/* The following buffers are generated by encoding a sequence of integers from
   0 to 100 using the corresponding encoding mechanism.

  They could be re-generated using the following script:

  generateZlibBuf():
    import zlib
    import brotli
    import zstd

    for i in range(100):
      a.append(i)

    encodedBufferExampleZlib = list(zlib.compress(bytes(a)))
    encodedBufferExampleBrotli = list(brotli.compress(bytes(a)))
    encodedBufferExampleZstd = list(zstd.compress(bytes(a)))
*/

uint8_t encodedBufferExampleZlib[108] = {
    0x78, 0x9c, 0x63, 0x60, 0x64, 0x62, 0x66, 0x61, 0x65, 0x63, 0xe7, 0xe0,
    0xe4, 0xe2, 0xe6, 0xe1, 0xe5, 0xe3, 0x17, 0x10, 0x14, 0x12, 0x16, 0x11,
    0x15, 0x13, 0x97, 0x90, 0x94, 0x92, 0x96, 0x91, 0x95, 0x93, 0x57, 0x50,
    0x54, 0x52, 0x56, 0x51, 0x55, 0x53, 0xd7, 0xd0, 0xd4, 0xd2, 0xd6, 0xd1,
    0xd5, 0xd3, 0x37, 0x30, 0x34, 0x32, 0x36, 0x31, 0x35, 0x33, 0xb7, 0xb0,
    0xb4, 0xb2, 0xb6, 0xb1, 0xb5, 0xb3, 0x77, 0x70, 0x74, 0x72, 0x76, 0x71,
    0x75, 0x73, 0xf7, 0xf0, 0xf4, 0xf2, 0xf6, 0xf1, 0xf5, 0xf3, 0x0f, 0x08,
    0x0c, 0x0a, 0x0e, 0x09, 0x0d, 0x0b, 0x8f, 0x88, 0x8c, 0x8a, 0x8e, 0x89,
    0x8d, 0x8b, 0x4f, 0x48, 0x4c, 0x4a, 0x06, 0x00, 0x8b, 0x7c, 0x13, 0x57,
};

uint8_t encodedBufferExampleBrotli[73] = {
    0x1b, 0x63, 0x0,  0xe8, 0x65, 0x60, 0x9e, 0x29, 0x8f, 0xd2, 0x59,
    0x86, 0xc8, 0x54, 0x48, 0x4c, 0xb0, 0x89, 0x63, 0x77, 0x7f, 0x0,
    0x40, 0x8,  0x46, 0x50, 0xc,  0x27, 0x48, 0x8a, 0x66, 0x58, 0x8e,
    0x17, 0x44, 0x49, 0x56, 0x54, 0x4d, 0x37, 0x4c, 0xcb, 0x76, 0x5c,
    0xcf, 0xf,  0xc2, 0x28, 0x4e, 0xd2, 0x2c, 0x2f, 0xca, 0xaa, 0x6e,
    0xda, 0xae, 0x1f, 0xc6, 0x69, 0x5e, 0xd6, 0x6d, 0x3f, 0xce, 0xeb,
    0x7e, 0x5e, 0xef, 0xcf, 0xf7, 0xf7, 0x7};

uint8_t encodedBufferExampleZstd[113] = {
    0x28, 0xb5, 0x2f, 0xfd, 0x24, 0x64, 0x21, 0x03, 0x00, 0x00, 0x01, 0x02,
    0x03, 0x04, 0x05, 0x06, 0x07, 0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e,
    0x0f, 0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17, 0x18, 0x19, 0x1a,
    0x1b, 0x1c, 0x1d, 0x1e, 0x1f, 0x20, 0x21, 0x22, 0x23, 0x24, 0x25, 0x26,
    0x27, 0x28, 0x29, 0x2a, 0x2b, 0x2c, 0x2d, 0x2e, 0x2f, 0x30, 0x31, 0x32,
    0x33, 0x34, 0x35, 0x36, 0x37, 0x38, 0x39, 0x3a, 0x3b, 0x3c, 0x3d, 0x3e,
    0x3f, 0x40, 0x41, 0x42, 0x43, 0x44, 0x45, 0x46, 0x47, 0x48, 0x49, 0x4a,
    0x4b, 0x4c, 0x4d, 0x4e, 0x4f, 0x50, 0x51, 0x52, 0x53, 0x54, 0x55, 0x56,
    0x57, 0x58, 0x59, 0x5a, 0x5b, 0x5c, 0x5d, 0x5e, 0x5f, 0x60, 0x61, 0x62,
    0x63, 0x97, 0x65, 0x16, 0x32};

struct CertificateCompressionTestCase {
  uint8_t* encodedBuf;
  unsigned int encodedBufLen;
  SECStatus (*decode)(const SECItem* input, unsigned char* output,
                      size_t outputLen, size_t* usedLen);
};

static const CertificateCompressionTestCase CertificateCompressionTestCases[] =
    {{encodedBufferExampleZlib, sizeof(encodedBufferExampleZlib),
      zlibCertificateDecode},
     {encodedBufferExampleBrotli, sizeof(encodedBufferExampleBrotli),
      brotliCertificateDecode},
     {encodedBufferExampleZstd, sizeof(encodedBufferExampleZstd),
      zstdCertificateDecode}};

class psm_CertificateCompression
    : public ::testing::Test,
      public ::testing::WithParamInterface<CertificateCompressionTestCase> {};

TEST_P(psm_CertificateCompression, DecoderCorrectlyDecodes) {
  const CertificateCompressionTestCase& testCase(GetParam());

  SECItem encodedItem = {siBuffer, testCase.encodedBuf, testCase.encodedBufLen};
  uint8_t decodedEncodedBuffer[100] = {0};
  size_t usedLen = 0;

  SECStatus rv = testCase.decode(&encodedItem, decodedEncodedBuffer,
                                 sizeof(decodedEncodedBuffer), &usedLen);
  ASSERT_EQ(SECSuccess, rv);

  ASSERT_EQ(usedLen, sizeof(decodedEncodedBuffer));
  for (size_t i = 0; i < usedLen; i++) {
    ASSERT_EQ(decodedEncodedBuffer[i], i);
  }
}

TEST_P(psm_CertificateCompression, DecodingFailsEmptyInputItem) {
  const CertificateCompressionTestCase& testCase(GetParam());
  uint8_t decodedEncodedBuffer[500] = {0};
  size_t usedLen = 0;
  /* nullptr instead of input */
  SECStatus rv = testCase.decode(nullptr, decodedEncodedBuffer,
                                 sizeof(decodedEncodedBuffer), &usedLen);
  ASSERT_EQ(SECFailure, rv);
}

TEST_P(psm_CertificateCompression, DecodingFailsEmptyInput) {
  const CertificateCompressionTestCase& testCase(GetParam());
  SECItem encodedItem = {siBuffer, nullptr,
                         (unsigned int)sizeof(encodedBufferExampleZlib)};
  uint8_t decodedEncodedBuffer[500] = {0};
  size_t usedLen = 0;
  /* nullptr instead of input.data */
  SECStatus rv = testCase.decode(&encodedItem, decodedEncodedBuffer,
                                 sizeof(decodedEncodedBuffer), &usedLen);
  ASSERT_EQ(SECFailure, rv);
}

TEST_P(psm_CertificateCompression, DecodingFails0LenInput) {
  const CertificateCompressionTestCase& testCase(GetParam());
  SECItem encodedItem = {siBuffer, encodedBufferExampleZlib, 0};
  uint8_t decodedEncodedBuffer[500] = {0};
  size_t usedLen = 0;
  /* 0 instead of input.len*/
  SECStatus rv = testCase.decode(&encodedItem, decodedEncodedBuffer,
                                 sizeof(decodedEncodedBuffer), &usedLen);
  ASSERT_EQ(SECFailure, rv);
}

TEST_P(psm_CertificateCompression, DecodingFailsEmptyBuffer) {
  const CertificateCompressionTestCase& testCase(GetParam());
  unsigned char encodedBuffer[500] = {0};
  SECItem encodedItem = {siBuffer, encodedBuffer, 500};
  uint8_t decodedEncodedBuffer[500] = {0};
  size_t usedLen = 0;
  /* Empty buffer will return an error if decoded. */
  SECStatus rv = testCase.decode(&encodedItem, decodedEncodedBuffer,
                                 sizeof(decodedEncodedBuffer), &usedLen);
  ASSERT_EQ(SECFailure, rv);
}

TEST_P(psm_CertificateCompression, DecodingFailsNullOutput) {
  const CertificateCompressionTestCase& testCase(GetParam());
  SECItem encodedItem = {siBuffer, encodedBufferExampleZlib,
                         (unsigned int)sizeof(encodedBufferExampleZlib)};
  uint8_t decodedEncodedBuffer[5] = {0};
  size_t usedLen = 0;
  /* Empty buffer will return an error if decoded. */
  SECStatus rv = testCase.decode(&encodedItem, nullptr,
                                 sizeof(decodedEncodedBuffer), &usedLen);
  ASSERT_EQ(SECFailure, rv);
}

TEST_P(psm_CertificateCompression, DecodingFailsOutputTooSmall) {
  const CertificateCompressionTestCase& testCase(GetParam());
  SECItem encodedItem = {siBuffer, encodedBufferExampleZlib,
                         (unsigned int)sizeof(encodedBufferExampleZlib)};
  uint8_t decodedEncodedBuffer[5] = {0};
  size_t usedLen = 0;
  /* Empty buffer will return an error if decoded. */
  SECStatus rv = testCase.decode(&encodedItem, decodedEncodedBuffer,
                                 sizeof(decodedEncodedBuffer), &usedLen);
  ASSERT_EQ(SECFailure, rv);
}

INSTANTIATE_TEST_SUITE_P(psm_CertificateCompression, psm_CertificateCompression,
                         ::testing::ValuesIn(CertificateCompressionTestCases));
