/*
   File: wff2ps.c
   Authors: Ron Blanford,
            K.R. Sloan,
	    James Painter
            Stephen Mann  
   Last Modified: 17 January 1992
   Purpose: Output an WFF image in postscript.
            This version accepts input on stdin
            This version handles color 
 */

#include <stdio.h>
#include <wff.h>
#include <assert.h>

typedef unsigned char byte;

#ifndef TRUE
#define TRUE 1
#define FALSE 0
#endif

static bit = 0;
static byte out = 0;
static width = 0;
static BitsPerPixel = 0;
static char *RoutineName;
static int FromFile = 0;
static char WFFfile[80];
static double xsize = -1, ysize = -1, dx = 0.5, dy = 0.5;
static int TeXembed = FALSE;
static int invert = FALSE;
static int color = 0;
extern double atof();

char hexlookup[] = {'0', '1', '2', '3',
                    '4', '5', '6', '7',
                    '8', '9', 'A', 'B',
                    'C', 'D', 'E', 'F' };

#define putnibble(val)\
{\
   putchar( hexlookup[val] );\
   if (++width >= 80)     {putchar('\n'); width=0;}\
   }

#define putpixel(inval) \
{\
  int i, j;\
  char buf[4];\
  int val = inval;\
  if (BitsPerPixel >= 4)\
    {\
       for(i=0, j=0; i<BitsPerPixel; i+=4, j++)\
	 {\
	   buf[j] = val & 0xF;\
	   val >>= 4;\
	 }\
       for( j-- ; j>=0; j--)\
	   putnibble( buf[j] );\
   } else {\
     out <<= BitsPerPixel;\
     out |= val;\
     bit += BitsPerPixel;\
     if (bit == 4)\
       {\
	 putnibble(out);\
	 out = 0;\
	 bit = 0;\
       }\
   }\
}


Usage()
 {
  fprintf(stderr,
"Usage:\n\t%s [WFF_filename] [-embed] [-invert] [-s sizex sizey] [-o dx dy]\n\
                                (Units are inches)\n"
   ,RoutineName);
 }

main(argc,argv)
 int argc;
 char *argv[];
 {
  int ScanLength, ScanCount;
  int x,y;
  byte *ScanLine;
  int mask, PixelsPerByte, ScanLineSlop;
  FrameBufferType *inFB;
  FILE *inFile;
  unsigned short Pixel[99];
  char str[80], WhatBands[80];
  int Left, Right, Top, Bottom, BitsPerBand;

  ParseCommandLine( argc, argv );

  /* Open input file */
  inFB = (FrameBufferType *)0;
  OpenFB (&inFB);

  if (FromFile)
   {
    inFile = fopen( WFFfile, "r" );
    if ((FILE *)0 == inFile)
     {
      sprintf(str,"Unable to open input file: %s");
      perror(str);
      exit(1);
     }
   }
  else inFile = stdin;

  PassImageIn (inFile, inFB);
  GetBounds (inFB, &Bottom, &Left, &Top, &Right);
  GetColorSystem (inFB, WhatBands, &BitsPerPixel);

  if      (0 == strncmp(WhatBands, "RGB", 3))  color = -1;
  else if (0 == strncmp(WhatBands, "I"  , 1))  color =  0;
  else 
   {
    color = 0;
    fprintf(stderr, 
            "Warning:  ColorSystem: %s.  Treating first band as Intensity\n",
	     WhatBands );
   }

  ScanLength   = Right - Left + 1;
  ScanCount    = Top - Bottom + 1;
  if (   (1 != BitsPerPixel)
      && (2 != BitsPerPixel)
      && (4 != BitsPerPixel)
      && (8 != BitsPerPixel))
   {
    fprintf( stderr, 
"Sorry, this program can only deal with images that have 1,2,4, or 8\n\
bits per pixel.\n" );
    exit(-1);
   }

  if (xsize < 0) 
   {
    xsize = 8.0*(double)ScanLength / 300.0;
    if (xsize > 7.5) xsize = 7.5;
    ysize = xsize * (double) ScanCount / (double) ScanLength;
    if (ysize > 10.5)
     {
      ysize = 10.5;
      xsize = ysize * (double) ScanLength / (double) ScanCount;
     } 
  }

  WritePostscriptHeader (ScanLength, ScanCount, BitsPerPixel,
                         xsize, ysize, dx, dy, invert );

  assert (ScanLine = (byte *) malloc( ScanLength ));

  mask = (1 << BitsPerPixel) - 1;
  PixelsPerByte = (8 / BitsPerPixel);

  if (color)
   ScanLineSlop  = ((3*ScanLength) % PixelsPerByte);
  else
   ScanLineSlop  = ( ScanLength    % PixelsPerByte);

  if (0 != ScanLineSlop)
   ScanLineSlop = PixelsPerByte - ScanLineSlop;

  for (y=ScanCount-1; y>=0; y--) 
   {
    for (x=0; x<ScanLength; x++) 
     {
      NextPixelIn( inFB, Pixel );
      putpixel (Pixel[0]);
      if (color) { putpixel(Pixel[1]); putpixel(Pixel[2]); }
     }
    for(x=0; x<ScanLineSlop; x++)  /* Scanlines must end on an even byte */
     { putpixel( 0 ); }

    if (0 != width) {putchar('\n'); width = 0;}
   }
  WritePostscriptTrailer();
  exit(0);
 }

WritePostscriptHeader (width, height, BitsPerPixel, sx, sy, dx, dy, invert)
 int width, height;
 int BitsPerPixel;
 double sx, sy, dx, dy;
 int invert;
 {
  char host[20];
  int curtime;

  gethostname(host, sizeof host);
  curtime = time(0);
  printf( "%%!PS-Adobe-1.0\n");
  printf( "%%%%Creator: %s:%s\n", host, getlogin());
  printf( "%%%%CreationDate: %s", ctime(&curtime));
  printf( "%%%%BoundingBox: %d %d %d %d\n",
	 (int)(72*dx),              (int)(72*dy), 
	 (int)(72*sx)+(int)(72*dx),(int)(72*sy)+(int)(72*dy));
  printf( "%%%%EndProlog\n");
  printf( "%%%%Page: ? 1\n\n");
  printf( "/MyDict 10 dict def\nMyDict begin\n");

  if (invert) 
   {
    printf( "\
%% Invert the intensity function\n\
{ 1 exch sub } settransfer\n\
\n\
");
   }

  if (color) 
   {
    printf("\
\n\
/picstr %d string def\n\
\n\
%d  %d  translate\n\
%d  %d  scale\n\
\n\
%d %d %d [%d 0 0 %d 0 0]\n\
{ currentfile picstr readhexstring pop } \n\
false 3 colorimage\n\
", 
    ((width * BitsPerPixel) + 7)/8,
    (int) (72*dx), 
    (int) (72*dy),
    (int) (72*sx), 
    (int) (72*sy), 
    width,  height, BitsPerPixel, width, height );
   }
  else 
   {
    printf("\
\n\
/picstr %d string def\n\
\n\
%d  %d  translate\n\
%d  %d  scale\n\
\n\
%d %d %d [%d 0 0 %d 0 0]\n\
{ currentfile picstr readhexstring pop } \n\
image\n\
", 
     ((width * BitsPerPixel) + 7)/8,
     (int) (72*dx), 
     (int) (72*dy),
     (int) (72*sx), 
     (int) (72*sy), 
     width,  height, BitsPerPixel, width, height );
  }
 }  


WritePostscriptTrailer()
 {
  if (!TeXembed)
   printf("\nshowpage\n");
  else
   printf("\n");
  printf("end\n");
 }


ParseCommandLine (argc, argv)
 int argc;
 char *argv[];
 {
  int nNames = 0;
  int ArgsParsed = 0;

  RoutineName = argv[ArgsParsed++];
  
  FromFile = 0;;

  for ( ; ArgsParsed < argc; ArgsParsed++ ) 
   {
    if (argv[ArgsParsed][0] == '-')
     { /*Switch*/
      switch ( argv[ArgsParsed][1] ) 
       {
        case 'e':
         if (strcmp(argv[ArgsParsed],"-embed") == 0) TeXembed = TRUE;
         else { Usage(); exit(1); }
         break;
        case 'i':
         if (0 == strcmp(argv[ArgsParsed],"-invert")) invert = TRUE;
         else { Usage(); exit(1); }
         break;
        case 'o':
         if (ArgsParsed < argc) dx = atof(argv[++ArgsParsed]);
         else { Usage(); exit(1); }
         if (ArgsParsed < argc)  dy = atof(argv[++ArgsParsed]);
         else { Usage(); exit(1); }
         break;
        case 's':
         if (ArgsParsed < argc) xsize = atof(argv[++ArgsParsed]);
         else { Usage(); exit(1); }
         if (ArgsParsed < argc) ysize = atof(argv[++ArgsParsed]);
         else { Usage(); exit(1); }
         break;
        default:
         Usage();
         exit(1);
       }
     }
    else
     {
      switch (++nNames) 
       {
        case 1:  strcpy(WFFfile, argv[ArgsParsed]); FromFile = -1; break;
       }
     }
   }
  switch (nNames) 
   {
    case 0:
    case 1:
     break;			/* OK */
    default:
     Usage();
     exit(1);
   }
 }
