/*
  File: Cubic.c 
  Authors: K.R. Sloan
  Last Modified: 28 June 1989
  Purpose: Convolve with a cubic filter, parameterized by:
             B, C - as described in Mitchell, SIGGRAPH '88 (default 1/3, 1/3)
             z = distance to the first zero-crossing (default 2.0)

             g = gain          (default 1.0)
             d = dc            (default 0.0)  
           Gain is the area under EACH of the two 1D filters.
           DC is the amount added to EACH of the two 1D filters.

           An important special case is when the images are offset
           so that middle-gray is considered to be the "zero" value.

           In this case, when we apply a gain factor g, we should
           automatically apply a DC correction of (g-1.0)/2.0.

           To do this, we offer a composite parameter:
             G = gain          (default 1.0)
                 dc = (gain-1.0)/2.0

           Yet another special case is (very) scattered data points
           in a field of gray.  Here, we want to set the gain (and DC)
           indirectly by setting the peak value of the kernel.  So...another
           parameter (-P). 

             P = peak [gain and dc derived indirectly]

 */

#include <stdio.h>
#include <math.h>
#include "wff.h"

#define ROUND(x) ((int)(0.5+(x)))

#define SwitchChar '-'
#define MaxFilterWidth 1000

double atof();


static int VERBOSE = 0;
static char *RoutineName;
static void usage()
 {
  fprintf(stderr, "Usage is\n\t%s [-h][-v][-B B][-C C][-z z][-g gain][-d DC][-G gain][-P p]\n",
              RoutineName);
 }

int main(argc, argv)
 int argc;
 char *argv[]; 
 {
  int ArgsParsed = 0;
  double B = 0.34;
  double C = 0.33;
  double z = 2.0;
  double gain = 1.0;
  double DC = 0.0;
  double Peak = 0.0;
  int n, HalfN, k, k0, v;
  int Kernel[MaxFilterWidth];
  double c, u, u2, u3, One;
  int Sum;
  double Correction;
  int Correction2;

  RoutineName = argv[ArgsParsed++];

  while (argc > ArgsParsed)
   {
    if (SwitchChar == argv[ArgsParsed][0])
     switch (argv[ArgsParsed++][1])
      {
       case 'B': B     = atof(argv[ArgsParsed++]); break;
       case 'C': C     = atof(argv[ArgsParsed++]); break;
       case 'z': z     = atof(argv[ArgsParsed++]); break;
       case 'g': gain  = atof(argv[ArgsParsed++]); break;
       case 'd': DC    = atof(argv[ArgsParsed++]); break;
       case 'G': gain  = atof(argv[ArgsParsed++]);
                 DC    = 0.5*(1.0-gain);
                 break;
       case 'P': Peak  = atof(argv[ArgsParsed++]); break;
       case 'v': VERBOSE = 1; break;
       case 'h':
       default:  usage(); exit(-1);  
      }
    else  { usage(); exit (-1); }	
   }

  HalfN = ROUND(2.0*z);
  n = 2+HalfN+HalfN;
  if (n>MaxFilterWidth)
    {
     n = MaxFilterWidth;
     HalfN = (n-2)/2;      /* belt */
     n = 2+HalfN+HalfN;    /* suspenders */
    } 

  if (0.0 != Peak)
   {
    gain = Peak*z;
    DC   = 0.5*(1.0-gain);
    c    = 2048.0*Peak;
   }
  else c = 2048.0*gain/z;

  c /= 6.0;
  Kernel[0] = ROUND(2048.0*DC);
  for(k=0;k<=HalfN;k++)
   {
    u = (double)k/z;    u2 = u*u;  u3 = u2*u;
    if (u<1.0)
     v = c*((( 12.0-( 9.0*B)-( 6.0*C))*u3)
           +((-18.0+(12.0*B)+( 6.0*C))*u2)
           +(   6.0-( 2.0*B)             ));
    else if (u<2.0)
     v = c*(((     -(     B)-( 6.0*C))*u3)
           +((      ( 6.0*B)+(30.0*C))*u2) 
           +((     -(12.0*B)-(48.0*C))*u )
           +((      ( 8.0*B)+(24.0*C))   )); 
    else 
     v = 0;
    Kernel[1+HalfN+k] = v;
    Kernel[1+HalfN-k] = v;
   }

  /* correct errors due to truncating the kernel */
   
  Sum = 0;  for(k=1;k<n;k++) Sum += Kernel[k];
  Correction = ROUND(2048.0*gain)/(double)Sum;
  if (VERBOSE)
   fprintf(stderr,
  "B = %f; C = %f; z = %f; gain = %f; DC = %f\nSum = %d; Correction = %f\n",
           B,C, z, gain, DC, Sum, Correction);

  /* correct round-off errors */

  for (k=1;k<n;k++) Kernel[k] = (int)((double)Kernel[k]*Correction);
  Sum = 0;  for(k=1;k<n;k++) Sum += Kernel[k];
  Correction2 = ROUND(2048.0*gain)-Sum;
  Kernel[1+HalfN] += Correction2;

  if (VERBOSE)
   {
    fprintf(stderr,"Secondary Correction = %f\n",Correction2/2048.0);
   }

  /* look for zero tails - these are common with wide filters */
  for(k0=0;(k0<HalfN) && (0 == Kernel[1+k0]);k0++) ;

  if (0<k0)
   {
    HalfN = HalfN-k0;
    n = 2+HalfN+HalfN;
    for(k=1;k<n;k++) Kernel[k] = Kernel[k+k0];
   }

  if (VERBOSE)
   {
    fprintf(stderr,"DC  : %15.12f\n",(double)Kernel[0]/2048.0);
    for(k=1;k<n;k++)
     fprintf(stderr,"%4d: %15.12f\n", k-HalfN-1,(double)Kernel[k]/2048.0);
   } 

  (void)wffConvHV(stdin,stdout,n,n,Kernel,Kernel);

  exit (0);
 }
