/*
  File: SGP.c
  Author: K.R. Sloan
  Last Modified: 18 December 1989
  Purpose: definitions for a Simple Graphics Package
           this is a generic SGP - device specific code is in
           the device drivers - see DEVICE.h

           Three display types are supported:
             Color - colors specified as R,G,B.
             Mono  - Intensity only - to get greater gray-scale
                     range out of displays with limited framebuffer
                     depth
             Debug - output a PostScript-style trace.  This is 
                     useful for debugging, or for piping into a 
                     suitable interpreter...


           Some DEVICEs send output to devOutputStream.  This version
           of SGP directs all DEVICE output to stdout.  A later version
           may allow the SGP user to specify this.  Gonzo hackers may
           want to experiment by changing devOutputStream AFTER calling
           sgpInit.

           Some DEVICEs have a locator device attached.  sgpPick lets
           you point and click.  The Cursor is handled automatically.
 */
#include <stdio.h>
#include <math.h>
#include "DEVICE.h"
#include "SGP.h"

/* SGP GLOBALS */
static sgpStateType CurrentState;
static int Active = 0;
static double epsilon = 0.00000000001;

/* Private functions */

extern int SGPSpline();

static double Interpolate(Old, OldMin, OldMax, NewMin, NewMax)
 double Old, OldMin, OldMax, NewMin, NewMax;
 {
  double OldRange;
  OldRange = OldMax-OldMin;
  if ( fabs(OldRange) < epsilon )
   return (NewMin+NewMax)/2.0;
  else
   return NewMin+((Old-OldMin)*(NewMax-NewMin)/OldRange);
 }

/* Exported Functions */

void sgpInit(UseDisplay)
 sgpDisplayType UseDisplay;
 {
  if (Active) sgpQuit();
  CurrentState.Display=UseDisplay;
  switch (CurrentState.Display)
   {
    case ColorDisplay: 
     if ( 0 != devInit(1,stdout))
      {
       fprintf(stderr,
       "sgpInit: ColorDisplay unavailable-continuing as DebugDisplay\n");
       CurrentState.Display=DebugDisplay;
      }  
     break;
    case MonochromeDisplay: 
     if ( 0 != devInit(0,stdout))
      {
       fprintf(stderr,
       "sgpInit: MonochromeDisplay unavailable-continuing as DebugDisplay\n");
       CurrentState.Display=DebugDisplay;
      }  
     break;
    case DebugDisplay:
     break;
    default: 
     fprintf(stderr,"sgpInit:unknown display, continuing as DebugDisplay");
     CurrentState.Display=DebugDisplay;
     break;
   }

  if (DebugDisplay == CurrentState.Display)
   fprintf(stdout,"%d sgpInit\n",UseDisplay);

  CurrentState.Window.Left   = 0.0;
  CurrentState.Window.Bottom = 0.0;
  CurrentState.Window.Right  = 1.0;
  CurrentState.Window.Top    = 1.0;
  CurrentState.Viewport.Left   = 0.0;
  CurrentState.Viewport.Bottom = 0.0;
  CurrentState.Viewport.Right  = 1.0;
  CurrentState.Viewport.Top    = 1.0;
  CurrentState.Normalized.Left   = 0.0;
  CurrentState.Normalized.Bottom = 0.0;
  CurrentState.Normalized.Right  = 1.0;
  CurrentState.Normalized.Top    = 1.0;
  if (abs(devRight-devLeft) > abs(devTop-devBottom))
   {
    CurrentState.Physical.Bottom = devBottom;
    CurrentState.Physical.Top    = devTop;
    if (devRight>devLeft)
     {
      CurrentState.Physical.Left 
          = (double)(devRight+devLeft-abs(devTop-devBottom))/2.0;
      CurrentState.Physical.Right
          = CurrentState.Physical.Left + (double)abs(devTop-devBottom);
     }
    else
     {
      CurrentState.Physical.Right 
          = (double)(devRight+devLeft-abs(devTop-devBottom))/2.0;
      CurrentState.Physical.Left
          = CurrentState.Physical.Left + (double)abs(devTop-devBottom);
     }   
   }
  else
   {
    CurrentState.Physical.Left  = devLeft;
    CurrentState.Physical.Right = devRight;
    if (devTop>devBottom)
     {
      CurrentState.Physical.Bottom
          = (double)(devTop+devBottom-abs(devLeft-devRight))/2.0;
      CurrentState.Physical.Top
          = CurrentState.Physical.Bottom + (double)abs(devLeft-devRight);
     }
    else
     {
      CurrentState.Physical.Top
          = (double)(devTop+devBottom-abs(devLeft-devRight))/2.0;
      CurrentState.Physical.Bottom
          = CurrentState.Physical.Top + (double)abs(devLeft-devRight);
     }   
   }
  CurrentState.Color.r = 0.8;
  CurrentState.Color.g = 0.8;
  CurrentState.Color.b = 0.8;
  Active = 1;
  switch (CurrentState.Display)
   {
    case ColorDisplay:
    case MonochromeDisplay:
     devSetClipWin((int) CurrentState.Physical.Left,
                   (int) CurrentState.Physical.Top,
                   (int) CurrentState.Physical.Right,
                   (int) CurrentState.Physical.Bottom
                  );
     break;
    default:
     break;
   }
 }

void sgpQuit()
 {
  if (!Active) return;
  switch (CurrentState.Display)
   {
    case ColorDisplay:
    case MonochromeDisplay:
     (void) devClose();
     break;
    case DebugDisplay:
     fprintf(stdout,"sgpQuit\n");
     break;
    default:
     break;
   }
  Active = 0;
 }

void sgpColor(NewColor)
 sgpColorType NewColor;
 {
  CurrentState.Color = NewColor;
  if (DebugDisplay == CurrentState.Display)
   fprintf(stdout,"[%f %f %f] sgpColor\n",
                   NewColor.r, NewColor.g, NewColor.b);
 }

void sgpGrayLevel(NewGrayLevel)
 sgpGrayType NewGrayLevel;
 {
  CurrentState.Color.r = NewGrayLevel;
  CurrentState.Color.g = NewGrayLevel;
  CurrentState.Color.b = NewGrayLevel;
  if (DebugDisplay == CurrentState.Display)
   fprintf(stdout,"%f sgpGrayLevel\n", NewGrayLevel);
 }

void sgpInquire(State)
 sgpStateType *State;
 {
  *State = CurrentState;
 }

void sgpSetWindow(NewWindow)
 sgpRectangleType NewWindow;
 {
  CurrentState.Window = NewWindow;
  if (DebugDisplay == CurrentState.Display)
   fprintf(stdout,"%f %f %f %f sgpSetWindow\n",
                     CurrentState.Window.Left,
                     CurrentState.Window.Bottom,
                     CurrentState.Window.Right,
                     CurrentState.Window.Top);
 }

void sgpSetViewport(NewViewport)
 sgpRectangleType NewViewport;
 {
  int dLeft, dRight, dBottom, dTop, t;
  if (!Active) return;
  CurrentState.Viewport = NewViewport;
  switch (CurrentState.Display)
   {
    case ColorDisplay:
    case MonochromeDisplay:
     dLeft   = CurrentState.Physical.Left 
                 + (CurrentState.Viewport.Left
                    *(CurrentState.Physical.Right-CurrentState.Physical.Left)
                   );
     dTop    = CurrentState.Physical.Bottom
                 + (CurrentState.Viewport.Top
                    *(CurrentState.Physical.Top-CurrentState.Physical.Bottom)
                   );
     dRight  = CurrentState.Physical.Left 
                 + (CurrentState.Viewport.Right
                    *(CurrentState.Physical.Right-CurrentState.Physical.Left)
                   );
     dBottom = CurrentState.Physical.Bottom
                 + (CurrentState.Viewport.Bottom
                    *(CurrentState.Physical.Top-CurrentState.Physical.Bottom)
                   );

     devSetClipWin(dLeft, dTop, dRight, dBottom);
     break;
    case DebugDisplay:
     fprintf(stdout,"%f %f %f %f sgpSetViewport\n",
                     CurrentState.Viewport.Left, 
                     CurrentState.Viewport.Bottom,
                     CurrentState.Viewport.Right,
                     CurrentState.Viewport.Top);
     break;
    default:
     break;
   }
 }

void sgpWorldToScreen(World, Screen)
 sgpPointType World, *Screen;
 {
  Screen->x = Interpolate(World.x,
                          CurrentState.Window.Left, 
                          CurrentState.Window.Right,
                          CurrentState.Viewport.Left,
                          CurrentState.Viewport.Right);
  Screen->y = Interpolate(World.y,
                          CurrentState.Window.Bottom, 
                          CurrentState.Window.Top,
                          CurrentState.Viewport.Bottom,
                          CurrentState.Viewport.Top);
 }

void sgpScreenToWorld(Screen, World)
 sgpPointType Screen, *World;
 {
  World->x = Interpolate(Screen.x,
                         CurrentState.Viewport.Left, 
                         CurrentState.Viewport.Right,
                         CurrentState.Window.Left,
                         CurrentState.Window.Right);
  World->y = Interpolate(Screen.y,
                         CurrentState.Viewport.Bottom, 
                         CurrentState.Viewport.Top,
                         CurrentState.Window.Bottom,
                         CurrentState.Window.Top);
 }

void sgpScreenToPhysical(Screen, Physical)
 sgpPointType Screen, *Physical;
 {
  Physical->x = Interpolate(Screen.x,
                            CurrentState.Normalized.Left, 
                            CurrentState.Normalized.Right,
                            CurrentState.Physical.Left,
                            CurrentState.Physical.Right);
  Physical->y = Interpolate(Screen.y,
                            CurrentState.Normalized.Bottom, 
                            CurrentState.Normalized.Top,
                            CurrentState.Physical.Bottom,
                            CurrentState.Physical.Top);
 }

void sgpPhysicalToScreen(Physical, Screen)
 sgpPointType Physical, *Screen;
 {
  Screen->x = Interpolate(Physical.x,
                          CurrentState.Physical.Left,
                          CurrentState.Physical.Right,
                          CurrentState.Normalized.Left, 
                          CurrentState.Normalized.Right);
  Screen->y = Interpolate(Physical.y,
                          CurrentState.Physical.Bottom,
                          CurrentState.Physical.Top,
                          CurrentState.Normalized.Bottom, 
                          CurrentState.Normalized.Top);
 }

void sgpClearScreen()
 {
  if (!Active) return;
  switch (CurrentState.Display)
   {
    case ColorDisplay:
    case MonochromeDisplay:
     (void)devClearScreen(devColorCode(CurrentState.Color.r,
                                       CurrentState.Color.g,
                                       CurrentState.Color.b));
     break;
    case DebugDisplay:
     fprintf(stdout,"sgpClearScreen\n");
     break;
    default: 
     break;
   }
 }

void sgpPoint(Point)
 sgpPointType Point;
 {
  sgpPointType Screen, Physical;
  int x, y;

  if (!Active) return;
  sgpWorldToScreen(Point, &Screen);
  sgpScreenToPhysical(Screen, &Physical);
  switch (CurrentState.Display)
   {
    case ColorDisplay:
    case MonochromeDisplay:
     x = Physical.x; y = Physical.y;
     (void)devSetColor(devColorCode(CurrentState.Color.r,
                                 CurrentState.Color.g,
                                 CurrentState.Color.b));
     (void)devRect(x, y, x, y);
     break;
    case DebugDisplay:
     fprintf(stdout,"[%f %f] sgpPoint\n", Point.x, Point.y);
     break;
    default:
      break;
   }
 }

void sgpDot(Point, ScreenSize)
 sgpPointType Point;
 int ScreenSize;
 {
  sgpPointType Screen, Physical;
  int x, y;

  if (!Active) return;
  sgpWorldToScreen(Point, &Screen);
  sgpScreenToPhysical(Screen, &Physical);
  switch (CurrentState.Display)
   {
    case ColorDisplay:
    case MonochromeDisplay:
     x = Physical.x; y = Physical.y;
     (void)devSetColor(devColorCode(CurrentState.Color.r,
                                 CurrentState.Color.g,
                                 CurrentState.Color.b));
     (void)devDisc((int)Physical.x, (int)Physical.y, ScreenSize);
     break;
    case DebugDisplay:
     fprintf(stdout,"[%f %f] %d sgpDot\n", Point.x, Point.y, ScreenSize);
     break;
    default:
      break;
   }
 }

void sgpLine(Point1, Point2)
 sgpPointType Point1, Point2;
 {
  sgpPointType Screen1, Screen2, Physical1, Physical2;
  int x0, y0, x1, y1;

  if (!Active) return;
  sgpWorldToScreen(Point1, &Screen1); 
  sgpScreenToPhysical(Screen1, &Physical1);
  sgpWorldToScreen(Point2, &Screen2);
  sgpScreenToPhysical(Screen2, &Physical2);
  switch (CurrentState.Display)
   {
    case ColorDisplay:
    case MonochromeDisplay:
     x0 = Physical1.x; y0 = Physical1.y;
     x1 = Physical2.x; y1 = Physical2.y;
     (void)devSetColor(devColorCode(CurrentState.Color.r,
                                 CurrentState.Color.g,
                                 CurrentState.Color.b));
     (void)devLine(x0, y0, x1, y1);
     break;
    case DebugDisplay:
     fprintf(stdout,"[%f %f] [%f %f] sgpLine\n", 
                     Point1.x, Point1.y, Point2.x, Point2.y); 
    default:

     break;
   }
 }

void sgpPolyLine(n, Points)
 int n; sgpPointType *Points;
 {
  int i;
  for (i=1; i<n; i++)
   sgpLine(Points[i-1], Points[i]);
 }

void sgpCurve(m, n, Points)
 int m, n; sgpPointType *Points;
 {
  float xIn[100], yIn[100], xOut[1000], yOut[1000];
  int i, N, M;
  sgpPointType Point0, Point1;

  if (n<100)  N = n; else N = 100;
  if (m<1000) M = m; else M = 1000;
  for (i=0; (i<N); i++)
   {
    xIn[i] = Points[i].x; yIn[i] = Points[i].y;
   }
  if (SGPSpline (N, xIn, M, xOut) && SGPSpline (N, yIn, M, yOut))
   {
    for (i=1; i<M; i++)
     {
      Point0.x = xOut[i-1]; Point0.y = yOut[i-1];
      Point1.x = xOut[i];   Point1.y = yOut[i];
      sgpLine(Point0, Point1);
     }  
   }
  else fprintf(stderr,"sgpCurve: oops\n");
 }

void sgpRectangle(Rectangle)
 sgpRectangleType Rectangle;
 {
  sgpPointType World1, World2, Screen1, Screen2, Physical1, Physical2;
  int Left, Bottom, Right, Top;

  if (!Active) return;
  World1.x = Rectangle.Left;  World1.y = Rectangle.Bottom;
  World2.x = Rectangle.Right; World2.y = Rectangle.Top;
  sgpWorldToScreen(World1, &Screen1); 
  sgpScreenToPhysical(Screen1, &Physical1);
  sgpWorldToScreen(World2, &Screen2);
  sgpScreenToPhysical(Screen2, &Physical2);
  switch (CurrentState.Display)
   {
    case ColorDisplay:
    case MonochromeDisplay:
     Left = Physical1.x; Right = Physical2.x;
     Bottom = Physical1.y; Top = Physical2.y; 
     (void)devSetColor(devColorCode(CurrentState.Color.r,
                                 CurrentState.Color.g,
                                 CurrentState.Color.b));
     (void)devRect(Left, Top, Right, Bottom);
     break;
    case DebugDisplay:
     fprintf(stdout,"[%f %f] [%f %f] sgpRectangle\n", 
                     Rectangle.Left,  Rectangle.Bottom,
                     Rectangle.Right, Rectangle.Top);
    default:
     break;
   }
 }

void sgpCircle(Center, Radius)
 sgpPointType Center;
 double Radius;
 {
  sgpPointType ScreenCenter, PhysicalCenter;
  sgpPointType WorldRadius, ScreenRadius, PhysicalRadius;

  if (!Active) return;

  sgpWorldToScreen(Center, &ScreenCenter);
  sgpScreenToPhysical(ScreenCenter, &PhysicalCenter);

  WorldRadius.x = Center.x+Radius; WorldRadius.y = Center.y;
  sgpWorldToScreen(WorldRadius, &ScreenRadius);
  sgpScreenToPhysical(ScreenRadius, &PhysicalRadius);
  switch (CurrentState.Display)
   {
    case ColorDisplay: 
    case MonochromeDisplay:
     (void)devSetColor(devColorCode(CurrentState.Color.r,
                                 CurrentState.Color.g,
                                 CurrentState.Color.b));
     (void)devCircle((int) PhysicalCenter.x, (int) PhysicalCenter.y,
              (int) (PhysicalRadius.x- PhysicalCenter.x));
     break;
    case DebugDisplay:
     fprintf(stdout,"[%f %f] %f sgpCircle\n",
                     Center.x, Center.y, Radius);
    default:
     break;
   }
 }

void sgpDisc(Center, Radius)
 sgpPointType Center;
 double Radius;
 {
  sgpPointType ScreenCenter, PhysicalCenter;
  sgpPointType WorldRadius, ScreenRadius, PhysicalRadius;
  
  if (!Active) return;

  sgpWorldToScreen(Center, &ScreenCenter);
  sgpScreenToPhysical(ScreenCenter, &PhysicalCenter);

  WorldRadius.x = Center.x+Radius; WorldRadius.y = Center.y;
  sgpWorldToScreen(WorldRadius, &ScreenRadius);
  sgpScreenToPhysical(ScreenRadius, &PhysicalRadius);
 
  switch (CurrentState.Display)
   {
    case ColorDisplay: 
    case MonochromeDisplay:
     (void)devSetColor(devColorCode(CurrentState.Color.r,
                                 CurrentState.Color.g,
                                 CurrentState.Color.b));
     (void)devDisc((int)PhysicalCenter.x, (int)PhysicalCenter.y,
                   (int)(PhysicalRadius.x - PhysicalCenter.x));
     break;
    case DebugDisplay:
     fprintf(stdout,"[%f %f] %f sgpDisc\n",
                     Center.x, Center.y, Radius);
    default:
     break;
   }
 }

void sgpShadeTriangle(Points, Colors)
 sgpPointType Points[3];
 sgpColorType Colors[3];
 {
  sgpPointType Screen[3], Physical[3];
  int i;
  int x0, y0, c0, x1, y1, c1, x2, y2, c2;

  if (!Active) return;
  for (i=0; i<3; i++)
   {
    sgpWorldToScreen(Points[i], &Screen[i]);
    sgpScreenToPhysical(Screen[i], &Physical[i]);
   }

  switch (CurrentState.Display)
   {
    case ColorDisplay: 
    case MonochromeDisplay:
     x0 = (int)Physical[0].x;     y0 = (int)Physical[0].y;
     x1 = (int)Physical[1].x;     y1 = (int)Physical[1].y;
     x2 = (int)Physical[2].x;     y2 = (int)Physical[2].y;
     c0 = devColorCode(Colors[0].r, Colors[0].g, Colors[0].b);
     c1 = devColorCode(Colors[1].r, Colors[1].g, Colors[1].b);
     c2 = devColorCode(Colors[2].r, Colors[2].g, Colors[2].b);

     (void)devShadeTriangle(x0, y0, c0, x1, y1, c1, x2, y2, c2);
     break;
    case DebugDisplay:
     fprintf(stdout,
"[[%f %f][%f %f][%f %f]][[%f %f %f][%f %f %f][%f %f %f]] sgpShadeTriangle\n",
 Points[0].x, Points[0].y,
 Points[1].x, Points[1].y,
 Points[2].x, Points[2].y,
 Colors[0].r,Colors[0].g,Colors[0].b,
 Colors[1].r,Colors[1].g,Colors[1].b,
 Colors[2].r,Colors[2].g,Colors[2].b);
    default:
     break;
   }
 }

void sgpSolidTriangle(Points)
 sgpPointType Points[3];
 {
  sgpPointType Screen[3], Physical[3];
  int x0, y0, x1, y1, x2, y2;
  int i;

  if (!Active) return;
  for (i=0; i<3; i++)
   {
    sgpWorldToScreen(Points[i], &Screen[i]);
    sgpScreenToPhysical(Screen[i], &Physical[i]);
   }

  switch (CurrentState.Display)
   {
     case ColorDisplay:
     case MonochromeDisplay:
     x0 = (int)Physical[0].x;     y0 = (int)Physical[0].y;
     x1 = (int)Physical[1].x;     y1 = (int)Physical[1].y;
     x2 = (int)Physical[2].x;     y2 = (int)Physical[2].y;
     (void)devSetColor(devColorCode(CurrentState.Color.r,
                                    CurrentState.Color.g,
                                    CurrentState.Color.b));
     (void)devSolidTriangle(x0, y0, x1, y1, x2, y2);
     break;
    case DebugDisplay:
     fprintf(stdout,"[[%f %f][%f %f][%f %f]] sgpSolidTriangle\n",
             Points[0].x, Points[0].y,
             Points[1].x, Points[1].y,
             Points[2].x, Points[2].y);
     break;
    default: break;
   }
 }

void sgpPick(World,button)
 sgpPointType *World;
 int *button;
 {
  int x, y;  
  sgpPointType Physical,Screen;
  int MyButton;

  if (!Active) return;
  switch (CurrentState.Display)
   {
    case ColorDisplay:
    case MonochromeDisplay:
     if (0 == devPick(&x, &y, &MyButton))
      {
       Physical.x = x; Physical.y = y;
       sgpPhysicalToScreen(Physical, &Screen);
       sgpScreenToWorld(Screen, World);     
       *button = MyButton;
      }
     else
      {
       fprintf(stderr,"sgpPick: no pointer attached!\n");
      }
     break;
    case DebugDisplay:
     fprintf(stdout,"sgpPick\n");
     break;
    default : break;
   }
 }
