/*
   File: Lighting.c
   Authors: Jinling Lee (Bill),
            K.R. Sloan
   Last Modified: 18 April 1990
   Purpose: Add lighting model to the 3D SGP
 */

#include <stdio.h>
#include <math.h>
#include "3D.h"
#include "Lighting.h"

#define MaxNumSources 50

/* PRIVATE state variables */

static int NumberOfSources = 0;
static double Specularity = 6.0;
static sgpColorType Ambient  = { 0.2, 0.2, 0.2 };
static sgpColorType Diffuse  = { 0.5, 0.5, 0.5 };
static sgpColorType Specular = { 0.5, 0.5, 0.5 };
static double Exposure = 1.0;

typedef struct source
                {                  
                 PointType3D  location[MaxNumSources];
                 sgpColorType color[MaxNumSources];      
                } SOURCE;
static SOURCE Sources;

/*
   PRIVATE functions
 */

/*
   EXPORTED procedures
 */

void ClearLightSources()
 {
  NumberOfSources = 0;
 }

void SetExposure(exposure)
 double exposure;
 {
  Exposure = exposure;
 }

void SetAmbient(color)
 sgpColorType color;
 {
  Ambient = color;
 }

void AddLightSource(location, color)
 PointType3D location;
 sgpColorType color;
 {
  if (MaxNumSources <=  NumberOfSources)
   {
    fprintf(stderr,"AddLightSource: too many light sources!\n");
    return;
   }
  Sources.location[NumberOfSources] = location;
  Sources.color[NumberOfSources++]  = color;
 }

void SetSurfaceReflectance(diffuse, specular, specularity)
 sgpColorType diffuse, specular;
 double specularity;
 {
  Diffuse  = diffuse;
  Specular = specular;
  Specularity = specularity;
 }

/*      
   Function ColorAtPoint returns the color of a point
   in the scene according its location, normal, and
   current light sources. The lighting model used here is:
  

     I = Ambient*(Diffuse+Specular)
          + Sum Intensity_source * Diffuse * cos(theta)
          sources
                                                 n
          + Sum Intensity_source * Specular * cos (alpha)
           sources
  
     Suppose N is the normal vector, L is the vector from P point to a light 
     source and V is the vector from P pointing to the eye point. Then:
  
          cos(theta) = N * L / |N| * |L|
  
          cos(alpha) = V * ( 2*cos(theta)*N - L ) 
  
 */

sgpColorType ColorAtPoint(Location, Normal)
 PointType3D Location;
 VectorType3D Normal;
 {
  int i, j, k, m;
  PointType3D EyePoint;
  VectorType3D OpticalAxis, Up;
  sgpColorType Color;
  VectorType3D L, N, V;
  double CosTheta, CosAlpha, s;

  GetCamera3D(&EyePoint, &OpticalAxis, &Up);
  Color.r = Ambient.r*(Diffuse.r+Specular.r);
  Color.g = Ambient.g*(Diffuse.g+Specular.g);
  Color.b = Ambient.b*(Diffuse.b+Specular.b);

  N = Normal;
  Normalize3D(&N);

  V.dx = EyePoint.x - Location.x;
  V.dy = EyePoint.y - Location.y;
  V.dz = EyePoint.z - Location.z;
  Normalize3D(&V);

  for(k=0; k < NumberOfSources; k++) 
   {
    L.dx = Sources.location[k].x - Location.x;
    L.dy = Sources.location[k].y - Location.y;
    L.dz = Sources.location[k].z - Location.z;

    Normalize3D(&L);

    CosTheta= Dot3D(L,N);

    /*
      If theta >= 180 degrees (CosTheta <= 0),
      then the only contribution to the 
      color (intensity) of that point is the ambient light,
      and we are done.  Otherwise...
     */

    if (CosTheta > 0)
     {
      CosAlpha =    V.dx*((2.0*CosTheta*N.dx)-L.dx)
                  + V.dy*((2.0*CosTheta*N.dy)-L.dy)
	          + V.dz*((2.0*CosTheta*N.dz)-L.dz);

      if (CosAlpha < 0.0) CosAlpha=0.0;     
      s = pow(CosAlpha, Specularity);

      Color.r += Sources.color[k].r*Diffuse.r*CosTheta;
      Color.g += Sources.color[k].g*Diffuse.g*CosTheta;
      Color.b += Sources.color[k].b*Diffuse.b*CosTheta;
    
      Color.r += Sources.color[k].r*Specular.r*s;
      Color.g += Sources.color[k].g*Specular.g*s;
      Color.b += Sources.color[k].b*Specular.b*s;
    }
  }

  /* Adjust exposure */

  Color.r *= Exposure; Color.g *= Exposure; Color.b *= Exposure;

  /* Clamp */

  if (Color.r < 0.0) Color.r = 0.0; else if (1.0 < Color.r) Color.r = 1.0;
  if (Color.g < 0.0) Color.g = 0.0; else if (1.0 < Color.g) Color.g = 1.0;
  if (Color.b < 0.0) Color.b = 0.0; else if (1.0 < Color.b) Color.b = 1.0;

  return(Color);
 }
