/*
  File: FindBtlnk.c
  Purpose:  Locate the column with the narrowest span between two previous
            paths in a toroidal graph and subdivide.
  Author:  James Painter, David Meyers
  Date  :  8 November 1989
*/

#include <TypeDefinitions.h>
#include <TorGraph.h>
#include <math.h>
#include <stdio.h>

#define rint(x) ((int)((x) + .5))


extern int  outputTrace;
extern PathType *BestPath;
extern REAL BiasThreshold;  /*	Lower bound for subdivision Bias
                            **  Should be in the range [0,.5]
   			    **  E.G. value of 1/2 means equal subdivision;
                            **  Bias is always away from the optimal path
             		    */
extern REAL BiasAdjustment;

void FindBottleNeck(), MeasureColumn();

/* Ordered3 returns  TRUE if a<=b<=c (allowing for wrap around) */
#define Ordered3(a,b,c) \
(((a)<(b) && (b)<=(c)) || ((b)<=(c) && (c)<(a)) || ((c)<(a) && (a)<(b)) )



void FindBottleNeck( LeftPath, RightPath, Top, Bottom, Left, Right, Size )
 PathType *LeftPath, *RightPath;
 int *Top, *Bottom, *Left, *Right; 
 int *Size;
 {
  int C;
  int Debug = FALSE;

  /* Subdivide bottleneck in the starting column */
  C = *Left = *Right = LeftPath->StartC;
  if(Debug)
   (void) fprintf(stderr,
	   "FindBottleNeck: LeftPath->StartR=%d, RightPath->StartR=%d\n",
	   LeftPath->StartR, RightPath->StartR);
  MeasureColumn( RightPath, LeftPath, C, Top, Bottom, Size );
 }

/* ------------------------------------------------------------------ */

void  MeasureColumn( TopPath, BottomPath, C, Top, Bottom, Height )
PathType *TopPath, *BottomPath;		/* Bounding paths */
int C;					/* Column number to check */
int *Top, *Bottom;  			/* Resulting bounds */
int *Height;                            /* And the height */
{
    int BNext;

    /* Find out where each path leaves/enters the column */

    *Top = TopPath->LeavesColInRow[C];
    *Bottom = (C == 0) ?   /* Handle wrap around */
	BottomPath->LeavesColInRow[2*BottomPath->Columns-1]
      : BottomPath->LeavesColInRow[C-1];	
    BNext = BottomPath->LeavesColInRow[C];

    /* Check for path overlap */
    if ( Ordered3(*Bottom,*Top,BNext) )
     {
      if ((C==TopPath->StartC) && Ordered3(*Bottom,TopPath->StartR,*Top))
        {  /* Search node in overlap zone...  Zero sized bottleneck. */
         *Top = TopPath->StartR;
         *Bottom = *Top-1;
         *Height = 0;
         return;
	}

      if ((C==BottomPath->StartC) && Ordered3(*Bottom,BottomPath->StartR,*Top))
        {  /* Search node in overlap zone...  Zero sized bottleneck. */
    
         *Top = BottomPath->StartR;
         *Bottom = *Top -1;
         *Height = 0;
         return;
	}
      *Top = *Bottom;
     }

    /* Check whether either end of the bottleneck is a search node and
    ** if so then narrow the bottleneck appropriatly.
    */

    if ((*Top == TopPath->StartR) && (C == TopPath->StartC)) 
     { (*Top)++;  if (*Top == 2*TopPath->Rows) *Top = 0;   }

    if ((*Bottom == BottomPath->StartR && (C == BottomPath->StartC))) 
     { (*Bottom)--;  if (*Bottom == -1) *Bottom = 2*BottomPath->Rows -1;  }

    /* Measure the height of the bottleneck */

    *Height = *Bottom - *Top +1;
    if (*Height < 0) *Height += 2*TopPath->Rows;
}

/* --------------------------------------------------------------------- */

void SubDivide( LeftPath, RightPath, R, C, Size, WhichWay )
 PathType *LeftPath, *RightPath;
 int      *R, *C, *Size;
 WhichWayType *WhichWay;
 {
  int LowD, HighD, LastC, BestHigh, BestLow;
  int Left, Right, Top, Bottom;
  REAL Hbias, Lbias;
  int Debug = FALSE;

  if(Debug)
   {
    (void) fprintf(stderr,"\nSubDivide calling FindBottleNeck\n");
    (void) fprintf(stderr,
	    "LeftStartC=%d LeftStartR=%d RightStartC=%d RightStartR=%d\n",
	    LeftPath->StartC, LeftPath->StartR, RightPath->StartC,
	    RightPath->StartR);
   }
  FindBottleNeck( LeftPath, RightPath, &Top, &Bottom, &Left, &Right, Size);
  if(Debug)
   (void) fprintf(stderr,
		  "Back from FindBottleNeck, Top=%d, Bottom=%d, Size=%d\n",
		  Top, Bottom, *Size);

  if (Left == Right  && Top == Bottom)   /* A quick kill, no work to do. */
   {
    *R = Top;
    if(Debug)
     (void) fprintf(stderr,
		    "Subdivide, quick kill code setting R to: %d\n", *R);
    *C = Left;
    return;
   }
  
  /* 
   **  See if the best path is above or below this bottleneck to guide the
   **  bias.
   */
  *C = Left;
  LastC = (*C == 0) ? 2*LeftPath->Columns-1 : *C -1;
  
  BestHigh = BestPath->LeavesColInRow[*C];
  BestLow = BestPath->LeavesColInRow[LastC];
  
  LowD = RightPath->LeavesColInRow[*C] - BestHigh;
  while (LowD >= BestPath->Rows) LowD -= BestPath->Rows;
  while (LowD < 0) LowD += BestPath->Rows;
  
  HighD = BestLow - LeftPath->LeavesColInRow[LastC];
  while (HighD >= BestPath->Rows) HighD -= BestPath->Rows;
  while (HighD < 0) HighD += BestPath->Rows;
  
  Lbias = .5;
  Hbias = .5;
  *WhichWay = DontKnow;
  if ( (LowD != 0 || HighD != 0)  && (LowD+HighD) > .1*BestPath->Rows )
   {  /* Compute Bias */
    Hbias = (REAL) HighD / (REAL) (LowD+HighD);
    if (Hbias < .5)
     { Hbias += BiasAdjustment; if (Hbias > .5) Hbias = .5;}
    else       
     { Hbias -= BiasAdjustment; if (Hbias < .5) Hbias = .5;}
    
    if (Hbias < BiasThreshold) 
     Hbias = BiasThreshold;
    
    if (Hbias > 1.-BiasThreshold) 
     Hbias = 1.-BiasThreshold;
    
    if (Hbias == .5)
     *WhichWay = DontKnow;
    else if (Hbias < .5)
     *WhichWay = TowardsRight;
    else
     *WhichWay = TowardsLeft;
   }
  
  Lbias = (1. - Hbias);
  *R = (int) rint( (double) ( (REAL) Top*Lbias + (REAL) Bottom*Hbias) );
  if(Debug)
   {
    (void) fprintf(stderr,"Subdivide, setting R to: %d\n", *R);
    (void) fprintf(stderr,"Top=%d, Bottom=%d Lbias=%f Hbias=%f\n",
	    Top, Bottom, Lbias, Hbias);
   }

  if (Bottom < Top)  /* Handle wrap-around */
   *R = (*R + LeftPath->Rows);
  *R %= (2*LeftPath->Rows);
}

/* ------------------------------------------------------------------------ */

/* Counts the number of nodes in the graph bounded by two paths */

int MeasureGraph ( LeftPath, RightPath )
PathType *LeftPath, *RightPath;
 {
  int C, LastC, Height, Result = 0, EndR, Left, Right, Max;
  int Debug = FALSE;

  LastC = 2*LeftPath->Columns -1;
  EndR = RightPath->LeavesColInRow[0] + LeftPath->Rows;
  for(C=0; C<= LeftPath->Columns; LastC = C++)
   {
    Left = LeftPath->LeavesColInRow[LastC];
    Right = RightPath->LeavesColInRow[C];
  
    Height = Left - Right +1;
    if (Height < 0) Height += 2*LeftPath->Rows;
    Max = EndR - Right;
    if (Max < 0) Max += 2*LeftPath->Rows;
    if (Height > Max) Height = Max;
    Result += Height;
   }
  if(Debug)
   (void) fprintf(stderr, "MeasureGraph returning Result=%d\n", Result);
  return Result;
 }
