/*
**                      Softwarepraktikum iMaze
**                              1993/94
**                Joerg Czeranski    Hans-Ulrich Kiel
**
** Datei: init_spieler.c
**
** Kommentar:
**  Schickt einem Client die Initialisierung
*/


#include <stdio.h>
#include <errno.h>
#include <string.h>
#include <signal.h>
#include <sys/time.h>

#include "global.h"
#include "speicher.h"
#include "labyrinth.h"
#include "protokoll.h"
#include "server_netz.h"
#include "init_spieler.h"

static char sccsid[] = "@(#)init_spieler.c	1.18 6/27/94";


#define PROGRAMM_NAME "iMaze server JC/HUK 1.1" /* Version des Servers */


/* bis hier lokaler Teil                       */
/***********************************************/
/* ab hier globaler Teil                       */


/*
** init_spieler
**  initialisiert einen Client (Prolog), hauptsaechlich werden das
**  Spielfeld gesendet und die Programmversionen ausgetauscht
**
** Parameter:
**  verbindung: Deskriptor fuer die Prolog-Verbindung zum Client
**  feldbreite: Breite des Spielfeldes
**  feldlaenge: Laenge des Spielfeldes
**  spielfeld: das Spielfeld
**
** Rueckgabewert:
**  1 fuer Fehler, 0 fuer korrekte Initialisierung
*/
int init_spieler(verbindung, feldbreite, feldlaenge, spielfeld)
void *verbindung;
u_int feldbreite, feldlaenge;
block **spielfeld;
{
	/* Flags: */
	int programmname_uebertragen; /* der Server hat sich bereits
	                                 identifiziert */
	int spiel_beginnt;            /* PT_SPIEL_BEGINNT gesendet */
	int leer_erlaubt;             /* der Client darf PT_LEER senden */

	/* Programmname wurde noch nicht uebertragen, PT_SPIEL_BEGINNT nicht
	   gesendet, der Client darf zu Anfang PT_LEER senden */
	programmname_uebertragen = 0;
	spiel_beginnt = 0;
	leer_erlaubt = 1;

	/* Schleife bis PT_SPIEL_BEGINNT gesendet ist */
	while (!spiel_beginnt)
	{
		int paket_laenge; /* Laenge des Pakets */
		int typ;          /* Typ des zu sendenden Pakets */
		int ignoriert;    /* das aktuelle Paket wird ignoriert (z.B. nicht
		                     verstanden) */
		u_char *paket;    /* zu sendendes oder empfangenes Paket */
		u_char kennung;   /* Kennung aus typ und ignoriert */

		/* ein Paket empfangen (der Client sendet das erste Paket) */
		if (prolog_paket_empfangen(verbindung, &paket_laenge,
			(void **)&paket))
			return 1;

		/* das Paket darf nicht leer sein, sonst ist das
		   ein Protokoll-Fehler */
		if (!paket_laenge)
			return 1;

		/* falls nichts mehr zum Senden ansteht, PT_LEER senden */
		typ = PT_LEER;

		/* das Paket gilt erstmal als nicht verstanden */
		ignoriert = PAKET_IGNORIERT;

		/* Paket vom Client auswerten */
		switch (PROL_PAKET_TYP(paket[0]))
		{
			case PT_LEER: /* leeres Paket */
				/* wenn kein PT_LEER vom Client haette kommen duerfen,
				   Verbindung abbrechen */
				if (!leer_erlaubt)
				{
					typ = PT_ABBRUCH;
					break;
				}

				/* keine Anfrage, also muss nicht geantwortet werden */
				break;

			case PT_WILL_LABYRINTH: /* Client fordert Labyrinth an */
				/* sofort antworten */
				typ = PT_LABYRINTH;
				ignoriert = PAKET_NICHT_IGNORIERT;
				break;

			case PT_BEREIT_FUER_SPIEL: /* Client bereit fuer Spiel */
				/* wenn keine Daten mehr zum Senden anstehen,
				   auf das Spiel schalten */
				typ = PT_SPIEL_BEGINNT;
				ignoriert = PAKET_NICHT_IGNORIERT;
				break;

			case PT_ABBRUCH: /* Abbruch durch den Client */
				/* Initialisierung fehlgeschlagen */
				return 1;
		}

		/* Paket vom Client wird nicht mehr benoetigt */
		speicher_freigeben((void **)&paket);

		/* falls keine Daten anliegen, die sofort uebertragen werden
		   muessen, und der Programmname noch nicht uebertragen
		   wurde, das jetzt tun */
		if ((typ == PT_LEER || typ == PT_SPIEL_BEGINNT) &&
			!programmname_uebertragen)
			typ = PT_PROGRAMM_NAME;

		/* Kennung fuer das Antwort-Paket erzeugen */
		kennung = PROL_PAKET_KENNUNG(ignoriert, typ);

		/* je nach Typ das Antwort-Paket zusammenstellen */
		switch (typ)
		{
			case PT_LABYRINTH: /* Spielfeld senden */
				/* Laenge des Pakets berechnen und Speicher anfordern */
				paket_laenge = 3 + 4 * feldbreite * feldlaenge;
				speicher_belegen((void **)&paket, paket_laenge);

				/* das Paket beginnt immer mit der Kennnung; hier
				   folgen die Dimensionen des Spielfelds */
				paket[0] = kennung;
				paket[1] = feldbreite;
				paket[2] = feldlaenge;

				{
					int x, y, i;   /* Indizes fuer das Spielfeld */
					u_char *daten; /* Zeiger in das Paket */

					/* Inhalt des Spielfeldes nach den Daten fester
					   Laenge anfuegen */
					daten = &paket[3];

					/* den Inhalt des Spielfeldes zeilenweise in das
					   Paket kopieren */
					for (y = 0; y < feldlaenge; y++)
						for (x = 0; x < feldbreite; x++)
							/* die Waende in allen 4 Himmelsrichtungen */
							for (i = 0; i < 4; i++)
								/* nur die Farbe der Wand/Tuer
								   uebertragen */
								*daten++ = spielfeld[x][y][i].farbe;
				}

				/* Paket senden */
				if (prolog_paket_senden(verbindung, paket_laenge, paket))
					/* bei Fehler Initialisierung abbrechen */
					return 1;

				/* gesandtes Paket wird nicht mehr benoetigt */
				speicher_freigeben((void **)&paket);

				break;

			case PT_ABBRUCH: /* Initialisierung abbrechen */
				/* Paket senden, enthaelt nur die Kennung */
				prolog_paket_senden(verbindung, 1, &kennung);

				/* Initialisierung ist fehlgeschlagen */
				return 1;

			case PT_PROGRAMM_NAME: /* Programmname senden */
				/* Laenge des Pakets berechnen und Speicher anfordern */
				paket_laenge = 1 + strlen(PROGRAMM_NAME);
				speicher_belegen((void **)&paket, paket_laenge);

				/* das Paket beginnt immer mit der Kennnung */
				paket[0] = kennung;

				/* Programmname in das Paket kopieren; ohne
				   abschliessendes Nullbyte */
				memcpy(&paket[1], PROGRAMM_NAME, strlen(PROGRAMM_NAME));

				/* Paket senden */
				if (prolog_paket_senden(verbindung, paket_laenge, paket))
					/* bei Fehler Initialisierung abbrechen */
					return 1;

				/* gesandtes Paket wird nicht mehr benoetigt */
				speicher_freigeben((void **)&paket);

				/* der Programmname wurde uebertragen */
				programmname_uebertragen = 1;

				break;

			case PT_SPIEL_BEGINNT: /* auf Spiel schalten */
				/* nach dem Senden dieses Pakets ist die Initialisierung
				   beendet */
				spiel_beginnt = 1;

				/* Paket senden: */

			default: /* Paket ohne Daten senden (z.B. PT_LEER) */
				/* Paket senden */
				if (prolog_paket_senden(verbindung, 1, &kennung))
					/* bei Fehler Initialisierung abbrechen */
					return 1;
		}

		/* PT_LEER vom Client ist nur erlaubt, wenn der Server kein
		   PT_LEER gesendet hat */
		leer_erlaubt = typ != PT_LEER;
	}

	/* Initialisierung ohne Fehler abgeschlossen */
	return 0;
}
