/*
  $Id$

  Copyright (C) 2003-2010 Virginia Tech.
  All rights reserved.

  SEE LICENSE FOR MORE INFORMATION

  Author:  Middleware Services
  Email:   middleware@vt.edu
  Version: $Revision$
  Updated: $Date$
*/
package edu.vt.middleware.ldap.ssl;

import java.io.IOException;
import java.io.InputStream;
import java.security.GeneralSecurityException;
import java.security.KeyStore;
import java.util.Arrays;

/**
 * Reads keystore credentials from a classpath, filepath, or stream resource.
 *
 * @author  Middleware Services
 * @version  $Revision$
 */
public class KeyStoreCredentialReader extends AbstractCredentialReader<KeyStore>
{


  /**
   * Reads a keystore from an input stream.
   *
   * @param  is  Input stream from which to read keystore.
   * @param  params  Two optional parameters are supported:
   *
   * <ul>
   *   <li>KeyStore password</li>
   *   <li>KeyStore type; defaults to JVM default keystore format if
   *     omitted</li>
   * </ul>
   *
   * <p>If only a single parameter is supplied, it is assumed to be
   * the password.</p>
   *
   * @return  KeyStore read from data in stream.
   *
   * @throws  IOException  On IO errors.
   * @throws  GeneralSecurityException  On errors with the credential data.
   */
  public KeyStore read(final InputStream is, final String... params)
    throws IOException, GeneralSecurityException
  {
    char[] password = null;
    if (params.length > 0 && params[0] != null) {
      password = params[0].toCharArray();
    }

    String type = KeyStore.getDefaultType();
    if (params.length > 1 && params[1] != null) {
      type = params[1];
    }

    final KeyStore keystore = KeyStore.getInstance(type);
    if (is != null) {
      keystore.load(this.getBufferedInputStream(is), password);
      if (password != null) {
        Arrays.fill(password, '0');
      }
    }
    return keystore;
  }
}
