/*
 * This file is part of LibEuFin.
 * Copyright (C) 2025 Taler Systems S.A.

 * LibEuFin is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation; either version 3, or
 * (at your option) any later version.

 * LibEuFin is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 * or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Affero General
 * Public License for more details.

 * You should have received a copy of the GNU Affero General Public
 * License along with LibEuFin; see the file COPYING.  If not, see
 * <http://www.gnu.org/licenses/>
 */

import org.w3c.dom.Document
import com.github.ajalt.clikt.core.CliktCommand
import com.github.ajalt.clikt.testing.test
import kotlinx.coroutines.runBlocking
import io.ktor.http.*
import io.ktor.http.content.*
import io.ktor.client.engine.mock.*
import org.junit.Test
import tech.libeufin.common.*
import tech.libeufin.common.crypto.CryptoUtil
import tech.libeufin.ebics.*
import tech.libeufin.ebics.test.*
import kotlin.io.path.*
import kotlin.test.*
import java.time.LocalDate

@OptIn(kotlin.io.path.ExperimentalPathApi::class)
class EbicsTest {
    private val ebicsLogger = EbicsLogger(null).tx("test").step("step")

    // POSTs an EBICS message to the mock bank.  Tests
    // the main branches: unreachable bank, non-200 status
    // code, and 200.
    @Test
    fun postMessage() {runBlocking {
        assertFailsWith<EbicsError.HTTP> {
            getMockedClient {
                respondError(HttpStatusCode.NotFound)
            }.postToBank("http://ignored.example.com/", ByteArray(0), "Test", ebicsLogger)
        }.run {
            assertEquals(HttpStatusCode.NotFound, status)
            assertEquals("Test: bank HTTP error: 404 Not Found", message)
        }
        assertFailsWith<EbicsError.Network> {
            getMockedClient {
                throw Exception("Simulate failure")
            }.postToBank("http://ignored.example.com/", ByteArray(0), "Test", ebicsLogger)
        }.run {
            assertEquals("Test: failed to contact bank", message)
            assertEquals("Simulate failure", cause!!.message)
        }
        assertFailsWith<EbicsError.Protocol> {
            getMockedClient {
                respondOk("<ebics broken></ebics>")
            }.postToBank("http://ignored.example.com/", ByteArray(0), "Test", ebicsLogger)
        }.run {
            assertEquals("Test: invalid XML bank response", message)
            assertEquals("Attribute name \"broken\" associated with an element type \"ebics\" must be followed by the ' = ' character.", cause!!.message)
        }
        getMockedClient {
            respondOk("<ebics></ebics>")
        }.postToBank("http://ignored.example.com/", ByteArray(0), "Test", ebicsLogger)
    }}

    // Tests that internal repr. of keys lead to valid PDF.
    // Mainly tests that the function does not throw any error.
    @Test
    fun keysPdf() {
        val pdf = generateKeysPdf(generateNewKeys(), object: EbicsHostConfig {
            override val baseUrl = "https://isotest.postfinance.ch/ebicsweb/ebicsweb"
            override val hostId = "PFEBICS"
            override val userId = "PFC00563"
            override val partnerId = "PFC00563"
        })
        Path("/tmp/libeufin-nexus-test-keys.pdf").writeBytes(pdf)
    }
}