#ifndef __LINUX_VIDEODEV2_H
#define __LINUX_VIDEODEV2_H
/*
 *	Video for Linux Two
 *
 *	Header file for v4l or V4L2 drivers and applications, for
 *	Linux kernels 2.2.x or 2.4.x.
 *
 *	See http://bytesex.org/v4l/ for API specs and other
 *	v4l2 documentation.
 *
 *	Author: Bill Dirks <bdirks@pacbell.net>
 *		Justin Schoeman
 *		et al.
 */
#include <linux/time.h> /* need struct timeval */

/*
 *	M I S C E L L A N E O U S
 */

/*  Four-character-code (FOURCC) */
#define v4l2_fourcc(a,b,c,d)\
        (((__u32)(a)<<0)|((__u32)(b)<<8)|((__u32)(c)<<16)|((__u32)(d)<<24))

/*
 *	E N U M S
 */
enum v4l2_field {
	V4L2_FIELD_ANY        = 0, /* driver can choose from none,
				      top, bottom, interlaced
				      depending on whatever it thinks
				      is approximate ... */
	V4L2_FIELD_NONE       = 1, /* this device has no fields ... */
	V4L2_FIELD_TOP        = 2, /* top field only */
	V4L2_FIELD_BOTTOM     = 3, /* bottom field only */
	V4L2_FIELD_INTERLACED = 4, /* both fields interlaced */
	V4L2_FIELD_SEQ_TB     = 5, /* both fields sequential into one
				      buffer, top-bottom order */
	V4L2_FIELD_SEQ_BT     = 6, /* same as above + bottom-top order */
	V4L2_FIELD_ALTERNATE  = 7, /* both fields alternating into
				      separate buffers */
};
#define V4L2_FIELD_HAS_TOP(field)	\
	((field) == V4L2_FIELD_TOP 	||\
	 (field) == V4L2_FIELD_INTERLACED ||\
	 (field) == V4L2_FIELD_SEQ_TB	||\
	 (field) == V4L2_FIELD_SEQ_BT)
#define V4L2_FIELD_HAS_BOTTOM(field)	\
	((field) == V4L2_FIELD_BOTTOM 	||\
	 (field) == V4L2_FIELD_INTERLACED ||\
	 (field) == V4L2_FIELD_SEQ_TB	||\
	 (field) == V4L2_FIELD_SEQ_BT)
#define V4L2_FIELD_HAS_BOTH(field)	\
	((field) == V4L2_FIELD_INTERLACED ||\
	 (field) == V4L2_FIELD_SEQ_TB	||\
	 (field) == V4L2_FIELD_SEQ_BT)

enum v4l2_buf_type {
	V4L2_BUF_TYPE_VIDEO_CAPTURE  = 1,
	V4L2_BUF_TYPE_VIDEO_OUTPUT   = 2,
	V4L2_BUF_TYPE_VIDEO_OVERLAY  = 3,
	V4L2_BUF_TYPE_VBI_CAPTURE    = 4,
	V4L2_BUF_TYPE_VBI_OUTPUT     = 5,
	V4L2_BUF_TYPE_PRIVATE        = 0x80,
};

enum v4l2_ctrl_type {
	V4L2_CTRL_TYPE_INTEGER	     = 1,
	V4L2_CTRL_TYPE_BOOLEAN	     = 2,
	V4L2_CTRL_TYPE_MENU	     = 3,
	V4L2_CTRL_TYPE_BUTTON	     = 4,
};

enum v4l2_tuner_type {
	V4L2_TUNER_RADIO	     = 1,
	V4L2_TUNER_ANALOG_TV	     = 2,
};

enum v4l2_memory {
	V4L2_MEMORY_MMAP             = 1,
	V4L2_MEMORY_USERPTR          = 2,
	V4L2_MEMORY_OVERLAY          = 3,
};

/* see also http://vektor.theorem.ca/graphics/ycbcr/ */
enum v4l2_colorspace {
	/* ITU-R 601 -- broadcast NTSC/PAL */
	V4L2_COLORSPACE_SMPTE170M     = 1,

	/* 1125-Line (US) HDTV */
	V4L2_COLORSPACE_SMPTE240M     = 2,

	/* HD and modern captures. */
	V4L2_COLORSPACE_REC709        = 3,
	
	/* broken BT878 extents (601, luma range 16-253 instead of 16-235) */
	V4L2_COLORSPACE_BT878         = 4,
	
	/* These should be useful.  Assume 601 extents. */
	V4L2_COLORSPACE_470_SYSTEM_M  = 5,
	V4L2_COLORSPACE_470_SYSTEM_BG = 6,
	
	/* I know there will be cameras that send this.  So, this is
	 * unspecified chromaticities and full 0-255 on each of the
	 * Y'CbCr components
	 */
	V4L2_COLORSPACE_JPEG          = 7,
	
	/* For RGB colourspaces, this is probably a good start. */
	V4L2_COLORSPACE_SRGB          = 8,
};

struct v4l2_rect {
	__s32   left;
	__s32   top;
	__s32   width;
	__s32   height;
};

struct v4l2_fract {
	__u32   numerator;
	__u32   denominator;
};

/*
 *	D R I V E R   C A P A B I L I T I E S
 */
struct v4l2_capability
{
	__u8	driver[16];	/* i.e. "bttv" */
	__u8	card[32];	/* i.e. "Hauppauge WinTV" */
	__u8	bus_info[32];	/* "PCI:" + pci_dev->slot_name */
	__u32   version;        /* should use KERNEL_VERSION() */
	__u32	capabilities;	/* Device capabilities */
	__u32	reserved[4];
};

/* Values for 'capabilities' field */
#define V4L2_CAP_VIDEO_CAPTURE	0x00000001  /* Is a video capture device */
#define V4L2_CAP_VIDEO_OUTPUT	0x00000002  /* Is a video output device */
#define V4L2_CAP_VIDEO_OVERLAY	0x00000004  /* Can do video overlay */
#define V4L2_CAP_VBI_CAPTURE	0x00000010  /* Is a VBI capture device */
#define V4L2_CAP_VBI_OUTPUT	0x00000020  /* Is a VBI output device */
#define V4L2_CAP_RDS_CAPTURE	0x00000100  /* RDS data capture */

#define V4L2_CAP_TUNER		0x00010000  /* Has a tuner */
#define V4L2_CAP_AUDIO		0x00020000  /* has audio support */

#define V4L2_CAP_READWRITE      0x01000000  /* read/write systemcalls */
#define V4L2_CAP_ASYNCIO        0x02000000  /* async I/O */
#define V4L2_CAP_STREAMING      0x04000000  /* streaming I/O ioctls */

/*
 *	V I D E O   I M A G E   F O R M A T
 */

struct v4l2_pix_format
{
	__u32         	 	width;
	__u32	         	height;
	__u32	         	pixelformat;
	enum v4l2_field  	field;
	__u32            	bytesperline;	/* for padding, zero if unused */
	__u32          	 	sizeimage;
        enum v4l2_colorspace	colorspace;
	__u32			priv;		/* private data, depends on pixelformat */
};

/*           Pixel format    FOURCC                  depth  Description   */
#define V4L2_PIX_FMT_RGB332  v4l2_fourcc('R','G','B','1') /*  8  RGB-3-3-2     */
#define V4L2_PIX_FMT_RGB555  v4l2_fourcc('R','G','B','O') /* 16  RGB-5-5-5     */
#define V4L2_PIX_FMT_RGB565  v4l2_fourcc('R','G','B','P') /* 16  RGB-5-6-5     */
#define V4L2_PIX_FMT_RGB555X v4l2_fourcc('R','G','B','Q') /* 16  RGB-5-5-5 BE  */
#define V4L2_PIX_FMT_RGB565X v4l2_fourcc('R','G','B','R') /* 16  RGB-5-6-5 BE  */
#define V4L2_PIX_FMT_BGR24   v4l2_fourcc('B','G','R','3') /* 24  BGR-8-8-8     */
#define V4L2_PIX_FMT_RGB24   v4l2_fourcc('R','G','B','3') /* 24  RGB-8-8-8     */
#define V4L2_PIX_FMT_BGR32   v4l2_fourcc('B','G','R','4') /* 32  BGR-8-8-8-8   */
#define V4L2_PIX_FMT_RGB32   v4l2_fourcc('R','G','B','4') /* 32  RGB-8-8-8-8   */
#define V4L2_PIX_FMT_GREY    v4l2_fourcc('G','R','E','Y') /*  8  Greyscale     */
#define V4L2_PIX_FMT_YVU410  v4l2_fourcc('Y','V','U','9') /*  9  YVU 4:1:0     */
#define V4L2_PIX_FMT_YVU420  v4l2_fourcc('Y','V','1','2') /* 12  YVU 4:2:0     */
#define V4L2_PIX_FMT_YUYV    v4l2_fourcc('Y','U','Y','V') /* 16  YUV 4:2:2     */
#define V4L2_PIX_FMT_UYVY    v4l2_fourcc('U','Y','V','Y') /* 16  YUV 4:2:2     */
#define V4L2_PIX_FMT_YUV422P v4l2_fourcc('4','2','2','P') /* 16  YVU422 planar */
#define V4L2_PIX_FMT_YUV411P v4l2_fourcc('4','1','1','P') /* 16  YVU411 planar */
#define V4L2_PIX_FMT_Y41P    v4l2_fourcc('Y','4','1','P') /* 12  YUV 4:1:1     */

/* two planes -- one Y, one Cr + Cb interleaved  */
#define V4L2_PIX_FMT_NV12    v4l2_fourcc('N','V','1','2') /* 12  Y/CbCr 4:2:0  */
#define V4L2_PIX_FMT_NV21    v4l2_fourcc('N','V','2','1') /* 12  Y/CrCb 4:2:0  */

/*  The following formats are not defined in the V4L2 specification */
#define V4L2_PIX_FMT_YUV410  v4l2_fourcc('Y','U','V','9') /*  9  YUV 4:1:0     */
#define V4L2_PIX_FMT_YUV420  v4l2_fourcc('Y','U','1','2') /* 12  YUV 4:2:0     */
#define V4L2_PIX_FMT_YYUV    v4l2_fourcc('Y','Y','U','V') /* 16  YUV 4:2:2     */
#define V4L2_PIX_FMT_HI240   v4l2_fourcc('H','I','2','4') /*  8  8-bit color   */

/* compressed formats */
#define V4L2_PIX_FMT_MJPEG    v4l2_fourcc('M','J','P','G') /* Motion-JPEG   */
#define V4L2_PIX_FMT_JPEG     v4l2_fourcc('J','P','E','G') /* JFIF JPEG     */
#define V4L2_PIX_FMT_DV       v4l2_fourcc('d','v','s','d') /* 1394          */
#define V4L2_PIX_FMT_MPEG     v4l2_fourcc('M','P','E','G') /* MPEG          */

/*  Vendor-specific formats   */
#define V4L2_PIX_FMT_WNVA    v4l2_fourcc('W','N','V','A') /* Winnov hw compres */

/*
 *	F O R M A T   E N U M E R A T I O N
 */
struct v4l2_fmtdesc
{
	__u32	            index;             /* Format number      */
	enum v4l2_buf_type  type;              /* buffer type        */
	__u32               flags;
	__u8	            description[32];   /* Description string */
	__u32	            pixelformat;       /* Format fourcc      */
	__u32	            reserved[4];
};

#define V4L2_FMT_FLAG_COMPRESSED 0x0001


/*
 *	T I M E C O D E
 */
struct v4l2_timecode
{
	__u32	type;
	__u32	flags;
	__u8	frames;
	__u8	seconds;
	__u8	minutes;
	__u8	hours;
	__u8	userbits[4];
};

/*  Type  */
#define V4L2_TC_TYPE_24FPS		1
#define V4L2_TC_TYPE_25FPS		2
#define V4L2_TC_TYPE_30FPS		3
#define V4L2_TC_TYPE_50FPS		4
#define V4L2_TC_TYPE_60FPS		5

/*  Flags  */
#define V4L2_TC_FLAG_DROPFRAME		0x0001 /* "drop-frame" mode */
#define V4L2_TC_FLAG_COLORFRAME		0x0002
#define V4L2_TC_USERBITS_field		0x000C
#define V4L2_TC_USERBITS_USERDEFINED	0x0000
#define V4L2_TC_USERBITS_8BITCHARS	0x0008
/* The above is based on SMPTE timecodes */


/*
 *	C O M P R E S S I O N   P A R A M E T E R S
 */
#if 0
/* ### generic compression settings don't work, there is too much
 * ### codec-specific stuff.  Maybe reuse that for MPEG codec settings
 * ### later ... */
struct v4l2_compression
{
	__u32	quality;
	__u32	keyframerate;
	__u32	pframerate;
	__u32	reserved[5];
};
#endif

struct v4l2_jpegcompression
{
	int quality;

	int  APPn;              /* Number of APP segment to be written,
				 * must be 0..15 */
	int  APP_len;           /* Length of data in JPEG APPn segment */
	char APP_data[60];      /* Data in the JPEG APPn segment. */
	
	int  COM_len;           /* Length of data in JPEG COM segment */
	char COM_data[60];      /* Data in JPEG COM segment */
	
	__u32 jpeg_markers;     /* Which markers should go into the JPEG
				 * output. Unless you exactly know what
				 * you do, leave them untouched.
				 * Inluding less markers will make the
				 * resulting code smaller, but there will
				 * be fewer aplications which can read it.
				 * The presence of the APP and COM marker
				 * is influenced by APP_len and COM_len
				 * ONLY, not by this property! */
	
#define V4L2_JPEG_MARKER_DHT (1<<3)    /* Define Huffman Tables */
#define V4L2_JPEG_MARKER_DQT (1<<4)    /* Define Quantization Tables */
#define V4L2_JPEG_MARKER_DRI (1<<5)    /* Define Restart Interval */
#define V4L2_JPEG_MARKER_COM (1<<6)    /* Comment segment */
#define V4L2_JPEG_MARKER_APP (1<<7)    /* App segment, driver will
                                        * allways use APP0 */
};


/*
 *	M E M O R Y - M A P P I N G   B U F F E R S
 */
struct v4l2_requestbuffers
{
	__u32	                count;
	enum v4l2_buf_type      type;
	enum v4l2_memory        memory;
	__u32	                reserved[2];
};

struct v4l2_buffer
{
	__u32			index;
	enum v4l2_buf_type      type;
	__u32			bytesused;
	__u32			flags;
	enum v4l2_field		field;
	struct timeval		timestamp;
	struct v4l2_timecode	timecode;
	__u32			sequence;

	/* memory location */
	enum v4l2_memory        memory;
	union {
		__u32           offset;
		unsigned long   userptr;
	} m;
	__u32			length;

	__u32			reserved[2];
};

/*  Flags for 'flags' field */
#define V4L2_BUF_FLAG_MAPPED	0x0001  /* Buffer is mapped (flag) */
#define V4L2_BUF_FLAG_QUEUED	0x0002	/* Buffer is queued for processing */
#define V4L2_BUF_FLAG_DONE	0x0004	/* Buffer is ready */
#define V4L2_BUF_FLAG_KEYFRAME	0x0008	/* Image is a keyframe (I-frame) */
#define V4L2_BUF_FLAG_PFRAME	0x0010	/* Image is a P-frame */
#define V4L2_BUF_FLAG_BFRAME	0x0020	/* Image is a B-frame */
#define V4L2_BUF_FLAG_TIMECODE	0x0100	/* timecode field is valid */

/*
 *	O V E R L A Y   P R E V I E W
 */
struct v4l2_framebuffer
{
	__u32			capability;
	__u32			flags;
/* FIXME: in theory we should pass something like PCI device + memory
 * region + offset instead of some physical address */
	void*                   base;
	struct v4l2_pix_format	fmt;
};
/*  Flags for the 'capability' field. Read only */
#define V4L2_FBUF_CAP_EXTERNOVERLAY	0x0001
#define V4L2_FBUF_CAP_CHROMAKEY		0x0002
#define V4L2_FBUF_CAP_LIST_CLIPPING     0x0004
#define V4L2_FBUF_CAP_BITMAP_CLIPPING	0x0008
/*  Flags for the 'flags' field. */
#define V4L2_FBUF_FLAG_PRIMARY		0x0001
#define V4L2_FBUF_FLAG_OVERLAY		0x0002
#define V4L2_FBUF_FLAG_CHROMAKEY	0x0004

struct v4l2_clip
{
	struct v4l2_rect        c;
	struct v4l2_clip	*next;
};

struct v4l2_window
{
	struct v4l2_rect        w;
	enum v4l2_field  	field;
	__u32			chromakey;
	struct v4l2_clip	*clips;
	__u32			clipcount;
	void			*bitmap;
};


/*
 *	C A P T U R E   P A R A M E T E R S
 */
struct v4l2_captureparm
{
	__u32		   capability;	  /*  Supported modes */
	__u32		   capturemode;	  /*  Current mode */
	struct v4l2_fract  timeperframe;  /*  Time per frame in .1us units */
	__u32		   extendedmode;  /*  Driver-specific extensions */
	__u32              readbuffers;   /*  # of buffers for read */
	__u32		   reserved[4];
};
/*  Flags for 'capability' and 'capturemode' fields */
#define V4L2_MODE_HIGHQUALITY	0x0001	/*  High quality imaging mode */
#define V4L2_CAP_TIMEPERFRAME	0x1000	/*  timeperframe field is supported */

struct v4l2_outputparm
{
	__u32		   capability;	 /*  Supported modes */
	__u32		   outputmode;	 /*  Current mode */
	struct v4l2_fract  timeperframe; /*  Time per frame in seconds */
	__u32		   extendedmode; /*  Driver-specific extensions */
	__u32              writebuffers; /*  # of buffers for write */
	__u32		   reserved[4];
};

/*
 *	I N P U T   I M A G E   C R O P P I N G
 */

struct v4l2_cropcap {
	enum v4l2_buf_type      type;	
        struct v4l2_rect        bounds;
        struct v4l2_rect        defrect;
        struct v4l2_fract       pixelaspect;
};

struct v4l2_crop {
	enum v4l2_buf_type      type;
	struct v4l2_rect        c;
};

/*
 *      A N A L O G   V I D E O   S T A N D A R D
 */

typedef __u64 v4l2_std_id;

/* one bit for each */
#define V4L2_STD_PAL_B          ((v4l2_std_id)0x00000001)
#define V4L2_STD_PAL_B1         ((v4l2_std_id)0x00000002)
#define V4L2_STD_PAL_G          ((v4l2_std_id)0x00000004)
#define V4L2_STD_PAL_H          ((v4l2_std_id)0x00000008)
#define V4L2_STD_PAL_I          ((v4l2_std_id)0x00000010)
#define V4L2_STD_PAL_D          ((v4l2_std_id)0x00000020)
#define V4L2_STD_PAL_D1         ((v4l2_std_id)0x00000040)
#define V4L2_STD_PAL_K          ((v4l2_std_id)0x00000080)

#define V4L2_STD_PAL_M          ((v4l2_std_id)0x00000100)
#define V4L2_STD_PAL_N          ((v4l2_std_id)0x00000200)
#define V4L2_STD_PAL_Nc         ((v4l2_std_id)0x00000400)
#define V4L2_STD_PAL_60         ((v4l2_std_id)0x00000800)

#define V4L2_STD_NTSC_M         ((v4l2_std_id)0x00001000)
#define V4L2_STD_NTSC_M_JP      ((v4l2_std_id)0x00002000)

#define V4L2_STD_SECAM_B        ((v4l2_std_id)0x00010000)
#define V4L2_STD_SECAM_D        ((v4l2_std_id)0x00020000)
#define V4L2_STD_SECAM_G        ((v4l2_std_id)0x00040000)
#define V4L2_STD_SECAM_H        ((v4l2_std_id)0x00080000)
#define V4L2_STD_SECAM_K        ((v4l2_std_id)0x00100000)
#define V4L2_STD_SECAM_K1       ((v4l2_std_id)0x00200000)
#define V4L2_STD_SECAM_L        ((v4l2_std_id)0x00400000)

/* ATSC/HDTV */
#define V4L2_STD_ATSC_8_VSB     ((v4l2_std_id)0x01000000)
#define V4L2_STD_ATSC_16_VSB    ((v4l2_std_id)0x02000000)

/* some common needed stuff */
#define V4L2_STD_PAL_BG		(V4L2_STD_PAL_B		|\
				 V4L2_STD_PAL_B1	|\
				 V4L2_STD_PAL_G)
#define V4L2_STD_PAL_DK		(V4L2_STD_PAL_D		|\
				 V4L2_STD_PAL_D1	|\
				 V4L2_STD_PAL_K)
#define V4L2_STD_PAL		(V4L2_STD_PAL_BG	|\
				 V4L2_STD_PAL_DK	|\
				 V4L2_STD_PAL_H		|\
				 V4L2_STD_PAL_I)
#define V4L2_STD_NTSC           (V4L2_STD_NTSC_M	|\
				 V4L2_STD_NTSC_M_JP)
#define V4L2_STD_SECAM		(V4L2_STD_SECAM_B	|\
				 V4L2_STD_SECAM_D	|\
				 V4L2_STD_SECAM_G	|\
				 V4L2_STD_SECAM_H	|\
				 V4L2_STD_SECAM_K	|\
				 V4L2_STD_SECAM_K1	|\
				 V4L2_STD_SECAM_L)

#define V4L2_STD_525_60		(V4L2_STD_PAL_M		|\
				 V4L2_STD_PAL_60	|\
				 V4L2_STD_NTSC)
#define V4L2_STD_625_50		(V4L2_STD_PAL		|\
				 V4L2_STD_PAL_N		|\
				 V4L2_STD_PAL_Nc	|\
				 V4L2_STD_SECAM)

#define V4L2_STD_UNKNOWN        0
#define V4L2_STD_ALL            (V4L2_STD_525_60	|\
				 V4L2_STD_625_50)

struct v4l2_standard
{
	__u32	       	     index;
	v4l2_std_id          id;
	__u8		     name[24];
	struct v4l2_fract    frameperiod; /* Frames, not fields */
	__u32		     framelines;
	__u32		     reserved[4];
};


/*
 *	V I D E O   I N P U T S
 */
struct v4l2_input
{
	__u32	     index;		/*  Which input */
	__u8	     name[32];	        /*  Label */
	__u32	     type;		/*  Type of input */
	__u32	     audioset;	        /*  Associated audios (bitfield) */
	__u32        tuner;             /*  Associated tuner */
	v4l2_std_id  std;
	__u32	     status;
	__u32	     reserved[4];
};
/*  Values for the 'type' field */
#define V4L2_INPUT_TYPE_TUNER		1
#define V4L2_INPUT_TYPE_CAMERA		2

/* field 'status' - general */
#define V4L2_IN_ST_NO_POWER    0x00000001  /* Attached device is off */
#define V4L2_IN_ST_NO_SIGNAL   0x00000002
#define V4L2_IN_ST_NO_COLOR    0x00000004

/* field 'status' - analog */
#define V4L2_IN_ST_NO_H_LOCK   0x00000100  /* No horizontal sync lock */
#define V4L2_IN_ST_COLOR_KILL  0x00000200  /* Color killer is active */

/* field 'status' - digital */
#define V4L2_IN_ST_NO_SYNC     0x00010000  /* No synchronization lock */
#define V4L2_IN_ST_NO_EQU      0x00020000  /* No equalizer lock */
#define V4L2_IN_ST_NO_CARRIER  0x00040000  /* Carrier recovery failed */

/* field 'status' - VCR and set-top box */
#define V4L2_IN_ST_MACROVISION 0x01000000  /* Macrovision detected */
#define V4L2_IN_ST_NO_ACCESS   0x02000000  /* Conditional access denied */
#define V4L2_IN_ST_VTR         0x04000000  /* VTR time constant */

/*
 *	V I D E O   O U T P U T S
 */
struct v4l2_output
{
	__u32	     index;		/*  Which output */
	__u8	     name[32];	        /*  Label */
	__u32	     type;		/*  Type of output */
	__u32	     audioset;	        /*  Associated audios (bitfield) */
	__u32	     modulator;         /*  Associated modulator */
	v4l2_std_id  std;
	__u32	     reserved[4];
};
/*  Values for the 'type' field */
#define V4L2_OUTPUT_TYPE_MODULATOR		1
#define V4L2_OUTPUT_TYPE_ANALOG			2
#define V4L2_OUTPUT_TYPE_ANALOGVGAOVERLAY	3

/*
 *	C O N T R O L S
 */
struct v4l2_control
{
	__u32		     id;
	__s32		     value;
};

/*  Used in the VIDIOC_QUERYCTRL ioctl for querying controls */
struct v4l2_queryctrl
{
	__u32	             id;
	enum v4l2_ctrl_type  type;
	__u8		     name[32];	/* Whatever */
	__s32		     minimum;	/* Note signedness */
	__s32		     maximum;
	__s32	             step;
	__s32		     default_value;
	__u32                flags;
	__u32		     reserved[2];
};

/*  Used in the VIDIOC_QUERYMENU ioctl for querying menu items */
struct v4l2_querymenu
{
	__u32		id;
	__u32		index;
	__u8		name[32];	/* Whatever */
	__u32		reserved;
};

/*  Control flags  */
#define V4L2_CTRL_FLAG_DISABLED		0x0001
#define V4L2_CTRL_FLAG_GRABBED		0x0002

/*  Control IDs defined by V4L2 */
#define V4L2_CID_BASE			0x00980900
/*  IDs reserved for driver specific controls */
#define V4L2_CID_PRIVATE_BASE		0x08000000

#define V4L2_CID_BRIGHTNESS		(V4L2_CID_BASE+0)
#define V4L2_CID_CONTRAST		(V4L2_CID_BASE+1)
#define V4L2_CID_SATURATION		(V4L2_CID_BASE+2)
#define V4L2_CID_HUE			(V4L2_CID_BASE+3)
#define V4L2_CID_AUDIO_VOLUME		(V4L2_CID_BASE+5)
#define V4L2_CID_AUDIO_BALANCE		(V4L2_CID_BASE+6)
#define V4L2_CID_AUDIO_BASS		(V4L2_CID_BASE+7)
#define V4L2_CID_AUDIO_TREBLE		(V4L2_CID_BASE+8)
#define V4L2_CID_AUDIO_MUTE		(V4L2_CID_BASE+9)
#define V4L2_CID_AUDIO_LOUDNESS		(V4L2_CID_BASE+10)
#define V4L2_CID_BLACK_LEVEL		(V4L2_CID_BASE+11)
#define V4L2_CID_AUTO_WHITE_BALANCE	(V4L2_CID_BASE+12)
#define V4L2_CID_DO_WHITE_BALANCE	(V4L2_CID_BASE+13)
#define V4L2_CID_RED_BALANCE		(V4L2_CID_BASE+14)
#define V4L2_CID_BLUE_BALANCE		(V4L2_CID_BASE+15)
#define V4L2_CID_GAMMA			(V4L2_CID_BASE+16)
#define V4L2_CID_WHITENESS		(V4L2_CID_GAMMA) /* ? Not sure */
#define V4L2_CID_EXPOSURE		(V4L2_CID_BASE+17)
#define V4L2_CID_AUTOGAIN		(V4L2_CID_BASE+18)
#define V4L2_CID_GAIN			(V4L2_CID_BASE+19)
#define V4L2_CID_HFLIP			(V4L2_CID_BASE+20)
#define V4L2_CID_VFLIP			(V4L2_CID_BASE+21)
#define V4L2_CID_HCENTER		(V4L2_CID_BASE+22)
#define V4L2_CID_VCENTER		(V4L2_CID_BASE+23)
#define V4L2_CID_LASTP1			(V4L2_CID_BASE+24) /* last CID + 1 */

/*
 *	T U N I N G
 */
struct v4l2_tuner
{
	__u32                   index;
	__u8			name[32];
	enum v4l2_tuner_type    type;
	__u32			capability;
	__u32			rangelow;
	__u32			rangehigh;
	__u32			rxsubchans;
	__u32			audmode;
	__s32			signal;
	__s32			afc;
	__u32			reserved[4];
};

struct v4l2_modulator
{
	__u32			index;
	__u8			name[32];
	__u32			capability;
	__u32			rangelow;
	__u32			rangehigh;
	__u32			txsubchans;
	__u32			reserved[4];
};

/*  Flags for the 'capability' field */
#define V4L2_TUNER_CAP_LOW		0x0001
#define V4L2_TUNER_CAP_NORM		0x0002
#define V4L2_TUNER_CAP_STEREO		0x0010
#define V4L2_TUNER_CAP_LANG2		0x0020
#define V4L2_TUNER_CAP_SAP		0x0020
#define V4L2_TUNER_CAP_LANG1		0x0040

/*  Flags for the 'rxsubchans' field */
#define V4L2_TUNER_SUB_MONO		0x0001
#define V4L2_TUNER_SUB_STEREO		0x0002
#define V4L2_TUNER_SUB_LANG2		0x0004
#define V4L2_TUNER_SUB_SAP		0x0004
#define V4L2_TUNER_SUB_LANG1		0x0008

/*  Values for the 'audmode' field */
#define V4L2_TUNER_MODE_MONO		0x0000
#define V4L2_TUNER_MODE_STEREO		0x0001
#define V4L2_TUNER_MODE_LANG2		0x0002
#define V4L2_TUNER_MODE_SAP		0x0002
#define V4L2_TUNER_MODE_LANG1		0x0003

struct v4l2_frequency
{
	__u32	              tuner;
	enum v4l2_tuner_type  type;
        __u32	              frequency;
	__u32	              reserved[8];
};

/*
 *	A U D I O
 */
struct v4l2_audio
{
	__u32	index;
	__u8	name[32];
	__u32	capability;
	__u32	mode;
	__u32	reserved[2];
};
/*  Flags for the 'capability' field */
#define V4L2_AUDCAP_STEREO		0x00001
#define V4L2_AUDCAP_AVL			0x00002

/*  Flags for the 'mode' field */
#define V4L2_AUDMODE_AVL		0x00001

struct v4l2_audioout
{
	__u32	index;
	__u8	name[32];
	__u32	capability;
	__u32	mode;
	__u32	reserved[2];
};

/*
 *	D A T A   S E R V I C E S   ( V B I )
 *
 *	Data services API by Michael Schimek
 */

struct v4l2_vbi_format
{
	__u32	sampling_rate;		/* in 1 Hz */
	__u32	offset;
	__u32	samples_per_line;
	__u32	sample_format;		/* V4L2_PIX_FMT_* */
	__s32	start[2];
	__u32	count[2];
	__u32	flags;			/* V4L2_VBI_* */
	__u32	reserved[2];		/* must be zero */
};

/*  VBI flags  */
#define V4L2_VBI_UNSYNC		(1<< 0)
#define V4L2_VBI_INTERLACED	(1<< 1)


/*
 *	A G G R E G A T E   S T R U C T U R E S
 */

/*	Stream data format
 */
struct v4l2_format
{
	enum v4l2_buf_type type;
	union
	{
		struct v4l2_pix_format	pix;  // V4L2_BUF_TYPE_VIDEO_CAPTURE
		struct v4l2_window	win;  // V4L2_BUF_TYPE_VIDEO_OVERLAY
		struct v4l2_vbi_format	vbi;  // V4L2_BUF_TYPE_VBI_CAPTURE
		__u8	raw_data[200];        // user-defined
	} fmt;
};


/*	Stream type-dependent parameters
 */
struct v4l2_streamparm
{
	enum v4l2_buf_type type;
	union
	{
		struct v4l2_captureparm	capture;
		struct v4l2_outputparm	output;
		__u8	raw_data[200];  /* user-defined */
	} parm;
};



/*
 *	I O C T L   C O D E S   F O R   V I D E O   D E V I C E S
 *
 */
#define VIDIOC_QUERYCAP		_IOR  ('V',  0, struct v4l2_capability)
#define VIDIOC_RESERVED		_IO   ('V',  1)
#define VIDIOC_ENUM_FMT         _IOWR ('V',  2, struct v4l2_fmtdesc)
#define VIDIOC_G_FMT		_IOWR ('V',  4, struct v4l2_format)
#define VIDIOC_S_FMT		_IOWR ('V',  5, struct v4l2_format)
#if 0
#define VIDIOC_G_COMP		_IOR  ('V',  6, struct v4l2_compression)
#define VIDIOC_S_COMP		_IOW  ('V',  7, struct v4l2_compression)
#endif
#define VIDIOC_REQBUFS		_IOWR ('V',  8, struct v4l2_requestbuffers)
#define VIDIOC_QUERYBUF		_IOWR ('V',  9, struct v4l2_buffer)
#define VIDIOC_G_FBUF		_IOR  ('V', 10, struct v4l2_framebuffer)
#define VIDIOC_S_FBUF		_IOW  ('V', 11, struct v4l2_framebuffer)
#define VIDIOC_OVERLAY		_IOWR ('V', 14, int)
#define VIDIOC_QBUF		_IOWR ('V', 15, struct v4l2_buffer)
#define VIDIOC_DQBUF		_IOWR ('V', 17, struct v4l2_buffer)
#define VIDIOC_STREAMON		_IOW  ('V', 18, int)
#define VIDIOC_STREAMOFF	_IOW  ('V', 19, int)
#define VIDIOC_G_PARM		_IOWR ('V', 21, struct v4l2_streamparm)
#define VIDIOC_S_PARM		_IOW  ('V', 22, struct v4l2_streamparm)
#define VIDIOC_G_STD		_IOR  ('V', 23, v4l2_std_id)
#define VIDIOC_S_STD		_IOW  ('V', 24, v4l2_std_id)
#define VIDIOC_ENUMSTD		_IOWR ('V', 25, struct v4l2_standard)
#define VIDIOC_ENUMINPUT	_IOWR ('V', 26, struct v4l2_input)
#define VIDIOC_G_CTRL		_IOWR ('V', 27, struct v4l2_control)
#define VIDIOC_S_CTRL		_IOW  ('V', 28, struct v4l2_control)
#define VIDIOC_G_TUNER		_IOWR ('V', 29, struct v4l2_tuner)
#define VIDIOC_S_TUNER		_IOW  ('V', 30, struct v4l2_tuner)
#define VIDIOC_G_AUDIO		_IOWR ('V', 33, struct v4l2_audio)
#define VIDIOC_S_AUDIO		_IOW  ('V', 34, struct v4l2_audio)
#define VIDIOC_QUERYCTRL	_IOWR ('V', 36, struct v4l2_queryctrl)
#define VIDIOC_QUERYMENU	_IOWR ('V', 37, struct v4l2_querymenu)
#define VIDIOC_G_INPUT		_IOR  ('V', 38, int)
#define VIDIOC_S_INPUT		_IOWR ('V', 39, int)
#define VIDIOC_G_OUTPUT		_IOR  ('V', 46, int)
#define VIDIOC_S_OUTPUT		_IOWR ('V', 47, int)
#define VIDIOC_ENUMOUTPUT	_IOWR ('V', 48, struct v4l2_output)
#define VIDIOC_G_AUDOUT		_IOWR ('V', 49, struct v4l2_audioout)
#define VIDIOC_S_AUDOUT		_IOW  ('V', 50, struct v4l2_audioout)
#define VIDIOC_G_MODULATOR	_IOWR ('V', 54, struct v4l2_modulator)
#define VIDIOC_S_MODULATOR	_IOW  ('V', 55, struct v4l2_modulator)
#define VIDIOC_G_FREQUENCY	_IOWR ('V', 56, struct v4l2_frequency)
#define VIDIOC_S_FREQUENCY	_IOW  ('V', 57, struct v4l2_frequency)
#define VIDIOC_CROPCAP		_IOR  ('V', 58, struct v4l2_cropcap)
#define VIDIOC_G_CROP		_IOWR ('V', 59, struct v4l2_crop)
#define VIDIOC_S_CROP		_IOW  ('V', 60, struct v4l2_crop)
#define VIDIOC_G_JPEGCOMP	_IOR  ('V', 61, struct v4l2_jpegcompression)
#define VIDIOC_S_JPEGCOMP	_IOW  ('V', 62, struct v4l2_jpegcompression)
#define VIDIOC_QUERYSTD      	_IOR  ('V', 63, v4l2_std_id)
#define VIDIOC_TRY_FMT      	_IOWR ('V', 64, struct v4l2_format)

#define BASE_VIDIOC_PRIVATE	192		/* 192-255 are private */


#ifdef __KERNEL__
/*
 *
 *	V 4 L 2   D R I V E R   H E L P E R   A P I
 *
 *	Some commonly needed functions for drivers (v4l2-common.o module)
 */
#include <linux/fs.h>

/*  Video standard functions  */
extern unsigned int v4l2_video_std_fps(struct v4l2_standard *vs);
extern int v4l2_video_std_construct(struct v4l2_standard *vs,
				    int id, char *name);

/*  Compatibility layer interface  */
typedef int (*v4l2_kioctl)(struct inode *inode, struct file *file,
			   unsigned int cmd, void *arg);
int v4l_compat_translate_ioctl(struct inode *inode, struct file *file,
			       int cmd, void *arg, v4l2_kioctl driver_ioctl);

/* names for fancy debug output */
extern char *v4l2_field_names[];
extern char *v4l2_type_names[];
extern char *v4l2_ioctl_names[];

#endif /* __KERNEL__ */
#endif /* __LINUX_VIDEODEV2_H */

/*
 * Local variables:
 * c-basic-offset: 8
 * End:
 */
#ifndef __LINUX_VIDEODEV2_H
#define __LINUX_VIDEODEV2_H
/*
 *	Video for Linux Two
 *
 *	Header file for v4l or V4L2 drivers and applications, for
 *	Linux kernels 2.2.x or 2.4.x.
 *
 *	See http://www.thedirks.org/v4l2/ for API specs and other
 *	v4l2 documentation.
 *
 *	Author: Bill Dirks <bdirks@pacbell.net>
 *		Justin Schoeman
 *		et al.
 */

#define V4L2_MAJOR_VERSION	0
#define V4L2_MINOR_VERSION	20


/*
 *	M I S C E L L A N E O U S
 */

/*  Four-character-code (FOURCC) */
#define v4l2_fourcc(a,b,c,d)\
        (((__u32)(a)<<0)|((__u32)(b)<<8)|((__u32)(c)<<16)|((__u32)(d)<<24))

/*  Open flag for non-capturing opens on capture devices  */
#define O_NONCAP	O_TRUNC
#define O_NOIO		O_TRUNC

/*  Timestamp data type, 64-bit signed integer, in nanoseconds  */
#ifndef STAMP_T
#define STAMP_T
typedef __s64 stamp_t;
#endif

/*
 *	D R I V E R   C A P A B I L I T I E S
 */
struct v4l2_capability
{
	char	name[32];	/* Descriptive, and unique */
	int	type;		/* Device type, see below */
	int	inputs;		/* Num video inputs */
	int	outputs;	/* Num video outputs */
	int	audios;		/* Num audio devices */
	int	maxwidth;
	int	maxheight;
	int	minwidth;
	int	minheight;
	int	maxframerate;
	__u32	flags;		/* Feature flags, see below */
	__u32	reserved[4];
};
/* Values for 'type' field */
#define V4L2_TYPE_CAPTURE	0	/* Is a video capture device */
#define V4L2_TYPE_CODEC		1	/* Is a CODEC device */
#define V4L2_TYPE_OUTPUT	2	/* Is a video output device */
#define V4L2_TYPE_FX		3	/* Is a video effects device */
#define V4L2_TYPE_VBI		4	/* Is a VBI capture device */
#define V4L2_TYPE_VTR		5	/* Is a tape recorder controller */
#define V4L2_TYPE_VTX		6	/* Is a teletext device */
#define V4L2_TYPE_RADIO		7	/* Is a radio device */
#define V4L2_TYPE_VBI_INPUT	4	/* Is a VBI capture device */
#define V4L2_TYPE_VBI_OUTPUT	9	/* Is a VBI output device */
#define V4L2_TYPE_PRIVATE	1000	/* Start of driver private types */
/* Flags for 'flags' field */
#define V4L2_FLAG_READ		0x00001 /* Can capture via read() call */
#define V4L2_FLAG_WRITE		0x00002 /* Can accept data via write() */
#define V4L2_FLAG_STREAMING	0x00004 /* Can capture streaming video */
#define V4L2_FLAG_PREVIEW	0x00008 /* Can do automatic preview */
#define V4L2_FLAG_SELECT	0x00010 /* Supports the select() call */
#define V4L2_FLAG_TUNER		0x00020 /* Can tune */
#define V4L2_FLAG_MONOCHROME	0x00040 /* Monochrome only */
#define V4L2_FLAG_DATA_SERVICE	0x00080 /* Has a related data service dev. */


/*
 *	V I D E O   I M A G E   F O R M A T
 */
struct v4l2_pix_format
{
	__u32	width;
	__u32	height;
	__u32	depth;
	__u32	pixelformat;
	__u32	flags;
	__u32	bytesperline;	/* only used when there are pad bytes */
	__u32	sizeimage;
	__u32	priv;		/* private data, depends on pixelformat */
};
/*           Pixel format    FOURCC                  depth  Description   */
#define V4L2_PIX_FMT_RGB332  v4l2_fourcc('R','G','B','1') /*  8  RGB-3-3-2     */
#define V4L2_PIX_FMT_RGB555  v4l2_fourcc('R','G','B','O') /* 16  RGB-5-5-5     */
#define V4L2_PIX_FMT_RGB565  v4l2_fourcc('R','G','B','P') /* 16  RGB-5-6-5     */
#define V4L2_PIX_FMT_RGB555X v4l2_fourcc('R','G','B','Q') /* 16  RGB-5-5-5 BE  */
#define V4L2_PIX_FMT_RGB565X v4l2_fourcc('R','G','B','R') /* 16  RGB-5-6-5 BE  */
#define V4L2_PIX_FMT_BGR24   v4l2_fourcc('B','G','R','3') /* 24  BGR-8-8-8     */
#define V4L2_PIX_FMT_RGB24   v4l2_fourcc('R','G','B','3') /* 24  RGB-8-8-8     */
#define V4L2_PIX_FMT_BGR32   v4l2_fourcc('B','G','R','4') /* 32  BGR-8-8-8-8   */
#define V4L2_PIX_FMT_RGB32   v4l2_fourcc('R','G','B','4') /* 32  RGB-8-8-8-8   */
#define V4L2_PIX_FMT_GREY    v4l2_fourcc('G','R','E','Y') /*  8  Greyscale     */
#define V4L2_PIX_FMT_YVU410  v4l2_fourcc('Y','V','U','9') /*  9  YVU 4:1:0     */
#define V4L2_PIX_FMT_YVU420  v4l2_fourcc('Y','V','1','2') /* 12  YVU 4:2:0     */
#define V4L2_PIX_FMT_YUYV    v4l2_fourcc('Y','U','Y','V') /* 16  YUV 4:2:2     */
#define V4L2_PIX_FMT_UYVY    v4l2_fourcc('U','Y','V','Y') /* 16  YUV 4:2:2     */
#if 0
#define V4L2_PIX_FMT_YVU422P v4l2_fourcc('4','2','2','P') /* 16  YVU422 planar */
#define V4L2_PIX_FMT_YVU411P v4l2_fourcc('4','1','1','P') /* 16  YVU411 planar */
#endif
#define V4L2_PIX_FMT_YUV422P v4l2_fourcc('4','2','2','P') /* 16  YVU422 planar */
#define V4L2_PIX_FMT_YUV411P v4l2_fourcc('4','1','1','P') /* 16  YVU411 planar */
#define V4L2_PIX_FMT_Y41P    v4l2_fourcc('Y','4','1','P') /* 12  YUV 4:1:1     */

/* two planes -- one Y, one Cr + Cb interleaved  */
#define V4L2_PIX_FMT_NV12    v4l2_fourcc('N','V','1','2') /* 12  Y/CbCr 4:2:0  */
#define V4L2_PIX_FMT_NV21    v4l2_fourcc('N','V','2','1') /* 12  Y/CrCb 4:2:0  */

/*  The following formats are not defined in the V4L2 specification */
#define V4L2_PIX_FMT_YUV410  v4l2_fourcc('Y','U','V','9') /*  9  YUV 4:1:0     */
#define V4L2_PIX_FMT_YUV420  v4l2_fourcc('Y','U','1','2') /* 12  YUV 4:2:0     */
#define V4L2_PIX_FMT_YYUV    v4l2_fourcc('Y','Y','U','V') /* 16  YUV 4:2:2     */
#define V4L2_PIX_FMT_HI240   v4l2_fourcc('H','I','2','4') /*  8  8-bit color   */

/* compressed formats */
#define V4L2_PIX_FMT_MJPEG    v4l2_fourcc('M','J','P','G') /* Motion-JPEG */

/*  Vendor-specific formats   */
#define V4L2_PIX_FMT_WNVA    v4l2_fourcc('W','N','V','A') /* Winnov hw compres */


/*  Flags */
#define V4L2_FMT_FLAG_COMPRESSED	0x0001	/* Compressed format */
#define V4L2_FMT_FLAG_BYTESPERLINE	0x0002	/* bytesperline field valid */
#define V4L2_FMT_FLAG_NOT_INTERLACED	0x0000
#define V4L2_FMT_FLAG_INTERLACED	0x0004	/* Image is interlaced */
#define V4L2_FMT_FLAG_TOPFIELD		0x0008	/* is a top field only */
#define V4L2_FMT_FLAG_BOTFIELD		0x0010	/* is a bottom field only */
#define V4L2_FMT_FLAG_ODDFIELD		V4L2_FMT_FLAG_TOPFIELD
#define V4L2_FMT_FLAG_EVENFIELD		V4L2_FMT_FLAG_BOTFIELD
#define V4L2_FMT_FLAG_COMBINED		V4L2_FMT_FLAG_INTERLACED
#define V4L2_FMT_FLAG_FIELD_field	0x001C
#define V4L2_FMT_CS_field		0xF000	/* Color space field mask */
#define V4L2_FMT_CS_601YUV		0x1000	/* ITU YCrCb color space */
#define V4L2_FMT_FLAG_SWCONVERSION	0x0800	/* used only in format enum. */
/*  SWCONVERSION indicates the format is not natively supported by the  */
/*  driver and the driver uses software conversion to support it  */


/*
 *	F O R M A T   E N U M E R A T I O N
 */
struct v4l2_fmtdesc
{
	int	index;			/* Format number */
	char	description[32];	/* Description string */
	__u32	pixelformat;		/* Format fourcc */
	__u32	flags;			/* Format flags */
	__u32	depth;			/* Bits per pixel */
	__u32	reserved[2];
};

struct v4l2_cvtdesc
{
	int	index;
	struct
	{
		__u32	pixelformat;
		__u32	flags;
		__u32	depth;
		__u32	reserved[2];
	}	in, out;
};

struct v4l2_fxdesc
{
	int	index;
	char 	name[32];
	__u32	flags;
	__u32	inputs;
	__u32	controls;
	__u32	reserved[2];
};


/*
 *	T I M E C O D E
 */
struct v4l2_timecode
{
	__u8	frames;
	__u8	seconds;
	__u8	minutes;
	__u8	hours;
	__u8	userbits[4];
	__u32	flags;
	__u32	type;
};
/*  Type  */
#define V4L2_TC_TYPE_24FPS		1
#define V4L2_TC_TYPE_25FPS		2
#define V4L2_TC_TYPE_30FPS		3
#define V4L2_TC_TYPE_50FPS		4
#define V4L2_TC_TYPE_60FPS		5


/*  Flags  */
#define V4L2_TC_FLAG_DROPFRAME		0x0001 /* "drop-frame" mode */
#define V4L2_TC_FLAG_COLORFRAME		0x0002
#define V4L2_TC_USERBITS_field		0x000C
#define V4L2_TC_USERBITS_USERDEFINED	0x0000
#define V4L2_TC_USERBITS_8BITCHARS	0x0008
/* The above is based on SMPTE timecodes */


/*
 *	C O M P R E S S I O N   P A R A M E T E R S
 */
#if 0
/* FIXME: generic compression settings don't work, there is too
          much codec-specific stuff.  Maybe reuse that for MPEG
          codec settings later ... */
struct v4l2_compression
{
	int	quality;
	int	keyframerate;
	int	pframerate;
	__u32	reserved[5];
};
#endif

/* for V4L2_PIX_FMT_MJPEG */
struct v4l2_jpegcompression
{
	int quality;

	int  APPn;              /* Number of APP segment to be written,
				 * must be 0..15 */
	int  APP_len;           /* Length of data in JPEG APPn segment */
	char APP_data[60];      /* Data in the JPEG APPn segment. */
	
	int  COM_len;           /* Length of data in JPEG COM segment */
	char COM_data[60];      /* Data in JPEG COM segment */
	
	__u32 jpeg_markers;     /* Which markers should go into the JPEG
				 * output. Unless you exactly know what
				 * you do, leave them untouched.
				 * Inluding less markers will make the
				 * resulting code smaller, but there will
				 * be fewer aplications which can read it.
				 * The presence of the APP and COM marker
				 * is influenced by APP_len and COM_len
				 * ONLY, not by this property! */
	
#define V4L2_JPEG_MARKER_DHT (1<<3)    /* Define Huffman Tables */
#define V4L2_JPEG_MARKER_DQT (1<<4)    /* Define Quantization Tables */
#define V4L2_JPEG_MARKER_DRI (1<<5)    /* Define Restart Interval */
#define V4L2_JPEG_MARKER_COM (1<<6)    /* Comment segment */
#define V4L2_JPEG_MARKER_APP (1<<7)    /* App segment, driver will
                                        * allways use APP0 */
};


/*
 *	M E M O R Y - M A P P I N G   B U F F E R S
 */
struct v4l2_requestbuffers
{
	int	count;
	__u32	type;
	__u32	reserved[2];
};
struct v4l2_buffer
{
	int			index;
	__u32			type;
	__u32			offset;
	__u32			length;
	__u32			bytesused;
	__u32			flags;
	stamp_t			timestamp;
	struct v4l2_timecode	timecode;
	__u32			sequence;
	__u32			reserved[3];
};
/*  Buffer type codes and flags for 'type' field */
#define V4L2_BUF_TYPE_field		0x00001FFF  /* Type field mask  */
#define V4L2_BUF_TYPE_CAPTURE		0x00000001
#define V4L2_BUF_TYPE_CODECIN		0x00000002
#define V4L2_BUF_TYPE_CODECOUT		0x00000003
#define V4L2_BUF_TYPE_EFFECTSIN		0x00000004
#define V4L2_BUF_TYPE_EFFECTSIN2	0x00000005
#define V4L2_BUF_TYPE_EFFECTSOUT	0x00000006
#define V4L2_BUF_TYPE_VIDEOOUT		0x00000007
#define V4L2_BUF_TYPE_FXCONTROL		0x00000008
#define V4L2_BUF_TYPE_VBI		0x00000009

/*  Starting value of driver private buffer types  */
#define V4L2_BUF_TYPE_PRIVATE		0x00001000

#define V4L2_BUF_ATTR_DEVICEMEM	0x00010000  /* Buffer is on device (flag) */

/*  Flags used only in VIDIOC_REQBUFS  */
#define V4L2_BUF_REQ_field	0xF0000000
#define V4L2_BUF_REQ_CONTIG	0x10000000  /* Map all buffers in one
					       contiguous mmap(). This flag
					       only used in VIDIOC_REQBUFS */

/*  Flags for 'flags' field */
#define V4L2_BUF_FLAG_MAPPED	0x0001  /* Buffer is mapped (flag) */
#define V4L2_BUF_FLAG_QUEUED	0x0002	/* Buffer is queued for processing */
#define V4L2_BUF_FLAG_DONE	0x0004	/* Buffer is ready */
#define V4L2_BUF_FLAG_KEYFRAME	0x0008	/* Image is a keyframe (I-frame) */
#define V4L2_BUF_FLAG_PFRAME	0x0010	/* Image is a P-frame */
#define V4L2_BUF_FLAG_BFRAME	0x0020	/* Image is a B-frame */
#define V4L2_BUF_FLAG_TOPFIELD	0x0040	/* Image is a top field only */
#define V4L2_BUF_FLAG_BOTFIELD	0x0080	/* Image is a bottom field only */
#define V4L2_BUF_FLAG_ODDFIELD	V4L2_BUF_FLAG_TOPFIELD
#define V4L2_BUF_FLAG_EVENFIELD	V4L2_BUF_FLAG_BOTFIELD
#define V4L2_BUF_FLAG_TIMECODE	0x0100	/* timecode field is valid */

/*
 *	O V E R L A Y   P R E V I E W
 */
struct v4l2_framebuffer
{
	__u32			capability;
	__u32			flags;
	void			*base[3];
	struct v4l2_pix_format	fmt;
};
/*  Flags for the 'capability' field. Read only */
#define V4L2_FBUF_CAP_EXTERNOVERLAY	0x0001
#define V4L2_FBUF_CAP_CHROMAKEY		0x0002
#define V4L2_FBUF_CAP_CLIPPING		0x0004
#define V4L2_FBUF_CAP_SCALEUP		0x0008
#define V4L2_FBUF_CAP_SCALEDOWN		0x0010
#define V4L2_FBUF_CAP_BITMAP_CLIPPING	0x0020
/*  Flags for the 'flags' field. */
#define V4L2_FBUF_FLAG_PRIMARY		0x0001
#define V4L2_FBUF_FLAG_OVERLAY		0x0002
#define V4L2_FBUF_FLAG_CHROMAKEY	0x0004

struct v4l2_clip
{
	int			x;
	int			y;
	int			width;
	int			height;
	struct v4l2_clip	*next;
};
struct v4l2_window
{
	int			x;
	int			y;
	int			width;
	int			height;
	__u32			chromakey;
	struct v4l2_clip	*clips;
	int			clipcount;
	void			*bitmap;
};


/*
 *	D E V I C E   P E R F O R M A N C E
 */
struct v4l2_performance
{
	int	frames;
	int	framesdropped;
	__u64	bytesin;
	__u64	bytesout;
	__u32	reserved[4];
};

/*
 *	C A P T U R E   P A R A M E T E R S
 */
struct v4l2_captureparm
{
	__u32		capability;	/*  Supported modes */
	__u32		capturemode;	/*  Current mode */
	unsigned long	timeperframe;	/*  Time per frame in .1us units */
	__u32		extendedmode;	/*  Driver-specific extensions */
	__u32		reserved[4];
};
/*  Flags for 'capability' and 'capturemode' fields */
#define V4L2_MODE_HIGHQUALITY	0x0001	/*  High quality imaging mode */
//#define V4L2_MODE_VFLIP		0x0002	/*  Flip image vertically */
//#define V4L2_MODE_HFLIP		0x0004	/*  Flip image horizontally */
#define V4L2_CAP_TIMEPERFRAME	0x1000	/*  timeperframe field is supported */

struct v4l2_outputparm
{
	__u32		capability;	/*  Supported modes */
	__u32		outputmode;	/*  Current mode */
	unsigned long	timeperframe;	/*  Time per frame in .1us units */
	__u32		extendedmode;	/*  Driver-specific extensions */
	__u32		reserved[4];
};

/*
 *	I N P U T   I M A G E   C R O P P I N G
 */
struct v4l2_cropcap
{
	__u32	capability;
	int	min_x;
	int	min_y;
	int	max_x;
	int	max_y;
	int	default_left;
	int	default_top;
	int	default_right;
	int	default_bottom;
	__u32	reserved[2];
};
struct v4l2_crop
{
	int	left;
	int	top;
	int	right;
	int	bottom;
	__u32	reserved;
};

/*
 *	D I G I T A L   Z O O M
 */
struct v4l2_zoomcap
{
	__u32	capability;
	__u32	maxwidth;
	__u32	maxheight;
	__u32	minwidth;
	__u32	minheight;
	__u32	reserved[2];
};
/*  Flags for the capability field  */
#define V4L2_ZOOM_NONCAP		0x0001
#define V4L2_ZOOM_WHILESTREAMING	0x0002

struct v4l2_zoom
{
	__u32	x;
	__u32	y;
	__u32	width;
	__u32	height;
	__u32	reserved;
};


/*
 *      A N A L O G   V I D E O   S T A N D A R D
 */
struct v4l2_standard
{
	__u8		name[24];
	struct {
		__u32	numerator;
		__u32	denominator;	/* >= 1 */
	} framerate;			/* Frames, not fields */
	__u32		framelines;
	__u32		reserved1;
	__u32		colorstandard;
	union {				
		struct {
			__u32		colorsubcarrier; /* Hz */
		} 		pal;
		struct {					
			__u32		colorsubcarrier; /* Hz */
		} 		ntsc;
		struct {
			__u32		f0b;	/* Hz (blue) */
			__u32		f0r;	/* Hz (red)  */
		} 		secam;
		__u8		reserved[12];
	} colorstandard_data;
	__u32		transmission;	/* Bit field. Must be zero for
					   non-modulators/demodulators. */
	__u32		reserved2;	/* Must be set to zero */
};

/*  Values for the 'colorstandard' field  */
#define V4L2_COLOR_STD_PAL		1
#define V4L2_COLOR_STD_NTSC		2
#define V4L2_COLOR_STD_SECAM		3
#define V4L2_COLOR_STD_AUTO		4

/*  Values for the color subcarrier fields  */
#define V4L2_COLOR_SUBC_PAL	4433619		/* PAL BGHI, NTSC-44 */
#define V4L2_COLOR_SUBC_PAL_M	3575611		/* PAL M (Brazil) */
#define V4L2_COLOR_SUBC_PAL_N	3582056		/* PAL N */
#define V4L2_COLOR_SUBC_NTSC	3579545		/* NTSC M, NTSC-Japan */
#define V4L2_COLOR_SUBC_SECAMB	4250000		/* SECAM B - Y carrier */
#define V4L2_COLOR_SUBC_SECAMR	4406250		/* SECAM R - Y carrier */

/*  Flags for the 'transmission' field  */
#define V4L2_TRANSM_STD_B		(1<<1)
#define V4L2_TRANSM_STD_D		(1<<3)
#define V4L2_TRANSM_STD_G		(1<<6)
#define V4L2_TRANSM_STD_H		(1<<7)
#define V4L2_TRANSM_STD_I		(1<<8)
#define V4L2_TRANSM_STD_K		(1<<10)
#define V4L2_TRANSM_STD_K1		(1<<11)
#define V4L2_TRANSM_STD_L		(1<<12)
#define V4L2_TRANSM_STD_M		(1<<13)
#define V4L2_TRANSM_STD_N		(1<<14)


/*  Used in the VIDIOC_ENUMSTD ioctl for querying supported standards  */
struct v4l2_enumstd
{
	int			index;
	struct v4l2_standard	std;
	__u32			inputs;  /* set of inputs that */
					 /* support this standard */
	__u32			outputs; /* set of outputs that */
					 /* support this standard */
	__u32			reserved[2];
};


/*
 *	V I D E O   I N P U T S
 */
struct v4l2_input
{
	int	index;		/*  Which input */
	char	name[32];	/*  Label */
	int	type;		/*  Type of input */
	__u32	capability;	/*  Capability flags */
	int	assoc_audio;	/*  Associated audio input */
	__u32	reserved[4];
};
/*  Values for the 'type' field */
#define V4L2_INPUT_TYPE_TUNER		1
#define V4L2_INPUT_TYPE_CAMERA		2

/*  Flags for the 'capability' field */
#define V4L2_INPUT_CAP_AUDIO		0x0001	/* assoc_audio */


/*
 *	V I D E O   O U T P U T S
 */
struct v4l2_output
{
	int	index;		/*  Which output */
	char	name[32];	/*  Label */
	int	type;		/*  Type of output */
	__u32	capability;	/*  Capability flags */
	int	assoc_audio;	/*  Associated audio */
	__u32	reserved[4];
};
/*  Values for the 'type' field */
#define V4L2_OUTPUT_TYPE_MODULATOR		1
#define V4L2_OUTPUT_TYPE_ANALOG			2
#define V4L2_OUTPUT_TYPE_ANALOGVGAOVERLAY	3

/*  Flags for the 'capability' field */
#define V4L2_OUTPUT_CAP_AUDIO		0x0001	/* assoc_audio */


/*
 *	C O N T R O L S
 */
struct v4l2_control
{
	__u32		id;
	int		value;
};

/*  Used in the VIDIOC_QUERYCTRL ioctl for querying controls */
struct v4l2_queryctrl
{
	__u32		id;
	__u8		name[32];	/* Whatever */
	int		minimum;	/* Note signedness */
	int		maximum;
	unsigned int	step;
	int		default_value;
	__u32		type;
	__u32		flags;
	__u32		category;	/* Automatically filled in by V4L2 */
	__u8		group[32];	/*   for pre-defined controls      */
	__u32		reserved[2];
};

/*  Used in the VIDIOC_QUERYMENU ioctl for querying menu items */
struct v4l2_querymenu
{
	__u32		id;
	int		index;
	__u8		name[32];	/* Whatever */
	int		reserved;
};

/*  Used in V4L2_BUF_TYPE_FXCONTROL buffers  */
struct v4l2_fxcontrol
{
	__u32	id;
	__u32	value;
};

/*  Control types  */
#define V4L2_CTRL_TYPE_INTEGER		0
#define V4L2_CTRL_TYPE_BOOLEAN		1
#define V4L2_CTRL_TYPE_MENU		2
#define V4L2_CTRL_TYPE_BUTTON		3

/*  Control flags  */
#define V4L2_CTRL_FLAG_DISABLED		0x0001
#define V4L2_CTRL_FLAG_GRABBED		0x0002

/*  Control categories	*/
#define V4L2_CTRL_CAT_VIDEO		1  /*  "Video"   */
#define V4L2_CTRL_CAT_AUDIO		2  /*  "Audio"   */
#define V4L2_CTRL_CAT_EFFECT		3  /*  "Effect"  */

/*  Control IDs defined by V4L2 */
#define V4L2_CID_BASE			0x00980900
/*  IDs reserved for driver specific controls */
#define V4L2_CID_PRIVATE_BASE		0x08000000
/*  IDs reserved for effect-specific controls on effects devices  */
#define V4L2_CID_EFFECT_BASE		0x0A00B000

#define V4L2_CID_BRIGHTNESS		(V4L2_CID_BASE+0)
#define V4L2_CID_CONTRAST		(V4L2_CID_BASE+1)
#define V4L2_CID_SATURATION		(V4L2_CID_BASE+2)
#define V4L2_CID_HUE			(V4L2_CID_BASE+3)
#define V4L2_CID_AUDIO_VOLUME		(V4L2_CID_BASE+5)
#define V4L2_CID_AUDIO_BALANCE		(V4L2_CID_BASE+6)
#define V4L2_CID_AUDIO_BASS		(V4L2_CID_BASE+7)
#define V4L2_CID_AUDIO_TREBLE		(V4L2_CID_BASE+8)
#define V4L2_CID_AUDIO_MUTE		(V4L2_CID_BASE+9)
#define V4L2_CID_AUDIO_LOUDNESS		(V4L2_CID_BASE+10)
#define V4L2_CID_BLACK_LEVEL		(V4L2_CID_BASE+11)
#define V4L2_CID_AUTO_WHITE_BALANCE	(V4L2_CID_BASE+12)
#define V4L2_CID_DO_WHITE_BALANCE	(V4L2_CID_BASE+13)
#define V4L2_CID_RED_BALANCE		(V4L2_CID_BASE+14)
#define V4L2_CID_BLUE_BALANCE		(V4L2_CID_BASE+15)
#define V4L2_CID_GAMMA			(V4L2_CID_BASE+16)
#define V4L2_CID_WHITENESS		(V4L2_CID_GAMMA) /* ? Not sure */
#define V4L2_CID_EXPOSURE		(V4L2_CID_BASE+17)
#define V4L2_CID_AUTOGAIN		(V4L2_CID_BASE+18)
#define V4L2_CID_GAIN			(V4L2_CID_BASE+19)
#define V4L2_CID_HFLIP			(V4L2_CID_BASE+20)
#define V4L2_CID_VFLIP			(V4L2_CID_BASE+21)
#define V4L2_CID_HCENTER		(V4L2_CID_BASE+22)
#define V4L2_CID_VCENTER		(V4L2_CID_BASE+23)
#define V4L2_CID_LASTP1			(V4L2_CID_BASE+24) /* last CID + 1 */
/*  Remember to change fill_ctrl_category() in videodev.c  */

/*
 *	T U N I N G
 */
struct v4l2_tuner
{
	int			input;
	char			name[32];
	struct v4l2_standard	std;
	__u32			capability;
	__u32			rangelow;
	__u32			rangehigh;
	__u32			rxsubchans;
	__u32			audmode;
	int			signal;
	int			afc;
	__u32			reserved[4];
};
struct v4l2_modulator
{
	int			output;
	char			name[32];
	struct v4l2_standard	std;
	__u32			capability;
	__u32			rangelow;
	__u32			rangehigh;
	__u32			txsubchans;
	__u32			reserved[4];
};
/*  Flags for the 'capability' field */
#define V4L2_TUNER_CAP_LOW		0x0001
#define V4L2_TUNER_CAP_NORM		0x0002
#define V4L2_TUNER_CAP_STEREO		0x0010
#define V4L2_TUNER_CAP_LANG2		0x0020
#define V4L2_TUNER_CAP_SAP		0x0020
#define V4L2_TUNER_CAP_LANG1		0x0040

/*  Flags for the 'rxsubchans' field */
#define V4L2_TUNER_SUB_MONO		0x0001
#define V4L2_TUNER_SUB_STEREO		0x0002
#define V4L2_TUNER_SUB_LANG2		0x0004
#define V4L2_TUNER_SUB_SAP		0x0004
#define V4L2_TUNER_SUB_LANG1		0x0008

/*  Values for the 'audmode' field */
#define V4L2_TUNER_MODE_MONO		0x0000
#define V4L2_TUNER_MODE_STEREO		0x0001
#define V4L2_TUNER_MODE_LANG2		0x0002
#define V4L2_TUNER_MODE_SAP		0x0002
#define V4L2_TUNER_MODE_LANG1		0x0003

struct v4l2_frequency
{
	int	input;
	__u32	frequency;
	__u32	reserved[2];
};

/*
 *	A U D I O
 */
struct v4l2_audio
{
	int	audio;
	char	name[32];
	__u32	capability;
	__u32	mode;
	__u32	reserved[2];
};
/*  Flags for the 'capability' field */
#define V4L2_AUDCAP_EFFECTS		0x0020
#define V4L2_AUDCAP_LOUDNESS		0x0040
#define V4L2_AUDCAP_AVL			0x0080

/*  Flags for the 'mode' field */
#define V4L2_AUDMODE_LOUDNESS		0x00002
#define V4L2_AUDMODE_AVL		0x00004
#define V4L2_AUDMODE_STEREO_field	0x0FF00
#define V4L2_AUDMODE_STEREO_LINEAR	0x00100
#define V4L2_AUDMODE_STEREO_PSEUDO	0x00200
#define V4L2_AUDMODE_STEREO_SPATIAL30	0x00300
#define V4L2_AUDMODE_STEREO_SPATIAL50	0x00400

struct v4l2_audioout
{
	int	audio;
	char	name[32];
	__u32	capability;
	__u32	mode;
	__u32	reserved[2];
};

/*
 *	D A T A   S E R V I C E S   ( V B I )
 *
 *	Data services API by Michael Schimek
 */

struct v4l2_vbi_format
{
	__u32	sampling_rate;		/* in 1 Hz */
	__u32	offset;
	__u32	samples_per_line;
	__u32	sample_format;		/* V4L2_VBI_SF_* */
	__s32	start[2];
	__u32	count[2];
	__u32	flags;			/* V4L2_VBI_* */
	__u32	reserved2;		/* must be zero */
};

/*  VBI sampling formats */
#define V4L2_VBI_SF_UBYTE	1

/*  VBI flags  */
#define V4L2_VBI_UNSYNC		(1<< 0)
#define V4L2_VBI_INTERLACED	(1<< 1)


/*
 *	A G G R E G A T E   S T R U C T U R E S
 */

/*	Stream data format
 */
struct v4l2_format
{
	__u32	type;
	union
	{
		struct v4l2_pix_format	pix;	/*  image format  */
		struct v4l2_vbi_format	vbi;	/*  VBI data  */
		/*  add more  */
		__u8	raw_data[200];  /* user-defined */
	} fmt;
};


/*	Stream type-dependent parameters
 */
struct v4l2_streamparm
{
	__u32	type;
	union
	{
		struct v4l2_captureparm	capture;
		struct v4l2_outputparm	output;
		/*  add more  */
		__u8	raw_data[200];  /* user-defined */
	} parm;
};



/*
 *	I O C T L   C O D E S   F O R   V I D E O   D E V I C E S
 *
 */
#define VIDIOC_QUERYCAP		_IOR  ('V',  0, struct v4l2_capability)
#define VIDIOC_RESERVED		_IO   ('V',  1)
#define VIDIOC_ENUM_PIXFMT	_IOWR ('V',  2, struct v4l2_fmtdesc)
#define VIDIOC_ENUM_FBUFFMT	_IOWR ('V',  3, struct v4l2_fmtdesc)
#define VIDIOC_G_FMT		_IOWR ('V',  4, struct v4l2_format)
#define VIDIOC_S_FMT		_IOWR ('V',  5, struct v4l2_format)
#if 0
#define VIDIOC_G_COMP		_IOR  ('V',  6, struct v4l2_compression)
#define VIDIOC_S_COMP		_IOW  ('V',  7, struct v4l2_compression)
#endif
#define VIDIOC_REQBUFS		_IOWR ('V',  8, struct v4l2_requestbuffers)
#define VIDIOC_QUERYBUF		_IOWR ('V',  9, struct v4l2_buffer)
#define VIDIOC_G_FBUF		_IOR  ('V', 10, struct v4l2_framebuffer)
#define VIDIOC_S_FBUF		_IOW  ('V', 11, struct v4l2_framebuffer)
#define VIDIOC_G_WIN		_IOR  ('V', 12, struct v4l2_window)
#define VIDIOC_S_WIN		_IOW  ('V', 13, struct v4l2_window)
#define VIDIOC_PREVIEW		_IOWR ('V', 14, int)
#define VIDIOC_QBUF		_IOWR ('V', 15, struct v4l2_buffer)
#define VIDIOC_DQBUF		_IOWR ('V', 17, struct v4l2_buffer)
#define VIDIOC_STREAMON		_IOW  ('V', 18, int)
#define VIDIOC_STREAMOFF	_IOW  ('V', 19, int)
#define VIDIOC_G_PERF		_IOR  ('V', 20, struct v4l2_performance)
#define VIDIOC_G_PARM		_IOWR ('V', 21, struct v4l2_streamparm)
#define VIDIOC_S_PARM		_IOW  ('V', 22, struct v4l2_streamparm)
#define VIDIOC_G_STD		_IOR  ('V', 23, struct v4l2_standard)
#define VIDIOC_S_STD		_IOW  ('V', 24, struct v4l2_standard)
#define VIDIOC_ENUMSTD		_IOWR ('V', 25, struct v4l2_enumstd)
#define VIDIOC_ENUMINPUT	_IOWR ('V', 26, struct v4l2_input)
#define VIDIOC_G_CTRL		_IOWR ('V', 27, struct v4l2_control)
#define VIDIOC_S_CTRL		_IOW  ('V', 28, struct v4l2_control)
#define VIDIOC_G_TUNER		_IOWR ('V', 29, struct v4l2_tuner)
#define VIDIOC_S_TUNER		_IOW  ('V', 30, struct v4l2_tuner)
#define VIDIOC_G_FREQ		_IOR  ('V', 31, int)
#define VIDIOC_S_FREQ		_IOWR ('V', 32, int)
#define VIDIOC_G_AUDIO		_IOWR ('V', 33, struct v4l2_audio)
#define VIDIOC_S_AUDIO		_IOW  ('V', 34, struct v4l2_audio)
#define VIDIOC_QUERYCTRL	_IOWR ('V', 36, struct v4l2_queryctrl)
#define VIDIOC_QUERYMENU	_IOWR ('V', 37, struct v4l2_querymenu)
#define VIDIOC_G_INPUT		_IOR  ('V', 38, int)
#define VIDIOC_S_INPUT		_IOWR ('V', 39, int)
#define VIDIOC_ENUMCVT		_IOWR ('V', 40, struct v4l2_cvtdesc)
#define VIDIOC_G_OUTPUT		_IOR  ('V', 46, int)
#define VIDIOC_S_OUTPUT		_IOWR ('V', 47, int)
#define VIDIOC_ENUMOUTPUT	_IOWR ('V', 48, struct v4l2_output)
#define VIDIOC_G_AUDOUT		_IOWR ('V', 49, struct v4l2_audioout)
#define VIDIOC_S_AUDOUT		_IOW  ('V', 50, struct v4l2_audioout)
#define VIDIOC_ENUMFX		_IOWR ('V', 51, struct v4l2_fxdesc)
#define VIDIOC_G_EFFECT		_IOR  ('V', 52, int)
#define VIDIOC_S_EFFECT		_IOWR ('V', 53, int)
#define VIDIOC_G_MODULATOR	_IOWR ('V', 54, struct v4l2_modulator)
#define VIDIOC_S_MODULATOR	_IOW  ('V', 55, struct v4l2_modulator)
#define VIDIOC_G_FREQUENCY	_IOWR ('V', 56, struct v4l2_frequency)
#define VIDIOC_S_FREQUENCY	_IOW  ('V', 57, struct v4l2_frequency)
#define VIDIOC_CROPCAP		_IOR  ('V', 58, struct v4l2_cropcap)
#define VIDIOC_G_CROP		_IOR  ('V', 59, struct v4l2_crop)
#define VIDIOC_S_CROP		_IOW  ('V', 60, struct v4l2_crop)
#define VIDIOC_G_JPEGCOMP	_IOR  ('V', 61, struct v4l2_jpegcompression)
#define VIDIOC_S_JPEGCOMP	_IOW  ('V', 62, struct v4l2_jpegcompression)

#define VIDIOC_ENUM_CAPFMT	VIDIOC_ENUM_PIXFMT
#define VIDIOC_ENUM_OUTFMT	VIDIOC_ENUM_PIXFMT
#define VIDIOC_ENUM_SRCFMT	VIDIOC_ENUM_PIXFMT
#define VIDIOC_ENUMFMT		VIDIOC_ENUM_PIXFMT

#define BASE_VIDIOC_PRIVATE	192		/* 192-255 are private */


#ifdef __KERNEL__
/*
 *
 *	V 4 L 2   D R I V E R   H E L P E R   A P I
 *
 *	Some commonly needed functions for drivers (v4l2-common.o module)
 */
#include <linux/fs.h>

extern void v4l2_version(int *major, int *minor);
extern int v4l2_major_number(void);
extern void v4l2_fill_ctrl_category(struct v4l2_queryctrl *qc);

/*  Simple queue management  */
struct v4l2_q_node
{
	struct v4l2_q_node 	*forw, *back;
};
struct v4l2_queue
{
	struct v4l2_q_node	*forw, *back;
	rwlock_t		qlock;
};
extern void  v4l2_q_init(struct v4l2_queue *q);
extern void  v4l2_q_add_head(struct v4l2_queue *q, struct v4l2_q_node *node);
extern void  v4l2_q_add_tail(struct v4l2_queue *q, struct v4l2_q_node *node);
extern void *v4l2_q_del_head(struct v4l2_queue *q);
extern void *v4l2_q_del_tail(struct v4l2_queue *q);
extern void *v4l2_q_peek_head(struct v4l2_queue *q);
extern void *v4l2_q_peek_tail(struct v4l2_queue *q);
extern void *v4l2_q_yank_node(struct v4l2_queue *q, struct v4l2_q_node *node);
extern int   v4l2_q_last(struct v4l2_queue *q);

/*  Time functions  */
extern unsigned long v4l2_timestamp_divide(stamp_t t,
					   unsigned long p_100ns);
extern unsigned long v4l2_timestamp_correct(stamp_t *t,
					    unsigned long p_100ns);

/*  Master Clock functions  */
struct v4l2_clock
{
	void	(*gettime)(stamp_t *);
};
extern int  v4l2_masterclock_register(struct v4l2_clock *clock);
extern void v4l2_masterclock_unregister(struct v4l2_clock *clock);
extern void v4l2_masterclock_gettime(stamp_t *curr);

/*  Video standard functions  */
extern unsigned int v4l2_video_std_fps(struct v4l2_standard *vs);
extern unsigned long v4l2_video_std_tpf(struct v4l2_standard *vs);
extern int v4l2_video_std_confirm(struct v4l2_standard *vs);
extern int v4l2_video_std_construct(struct v4l2_standard *vs,
				    int id, __u32 transmission);

#define V4L2_STD_PAL		1	/* PAL B, G, H, I (...) */
#define V4L2_STD_PAL_M		5	/* (Brazil) */
#define V4L2_STD_PAL_N		6	/* (Argentina, Paraguay, Uruguay) */
#define V4L2_STD_PAL_60		10	/* PAL/NTSC hybrid */
#define V4L2_STD_NTSC		11	/* NTSC M (USA, ...) */
#define V4L2_STD_NTSC_N		12	/* (Barbados, Bolivia, Colombia, 
					   S. Korea) */
#define V4L2_STD_NTSC_44	15	/* PAL/NTSC hybrid */
#define V4L2_STD_SECAM		21	/* SECAM B, D, G, K, K1 (...) */
//#define V4L2_STD_SECAM_H	27	/* (Greece, Iran, Morocco) */ 
//#define V4L2_STD_SECAM_L	28	/* (France, Luxembourg, Monaco) */
//#define V4L2_STD_SECAM_M	29	/* (Jamaica) */

/*  Compatibility layer interface  */
typedef int (*v4l2_kioctl)(struct inode *inode, struct file *file,
			   unsigned int cmd, void *arg);
int v4l_compat_translate_ioctl(struct inode *inode, struct file *file,
			       int cmd, void *arg, v4l2_kioctl driver_ioctl);

#endif /* __KERNEL__ */
#endif /* __LINUX_VIDEODEV2_H */
