#include "gradm.h"

extern FILE *gradmin;
extern int gradmparse(void);

static struct deleted_file *
is_deleted_file_dupe(const char *filename)
{
	struct deleted_file *tmp;

	tmp = deleted_files;

	do {
		if (!strcmp(filename, tmp->filename))
			return tmp;
	} while ((tmp = tmp->next));

	return NULL;
}

static struct deleted_file *
add_deleted_file(char *filename)
{
	struct deleted_file *dfile;
	struct deleted_file *retfile;
	static ino_t ino = 0;

	ino++;

	if (!deleted_files) {
		deleted_files = malloc(sizeof (struct deleted_file));
		if (!deleted_files)
			failure("malloc");
		deleted_files->filename = filename;
		deleted_files->ino = ino;
		deleted_files->next = NULL;
	} else {
		retfile = is_deleted_file_dupe(filename);
		if (retfile)
			return retfile;
		dfile = malloc(sizeof (struct deleted_file));
		if (!dfile)
			failure("malloc");
		dfile->filename = filename;
		dfile->ino = ino;
		dfile->next = deleted_files;
		deleted_files = dfile;
	}

	return deleted_files;
}

static struct file_acl *
is_proc_object_dupe(struct file_acl *filp, struct file_acl *filp2)
{
	struct file_acl *tmp = filp;

	for_each_object(tmp, filp)
	    if (((tmp->inode == filp2->inode) && (tmp->dev == filp2->dev)) ||
		!strcmp(tmp->filename, filp2->filename))
		return tmp;

	return NULL;
}

static struct proc_acl *
is_proc_subject_dupe(struct acl_tmps *acl_tmp, struct proc_acl *proc)
{
	struct proc_acl *tmp = acl_tmp->proc;

	for_each_subject(tmp, acl_tmp)
	    if (((tmp->inode == proc->inode) && (tmp->dev == proc->dev)) ||
		!strcmp(tmp->filename, proc->filename))
		return tmp;

	return NULL;
}

static int
is_include_dupe(const char *filename)
{
	struct include_line *tmp = includes;

	for_each_include(tmp)
	    if (!strcmp(tmp->filename, filename))
		return 1;

	return 0;
}

static int
add_globbing_file(struct file_acl **filp, char *filename, __u32 mode, int type)
{
	char tmp[PATH_MAX] = { 0 };
	unsigned int len = strlen(filename);
	unsigned long i;
	char *last;
	glob_t pglob;
	int err;
	struct file_acl ftmp, ftmp2;
	struct stat fstat;

	while (len--) {
		switch (*(filename + len)) {
		case '*':
		case '?':
		case '/':
			goto out;
		}
	}
      out:
	/* this means the globbing was done before the last path component
	   therefore we want to add objects even if they don't exist
	   eg: /home/ * /test would match /home/user1/test, /home/user2/test,
	   /home/user3/test....etc even if the "test" file did not exist
	   in their homedirs
	 */
	if (*(filename + len) == '/') {
		len = strlen(filename);
		while (len--) {
			switch (*(filename + len)) {
			case '*':
			case '?':
				goto out2;
			}
		}
	      out2:
		last = strchr(filename + len, '/');
		*last = '\0';
		last++;

		err = glob(filename, GLOB_PERIOD | GLOB_ONLYDIR, NULL, &pglob);

		if (err) {
			fprintf(stderr, "glob() error \"%s\" encountered"
				" on line %lu of %s.\n"
				"The ACL system will not load until this"
				" error is fixed.\n", strerror(errno), lineno,
				current_acl_file);
			exit(EXIT_FAILURE);
		}

		for (i = 0; i < pglob.gl_pathc; i++) {
			len = strlen(*(pglob.gl_pathv + i));

			if (len > 3) {
				char *p;
				p = (*(pglob.gl_pathv + i) + len - 3);
				if (!strcmp(p, "/.."))
					continue;
				p++;
				if (!strcmp(p, "/."))
					continue;
			}

			snprintf(tmp, sizeof (tmp), "%s/%s",
				 *(pglob.gl_pathv + i), last);

			ftmp.filename = tmp;

			if (!lstat(ftmp.filename, &fstat)) {
				ftmp.inode = fstat.st_ino;
				ftmp.dev =
				    MKDEV(MAJOR(fstat.st_dev),
					  MINOR(fstat.st_dev));

				if (S_ISLNK(fstat.st_mode)) {
					char buf[PATH_MAX];
					memset(&buf, 0, sizeof (buf));
					realpath(ftmp.filename, buf);
					ftmp2.filename = buf;

					if (!stat(ftmp2.filename, &fstat)) {
						ftmp2.inode = fstat.st_ino;
						ftmp2.dev =
						    MKDEV(MAJOR(fstat.st_dev),
							  MINOR(fstat.st_dev));

						if (is_proc_object_dupe
						    (*filp, &ftmp2))
							goto add_link1;
						if (!add_proc_object_acl
						    (filp, strdup(buf), mode,
						     type | GR_GLOB))
							return 0;
					}
				}
add_link1:
				if (is_proc_object_dupe(*filp, &ftmp))
					continue;
				if (!add_proc_object_acl
				    (filp, strdup(tmp), mode, type))
					return 0;
			}
		}
		globfree(&pglob);
	} else {
		err = glob(filename, GLOB_PERIOD, NULL, &pglob);
		if (err) {
			fprintf(stderr, "glob() error \"%s\" encountered"
				" on line %lu of %s.\n"
				"The ACL system will not load until this"
				" error is fixed.\n", strerror(errno), lineno,
				current_acl_file);
			exit(EXIT_FAILURE);
		}

		for (i = 0; i < pglob.gl_pathc; i++) {
			len = strlen(*(pglob.gl_pathv + i));

			if (len > 3) {
				char *p;
				p = (*(pglob.gl_pathv + i) + len - 3);
				if (!strcmp(p, "/.."))
					continue;
				p++;
				if (!strcmp(p, "/."))
					continue;
			}

			ftmp.filename = *(pglob.gl_pathv + i);

			if (!lstat(ftmp.filename, &fstat)) {
				ftmp.inode = fstat.st_ino;
				ftmp.dev =
				    MKDEV(MAJOR(fstat.st_dev),
					  MINOR(fstat.st_dev));

				if (S_ISLNK(fstat.st_mode)) {
					char buf[PATH_MAX];
					memset(&buf, 0, sizeof (buf));
					realpath(ftmp.filename, buf);
					ftmp2.filename = buf;

					if (!stat(ftmp2.filename, &fstat)) {
						ftmp2.inode = fstat.st_ino;
						ftmp2.dev =
						    MKDEV(MAJOR(fstat.st_dev),
							  MINOR(fstat.st_dev));

						if (is_proc_object_dupe
						    (*filp, &ftmp2))
							goto add_link2;
						if (!add_proc_object_acl
						    (filp, strdup(buf), mode,
						     type | GR_GLOB))
							return 0;
					}
				}
add_link2:
				if (is_proc_object_dupe(*filp, &ftmp))
					continue;
				if (!add_proc_object_acl
				    (filp, *(pglob.gl_pathv + i), mode,
				     type))
					return 0;

			}
		}
	}

	return 1;
}

static void
display_all_dupes(struct file_acl *filp, struct file_acl *filp2)
{
	struct file_acl *tmp = filp;
	struct stat fstat;
	struct file_acl ftmp;

	for_each_object(tmp, filp)
	    if (!stat(tmp->filename, &fstat)) {
		ftmp.inode = fstat.st_ino;
		ftmp.dev = MKDEV(MAJOR(fstat.st_dev), MINOR(fstat.st_dev));
		if (ftmp.inode == filp2->inode && ftmp.dev == filp2->dev)
			fprintf(stderr, "%s\n", tmp->filename);
	}

	return;
}

int
add_proc_object_acl(struct file_acl **filp, char *filename,
		    __u32 mode, int type)
{
	struct file_acl *p;
	struct file_acl *p2;
	struct stat fstat;
	struct deleted_file *dfile;
	unsigned int file_len = strlen(filename) + 1;

	if (!filename) {
		fprintf(stderr, "Out of memory.\n");
		exit(EXIT_FAILURE);
	}

	if (strchr(filename, '?') || strchr(filename, '*'))
		return add_globbing_file(filp, filename, mode, type);

	if (lstat(filename, &fstat)) {
		dfile = add_deleted_file(filename);
		fstat.st_ino = dfile->ino;
		fstat.st_dev = 0;
		mode |= GR_DELETED;
	} else if (S_ISLNK(fstat.st_mode) && !(type & GR_GLOB)) {
		char buf[PATH_MAX];

		memset(&buf, 0, sizeof (buf));
		realpath(filename, buf);
		add_proc_object_acl(filp, strdup(buf), mode, type | GR_SYMLINK);
	}

	if ((p =
	     (struct file_acl *) calloc(1, sizeof (struct file_acl))) == NULL)
		failure("calloc");

	if (*filp)
		(*filp)->next = p;

	p->prev = *filp;

	if ((filename[file_len - 2] == '/') && file_len != 2)
		filename[file_len - 2] = '\0';

	if (file_len > PATH_MAX) {
		fprintf(stderr, "Filename too long on line %lu of file %s.\n",
			lineno, current_acl_file);
		exit(EXIT_FAILURE);
	}

	p->filename = filename;
	p->mode = mode;
	p->inode = fstat.st_ino;
	p->dev = MKDEV(MAJOR(fstat.st_dev), MINOR(fstat.st_dev));

	if (type & GR_FLEARN) {
		struct file_acl *tmp = *filp;

		for_each_object(tmp, *filp) {
			if (!strcmp(tmp->filename, p->filename) ||
			    ((tmp->inode == p->inode)
			     && (tmp->dev == p->dev))) {
				tmp->mode |= mode;
				return 1;
			}
		}
	} else if ((p2 = is_proc_object_dupe(*filp, p))) {
		if (type & GR_SYMLINK)
			return 1;
		fprintf(stderr, "Duplicate ACL entry found for \"%s\""
			" on line %lu of %s.\n"
			"\"%s\" references the same object as the following object(s):\n",
			p->filename, lineno, current_acl_file, p->filename);
		display_all_dupes(*filp, p);
		fprintf(stderr, "specified on an earlier line."
			"The ACL system will not load until this"
			" error is fixed.\n");
		return 0;
	}

	*filp = p;

	return 1;
}

/*
void rem_proc_object_acl(struct proc_acl *proc, struct file_acl *filp)
{
	if(filp->next)
		filp->next->prev = filp->prev;
	else if(filp->prev) // we're in the first run of the for loop
		filp->prev->next = NULL;
	// else - we're the only acl in this subject
	free(filp); 

	return;
}
*/

int
add_proc_subject_acl(struct acl_tmps **acl_tmp, char *filename, __u32 mode)
{
	struct proc_acl *p;
	struct proc_acl *p2;
	struct deleted_file *dfile;
	struct ip_acl *tmpi;
	struct stat fstat;
	unsigned short i;
	unsigned int file_len;

	if (!filename) {
		fprintf(stderr, "Out of memory.\n");
		exit(EXIT_FAILURE);
	}

	file_len = strlen(filename) + 1;

	if ((p =
	     (struct proc_acl *) calloc(1, sizeof (struct proc_acl))) == NULL)
		failure("calloc");

	if ((*acl_tmp)->proc)
		(*acl_tmp)->proc->next = p;

	p->prev = (*acl_tmp)->proc;

	if ((filename[file_len - 2] == '/') && file_len != 2)
		filename[file_len - 2] = '\0';

	if (file_len > PATH_MAX) {
		fprintf(stderr, "Filename too long on line %lu of file %s.\n",
			lineno, current_acl_file);
		exit(EXIT_FAILURE);
	}

	if (!strcmp(filename, "god") || stat(filename, &fstat)) {
		dfile = add_deleted_file(filename);
		fstat.st_ino = dfile->ino;
		fstat.st_dev = 0;
		mode |= GR_DELETED;
	}

	p->filename = filename;
	p->mode = mode;

	p->proc_object = (*acl_tmp)->filp;

	p->ip_object = (*acl_tmp)->ipp;
	if (p->ip_object) {
		for_each_object(tmpi, p->ip_object) {
			for (i = 0; i < 8; i++)
				p->ip_proto[i] |= tmpi->proto[i];
			p->ip_type |= tmpi->type;
		}
	}

	p->cap_raise = (*acl_tmp)->cap_raise_tmp;
	p->cap_drop = (*acl_tmp)->cap_drop_tmp;

	memcpy(p->res, (*acl_tmp)->res, sizeof ((*acl_tmp)->res));
	p->resmask = (*acl_tmp)->resmask;

	if ((p->resmask & (1 << RLIM_NLIMITS)) &&	// check for RES_CRASH
	    S_ISDIR(fstat.st_mode) && !(mode & GR_DELETED)) {
		fprintf(stderr, "RES_CRASH is only valid for binary "
			"ACL subjects.\n"
			"The ACL system will not load until this "
			"error in subject ACL: %s is fixed.\n\n", p->filename);
		exit(EXIT_FAILURE);
	}

	p->dev = MKDEV(MAJOR(fstat.st_dev), MINOR(fstat.st_dev));
	p->inode = fstat.st_ino;

	if ((p2 = is_proc_subject_dupe(*acl_tmp, p))) {
		fprintf(stderr, "Duplicate ACL entry found for \"%s\""
			" on line %lu of %s.\n"
			"\"%s\" references the same subject as \"%s\""
			" specified on an earlier line.\n"
			"The ACL system will not load until this"
			" error is fixed.\n", p->filename, lineno,
			current_acl_file, p->filename, p2->filename);
		return 0;
	}

	(*acl_tmp)->cap_raise_tmp = 0;
	(*acl_tmp)->cap_drop_tmp = 0;
	memset((*acl_tmp)->res, 0, sizeof ((*acl_tmp)->res));
	memset((*acl_tmp)->ip_proto, 0, sizeof ((*acl_tmp)->ip_proto));
	(*acl_tmp)->ip_type = 0;
	(*acl_tmp)->resmask = 0;
	(*acl_tmp)->filp = NULL;
	(*acl_tmp)->ipp = NULL;
	(*acl_tmp)->proc = p;

	return 1;
}

/*
void rem_proc_subject_acl(struct proc_acl * proc)
{
	struct file_acl * filp = proc->proc_object;

	while(filp && filp->prev) {
		filp = filp->prev;
		free(filp->next);
	}

	if(filp)
		free(filp);
	
	if(proc->next)
		proc->next->prev = proc->prev;
	else if(proc->prev) // we're at the end of process acls
		proc->prev->next = NULL;
	// else - we're the only subject acl, impossible

	free(proc);

	return;
}
*/

__u32
proc_subject_mode_conv(const char *mode)
{
	int i;
	__u32 retmode = 0;

	retmode |= GR_FIND;

	for (i = 0; i < strlen(mode); i++) {
		switch (mode[i]) {
		case 'T':
			retmode |= GR_NOTROJAN;
			break;
		case 'K':
			retmode |= GR_KILLPROC;
			break;
		case 'C':
			retmode |= GR_KILLIPPROC;
			break;
		case 'A':
			retmode |= GR_PROTSHM;
			break;
		case 'P':
			retmode |= GR_PAXPAGE;
			break;
		case 'R':
			retmode |= GR_PAXRANDMMAP;
			break;
		case 'M':
			retmode |= GR_PAXMPROTECT;
			break;
		case 'S':
			retmode |= GR_PAXSEGM;
			break;
		case 'G':
			retmode |= GR_PAXGCC;
			break;
		case 'X':
			retmode |= GR_PAXRANDEXEC;
			break;
		case 'O':
			retmode |= GR_IGNORE;
			break;
		case 'o':
			retmode |= GR_OVERRIDE;
			break;
		case 'l':
			retmode |= GR_LEARN;
			break;
		case 'h':
			retmode &= ~GR_FIND;
			break;
		case 'p':
			retmode |= GR_PROTECTED;
			break;
		case 'k':
			retmode |= GR_KILL;
			break;
		case 'v':
			retmode |= GR_VIEW;
			break;
		case 'd':
			retmode |= GR_PROTPROCPID;
			break;
		case 'b':
			retmode |= GR_PROCACCT;
			break;
		default:
			fprintf(stderr, "Invalid proc subject mode "
				"\'%c\' found on line %lu "
				"of %s\n", mode[i], lineno, current_acl_file);
		}
	}

	return retmode;
}

__u32
proc_object_mode_conv(const char *mode)
{
	int i;
	__u32 retmode = 0;

	retmode |= GR_FIND;

	for (i = 0; i < strlen(mode); i++) {
		switch (mode[i]) {
		case 'r':
			retmode |= GR_READ;
			break;
		case 'w':
			retmode |= GR_WRITE;
			retmode |= GR_APPEND;
			break;
		case 'x':
			retmode |= GR_EXEC;
			break;
		case 'a':
			retmode |= GR_APPEND;
			break;
		case 'h':
			retmode &= ~GR_FIND;
			break;
		case 'i':
			retmode |= GR_INHERIT;
			break;
		case 't':
			retmode |= GR_PTRACERD;
			break;
		case 'F':
			retmode |= GR_AUDIT_FIND;
			break;
		case 'R':
			retmode |= GR_AUDIT_READ;
			break;
		case 'W':
			retmode |= GR_AUDIT_WRITE;
			retmode |= GR_AUDIT_APPEND;
			break;
		case 'X':
			retmode |= GR_AUDIT_EXEC;
			break;
		case 'A':
			retmode |= GR_AUDIT_APPEND;
			break;
		case 'I':
			retmode |= GR_AUDIT_INHERIT;
			break;
		case 's':
			retmode |= GR_SUPPRESS;
			break;
		default:
			fprintf(stderr, "Invalid proc object mode "
				"\'%c\' found on line %lu "
				"of %s\n", mode[i], lineno, current_acl_file);
		}
	}

	return retmode;
}

void
add_include(const char *filename)
{
	int n;
	struct stat fstat;
	char *dir;
	struct include_line *p;

	if (filename[0] == '/') {
		if ((dir =
		     (char *) calloc(strlen(filename) + 1,
				     sizeof (char))) == NULL)
			failure("calloc");
		strcpy(dir, filename);
	} else {
		fprintf(stderr,
			"Only absolute paths are supported for include.\n"
			"The include of %s is therefore invalid.\n", filename);
		exit(EXIT_FAILURE);
	}

	if (stat(dir, &fstat) < 0) {
		fprintf(stderr, "Unable to stat include \"%s\".\n"
			"Error: %s\n", dir, strerror(errno));
		exit(EXIT_FAILURE);
	}

	if (S_ISDIR(fstat.st_mode)) {
		struct dirent **namelist;
		char path[PATH_MAX];

		n = scandir(dir, &namelist, 0, alphasort);
		if (n >= 0) {
			while (n--) {
				if (strcmp(namelist[n]->d_name, ".")
				    && strcmp(namelist[n]->d_name, "..")) {
					memset(&path, 0, sizeof (path));
					snprintf(path, PATH_MAX - 1, "%s/%s",
						 dir, namelist[n]->d_name);
					add_include(path);
				}
			}
		}
		free(dir);
		return;
	}

	if (is_include_dupe(dir)) {
		fprintf(stderr,
			"Duplicate include entry \"%s\" found on line %lu "
			"of %s.\n" "The ACL system will not load until this "
			"error is fixed.\n", dir, lineno, current_acl_file);
		exit(EXIT_FAILURE);
	}

	if ((p =
	     (struct include_line *) calloc(1,
					    sizeof (struct include_line))) ==
	    NULL)
		failure("calloc");

	if (includes)
		includes->next = p;

	p->prev = includes;

	p->filename = dir;
	p->left = includeno++;

	includes = p;

	return;
}

void
parse_acls(void)
{
	struct include_line *p;
	struct include_line *tmp;

	if (chdir(GRSEC_DIR) < 0) {
		fprintf(stderr, "Error changing directory to %s\n"
			"Error: %s\n", GRSEC_DIR, strerror(errno));
		exit(EXIT_FAILURE);
	}

	add_admin_acl();
	add_kernel_acl();
	add_gradm_acl();

	gradmin = open_acl_file(GR_ACL_PATH);
	add_include(GR_ACL_PATH);
	change_current_acl_file(GR_ACL_PATH);
	gradmparse();
	fclose(gradmin);

	if (includeno == 1)
		return;

	do {

		tmp = includes;
		includeno = 1;

		p = includes;
		do {
			lineno = 1;
			def_acl_tmp->cap_raise_tmp = 0;
			def_acl_tmp->cap_drop_tmp = 0;
			memset(def_acl_tmp->res, 0, sizeof (def_acl_tmp->res));
			memset(def_acl_tmp->ip_proto, 0,
			       sizeof (def_acl_tmp->ip_proto));
			def_acl_tmp->ip_type = 0;
			def_acl_tmp->resmask = 0;

			gradmin = open_acl_file(p->filename);
			change_current_acl_file(p->filename);
			gradmparse();
			fclose(gradmin);

			p = p->prev;
		} while (p->left);
		tmp->left = 0;
	} while (tmp != includes);

	return;
}

struct gr_arg *
conv_user_to_kernel(struct gr_pw_entry *entry)
{
	struct gr_arg *retarg;
	struct user_acl_subject_db *subj_db;
	struct proc_acl *tmp = NULL;
	struct file_acl *tmpf = NULL;
	struct ip_acl *tmpi = NULL;
	struct ip_acl *i_tmp = NULL;
	struct proc_acl **s_tmp = NULL;
	struct ip_acl **i_table = NULL;
	unsigned long facls = 0;
	unsigned long pacls = 0;
	unsigned long iacls = 0;
	unsigned long tiacls = 0;
	unsigned long i = 0;
	int err;

	for_each_subject(tmp, def_acl_tmp) {
		pacls++;
		for_each_object(tmpi, tmp->ip_object)
		    tiacls++;
		for_each_object(tmpf, tmp->proc_object)
		    facls++;
	}

	if ((retarg =
	     (struct gr_arg *) calloc(1, sizeof (struct gr_arg))) == NULL)
		failure("calloc");

	err = mlock(retarg, sizeof (struct gr_arg));
	if (err)
		fprintf(stderr, "Warning, unable to lock authentication "
			"structure in physical memory.\n");

	if (!pacls && !facls)	// we are disabling, don't want to calloc 0
		goto set_pw;

	if ((subj_db =
	     (struct user_acl_subject_db *) calloc(1,
						   sizeof (struct
							   user_acl_subject_db)))
	    == NULL)
		failure("calloc");

	subj_db->s_entries = pacls;
	subj_db->i_entries = tiacls;
	subj_db->o_entries = facls;

	if ((s_tmp = subj_db->s_table =
	     (struct proc_acl **) calloc(pacls,
					 sizeof (struct proc_acl *))) == NULL)
		failure("calloc");

	for_each_subject(tmp, def_acl_tmp) {
		*s_tmp = tmp;
		iacls = 0;
		for_each_object(tmpi, tmp->ip_object)
		    iacls++;
		if (iacls) {
			i_table =
			    (struct ip_acl **) calloc(iacls,
						      sizeof (struct ip_acl *));
			if (!i_table)
				failure("calloc");
			i = 0;
			for_each_object(tmpi, tmp->ip_object) {
				i_tmp =
				    (struct ip_acl *) calloc(1,
							     sizeof (struct
								     ip_acl));
				memcpy(i_tmp, tmpi, sizeof (struct ip_acl));
				*(i_table + i) = i_tmp;
				i++;
			}
			(*s_tmp)->ips = i_table;
			(*s_tmp)->ip_num = iacls;
		} else {
			(*s_tmp)->ips = NULL;
			(*s_tmp)->ip_num = 0;
		}
		s_tmp++;
	}

	memcpy(&retarg->subj_db, subj_db, sizeof (struct user_acl_subject_db));
      set_pw:

	strncpy(retarg->pw, entry->passwd, GR_PW_LEN - 1);
	retarg->pw[GR_PW_LEN - 1] = '\0';

	retarg->mode = entry->mode;
	retarg->segv_inode = entry->segv_inode;
	retarg->segv_dev = entry->segv_dev;
	retarg->segv_uid = entry->segv_uid;

	memset(entry, 0, sizeof (struct gr_pw_entry));

	return retarg;
}
