//
//  kjots
//
//  Copyright (C) 1997 Christoph Neerfeld
//  email:  Christoph.Neerfeld@home.ivm.de or chris@kde.org
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
//  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
//

#include <qbuttongroup.h>
#include <qcursor.h>
#include <qobjectlist.h>
#include <qdatetime.h>
#include <qlabel.h>
#include <qpainter.h>
#include <qpaintdevicemetrics.h>
#include <qtimer.h>

#include <kaccel.h>
#include <kaction.h>
#include <kstdaccel.h>
#include <kstdaction.h>
#include <kapplication.h>
#include <kfiledialog.h>
#include <kdebug.h>
#include <kpopupmenu.h>
#include <kkeydialog.h>
#include <klineeditdlg.h>
#include <klocale.h>
#include <kmenubar.h>
#include <kmessagebox.h>
#include <kopenwith.h>
#include <kpopupmenu.h>
#include <krun.h>
#include <kstandarddirs.h>
#include <kurl.h>
#include <keditcl.h>
#include <klocale.h>
#include <kprinter.h>
#include <kglobal.h>

#include "KJotsMain.h"
#include "SubjList.h"
#include "optiondialog.h"


#define KJOTSVERSION "0.4"

const unsigned int HOT_LIST_SIZE = 8;
const unsigned int BUTTON_WIDTH = 56;


//----------------------------------------------------------------------
// MYMULTIEDIT
//----------------------------------------------------------------------
MyMultiEdit::MyMultiEdit (QWidget* parent, const char* name)
  : KTextEdit(parent, name),
   mSearchDialog(0),
   mReplaceDialog(0)
{
  setWordWrap(QTextEdit::WidgetWidth);
  web_menu = new KPopupMenu(this);;
  web_menu->insertItem(i18n("Open URL"), this, SLOT(openUrl()) );
}

MyMultiEdit::~MyMultiEdit ()
{
  delete mReplaceDialog;
}

void MyMultiEdit::search()
{
    if (!mSearchDialog)
    {
        mSearchDialog = new KEdFind(this);
        connect(mSearchDialog, SIGNAL(search()), this, SLOT(slotSearch()));
    }
    mSearchDialog->show();
}

void MyMultiEdit::repeatSearch()
{
    if (mSearchFor == QString::null)
    {
        return;
    }
    
    int para = 0;
    int index = 0;
    getCursorPosition(&para, &index);
    removeSelection();

    if (find(mSearchFor, mSearchDialog->case_sensitive(), false, !mSearchDialog->get_direction(), &para, &index))
    {
        setSelection(para, index, para, index + mSearchFor.length());
    }
    else
    {
        if (KMessageBox::questionYesNo(this, i18n("End of page reached.\n\nContinue from beginning of page?")) == KMessageBox::Yes)
        {
            setCursorPosition(0, 0);
            repeatSearch();
        }
    }
 }

void MyMultiEdit::slotSearch()
{
    int para = 0;
    int index = 0;
    getCursorPosition(&para, &index);
    mSearchFor = mSearchDialog->getText();
    removeSelection();

    if (find(mSearchFor, mSearchDialog->case_sensitive(), false, !mSearchDialog->get_direction(), &para, &index))
    {
         setSelection(para, index, para, index + mSearchFor.length());
    }
    else
    {
        if (KMessageBox::questionYesNo(this, i18n("End of page reached.\n\nContinue from beginning of page?")) == KMessageBox::Yes)
        {
            setCursorPosition(0, 0);
            slotSearch();
        }
        else
        {
            mSearchDialog->hide();
        }
    }
}

void MyMultiEdit::mousePressEvent( QMouseEvent *e )
{
  if(e->button() == RightButton &&
     hasSelectedText())
  {
      KURL url(selectedText());

      if(url.isValid())
      {
         web_menu->popup(QCursor::pos());
     }
      
      return;
  }

  KTextEdit::mousePressEvent(e);
}

void MyMultiEdit::openUrl()
{
  if (hasSelectedText())
  {
    KURL url(selectedText());
    if(url.isValid())
    {
      KFileOpenWithHandler fowh;  
      new KRun(url);
    }
  }
}

void MyMultiEdit::print(QString title)
{
    KPrinter printer;
    printer.setDocName(title);
    printer.setFullPage(false);
    printer.setCreator("KJots");

    if (printer.setup(this))
    {
        QFont printFont = font();
        QPainter painter( &printer );
        QPaintDeviceMetrics metrics( &printer );
        int y = 0;
        int maxWidth = metrics.width();
        int maxHeight = metrics.height();
        QString currentParagraph;

        for (int paragraphCount = 0; paragraphCount < paragraphs(); ++paragraphCount )
        {
            currentParagraph = text(paragraphCount);
            QRect r = painter.boundingRect(0, y, maxWidth, maxHeight,
                                                                QPainter::ExpandTabs | QPainter::WordBreak, currentParagraph);

            if ((y + r.height()) > maxHeight)
            {
                printer.newPage();
                y = 0;
            }

            painter.drawText(0, y, maxWidth, maxHeight - y,
                                        QPainter::ExpandTabs | QPainter::WordBreak, currentParagraph);
            y += r.height();
        }
        painter.end();
    }
}

//----------------------------------------------------------------------
// KJOTSMAIN
//----------------------------------------------------------------------
KJotsMain::KJotsMain(const char* name)
  : KMainWindow( 0, name ),
    current(0),
    currentFolder(0),
    currentFolderID(0),
    folderOpen(false),
    unique_id(0),
    mOptionDialog(0)
{
  //
  // Main widget
  //
  QFrame* f_main = new QFrame( this, "Frame_0" );
  f_main->setFrameStyle( QFrame::NoFrame );
  setCentralWidget(f_main);

  //
  // Text area
  //
  QFrame* f_text = new QFrame( f_main, "Frame_1" );
  f_text->setFrameStyle( QFrame::NoFrame );

  //
  // Status bar area
  //
  QFrame* f_labels = new QFrame( f_main, "Frame_2" );
  f_labels->setFrameStyle( QFrame::NoFrame );

  //
  // Data widgets.
  //
  s_bar = new QScrollBar( f_main, "ScrollBar_1" );
  s_bar->setOrientation( QScrollBar::Horizontal );
  s_bar->setRange(0,0);
  s_bar->setValue(0);
  s_bar->setSteps(1,1);
  connect( s_bar, SIGNAL(valueChanged(int)), this, SLOT(barMoved(int)) );

  me_text = new MyMultiEdit( f_text, "me_text" );
  me_text->setFocusPolicy(QWidget::StrongFocus);
  me_text->setEnabled(false);
  
  QLabel* l_book = new QLabel( f_labels, "l_book" );
  l_book->setText(i18n("Book:"));
  
  l_folder = new QLabel( f_labels, "Label_4" );
  l_folder->setFrameStyle( QFrame::WinPanel | QFrame::Sunken );
  QFont font_label(l_folder->fontInfo().family());
  font_label.setBold(true);
  l_folder->setFont(font_label);

  QLabel* l_subject = new QLabel( f_labels, "l_subject" );
  l_subject->setText(i18n("Page title:"));
  le_subject = new QLineEdit( f_labels, "le_subject" );
  le_subject->setFocusPolicy(QWidget::ClickFocus);
  le_subject->setEnabled(false);

  l_folder->setMinimumWidth( fontMetrics().maxWidth()*8 );
  l_folder->setFixedHeight( le_subject->sizeHint().height() );

  // create the hotlist
  QPushButton tp(0, "Test Button");
  bg_top = new QButtonGroup(i18n("Hotlist"), f_main, "ButtonGroup_2");
  bg_top->setFrameStyle( QFrame::Box | QFrame::Sunken );
  bg_top->setExclusive(true);
  bg_top->setMinimumHeight( tp.sizeHint().height() + 6 );

  // set up the layout for the hotlist
  QHBoxLayout* tempLayout = new QHBoxLayout(bg_top, KDialog::spacingHint());
  bg_top_layout = new QHBoxLayout(bg_top, KDialog::spacingHint());
  bg_top_layout->setMargin(KDialog::marginHint());
  tempLayout->addLayout(bg_top_layout);
  tempLayout->addItem(new QSpacerItem(30, 20, QSizePolicy::Expanding, QSizePolicy::Minimum ));
  //
  // Put data widgets into layouts
  //
  QVBoxLayout *l1 = new QVBoxLayout(f_text, 0, 0);
  l1->addWidget(me_text);
  
  QHBoxLayout *l2 = new QHBoxLayout(f_labels);
  l2->addWidget(l_book);
  l2->addWidget(l_folder);
  l2->addItem(new QSpacerItem(30, 20, QSizePolicy::Fixed, QSizePolicy::Minimum ));
  l2->addWidget(l_subject);
  l2->addWidget(le_subject, 1 );

  QVBoxLayout *tl = new QVBoxLayout(f_main, 4);
  tl->addWidget(f_labels);
  tl->addWidget(f_text, 1);
  tl->addWidget(s_bar);
  tl->addWidget(bg_top);


  entrylist.setAutoDelete(true);
  entrylist.append(new TextEntry);
  button_list.setAutoDelete(true);

  subj_list = new SubjList( topLevelWidget(), 0, false );
  connect( this, SIGNAL(folderChanged(QPtrList<TextEntry> *)),
	   subj_list, SLOT(rebuildList( QPtrList<TextEntry> *)) );
  connect( this, SIGNAL(entryMoved(int)),
	   subj_list, SLOT( select(int)) );
  connect( subj_list, SIGNAL(entryMoved(int)),
	   this, SLOT( barMoved(int)) );
  connect( le_subject, SIGNAL(textChanged(const QString &)),
	   subj_list, SLOT(entryChanged(const QString &)) );



  KConfig *config = kapp->config();
  config->setGroup("kjots");

  // read hotlist

  hotlist = config->readListEntry( "Hotlist" );
  while( hotlist.count() > HOT_LIST_SIZE )
    hotlist.remove(hotlist.last());
  // read list of folders
  QDir dir(locateLocal("appdata", ""));
  QString absPath = dir.absPath() + '/';
  QStringList files = dir.entryList(QDir::Files |  QDir::Readable); // config->readListEntry( "Folders" );

  // create hotlist buttons and hotlist menu
  folders_menu = new QPopupMenu(this);
  QPushButton *temp_button;
  for ( QStringList::Iterator it = files.begin();
        it != files.end();
        ++it)
  {
    QFile folder(absPath + *it);
    if(!folder.open(IO_ReadWrite))
    {
      continue;
    }

    QTextStream st( (QIODevice *) &folder);
    QString buf = st.readLine();
    
    if(buf.left(9) != "\\NewEntry")
    {
      continue;
    }
    
    QString decodedFilename = QFile::decodeName((*it).latin1());
    folder_list.push_back(decodedFilename);
    folders_menu->insertItem(decodedFilename, unique_id);
    if(hotlist.contains(decodedFilename))
    {
      if (decodedFilename.length() > 15)
      {
        decodedFilename.truncate(12);
        decodedFilename += "...";
      }

      temp_button = new QPushButton(decodedFilename, bg_top);
      temp_button->setToggleButton(true);
      bg_top->insert(temp_button, unique_id);
      bg_top_layout->addWidget(temp_button);
      button_list.append(temp_button);
    }
 
    ++unique_id;
  }

  connect( folders_menu, SIGNAL(activated(int)), this, SLOT(openFolder(int)) );
  connect( bg_top, SIGNAL(clicked(int)), this, SLOT(openFolder(int)) );

  updateConfiguration();

  // creat keyboard shortcuts
  // CTRL+Key_J := insert date
  // CTRL+Key_J := previous page
  // CTRL+Key_K := next page
  // CTRL+Key_L := show subject list
  // CTRL+Key_A := add new page
  // CTRL+Key_M := move focus

  keys = new KAccel( this );
  keys->insertItem(i18n("Move Focus"),       "MoveFocus",       CTRL+Key_M);
  keys->connectItem( "MoveFocus", this, SLOT(moveFocus()) );

  // create menu
  KAction *action;
  QPopupMenu *file_menu = new QPopupMenu(this);
  
  // New Book
  action = KStdAction::action(KStdAction::New, this, SLOT(createFolder()), actionCollection());
  action->setText(i18n("&New Book..."));
  action->plug(file_menu);

  // New Page
  action = new KAction(i18n("New P&age"), "filenew", CTRL+SHIFT+Key_N, this,
                       SLOT(newEntry()), actionCollection(), "new_page");
  action->plug(file_menu);
  action->plug(toolBar(0));
  action->plugAccel(keys);
 
  // Save Book
  action = KStdAction::action(KStdAction::Save, this, SLOT(saveFolder()), actionCollection());
  action->setText(i18n("Save Current Book"));
  action->plug(file_menu);
  
  id_f_save_book_ascii = file_menu->insertItem(i18n("Save Book to ASCII File..."),
                                                                              this, SLOT(writeBook()));
  id_f_save_page_ascii = file_menu->insertItem(i18n("Save Page to ASCII File..."),
                                                                              this, SLOT(writePage()));

  file_menu->insertSeparator();

  action = new KAction(i18n("Delete Current Book"), "editdelete", 0, this,
                       SLOT(deleteFolder()), actionCollection(), "del_folder");
  action->plug(file_menu);
  action->plugAccel(keys);

  action = new KAction(i18n("&Delete Page"), "editdelete", CTRL+Key_D, this,
                       SLOT(deleteEntry()), actionCollection(), "del_page");
  action->plug(file_menu);
  action->plug(toolBar(0));
  action->plugAccel(keys);

  file_menu->insertSeparator();

  // Print
  action = KStdAction::action(KStdAction::Print, this, SLOT(slotPrintPage()), actionCollection());
  action->plug(file_menu);
  action->plug(toolBar(0));
  action->plugAccel(keys);

  file_menu->insertSeparator();

  // Quit
  action = KStdAction::action(KStdAction::Quit, this, SLOT(slotQuit()), actionCollection());
  action->plug(file_menu);

  QPopupMenu *edit_menu = new QPopupMenu(this);

  action = KStdAction::action(KStdAction::Cut, me_text, SLOT(cut()), actionCollection());
  action->plug(edit_menu);
  
  action  = KStdAction::action(KStdAction::Copy, me_text, SLOT(copy()), actionCollection());
  action->plug(edit_menu);

  action = new KAction(i18n("Copy &into Page Title"), "editcopy", CTRL+Key_T, this,
                       SLOT(copySelection()), actionCollection(), "copyIntoTitle");
  action->plug(edit_menu);
  action->plugAccel(keys);

  action  = KStdAction::action(KStdAction::Paste, me_text, SLOT(paste()), actionCollection());
  action->plug(edit_menu);
  
  edit_menu->insertSeparator();

  // added actionized items (pfeiffer)
  action = KStdAction::find( this, SLOT( slotSearch() ), actionCollection() );
  action->plug( edit_menu );
  action = KStdAction::findNext( this, SLOT( slotRepeatSearch() ), actionCollection() );
  action->plug( edit_menu );
  action = KStdAction::replace( this, SLOT( slotReplace() ), actionCollection() );
  action->plug( edit_menu );
  edit_menu->insertSeparator();
  action = new KAction(i18n("Rename Book..."), KShortcut(), this,
                       SLOT(slotEditBookTitle()), actionCollection(), "edit_folderName");
  action->plug(edit_menu);
  action->plugAccel( keys );
  action = new KAction(i18n("Insert Date"), "date", CTRL+Key_I, this,
                       SLOT(insertDate()), actionCollection(), "insert_date");
  action->plug(edit_menu);
  action->plug(toolBar(0));
  action->plugAccel( keys );

  QPopupMenu *go_menu = new QPopupMenu(this);

  action = KStdAction::prior(this, SLOT(prevEntry()), actionCollection());
  action->plug( go_menu );
  action->plug( toolBar(0) );
  action->plugAccel( keys );
  action = KStdAction::next(this, SLOT(nextEntry()), actionCollection());
  action->plug( go_menu );
  action->plug( toolBar(0) );
  action->plugAccel( keys );
  action = KStdAction::gotoPage(this, SLOT(toggleSubjList()), actionCollection());
  action->setAccel(CTRL + Key_L);
  action->plug( go_menu );
  action->plug( toolBar(0) );
  action->plugAccel( keys );

  QPopupMenu *options_menu = new QPopupMenu(this);
  action = KStdAction::keyBindings(this, SLOT(configureKeys()), actionCollection());
  action->plug(options_menu);
  /*
   * TODO: configure toolbar? need to use xmlgui
   * action = KStdAction::configureToolbars(this, SLOT(configureToolbars()), actionCollection());
   */
  action = KStdAction::preferences(this, SLOT(configure()), actionCollection());
  action->plug(options_menu);

  QPopupMenu *hotlist_menu = new QPopupMenu(this);
  id_fav_add = hotlist_menu->insertItem(i18n("Add Current Book to Hotlist"),
				                                this, SLOT(addToHotlist()) );
  id_fav_del = hotlist_menu->insertItem(i18n("Remove Current Book From Hotlist"),
				                                this, SLOT(removeFromHotlist()) );
  menuBar()->insertItem( i18n("&File"), file_menu );
  menuBar()->insertItem( i18n("&Edit"), edit_menu );
  menuBar()->insertItem( i18n("&Go"), go_menu );
  menuBar()->insertItem( i18n("&Books"), folders_menu );
  menuBar()->insertItem( i18n("Hot&list"), hotlist_menu );
  menuBar()->insertItem( i18n("&Settings"), options_menu );
  menuBar()->insertSeparator();
  menuBar()->insertItem( i18n("&Help"), helpMenu() );

  int nr = folder_list.findIndex(QFile::decodeName(config->readEntry("LastOpenFolder").latin1()));
  if( nr >= 0 )
  {
    openFolder(nr);
  }

  //
  // Set startup size.
  //
  int w = config->readNumEntry("Width");
  int h = config->readNumEntry("Height");
  w = QMAX( w, sizeHint().width() );
  h = QMAX( h, sizeHint().height() );
  resize(w, h);

  keys->readSettings();
  updateMenu();
  
  if (folder_list.count() == 0)
  {
    QTimer::singleShot(0, this, SLOT(createFolder())); 
  }
}



KJotsMain::~KJotsMain()
{
  delete mOptionDialog;
  saveProperties();
}


void KJotsMain::saveProperties( void )
{
  KConfig* config = kapp->config();
  config->setGroup("kjots");
  button_list.clear();
  if( folderOpen )
  {
    QFileInfo fi(current_folder_name);
    config->writeEntry("LastOpenFolder", fi.fileName());
  }
  saveFolder();
  config->writeEntry("Width", width());
  config->writeEntry("Height", height());
  toolBar(0)->saveSettings(config, "MainToolbar");
  config->sync();
}

int KJotsMain::readFile( QString name )
{
  int pos;
  QString buf, subj;
  TextEntry *entry;

  entrylist.clear();
  QFile folder(name);
  if( !folder.open(IO_ReadWrite) )
    return -1;

  QTextStream st( (QIODevice *) &folder);
  buf = st.readLine();
  if( buf.left(9) != "\\NewEntry" )
    return -1;

  entry = new TextEntry;
  entrylist.append( entry );
  subj = buf.mid(10, buf.length() );
  entry->subject = subj.isNull() ? (QString) "" : (QString) subj;
  while( !st.eof() )
    {
      buf = st.readLine();
      if( buf.left(9) == "\\NewEntry" )
	{
	  entry = new TextEntry;
	  entrylist.append(entry);
	  subj = buf.mid(10, buf.length() );
	  entry->subject = subj.isNull() ? (QString) "" : (QString) subj;
	  buf = st.readLine();
	}
      pos = 0;
      while( (pos = buf.find( '\\', pos )) != -1 )
	if( buf[++pos] == '\\' )
	  buf.remove( pos, 1 );
      entry->text.append( buf );
      entry->text.append("\n");
    }
  folder.close();
  while( entry->text.right(1) == "\n" )
    entry->text.truncate(entry->text.length() - 1);
  entry->text.append("\n");

  updateMenu();
  return 0;
}

int KJotsMain::writeFile( QString name ) 
{
  int pos;
  TextEntry *entry = entrylist.first();
  QString buf;
  QFile folder(name);
  if( !folder.open(IO_WriteOnly | IO_Truncate) )
    return -1;
  QTextStream st( (QIODevice *) &folder);

  if (entry == 0)
  {
    st << "\\NewEntry \n\n";
  }
  else  
  {
    for(; entry != NULL; entry = entrylist.next())
    {
      st << "\\NewEntry ";
      st << entry->subject;
      st << "\n";
      buf = entry->text;

      pos = 0;
      while( (pos = buf.find( '\\', pos )) != -1 )
     	{
    	  buf.insert( ++pos, '\\' );
     	  pos++;
     	}
      st << buf;
      if( buf.right(1) != "\n" )
       	st << '\n';
    }
  }

  folder.close();
  return 0;
}

void KJotsMain::openFolder(int id) 
{
  if (folders_menu->indexOf(id) < 0)
  {
    return;
  }
 
  QPushButton *but;
  for( but = button_list.first(); but != NULL; but = button_list.next() )
    but->setOn(false);
  but = static_cast<QPushButton *>(bg_top->find(id));
  if( but )
    but->setOn(true);

  currentFolder = folders_menu->indexOf(id);
  currentFolderID = id;
  QDir dir = QDir( locateLocal("appdata", "") );
  QString file_name = dir.absPath();
  file_name += '/';
  file_name += QFile::encodeName(*folder_list.at(currentFolder));
  if( current_folder_name == file_name )
    return;

  if( folderOpen )
    saveFolder();
  current_folder_name = file_name;

  if( readFile(current_folder_name) < 0)
    {
      folderOpen = false;
      kdDebug() << "Unable to open folder" << endl;
      return;
    }

  current = 0;
  me_text->removeSelection();
  me_text->setText(entrylist.first()->text);
  emit folderChanged(&entrylist);
  emit entryMoved(current);
  le_subject->setText(entrylist.first()->subject);
  folderOpen = true;
  l_folder->setText( *folder_list.at(folders_menu->indexOf(id)) );
  me_text->setEnabled(true);
  le_subject->setEnabled(true);
  me_text->setFocus();
  s_bar->setValue(0);
  s_bar->setRange(0,entrylist.count()-1);

  updateMenu();
}


void KJotsMain::createFolder()
{
  KLineEditDlg dlg(i18n("Book name:"), QString::null, this);
  dlg.setCaption(i18n("New Book"));
  if (!dlg.exec())
    return;
  QString name = dlg.text();
  
  if( folder_list.contains(name) )
  {
    QString msg = i18n("A book with this name already exists.");
    KMessageBox::sorry( topLevelWidget(), msg );
    return;
  }
  saveFolder();
  entrylist.clear();
  folderOpen = true;
  me_text->setEnabled(true);
  le_subject->setEnabled(true);
  me_text->setFocus();
  me_text->clear();
  me_text->removeSelection();
  TextEntry *new_entry = new TextEntry;
  entrylist.append(new_entry);
  new_entry->subject = "";
  current = 0;
  s_bar->setValue(0);
  s_bar->setRange(0,0);
  emit folderChanged(&entrylist);
  emit entryMoved(current);
  le_subject->setText(entrylist.first()->subject);
  folder_list.append(name);

  ++unique_id;
  currentFolderID = unique_id;
  folders_menu->insertItem(name, currentFolder);
  currentFolder = folders_menu->indexOf(currentFolderID);
  QDir dir = QDir( locateLocal("appdata", "") );
  current_folder_name = dir.absPath();
  current_folder_name += '/';
  current_folder_name += QFile::encodeName(name);
  l_folder->setText(name);
  QPushButton *but;
  for( but = button_list.first(); but != 0; but = button_list.next() )
    but->setOn(false);

  updateMenu();
}

void KJotsMain::deleteFolder()
{
  if( !folderOpen )
    return;

  QString msg = i18n("Are you sure you want to delete the current book?");
  int result = KMessageBox::warningYesNo( topLevelWidget(), msg, i18n("Delete Book") );
  if( result != KMessageBox::Yes )
  {
    return;
  }

  QFileInfo fi(current_folder_name);
  QDir dir = fi.dir(true);
  QString name = fi.fileName();
  int index = folder_list.findIndex(name);
  if( index < 0 )
    return;
  dir.remove(current_folder_name);
  folder_list.remove(folder_list.at(index));
  int id = folders_menu->idAt(index);
  folders_menu->removeItemAt(index);
  if( hotlist.contains(name) )
    {
      hotlist.remove(name);
      QButton *but = bg_top->find(id);
      bg_top->remove(but);
      button_list.remove( (QPushButton *) but );
      resize(size());
    }
  KConfig *config = KApplication::kApplication()->config();
  config->setGroup("kjots");
  config->writeEntry( "Folders", folder_list );
  config->writeEntry( "Hotlist", hotlist );
  config->sync();
  entrylist.clear();
  current_folder_name = "";
  currentFolder = 0;
  currentFolderID = 0;
  folderOpen = false;
  me_text->setEnabled(false);
  me_text->clear();
  me_text->removeSelection();
  le_subject->setEnabled(false);
  le_subject->clear();
  emit folderChanged(&entrylist);
  s_bar->setValue(0);
  s_bar->setRange(0,0);
  l_folder->clear();
  subj_list->repaint(true);

  openFolder(folders_menu->idAt(0));
  updateMenu();
}

void KJotsMain::saveFolder()
{
  if( !folderOpen )
    return;
  entrylist.at(current)->text = me_text->text();
  entrylist.at(current)->subject = le_subject->text();
  writeFile(current_folder_name);
}

void KJotsMain::nextEntry()
{
  if( !folderOpen )
    return;
  if( current+1 >= (int) entrylist.count() )
    return;
  entrylist.at(current)->text = me_text->text();
  entrylist.at(current)->subject = le_subject->text();
  me_text->setText( entrylist.at(++current)->text );
  me_text->removeSelection();
  me_text->repaint();
  emit entryMoved(current);
  le_subject->setText( entrylist.at(current)->subject );
  s_bar->setValue(current);

  updateMenu();
}

void KJotsMain::prevEntry()
{
  if( !folderOpen )
    return;
  if( current-1 < 0 )
    return;
  entrylist.at(current)->text = me_text->text();
  entrylist.at(current)->subject = le_subject->text();
  me_text->setText( entrylist.at(--current)->text );
  me_text->removeSelection();
  me_text->repaint();
  emit entryMoved(current);
  le_subject->setText( entrylist.at(current)->subject );
  s_bar->setValue(current);

  updateMenu();
}

void KJotsMain::newEntry()
{
  if( !folderOpen )
    return;
  entrylist.at(current)->text = me_text->text();
  entrylist.at(current)->subject = le_subject->text();
  me_text->clear();
  le_subject->clear();
  TextEntry *new_entry = new TextEntry;
  entrylist.append(new_entry);
  new_entry->subject = "";
  current = entrylist.count()-1;
  s_bar->setRange(0,current);
  s_bar->setValue(current);
  emit folderChanged(&entrylist);
  emit entryMoved(current);

  updateMenu();
}

void KJotsMain::deleteEntry()
{
  if( !folderOpen )
    return;
  if( entrylist.count() == 0 )
    return;
  if (KMessageBox::warningYesNo(this, 
                                i18n("Are you sure you want to delete the current page?"),
                                i18n("Delete Page"),
                                KStdGuiItem::yes(),
                                KStdGuiItem::no(),
                                "deletePageWarning") != KMessageBox::Yes)
  {
    return;
  }

  if( entrylist.count() == 1 )
    {
      entrylist.at(0)->text = "";
      entrylist.at(0)->subject = "";
      s_bar->setValue(0);
      me_text->clear();
      le_subject->clear();
      return;
    }
  entrylist.remove(current);
  if( current >= (int) entrylist.count() - 1 )
    {
      if( current )
      current--;
      s_bar->setValue(current);
      s_bar->setRange(0, entrylist.count()-1 );
    }
  me_text->setText( entrylist.at(current)->text );
  emit entryMoved(current);
  le_subject->setText( entrylist.at(current)->subject );
  s_bar->setRange(0, entrylist.count()-1 );
  emit folderChanged(&entrylist);

  updateMenu();
}

void KJotsMain::barMoved( int new_value )
{
  if( !folderOpen )
    return;
  if( current == new_value )
    return;
  entrylist.at(current)->text = me_text->text();
  entrylist.at(current)->subject = le_subject->text();
  current = new_value;
  me_text->setText( entrylist.at(current)->text );
  me_text->removeSelection();
  me_text->repaint();
  emit entryMoved(current);
  le_subject->setText( entrylist.at(current)->subject );
  s_bar->setValue(new_value);

  updateMenu();
}

void KJotsMain::addToHotlist()
{
  if( hotlist.count() == HOT_LIST_SIZE )
    return;
  QFileInfo fi(current_folder_name);
  QString name = fi.fileName();
  if( hotlist.contains(name) )
    return;
  hotlist.append(name);
  int index = folder_list.findIndex(name);
  if( index < 0 )
    return;
  int id = folders_menu->idAt(index);
  QString buttonLabel = name;
  if (buttonLabel.length() > 15)
  {
    buttonLabel.truncate(12);
    buttonLabel += "...";
  }
  QPushButton *but = new QPushButton(buttonLabel, bg_top);
  button_list.append(but);
  bg_top->insert(but, id);
  bg_top_layout->addWidget(but);
  KConfig *config = KApplication::kApplication()->config();
  config->setGroup("kjots");
  config->writeEntry( "Hotlist", hotlist );
  config->sync();
  but->setToggleButton(true);
  but->show();
  but->setOn(true);
}

void KJotsMain::removeFromHotlist()
{
  QFileInfo fi(current_folder_name);
  QString name = fi.fileName();
  if( !hotlist.contains(name) )
    return;
  hotlist.remove(name);
  int index = folder_list.findIndex(name);
  if( index < 0 )
    return;
  int id = folders_menu->idAt(index);
  QButton *but = bg_top->find(id);
  bg_top->remove(but);
  button_list.remove( (QPushButton *) but );
  KConfig *config = KApplication::kApplication()->config();
  config->setGroup("kjots");
  config->writeEntry( "Hotlist", hotlist );
  config->sync();
}

void KJotsMain::toggleSubjList()
{
  if( subj_list->isVisible() )
  {
    subj_list->hide();
  }
  else
  {
    subj_list->setInitialSize( QSize(width()/2,height()) );
   //subj_list->resize( QSize(width() / 2, height()) );
    subj_list->show();
  }
}

void KJotsMain::configure()
{
  if( mOptionDialog == 0 )
  {
    mOptionDialog = new ConfigureDialog( topLevelWidget(), 0, false );
    if( mOptionDialog == 0 ) { return; }
    connect( mOptionDialog, SIGNAL(hidden()),this,SLOT(configureHide()));
    connect( mOptionDialog, SIGNAL(valueChanged()),
	     this, SLOT(updateConfiguration()) );
  }

  mOptionDialog->show();
}

void KJotsMain::configureHide()
{
  QTimer::singleShot( 0, this, SLOT(configureDestroy()) );
}

void KJotsMain::configureDestroy()
{
  if( mOptionDialog != 0 && mOptionDialog->isVisible() == false )
  {
    delete mOptionDialog; mOptionDialog = 0;
  }
}

void KJotsMain::slotSearch()
{
    me_text->search();
}

void KJotsMain::slotRepeatSearch()
{
    me_text->repeatSearch();
}

void KJotsMain::slotReplace()
{
    me_text->replace();
}

void KJotsMain::updateConfiguration()
{
  KConfig* config = kapp->config();

  config->setGroup("kjots");
  QFont font( config->readEntry("EFontFamily"),
                     config->readNumEntry("EFontSize"),
                    config->readNumEntry("EFontWeight"),
                    config->readNumEntry("EFontItalic"));
  me_text->setFont(font);
  toolBar(0)->applySettings(config, "MainToolbar");
}

void KJotsMain::writeBook()
{
  saveFolder();
  QString name;
  while( name.isNull() )
  {
    KURL url = KFileDialog::getSaveURL();

    if ( url.isEmpty() )
      return;

    if ( !url.isLocalFile() )
    {
      KMessageBox::sorry( 0L, i18n( "Only local files are currently supported." ) );
      return;
    }

    name = url.path();

    QFileInfo f_info(name);
    if( f_info.exists() )
    {
      QString msg = i18n("File already exists.\n"
	                 "Do you want to overwrite it?");
      int result = KMessageBox::warningYesNo( topLevelWidget(), msg );
      if( result != KMessageBox::Yes )
      {
	name = "";
      }
    }
  }
  QFile ascii_file(name);
  if( !ascii_file.open(IO_WriteOnly | IO_Truncate) )
    return;
  QTextStream st( (QIODevice *) &ascii_file);
  TextEntry *entry;
  for( entry = entrylist.first(); entry != NULL; entry = entrylist.next() )
    {
      writeEntry( st, entry );
    }
  ascii_file.close();
}

void KJotsMain::writePage()
{
  saveFolder();
  QString name;
  while( name.isNull() )
  {
    KURL url = KFileDialog::getSaveURL();

    if( url.isEmpty() )
      return;

    if( !url.isLocalFile() )
    {
      KMessageBox::sorry( 0L, i18n( "Only local files are currently supported." ) );
      return;
    }

    name = url.path();

    QFileInfo f_info(name);
    if( f_info.exists() )
    {
      QString msg = i18n("File already exists.\n"
	                 "Do you want to overwrite it?");
      int result = KMessageBox::warningYesNo( topLevelWidget(), msg );
      if( result != KMessageBox::Yes )
      {
	name = "";
      }
    }
  }
  QFile ascii_file(name);
  if( !ascii_file.open(IO_WriteOnly | IO_Truncate) )
    return;
  QTextStream st( (QIODevice *) &ascii_file);
  writeEntry( st, entrylist.at(current) );
  ascii_file.close();
}

void KJotsMain::writeEntry( QTextStream &st, TextEntry *entry )
{
  QString line, buf;
  line.fill('#', entry->subject.length()+2 );
  st << line << '\n';
  st << "# ";
  st << entry->subject << '\n';
  st << line << '\n';
  buf = entry->text;
  st << buf;
  if( buf.right(1) != "\n" )
    st << '\n';
  st << '\n';
}

void KJotsMain::moveFocus()
{
  if( me_text->hasFocus() )
    le_subject->setFocus();
  else
    me_text->setFocus();
}

void KJotsMain::configureKeys()
{
  if (KKeyDialog::configureKeys(keys, true, topLevelWidget()))
  {
    keys->writeSettings();
    kapp->config()->sync();
    keys->readSettings();
  }
}

void KJotsMain::copySelection()
{
  if(me_text->hasSelectedText())
  {
    le_subject->setText(me_text->selectedText());
  }
}

void KJotsMain::slotPrintPage()
{
    QString docName = current_folder_name;
    if (le_subject->text() != QString::null)
    {
        docName += ": " + le_subject->text();
    }
    me_text->print(docName);
 }

void KJotsMain::slotQuit() 
{
    // just save on exit...
    saveFolder();
    close();
}

void KJotsMain::slotEditBookTitle()
{
  if (!folderOpen)
  {
    return;
  }
  
  bool accepted(false);
  QString newName = KLineEditDlg::getText(i18n("Rename Book"), i18n("Title of this book:"),
                                          l_folder->text(), &accepted, this);
  if (accepted)
  {
    QFileInfo folderInfo(current_folder_name);
    QString temp = folderInfo.dirPath(true);
    QDir dir(temp);
    temp += "/" + QFile::encodeName(newName);
    
    if (QFile::exists(temp))
    {
        KMessageBox::sorry(this, i18n("A book by that name already exists."), i18n("Rename Book")) ;
        return;
    }
    
    dir.rename(current_folder_name, temp);
    current_folder_name = temp;
    
    if (hotlist.contains(l_folder->text()))
    {
        (*hotlist.find(l_folder->text())) = newName;
         KConfig *config = KApplication::kApplication()->config();
         config->setGroup("kjots");
         config->writeEntry( "Hotlist", hotlist );
         config->sync();
    }
    
    l_folder->setText(newName);
    (*folder_list.at(currentFolder)) = newName;
    QPushButton* but = dynamic_cast<QPushButton*>(bg_top->find(currentFolderID));
    if (but)
    {
        if (newName.length() > 15)
        {
            newName.truncate(12);
            newName += "...";
        }
        but->setText(newName);
    }
  }
}

void KJotsMain::insertDate()
{
  me_text->insert(KGlobal::locale()->formatDateTime(QDateTime::currentDateTime(), true) + " ");
}

void KJotsMain::updateMenu()
{
  actionCollection()->action("del_folder")->setEnabled(folderOpen);
  actionCollection()->action("del_page")->setEnabled(folderOpen);
  actionCollection()->action("new_page")->setEnabled(folderOpen);
  actionCollection()->action("edit_folderName")->setEnabled(folderOpen);

  actionCollection()->action(KStdAction::stdName(KStdAction::Save))->setEnabled(folderOpen);
  menuBar()->setItemEnabled(id_f_save_book_ascii, folderOpen);
  menuBar()->setItemEnabled(id_f_save_page_ascii, folderOpen);

  actionCollection()->action(KStdAction::stdName(KStdAction::Cut))->setEnabled(folderOpen);
  actionCollection()->action(KStdAction::stdName(KStdAction::Copy))->setEnabled(folderOpen);
  actionCollection()->action(KStdAction::stdName(KStdAction::Paste))->setEnabled(folderOpen);

  menuBar()->setItemEnabled(id_fav_add, folderOpen);
  menuBar()->setItemEnabled(id_fav_del, folderOpen);
}
#include "KJotsMain.moc"
