/*************************************************************************
  (c) Copyright.  Digital Equipment Corporation, 1995.  All Rights
  Reserved.

  Permission is hereby granted to use, copy, modify, or enhance this 
  software freely, as long as the foregoing copyright of Digital Equipment
  Corporation and this notice are retained on the software.  This 
  software may not be distributed or sublicensed for a fee.  Digital      
  makes this software available "AS IS" and without warranties of any
  kind.  
 *************************************************************************/
/*
 * Marko Kiiskila carnil@cs.tut.fi 
 * 
 * Tampere University of Technology - Telecommunications Laboratory
 *
 * Permission to use, copy, modify and distribute this
 * software and its documentation is hereby granted,
 * provided that both the copyright notice and this
 * permission notice appear in all copies of the software,
 * derivative works or modified versions, and any portions
 * thereof, that both notices appear in supporting
 * documentation, and that the use of this software is
 * acknowledged in any publications resulting from using
 * the software.
 * 
 * TUT ALLOWS FREE USE OF THIS SOFTWARE IN ITS "AS IS"
 * CONDITION AND DISCLAIMS ANY LIABILITY OF ANY KIND FOR
 * ANY DAMAGES WHATSOEVER RESULTING FROM THE USE OF THIS
 * SOFTWARE.
 * 
 */

/*
* Module Name:
*   lec_arp.h
*   
* Overview:
*   This file specifies the interface to the LE-ARP cache module (implemented
*   in lec_arp.c).
*
* Authors:
*   TLR - Theodore L. Ross
*
* Modification History:
*   Date       Name  Description 
*    2-Feb-95  TLR   Created.
*
* Description:
*   The LE-ARP cache module maintains the relationships between MAC addresses
*   and VCCs.  It shares the ATM address list and VC list owned by the
*   Connection Manager.  This module performs the following tasks:
*
*       1.  Resolves MAC addresses to VCCs for every transmitted data frame.
*
*       2.  Performs the two-tiered ARP refresh algorithm specified in the
*           ATM Forum Specification.
*
*       3.  Keeps track of data frames flooded and limits the rate based on
*           ATM Forum parameters C10 and C11.
*
*       4.  Drives the Flush protocol when data path switching takes place.
*
*   Note that this module does not deal with frame formats or data transmission
*   or reception.  It uses the services of the LEC_CTRL module for these.
*
* Call Sequence Requirements:
*
*   The following diagram depicts the call sequence constraints:  Functions
*   with lesser indentation must be called before function with greater
*   indentation.
*
*   - la_create
*     - la_register
*       - la_config
*         - la_arp_resolve
*         - la_addr_delete
*         - la_arp_update
*         - la_flush_complete
*         - la_topology_change_set
*         - la_vc_added
*         - la_vc_deleted
*       - la_deregister
*     - la_destroy
*
* Example Usage Scenario:
*
*   la_create...............This call is made as a result of the creation of
*                           the LAN Emulation Client super-component.
*                           la_create is called by lc_create.
*
*   la_register.............When a protocol registers with the LEC and requests
*                           that an ELAN be joined, this call results.  A new
*                           LE-ARP cache instance is allocated and initialized
*                           to contain no entries.
*
*   la_config...............This call is used to configure the LE-ARP cache
*                           with parameters needed to operate.  This call is
*                           made once the configuration phase of the join is
*                           complete and the parameters are known.
*
*   la_arp_resolve..........This is called each time a transmit data packet is
*                           sent by the protocol.  In this case, the
*                           destination MAC address is not in the LE-ARP cache
*                           so a new entry is created and the cache begins to
*                           attempt to resolve the address to a VC.
*
*   (arp_xmt_callback)......Still in the context of the above call, the cache
*                           uses this callback to request the transmission of
*                           an LE-ARP request frame.
*
*   la_arp_resolve..........This call is for the second data packet to the same
*                           MAC address destination.  This time, the cache
*                           finds an entry with the MAC address but since the
*                           resolution to a VC is not complete, the packet must
*                           be either flooded or discarded depending on how
*                           many packets have been flooded recently.
*
*   la_arp_update...........The network has responded to the LE-ARP request
*                           with an LE-ARP response.  The cache now knows the
*                           mapping from the MAC address to the ATM address.
*
*   (svc_setup_callback)....Still in the context of the above call, the cache
*                           finds that there is no SVC associated with the
*                           ATM address in the new mapping so it uses this
*                           callback to request that a VC be set up.
*
*   la_vc_added.............When the new SVC is finally set up, the LEC_CTRL
*                           module uses this call to notify the LE-ARP cache.
*                           The MAC address entry is now associated with a
*                           data-direct VCC.
*
*   (flush_xmt_callback)....Still in the context of the above call, the cache
*                           uses this callback to request the transmission of
*                           a flush request on the broadcast channel.
*
*   la_arp_resolve..........The next packet to the same MAC address is being
*                           resolved.  Since the fluch protocol is in process,
*                           this packet must be discarded.  The return value
*                           of this call indicates MUST_DISCARD.
*
*   la_flush_complete.......The flush protocol is complete because a flush
*                           response has been received for the MAC address.
*                           The cache updates the entry to allow data-direct
*                           transmission.
*
*   la_arp_resolve..........The next packet is finally resolved to an open
*                           data-direct VCC.
*
*   la_deregister...........This call is made when the ELAN is unjoined by the
*                           protocol.
*
*   la_destroy..............This call destroys the instance of the LE-ARP
*                           cache.
*/

/*************************************************************************
 * Types that are specific to this interface.                            *
 *************************************************************************/

/* XMT_ACTION
 *  This type enumerates the actions that may be indicated by the arp_resolve
 *  function.
 *
 *  XMT_DISCARD.........The packet must be discarded.  This occurs when there
 *                      is no data-direct VCC and the max unknown frame count
 *                      has been exceeded or the flush protocol is in process.
 *
 *  XMT_FLOOD...........The packet may be flooded on the broadcast channel.
 *
 *  XMT_FORWARD.........The packet may be forwarded on the data-direct VCC.
 */
typedef enum
   {
   XMT_DISCARD,
   XMT_FLOOD,
   XMT_FORWARD
   } XMT_ACTION;


/*************************************************************************
 * Definition of Interface that is used by the LEC_CTRL module
 *************************************************************************/


/*++
* ====================
* = ARP_XMT_CALLBACK =
* ====================
*
* Overview:
*   Callback to the LEC_CTRL client.  Requests that an LE-ARP be sent on 
*   the network.
*
* Arguments:
*   lc_elan_handle  - (IN)  Handle of LEC CTRL instance to be used.
*   esi             - (IN)  48-bit address for LE-ARP.
*   sizeoftlvs      - (IN)  LANE2: total length of TLVs to send
*              
* Returns:
*   None
*
* Preconditions:
*   None
*
* Postconditions:
*   None
*
--*/
typedef void (*ARP_XMT_CALLBACK) (HANDLE    lc_elan_handle,
                                  ESI       esi,
                                  int sizeoftlvs);

/*++
* ======================
* = FLUSH_XMT_CALLBACK =
* ======================
*
* Overview:
*   Callback to the LEC_CTRL client.  Requests that a flush request be sent
*   on the network for the specified MAC address.
*
* Arguments:
*   lc_elan_handle  - (IN)  Handle of LEC_CTRL instance to be used.
*   addr_atm        - (IN)  ATM address for the flush.
*   esi_handle      - (IN)  Handle of the ESI record for the ESI being flushed.
*                           This value is passed in the transaction_id field
*                           of the flush request packet.
*              
* Returns:
*   None
*
* Preconditions:
*   None
*
* Postconditions:
*   None
*
--*/
typedef void (*FLUSH_XMT_CALLBACK) (HANDLE    lc_elan_handle,
                                    unsigned char*  addr_atm,
                                    HANDLE    esi_handle);

/*++
* ======================
* = SVC_SETUP_CALLBACK =
* ======================
*
* Overview:
*   Callback to the LEC_CTRL client.  Requests that a data-direct SVC be set up
*   to the indicated ATM address.
*
* Arguments:
*   lc_elan_handle  - (IN)  Handle of LEC_CTRL instance to be used.
*   addr_atm        - (IN)  Callee ATM address for the setup.
*              
* Returns:
*   None
*
* Preconditions:
*   None
*
* Postconditions:
*   None
*
--*/
typedef void (*SVC_SETUP_CALLBACK) (HANDLE   lc_elan_handle,
                                    ADDR_ATM addr_atm);



/*++
* =============
* = la_create =
* =============
*
* Overview:
*   Creates an instance of the LEC ARP-Cache.  This create is done once for
*   each physical interface that is supported by the LEC.  Subsequent calls
*   to la_register are required for each ELAN instance.
*
* Arguments:
*   os_handle           - (IN)  OS Handle to be used for memory and timer
*                               alloctaion.
*
*   lc_handle           - (IN)  Handle of the LEC_CTRL instance that creates
*                               this ARP instance.
*
*   cm_handle           - (IN)  Handle of the associated CM instance.
*
*   arp_xmt_callback    - (IN)  Callback routine for ARP transmits.
*
*   flush_xmt_callback  - (IN)  Callback routine for Flush transmits.
*
*   svc_setup_callback  - (IN)  Callback for data-direct SVC setups.
*
*   p_la_handle         - (OUT) Handle of the created ARP instance.
*
* Returns:
*   STATUS_K_SUCCESS    - Successful Completion, see postconditions.
*   STATUS_K_RESOURCES  - Insufficient resources available to complete the
*                         operation.
*
* Preconditions:
*   None
*
* Postconditions:
*   The p_la_handle is valid.
--*/
STATUS la_create (HANDLE              lc_handle,
                  ARP_XMT_CALLBACK    arp_xmt_callback,
                  FLUSH_XMT_CALLBACK  flush_xmt_callback,
                  SVC_SETUP_CALLBACK  svc_setup_callback,
                  void                (*associate_req_callback)(HANDLE lc_elan_handle,
                                                                int sizeoftlvs),  /* LANE2 */
                  HANDLE              *p_la_handle);

/*++
* ==============
* = la_destroy =
* ==============
*
* Overview:
*   Destroys an instance of the ARP module.
*
* Arguments:
*   la_handle   - (IN) Handle returned by la_create.
*
* Returns:
*   None
*
* Preconditions:
*   The la_handle is valid.
*
* Postconditions:
*   The la_handle is invalid.
--*/
void la_destroy (HANDLE  la_handle);

/*++
* ===============
* = la_register =
* ===============
*
* Overview:
*   Registers an ELAN instance of the ARP module.  This routine is called
*   by the LEC_CTRL module for each ELAN instance of the LEC.  It creates an
*   instance of the LE-ARP cache to be used to support a single ELAN
*   membership.
*
* Arguments:
*   la_handle           - (IN)  Handle returned by la_create.
*   lc_elan_handle      - (IN)  Handle of associated lc_elan instance.
*   lport_handle        - (IN)  Logical Port handle for calls to proxy fast
*                               path.
*   p_text              - (IN)  Textual description for event logging.
*   p_la_elan_handle    - (OUT) Handle that represents this LE-ARP cache
*                               instance.
*
* Returns:
*   STATUS_K_SUCCESS    - Successful Completion, see postconditions.
*   STATUS_K_RESOURCES  - Insufficient resources available to complete the
*                         operation.
*
* Preconditions:
*   None
*
* Postconditions:
*   The p_la_elan_handle is valid and ready for configuration.
--*/
STATUS la_register (HANDLE    la_handle,
                    HANDLE    lc_elan_handle,
                    HANDLE    lport_handle,
                    char     *p_text,
                    HANDLE   *p_la_elan_handle);

/*++
* =================
* = la_deregister =
* =================
*
* Overview:
*   Deletes a single LE-ARP cache entry.  This is called when membership in
*   an ELAN is to be cancelled by request of higher layers.
*
* Arguments:
*   la_elan_handle  - (IN) Handle returned by la_register.
*
* Returns:
*   None
*
* Preconditions:
*   The la_elan_handle is valid.
*
* Postconditions:
*   The la_elan_handle is not valid.
--*/
void la_deregister (HANDLE la_elan_handle);

/*++
* =============
* = la_config =
* =============
*
* Overview:
*   Configure the LE-ARP cache with the required operating parameters.
*   This may be used repeatedly to change the operating parameters.  An example
*   of when this might happen is if connection to a LES is lost and the rejoin
*   operation results in a connection to a LES with different parameters.
*
* Arguments: (The C* specifications are from the ATM Forum LAN Emulation
*             Specification V1.0).
*
*   la_elan_handle          - (IN) Handle returned by la_register.
*   lan_type                - (IN) C2  - Lan Type for this ELAN.
*   proxy_flag              - (IN) C4  - Indicates whether this LEC is a proxy.
*   max_unknown_frame_count - (IN) C10 - For rate-limiting flooded unknowns.
*   max_unknown_frame_time  - (IN) C11 - For rate-limiting flooded unknowns.
*   vcc_timeout             - (IN) C12 - For aging out VCCs.
*   max_retry_count         - (IN) C13 - Max retries for LE-ARP requests.
*   aging_time              - (IN) C17 - Aging time for all addresses.
*   forward_delay_time      - (IN) C18 - Aging time for remote addresses during
*                                        topology changes.
*   le_arp_response_time    - (IN) C20 - Expected response time for LE-ARPs.
*   flush_timeout           - (IN) C21 - Timeout for flush operation.
*   path_switching_delay    - (IN) C22 - Path switching time when flush is not
*                                        successful.
*
* Returns:
*   None
*
* Preconditions:
*   The la_elan_handle must be valid.
*
* Postconditions:
*   The new parameters have taken effect and the ARP cache is ready for use.
--*/
void la_config (HANDLE     la_elan_handle,
                LAN_TYPE   lan_type,
                BOOLEAN    proxy_flag,
                UINT16     max_unknown_frame_count,
                UINT16     max_unknown_frame_time,
                UINT32     vcc_timeout,
                UINT16     max_retry_count,
                UINT32     aging_time,
                UINT16     forward_delay_time,
                UINT16     le_arp_response_time,
                UINT16     flush_timeout,
                UINT16     path_switching_delay,
                unsigned int lane_version_num); /* LANE2 */

/*++
* =================
* = la_elan_reset =
* =================
*
* Overview:
*   Resets an instance of the arp cache.  All entries in the cache are removed.
*
* Arguments:
*   la_elan_handle   - (IN) Handle returned by la_register.
*
* Returns:
*   None
*
* Preconditions:
*   The la_elan_handle must be valid.
*
* Postconditions:
*   The arp cache is empty.
--*/
void la_elan_reset (HANDLE   la_elan_handle);

/*++
* ==================
* = la_addr_delete =
* ==================
*
* Overview:
*   Delete an entry from the LE-ARP cache.  Both permanent and non-permanent
*   entries may be deleted.  This allows permanent addresses to be deleted by
*   network management software and non-permanent addresses to be deleted via
*   a Negative ARP (NARP) request.
*
* Arguments:
*   la_elan_handle  - (IN) Handle returned by la_register.
*   esi             - (IN) MAC Address of entry to be deleted.
*   permanent       - (IN) If TRUE,  address may be permanent or non-permanent.
*                          if FALSE, address must be non-permanent (NARP).
*
* Returns:
*   STATUS_K_SUCCESS    - Successful Completion, see postconditions.
*   STATUS_K_NOT_FOUND  - The proper kind of address was not found.  This
*                         status will be returned in the following cases:
*
*                         1. No entry was found with a matching MAC address,
*                         2. The permanent flag was FALSE and a matching
*                            permanent entry was found.
*
* Preconditions:
*   None
*
* Postconditions:
*   If the permanent flag is set, there is no entry in the LE-ARP cache with 
*   a matching ESI.
*
*   If the permanent flag is not set, there is no non-permanent entry in the
*   LE-ARP cache with a matching ESI.
--*/
STATUS la_addr_delete (HANDLE    la_elan_handle,
                       unsigned char *atm_addr,
                       BOOLEAN   permanent);

/*++
* =================
* = la_arp_update =
* =================
*
* Overview:
*   Updates the LE-ARP cache with a fresh MAC address to ATM address mapping.
*   This is generally called when a LE-ARP response is received.
*
* Arguments:
*   la_elan_handle  - (IN) Handle returned by la_register.
*   esi             - (IN) MAC Address in updated mapping.
*   addr_atm        - (IN) ATM Address in the updated mapping.
*   remote_flag     - (IN) Remote flag contained in the LE-ARP response frame.
*   tlv_list        - (IN) LANE2: LE-ARP frames can now contain TLVs
*   sizeoftlvs      - (IN) LANE2: The length of tlv_list in bytes
*
* Returns:
*   None
*
* Preconditions:
*   None
*
* Postconditions:
*   The arp entry for this MAC address is up-to-date.  If this call indicates
*   a change from the original mapping (i.e. the ATM address is different from
*   what was originally in the mapping), the old mapping is discarded and the
*   flush mechanism is invoked to facilitate the path switch.
*   LANE2: The TLV association is also up-to-date
--*/
void la_arp_update (HANDLE       la_elan_handle,
                    ESI          esi,
                    ADDR_ATM     addr_atm,
                    BOOLEAN      remote_flag,
                    char         *tlv_list,
                    int          sizeoftlvs,
                    int          targetless_le_arp,
                    int          no_source_le_narp);

/*++
* =====================
* = la_flush_complete =
* =====================
*
* Overview:
*   This call is made when the LEC_CTRL module receives a flush response frame
*   from the network.  It notifies the LE-ARP cache that the flush protocol
*   has successfully completed.
*
* Arguments:
*   la_elan_handle  - (IN) Handle returned by la_register.
*   addr_atm        - (IN) ATM address that was indicated in the flush
*                          response.
*   tran_id         - (IN) Transaction ID of original flush request.
*
* Returns:
*   None
*
* Preconditions:
*   None
*
* Postconditions:
*   The LE-ARP cache data structure specifies that packets sent to the
*   requested MAC address may now be forwarded to the data-direct VCC.
--*/
void la_flush_complete (HANDLE   la_elan_handle,
                        UINT32   tran_id);

/*++
* ==========================
* = la_topology_change_set =
* ==========================
*
* Overview:
*   This call indicates to the LE-ARP cache that the topology change flag
*   has changed value.  This causes the LE-ARP cache to change its policy
*   regarding the aging and refreshing of remote address mappings.
*
* Arguments:
*   la_elan_handle       - (IN) The handle returned by la_register.
*   topology_change_flag - (IN) The new value of the topology change flag.
*
* Returns:
*   None
*
* Preconditions:
*   None
*
* Postconditions:
*   The aging and refresh policy for the indicated address has been changed
*   to be appropriate to the value of the topology change flag.
--*/
void la_topology_change_set (HANDLE  la_elan_handle,
                             BOOLEAN topology_change_flag);

void kernel_flush_xmt(unsigned char *to_pass);
